/*
 * makeflash.c  -  Generate boot image file for programming a FlashCard
 *
 * Copyright (C) 1997,1998 Gero Kuhlmann   <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "common.h"
#include "nblib.h"
#include "makerom.h"
#include "makeflash.h"



/*
 * Variables private to this module
 */
static char *outname;			/* Name of output file */
static int outfile;			/* File handle for output file */
static int cur_rec_num = -1;		/* Number of current load record */
static struct load_header header;	/* Load header */
static struct load_record *cur_rec;	/* Pointer to current load record */



/*
 * Read one sector from the temporary rom image file
 */
static int readsect(buf, romimage)
unsigned char *buf;
int romimage;
{
  int i;

  if ((i = nbread(buf, SECTSIZE, romimage)) == 0)
	return(FALSE);
  else if (i != SECTSIZE) {
	/* The ROM image file has to have a multiple of SECTSIZE bytes */
	fprintf(stderr, "%s: Unexpected end of temporary rom image file\n",
								progname);
	exit(EXIT_MAKEROM_ROMEOF);
  }
  return(TRUE);
}



/*
 * Write a buffer into the output file and update the load record
 */
static void putrec(recnum, src, size)
int recnum;
char *src;
int size;
{
  unsigned long l;
  size_t isize;
  unsigned char *buf;

  if (cur_rec_num != recnum) {
	fprintf(stderr, "%s: Internal error: image chunks mis-ordered!\n",
								progname);
	exit(EXIT_MAKEROM_MISORDER);
  }
  isize = ((size / (SECTSIZE + 1)) + 1) * SECTSIZE;
  buf = (unsigned char *)nbmalloc(isize);
  memcpy(buf, src, size);
  (void)nbwrite(buf, isize, outfile);
  free(buf);
  l = get_long(cur_rec->ilength) + isize;
  assign(cur_rec->ilength.low, htot(low_word(l)));
  assign(cur_rec->ilength.high, htot(high_word(l)));
  l = get_long(cur_rec->mlength) + isize;
  assign(cur_rec->mlength.low, htot(low_word(l)));
  assign(cur_rec->mlength.high, htot(high_word(l)));
}



/*
 * Initialize a load record
 */
static void initrec(recnum, segment, flags)
int recnum;
int segment;
int flags;
{
  if (++cur_rec_num != recnum) {
	fprintf(stderr, "%s: Internal error: image chunks mis-ordered!\n",
								progname);
	exit(EXIT_MAKEROM_MISORDER);
  }

  if (cur_rec_num > 0)
	cur_rec = (struct load_record *)((unsigned char *)cur_rec +
					((cur_rec->rlength << 2) & 0x3c) +
					((cur_rec->rlength >> 2) & 0x3c));
  cur_rec->rlength = (sizeof(struct load_record)) >> 2;
  cur_rec->rtag1   = recnum + VENDOR_OFF;
  cur_rec->rflags  = flags;
  assign(cur_rec->address.low, htot(low_word((unsigned long) segment << 4)));
  assign(cur_rec->address.high, htot(high_word((unsigned long) segment << 4)));
}



/*
 * Dump the load record information to stderr
 */
static void dump_header(lh)
struct load_header *lh;
{
  static char *s_tags[] = { /* PROGRAMNUM */  "flash EPROM programmer",
			    /* ROMIMAGENUM */ "boot rom image"};
  static char *s_flags[]= { "absolute address", "after previous segment",
			    "at end of memory", "before previos segment"};
  struct load_record *lr;
  char *vendstr = NULL;
  int i, num = 0;

  i = (lh->hlength >> 2) & 0x3c;
  vendstr = nbmalloc(i + 2);
  memcpy(vendstr, lh->dummy, i);

  printf("\nLoad record information:\n"
	  "  Magic number:     0x%08lX\n"
	  "  Length of header: %d bytes\n"
	  "  Flags:            0x%08lX\n"
	  "  Location address: %04X:%04X\n"
	  "  Execute address:  %04X:%04X\n"
	  "  Vendor data:      %s\n"
	  "\n",
	  get_long(lh->magic),
	  (lh->hlength << 2) & 0x3c,
	  (unsigned long)lh->hflags1 +
		((unsigned long)lh->hflags2 << 8) +
		((unsigned long)lh->hflags3 << 16),
	  ttoh(getval(lh->locn.segment)), ttoh(getval(lh->locn.offset)),
	  ttoh(getval(lh->execute.segment)), ttoh(getval(lh->execute.offset)),
	  vendstr);

  i  = ((lh->hlength >> 2) & 0x3c) + ((lh->hlength << 2) & 0x3c);
  lr = (struct load_record *)&(((__u8 *)lh)[i]);

  for (;;) {
	printf("Record #%d:\n"
	    "  Length of header: %d bytes (standard) + %d bytes (vendor)\n"
	    "  Vendor tag:       0x%02X (%s)\n"
	    "  Reserved flags:   0x%02X\n"
	    "  Flags:            0x%02X (%s%s)\n"
	    "  Load address:     0x%08lX%s\n"
	    "  Image length:     0x%08lX bytes\n"
	    "  Memory length:    0x%08lX bytes\n",
	    ++num,
	    (lr->rlength << 2) & 0x3c,
	    (lr->rlength >> 2) & 0x3c,
	    (int)lr->rtag1,
	    lr->rtag1 < 16 || lr->rtag1-16 >= NUM_RECORDS ? "unknown" : s_tags[lr->rtag1-16],
	    (int)lr->rtag2,
	    (int)lr->rflags, s_flags[lr->rflags & 0x03],
	    lr->rflags & FLAG_EOF ? ", last record" : "",
	    get_long(lr->address),
	    get_long(lr->address) >= 0x100000L &&
	    (lr->rflags & 0x03) == 0? " (high memory)" : "",
	    get_long(lr->ilength),
	    get_long(lr->mlength));

	if (lr->rflags & FLAG_EOF)
		break;

	i  = ((lr->rlength >> 2) & 0x3c) + ((lr->rlength << 2) & 0x3c);
	lr = (struct load_record *)&(((__u8 *)lr)[i]);
  }
  free(vendstr);
}



/*
 * Routine to generate a boot image file which can program a FlashCard with
 * a new boot rom image.
 */
void makeflash(fname, romimage)
char *fname;
int romimage;
{
  unsigned char copyrec_buf[SECTSIZE];
  int vendor_size;

  /* Open the input and output files */
  outname = fname;
  if ((outfile = open(outname, O_CREAT | O_WRONLY | O_TRUNC | O_BINARY,
								0644)) < 0) {
	perror(outname);
	exit(EXIT_CREATE);
  }

  /* Initialize the boot header */
  vendor_size = (sizeof(VENDOR_ID) / sizeof(__u32) + 1) * sizeof(__u32);
  memset(&header, 0, sizeof(header));
  assign(header.magic.low,       htot(low_word(HEADER_MAGIC)));
  assign(header.magic.high,      htot(high_word(HEADER_MAGIC)));
  assign(header.locn.segment,    htot(HEADERSEG));
  assign(header.locn.offset,     htot(0));
  assign(header.execute.segment, htot(PROGRAMSEG));
  assign(header.execute.offset,  htot(0));
  assign(header.bootsig,         htot(BOOT_SIGNATURE));
  header.hlength  = (__u8)(((int)header.dummy - (int)&header)
                    / sizeof(__u32)) & 0x0f;
  header.hlength |= (__u8)((vendor_size/sizeof(__u32)) << 4) & 0xf0;
  memcpy(header.dummy, VENDOR_ID, sizeof(VENDOR_ID));
  (void)nbwrite((unsigned char *)&header, sizeof(header), outfile);

  /* Initialize pointer to first load record */
  cur_rec = (struct load_record *)&(header.dummy[vendor_size]);

  /* Process the FlashCard programmer record */
  if (flash_data_size > PROGRAMLSIZE) {
	fprintf(stderr, "%s: Boot loader too large\n", progname);
	exit(EXIT_MAKEROM_PROGLSIZE);
  }
  initrec(PROGRAMNUM, PROGRAMSEG, 0);
  putrec(PROGRAMNUM, (char *) flash_data, flash_data_size);
  assign(cur_rec->mlength.low,  htot(low_word(PROGRAMMSIZE)));
  assign(cur_rec->mlength.high, htot(high_word(PROGRAMMSIZE)));

  /* Process the boot rom image */
  initrec(ROMIMAGENUM, ROMIMAGESEG, FLAG_EOF);
  while (readsect(copyrec_buf, romimage))
	putrec(ROMIMAGENUM, (char *)copyrec_buf, SECTSIZE);

  /* After writing out all this stuff, finally update the boot header */
  if (lseek(outfile, 0, 0) != 0) {
	perror(outname);
	exit(EXIT_SEEK);
  }
  (void)nbwrite((unsigned char *)&header, sizeof(header), outfile);
  close(outfile);

  /* If user asked for detailed output, parse the header and output all of */
  /* the load record information */
  if (verbose > 1)
	dump_header(&header);
}
