# Copyright (c) 2004 Divmod.
# See LICENSE for details.


from __future__ import generators

import os
import os.path
import warnings

from nevow import inevow
from nevow.stan import slot, specialMatches
from nevow.tags import *
from nevow import util
from nevow import compy
from nevow.context import NodeNotFound

import formless
from formless import iformless
from formless.formutils import enumerate, FormDefaults, FormErrors, calculatePostURL, keyToXMLID, getError

def _locateDefaultCSS():
    """Calculate and return the full path to the default freeform CSS.
    """
    dirname = os.path.abspath(os.path.dirname(__file__))
    return os.path.join(dirname, 'freeform-default.css')



try:
    from nevow.static import File
except ImportError:
    class File(object):
        __implements__ = inevow.IResource
        def __init__(self, path, content_type='text/plain'):
            self.path = path
            self.content_type = content_type

        def locateChild(self, *args):
            from nevow import rend
            return rend.NotFound

        def renderHTTP(self, ctx):
            inevow.IRequest(ctx).setHeader('Content-type', self.content_type)
            return open(self.path).read()


defaultCSS = File(_locateDefaultCSS(), 'text/css')


class DefaultRenderer(object):
    __implements__ = inevow.IRenderer, iformless.ITypedRenderer
    complexType = False
    def rend(self, context, data):
        return StringRenderer(data)
defaultBindingRenderer = DefaultRenderer()


class BaseInputRenderer(compy.Adapter):
    __implements__ = inevow.IRenderer, iformless.ITypedRenderer
    complexType = False
    def rend(self, context, data):
        defaults = context.locate(iformless.IFormDefaults)
        value = defaults.getDefault(context.key, context)

        if data.typedValue.getAttribute('immutable'):
            inp = span(id=keyToXMLID(context.key))[value]
        else:
            ##value may be a deferred; make sure to wait on it properly before calling self.input
            ## TODO: If flattening this results in an empty string, return an empty string
            inp = invisible(
                render=lambda c, value: self.input( context, invisible(), data, data.name, value ),
                data=value)

        if data.typedValue.getAttribute('hidden') or data.typedValue.getAttribute('compact'):
            return inp

        context.fillSlots( 'label', data.label )
        context.fillSlots( 'name', data.name )
        context.fillSlots( 'input', inp )
        context.fillSlots( 'error', getError(context) )
        context.fillSlots( 'description', data.description )
        context.fillSlots( 'id', keyToXMLID(context.key) )
        context.fillSlots( 'value', value )

        return context.tag

    def input(self, context, slot, data, name, value):
        raise NotImplementedError, "Implement in subclass"


class PasswordRenderer(BaseInputRenderer):
    def input(self, context, slot, data, name, value):
        return [
            input(id=keyToXMLID(context.key), name=name, type="password", _class="freeform-input-password"),
            " Again ",
            input(name="%s____2" % name, type="password", _class="freeform-input-password"),
        ]


class PasswordEntryRenderer(BaseInputRenderer):
    def input(self, context, slot, data, name, value):
        return slot[
            input(id=keyToXMLID(context.key), type='password', name=name,
                  _class='freeform-input-password')]


class StringRenderer(BaseInputRenderer):
    def input(self, context, slot, data, name, value):
        if data.typedValue.getAttribute('hidden'):
            T="hidden"
        else:
            T="text"
        return slot[
            input(id=keyToXMLID(context.key), type=T, name=name, value=value,
                  _class='freeform-input-%s' % T)]


class TextRenderer(BaseInputRenderer):
    def input(self, context, slot, data, name, value):
        return slot[
            textarea(id=keyToXMLID(context.key), name=name, _class="freeform-textarea", rows=8, cols=40)[
                value or '']]


class BooleanRenderer(BaseInputRenderer):
    def input(self, context, slot, data, name, value):
        ## The only difference here is the "checked" attribute; the value is still the same because
        ## we want true to be passed to the server when the checkbox is checked and the form
        ## is posted.
        node = input(id=keyToXMLID(context.key), type="checkbox", name=name, value='True', _class="freeform-input-checkbox")
        if value:
            node(checked="checked")

        # HTML forms are so weak. If the checkbox is not checked, no value at all will be
        # in request.args with the name data.name. So let's force the value False to always
        # be in request.args[data.name]. If the checkbox is checked, the value True will
        # be first, and we will find that.
        return slot[node, input(type="hidden", name=name, value="False")]


class FileUploadRenderer(BaseInputRenderer):
    def input(self, context, slot, data, name, value):
        return slot[input(id=keyToXMLID(context.key), type="file", name=name,
                          _class='freeform-input-file')]


class ChoiceRenderer(BaseInputRenderer):
    def input(self, context, slot, data, name, value):
        tv = data.typedValue
        if tv.choicesAttribute:
            choices = getattr(context.locate(iformless.IConfigurable).boundTo, tv.choicesAttribute)
        else:
            choices = tv.choices

        numChoices = len(choices)
        if numChoices == 0:
            return None

        selector = select(id=keyToXMLID(context.key), name=name)
        stringify = tv.stringify

        for index, val in enumerate(choices):
            if val == value:
                selector[option(value=str(index), selected="selected")[stringify(val)]]
            else:
                selector[option(value=str(index))[stringify(val)]]
        return slot[selector]


class ObjectRenderer(compy.Adapter):
    __implements__ = inevow.IRenderer, iformless.ITypedRenderer
    complexType = True
    def rend(self, context, data):
        configurable = context.locate(iformless.IConfigurable)
        return getattr(configurable, data.name)


class NullRenderer(compy.Adapter):
    """Use a NullRenderer as the ITypedRenderer adapter when nothing should
    be included in the output.
    """
    __implements__ = inevow.IRenderer, iformless.ITypedRenderer
    def rend(self, context, data):
        return ''


class GroupBindingRenderer(compy.Adapter):
    __implements__ = inevow.IRenderer,

    def rend(self, context, data):
        context.remember(data, iformless.IBinding)

        from formless import configurable as conf

        configurable = conf.GroupConfigurable(data.boundTo, data.typedValue.interface)
        context.remember(configurable, iformless.IConfigurable)

        bindingNames = configurable.getBindingNames(context)

        def generateBindings():
            for name in bindingNames:
                bnd = configurable.getBinding(context, name)
                renderer = iformless.IBindingRenderer(bnd, defaultBindingRenderer, persist=False)
                renderer.isGrouped = True
                renderer.needsSkin = True
                yield invisible(
                    data=bnd,
                    render=renderer,
                    key=name)

        return getError(context), form(
            id=keyToXMLID(context.key),
            enctype="multipart/form-data",
            action=calculatePostURL(context, data), 
            method="post")[
                div(_class="freeform-form-label")[data.label],
                fieldset[
                    generateBindings(),
                    input(type="submit")]]


class BaseBindingRenderer(compy.Adapter):
    __implements__ = inevow.IRenderer,

    isGrouped = False
    needsSkin = False
    def calculateDefaultSkin(self, context):
        if self.isGrouped:
            frm = invisible
            butt = ''
            fld = invisible
        else:
            frm = form(
                id=slot('form-id'), action=slot('form-action'), method="post", enctype="multipart/form-data")
            butt = slot('form-button')
            fld = fieldset

        ## Provide default skin since no skin was provided for us.
        context.tag.clear()[
            frm[
                div(_class="freeform-form-label")[ slot('form-label') ],
                div(_class="freeform-form-description")[ slot('form-description') ],
                div(_class="freeform-form-error")[ slot('form-error') ],
                fld[ slot('form-arguments'), butt ]]]

    def fillForm(self, context, data):
        context.fillSlots( 'form-id', keyToXMLID(context.key) )
        context.fillSlots( 'form-action', calculatePostURL(context, data) )
        context.fillSlots( 'form-name', data.name )
        context.fillSlots( 'form-error', getError(context) )


class PropertyBindingRenderer(BaseBindingRenderer):
    def rend(self, context, data):
        context.remember(data, iformless.IBinding)
        typedRenderer = iformless.ITypedRenderer(data.typedValue, defaultBindingRenderer, persist=False)
        if typedRenderer.complexType:
            return invisible(data=data, render=typedRenderer)

        if self.needsSkin or not context.tag.children:
            self.calculateDefaultSkin(context)

        if self.isGrouped or data.typedValue.getAttribute('immutable'):
            subm = ''
        else:
            subm = input(type="submit", name="change", value="Change")

        self.fillForm(context, data)
        context.fillSlots( 'form-label', '' )
        context.fillSlots( 'form-description', '' )
        context.fillSlots(
            'form-arguments', 
            freeformDefaultContentPattern(data=data, render=typedRenderer, key=data.name))
        context.fillSlots('form-button', subm)

        return context.tag


freeformDefaultContentPattern = invisible[
    label(_class="freeform-label", _for=slot('id'))[ slot('label') ],
    span(_class="freeform-input")[ slot('input') ],
    div(_class="freeform-error")[ slot('error') ],
    div(_class="freeform-description")[label(_for=slot('id'))[ slot('description') ]]].freeze()


class MethodBindingRenderer(BaseBindingRenderer):
    def rend(self, context, data):
        if data.getAttribute('invisible'):
            return ''

        context.remember(data, iformless.IBinding)

        if self.needsSkin or not context.tag.children:
            self.calculateDefaultSkin(context)

        self.fillForm(context, data)
        context.fillSlots( 'form-label', data.label )
        context.fillSlots( 'form-description', data.description )
        context.fillSlots( 'form-arguments', list(self.generateArguments(context, data.getArgs())))

        if not self.isGrouped:
            try:
                button_pattern = context.tag.onePattern( 'form-button' )
            except NodeNotFound:
                button_pattern = invisible[ slot('input') ]

            button_pattern.fillSlots( 'input', input(type='submit', value=data.action or data.label) )

            context.fillSlots( 'form-button', button_pattern )

        return context.tag(key=None)

    def generateArguments(self, context, args):
        default_content_pattern = None
        content_pattern = None
        for argument in args:
            try:
                content_pattern = context.tag.patternGenerator( 'argument!!%s' % argument.name )
            except NodeNotFound:
                if default_content_pattern is None:
                    try:
                        default_content_pattern = context.tag.patternGenerator( 'argument' )
                    except NodeNotFound:
                        default_content_pattern = freeformDefaultContentPattern
                content_pattern = default_content_pattern
            pat = content_pattern(
                key=argument.name,
                data=argument,
                render= iformless.ITypedRenderer(
                    argument.typedValue, defaultBindingRenderer, persist=False))
            context.fillSlots( 'argument!!%s' % argument.name, pat )
            yield pat


class ButtonRenderer(compy.Adapter):
    __implements__ = inevow.IRenderer,

    def rend(self, context, data):
        return input(type='submit', value=data.label)


freeformDefaultForm = div(_class="freeform-form").freeze()


def renderForms(configurableKey='', bindingNames=None, formDefaults=None):
    """Render forms for either the named configurable, or, if no configurableKey is given,
    the main configurable. If no bindingNames are given, forms will be
    rendered for all bindings described by the configurable.
    """
    def formRenderer(context, data):
        # Remember the key for when the form post URL is generated.
        context.remember(configurableKey, iformless.IConfigurableKey)

        cf = context.locate(iformless.IConfigurableFactory)
        configurable = cf.locateConfigurable(context, configurableKey)
        if configurable is None:
            warnings.warn(
                "No configurable was found which provides enough type information for freeform to be able to render forms; %r" % (cf, ))
            yield ''
            return
        context.remember(configurable, iformless.IConfigurable)

        available = configurable.getBindingNames(context)
        bindings = []

        default_binding_pattern = None

        for name in available:
            if bindingNames is not None and name not in bindingNames:
                continue
            bnd = configurable.getBinding(context, name)
            renderer = iformless.IBindingRenderer(bnd, defaultBindingRenderer, persist=False)
            try:
                binding_pattern = context.tag.patternGenerator( 'freeform-form!!%s' % name )
            except NodeNotFound:
                if default_binding_pattern is None:
                    try:
                        default_binding_pattern = context.tag.patternGenerator( 'freeform-form' )
                    except NodeNotFound:
                        default_binding_pattern = freeformDefaultForm
                binding_pattern = default_binding_pattern
            if binding_pattern is freeformDefaultForm:
                renderer.needsSkin = True
            yield binding_pattern(data = bnd, render = renderer, key = name)
    return invisible(render=formRenderer)

