#!/bin/sh 

############################################################################
# A proposal for the newbiedoc project archive creator 
#  version alpha0.4
#
# This code is protected by the GPL license: www.gnu.org/copyleft/gpl.html
#  
#
# How it works:
#
#   just cd to a directory where you have write permission
#   and call make_archive.sh filename.tar your_local_cvs:
#
#   make_archive.sh  hop.tar ~/NewbieDocDVS/newbiedoc
#
#
#   As usual E-mail me for comments, bugs, improvements etc
#   rom1 at users.sourceforge.net
#
#############################################################################

# Some variable definitions

#VERBOSE_TAR=--verbose
#DEBUG=yes

# DEBUG is really only a talkative 'verbose'. Use it to display additional 
# messages.

# so we know what to call this script when displaying an error message
	PROG_NAME=$(basename $0)

# checking for the correct number of arguments
	if [ $# -lt 2 ]; then
		echo "Usage: $PROG_NAME archive_file source_directory"
		exit  0
	fi
# we're going to move around a bit, so let's remember where we started from
	INITIAL_DIR=$PWD

# we're going to temporarily write to $TEMP_DIR/$ND_DIR/ and then
# tar all the files into $TEMP_DIR/$TEMP_TAR:
	TEMP_DIR=/tmp
	ND_DIR=newbiedoc    # only 1 dir deep!
	TEMP_TAR=newbie.tmp.tar

# the output will be:
	WRITE_FILE=$1
	WRITE_DIR=$(dirname $WRITE_FILE)

# that's where the user said the source files are
	SOURCE_DIR=$2

#---- Functions used in the code -------------------------------------------#

#it's a " multiple backspace"
	function back_echo ()
	{
# usage: back_echo number_of_backspaces
		if [ $# != 1 ]; then
			echo "Internal error: incorrect number of parameters for back_echo"
			exit 2
		fi
		local NUM_CHARS=$1
		local i
		let i=0
		while [ $i -lt $NUM_CHARS ]; do
			echo -n -e '\b'
			# -e for 'understand the \xxx characters'
			# -n for 'forget about the trailing newline'
			let i=$i+1
		done
	}

# this is where we actually call 'ndb2html' to compile the sgml source to html
# it is a bit complicated because I wanted to have a "xx% completed" counter.
# Note: the counter works on files, not size, so it might not represent 
# anything close to the remaining compilation time...
	function actual_compilation()
	{
		local COMPLETED_STRING
		local completed_string_size
		local FILE

		for FILE in $SOURCE_FILES; do
# this is the string that display xx% completed
			COMPLETED_STRING="$(( $NUM_COMPILED / $NUM_SOURCE_FILES ))% completed"
			echo -n $COMPLETED_STRING

# get string size:
			completed_string_size=$( echo -n $COMPLETED_STRING | wc -c )
		
#    Backspaceing  over exactly the "xx% Completed" part
			back_echo $completed_string_size

# so we don't have to multiply current/total_number by 100 to get percents
# we just add 100. Since we don't need NUM_COMPILED for anything else,
# this is not a problem...
			let NUM_COMPILED=$NUM_COMPILED+100

# ndb2html builds in the current dir, in this case $TEMP_DIR/$ND_DIR
			ndb2html single $FILE
		done
# this prints the last 100% completed 
		COMPLETED_STRING="$(( $NUM_COMPILED / $NUM_SOURCE_FILES ))% completed"
		echo -e -n $COMPLETED_STRING
		completed_string_size=$(echo -n $COMPLETED_STRING | wc -c )
		back_echo $completed_string_size
# no, it's not useless to display 100% completed if we echo 'done'
# right after. I tried on a slow standalone X terminal, and the echo was so
# slow I could actually see the 100%.
# Besides, we might one day do fancy things before echo-ing 'done', so the
# 100% will lot cleaner.

# see the few spaces after done ? That's cuz if you don't you'll see
# "done completed" instead of "done".
		echo 'done                        '
	}


#---- Start of the main code: ----------------------------------------------#


# checking for permissions...
	if [ -d $WRITE_FILE ]; then
		echo "Fatal error in $PROG_NAME:"
		echo "$WRITE_FILE is a directory"
		exit 1
	fi

	if [ ! -w $WRITE_DIR ]; then
 		echo "Fatal error in $PROG_NAME:"
		echo "You do not have write permissions for directory $WRITE_DIR"
		exit 1
	fi

	if [ ! -r $SOURCE_DIR ]; then
		echo "Fatal error in $PROG_NAME:"
		echo "Cannot read from $SOURCE_DIR"
		exit 1
	fi
# checking that ND_DIR is one-deep:
	if [ $( dirname $ND_DIR ) != "." ]; then
		echo "Internal error: variable ND_DIR should be a depth 1 directory"
		echo "Current value is $ND"
		exit 2
	fi
# Fully qualify $SOURCE_DIR (get rid of ./ ../ etc.)
#   we're going to need an absolute pathname  because when we move to 
#$TEMP_DIR, a relative pathname will be useless.

	cd $SOURCE_DIR
	SOURCE_DIR=$PWD
	test $DEBUG && echo "Fully qualified source directory is $SOURCE_DIR"

# Now make a $TEMP_DIR/$ND_DIR directory to build in
	test $DEBUG && echo "Moving to $TEMP_DIR from $INITIAL_DIR"
	cd $TEMP_DIR
	if [ ! -d $ND_DIR ]; then
 		test $BUILD && mkdir $ND_DIR
 		test $DEBUG && echo "Creating a $TEMP_DIR/$ND_DIR directory"
		WE_MADE_IT=yes
	else
 		test $DEBUG && echo "Using existing $TEMP_DIR/$ND_DIR directory"
		# we might want to clean this
		WE_MADE_IT=no
	fi

# move to that directory
	cd $ND_DIR
	test $DEBUG && echo "Moving to $ND_DIR"

# after this line will be appended during the compilation something like
# "xx% completed" :
	echo -n "Compiling sgml sources in $TEMP_DIR/$ND_DIR..."

#-------------------------------------------------------------------------
#             Compilation process

# $SOURCE_DIR is the source directory
#   What are the files to compile ?
	SOURCE_FILES=$( find $SOURCE_DIR -follow -name *.sgml )
#   Their number:
	NUM_SOURCE_FILES=$( echo $SOURCE_FILES | wc -w )
	test $DEBUG && echo "Source files are: "$SOURCE_FILES $NUM_SOURCE_FILES
#   The number of compiled files so far:
	NUM_COMPILED=0

# this call the function defined above
	actual_compilation

#------------------------------------------------------------------------


#moving out of $TEMP_DIR/$ND_DIR to $TEMP_DIR

	cd ..
	# now we should be in $TEMP_DIR

# We could also just say "cd $TEMP_DIR" and forget about the depth one
# limitation, couldn't we ? Is this the only place where we have this problem ?
# I'll check.

	if [ $PWD != $TEMP_DIR ]; then
		echo "Internal error: \$ND_DIR should only be one deep!"
		exit 2
	fi

	test $DEBUG && echo "Exiting directory $TEMP_DIR/$ND_DIR to $PWD"
	echo -n 'Creating archive with html files...'

#we could even build tgz archives, but I prefer tar and then gzip it manually

	tar $VERBOSE_TAR -cf $TEMP_TAR ./$ND_DIR/*.html

#the idea of having $TEMP_DIR/$ND_DIR is that when tarring from
#$TEMP_DIR all the files are in a ./$ND_DIR subdir. Besides I know I
#have write permissions to /tmp (= $TEMP_DIR)

	echo 'done'
# if this shell created the $TEMP_DIR shell, then it is its responsibility
# to remove it.:
	if [ $WE_MADE_IT = yes ]; then
		echo "Cleaning up: removing directory $TEMP_DIR/$ND_DIR"
		rm -rf $TEMP_DIR/$ND_DIR
	fi

# moving back to where the script was called
	test $DEBUG && echo "Moving back to $INITIAL_DIR"
	cd $INITIAL_DIR

# create the final archive
	echo -n 'Creating final archive...'
	mv $TEMP_DIR/$TEMP_TAR $WRITE_FILE
	echo 'done'

	echo "Process complete: archive $WRITE_FILE created"



