/*
 	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com).
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


#ifndef ASN1RESP_H
#define ASN1RESP_H

#include <mVector.h>
#include <ASN1/NewPKIObject.h>
#include <openssl/asn1.h>
#include <openssl/x509.h>
#include <openssl/asn1.h>
#include <openssl/asn1t.h>
#include <openssl/x509v3.h>
#include <openssl/rand.h>
#include <openssl/evp.h>
#include <openssl/err.h>
#include <openssl/pkcs7.h>

#include <PKI_CERT.h>
#include <PKI_P7B.h>
#include "Asn1Logs.h"
#include "Asn1Err.h"
#include "Asn1User.h"
#include "Asn1Entity.h"
#include "Asn1Conf.h"
#include "Asn1Cert.h"

/*!
	This structure represents the certificate generated by a CA.
*/
typedef struct st_INTERNAL_CA_CERT
{
	ASN1_INTEGER * ca_type;			//!< The CA type
	ASN1_INTEGER * serial;			//!< The serial
	ASN1_INTEGER * state;			//!< The certificate state
	ASN1_UTF8STRING * dn;			//!< The certificate DN
	ASN1_UTF8STRING * uid;			//!< The optional LDAP uid
	ASN1_INTEGER * begin_date;		//!< The begin date
	ASN1_INTEGER * end_date;		//!< The expiration date
	ASN1_INTEGER * rev_date;		//!< The optional revocation date
	ASN1_INTEGER * susp_date;		//!< The optional suspension date
	X509 * cert;					//!< The certificate
}INTERNAL_CA_CERT;
DECLARE_ASN1_ITEM(INTERNAL_CA_CERT)

/*!
	This structure represents a LDAP attribute.
*/
typedef struct st_LDAP_OBJECT
{
	ASN1_UTF8STRING * name;			//!< The attribute name
	ASN1_UTF8STRING * value;		//!< The attribute value
}LDAP_OBJECT;
DECLARE_ASN1_ITEM(LDAP_OBJECT)


/*!
	This structure represents a LDAP search result.
*/
typedef struct st_LDAP_RESULT
{
	ASN1_UTF8STRING * rdn;				//!< The result RDN
	ASN1_UTF8STRING * uid;				//!< The result UID
	STACK_OF(LDAP_OBJECT) * objects;	//!< The LDAP attributes
}LDAP_RESULT;
DECLARE_ASN1_ITEM(LDAP_RESULT)



#define STATUS_CA_OK			1		//!< The status of the CA is OK
#define STATUS_CA_AWAITING_CERT	2		//!< The status of the CA is waiting for its certificate
#define STATUS_CA_NO_CERT		3		//!< The status of the CA is not initialized



#define ADMIN_RESP_TYPE_ERRORS				0	//!< The response is an error
#define ADMIN_RESP_TYPE_NONE				1	//!< There is no result
#define ADMIN_RESP_TYPE_USER_TYPE			2	//!< The response is a user type
#define ADMIN_RESP_TYPE_ENUM_USERS			3	//!< The response is a list of Server users
#define ADMIN_RESP_TYPE_ENUM_ENTITIES		4	//!< The response is a list of entities
#define ADMIN_RESP_TYPE_ENUM_LOGS			5	//!< The response is a list of logs
#define ADMIN_RESP_TYPE_CREATE_ENTITY		6	//!< The response is the public key(s) of the created entity
#define ADMIN_RESP_TYPE_SIGN_ENTITY			7	//!< The response is the certificate(s) of the sign entity
#define ADMIN_RESP_TYPE_ENUM_USERS_CERT		8	//!< The response is a list of PKI users
#define ADMIN_RESP_TYPE_CREATE_PKI_USER		9	//!< The response is the certificate of the created user
#define ADMIN_RESP_TYPE_STATUS				10	//!< The response is the status (online/offline) de l'entit PKI
#define ADMIN_RESP_TYPE_ACLS				11	//!< The response is a list of ACLs types
#define ADMIN_RESP_TYPE_MY_ACL				12	//!< The response is the ACLs of the logged in user
#define ADMIN_RESP_TYPE_CONF				13	//!< The response is the global PKI conf
#define ADMIN_RESP_TYPE_MY_CONF				14	//!< The response is an entity conf (crypted)
#define ADMIN_RESP_TYPE_GET_REPOSITORIES	15	//!< The response is the list of repositories
#define ADMIN_RESP_TYPE_LOGS_TYPE			16	//!< The response is a supported loggs type
#define ADMIN_RESP_TYPE_ENTITY_CONF			17	//!< The response is a the conf of an entity (not crypted)
#define ADMIN_RESP_TYPE_MAIL_CONF			18	//!< The response is the email conf
#define ADMIN_RESP_TYPE_AUDITS				19	//!< The response is a list of audits
#define ADMIN_RESP_TYPE_ENTITIES_LINKS		20	//!< The response is the list of links
#define ADMIN_RESP_TYPE_REQS				21	//!< The response is a list of requests
#define ADMIN_RESP_TYPE_CERTS				22	//!< The response is a list of certificates
#define ADMIN_RESP_TYPE_P7B					23	//!< The response is a P7B
#define ADMIN_RESP_TYPE_CSR					24	//!< The response is a CSR
#define ADMIN_RESP_TYPE_CRLS				25	//!< The response is a list of CRLs
#define ADMIN_RESP_TYPE_GROUPS				26	//!< The response is a list of groups
#define ADMIN_RESP_TYPE_PROFILE_ID			27	//!< The response is a profile ID
#define ADMIN_RESP_TYPE_TRANSACTION_ID		28	//!< The response is a transaction ID
#define ADMIN_RESP_TYPE_RESPS				29	//!< The response is a list of responses
#define ADMIN_RESP_TYPE_PROFILES			30	//!< The response is a list of profiles
#define ADMIN_RESP_TYPE_LOCAL_ENTITY_CONF	31	//!< The response is the conf of an entity
#define ADMIN_RESP_TYPE_KNOWN_OBJECTS		32	//!< The response is a list of known objects
#define ADMIN_RESP_TYPE_REQ_ID				33	//!< The response is a request ID
#define ADMIN_RESP_TYPE_CERT				34	//!< The response is a certificate
#define ADMIN_RESP_TYPE_LOGS_COUNT			35	//!< The response is the number of logs an entity has
#define ADMIN_RESP_TYPE_LDAP_RESULTS		36	//!< The response is a LDAP search result
#define ADMIN_RESP_TYPE_OBJECT_STATUS		37	//!< The response is the status of an object
#define ADMIN_RESP_TYPE_DN_VALIDATION		38	//!< The response is a DN validation request
#define ADMIN_RESP_TYPE_WEBUSER_ID			39	//!< The response is a web user ID
#define ADMIN_RESP_TYPE_WEBUSER_CERTS		40	//!< The response is a web user's certs
#define ADMIN_RESP_TYPE_DN					41	//!< The response is a DN
#define ADMIN_RESP_TYPE_EE_USERS			42	//!< The response is a list of webusers


/*!
	This structure represents the body of a response.
*/
typedef struct st_ADMIN_RESPONSE_BODY
{
	int type;
	union
	{
		STACK_OF(ERROR_ENTRY) * errors;							//!< A list of errors
		ASN1_INTEGER * usertype;								//!< The user type
		STACK_OF(ENTITY_ENTRY_INFO) * entities;					//!< A list of entities
		STACK_OF(USER_ENTRY) * users;							//!< A list of server users
		STACK_OF(LOG_ENTRY) * logs;								//!< A list of logs
		ENTITY_CREATION_RESP * cre_entity;						//!< A creation response
		ENTITY_SIGNATURE_RESP * sign_entity;					//!< A signature response
		STACK_OF(USER_ENTRY_CERT) * users_cert;					//!< A list of PKI users
		CREATE_PKI_USER_RESPONSE * create_pki_user;				//!< A list of certificates
		ASN1_INTEGER * status;									//!< The PKI entity status
		X509_ACL * acls_conf;									//!< The ACLs
		ASN1_BIT_STRING * my_acl;								//!< A user's ACLs
		EXPORTED_PKI_CONF * conf;								//!< The global PKI conf
		ENTITY_CONF_CRYPTED * my_conf;							//!< The crypted entity conf
		STACK_OF(REP_ENTRY_INFO) * repositories;				//!< The list of repositories
		STACK_OF(ASN1_INTEGER) * logs_type;						//!< A list of entity logs type
		ENTITY_CONF_BODY * entity_conf;							//!< An entity conf body
		EMAIL_CONF * mail_conf;									//!< An entity email conf
		STACK_OF(ENTITY_AUDIT_ENTRY) * audits;					//!< An entity audits conf
		STACK_OF(ENTITY_LINKS) * entities_links;				//!< The PKI links
		CRYPTED_NEWPKI_REQUESTS * object_reqs;					//!< The inter-entities requests
		CRYPTED_NEWPKI_RESPONSES * object_resps;				//!< The inter-entities responses
		STACK_OF(INTERNAL_CA_CERT) * certs;						//!< A list of internal certificates
		PKCS7 * p7b;											//!< A P7B
		X509_REQ * csr;											//!< A CSR
		STACK_OF(X509_CRL) * crls;								//!< A list of CRLs
		STACK_OF(USERS_GROUP) * groups;							//!< A list of groups
		ASN1_INTEGER * id;										//!< An ID
		ASN1_OCTET_STRING * transactionID;						//!< A transaction ID
		STACK_OF(NEWPKI_PROFILE_DATAS) * profiles;				//!< A list of profiles
		ENTITY_CONF * local_entity_conf;						//!< An entity conf (not crypted)
		OBJECTS_LIST_SYNCHRO * known_objects;					//!< The list of repositories' known objects
		X509 * cert;											//!< A certificate
		STACK_OF(LDAP_RESULT) * ldap_results;					//!< A list of LDAP results
		X509_NAME * dn;											//!< A Dn
		STACK_OF(NEWPKI_PROFILE_DATAS_CERT) * webuser_certs;	//!< The list of webuser's certs
		STACK_OF(NEWPKI_EE_USER) * ee_users;					//!< The list of webusers
		ASN1_NULL * other;										//!< Any other type of response
	}d;
}ADMIN_RESPONSE_BODY;
DECLARE_ASN1_ITEM(ADMIN_RESPONSE_BODY)


/*!
	This structure represents a response.
*/
typedef struct st_ADMIN_RESPONSE
{
	ADMIN_RESPONSE_BODY * body;				//!< The response body
	X509 * signing_cert;					//!< The signing certificate
	X509_ALGOR * sig_algo;					//!< The signature algo
	ASN1_BIT_STRING * signature;			//!< The signature
}ADMIN_RESPONSE;
DECLARE_ASN1_ITEM(ADMIN_RESPONSE)


#define d2i_ADMIN_RESPONSE_bio(bp,msg) (ADMIN_RESPONSE *)ASN1_item_d2i_bio(AdminResponse::get_ASN1_ITEM(), bp,(unsigned char **)(msg))

#define i2d_ADMIN_RESPONSE_bio(bp,msg) ASN1_item_i2d_bio(AdminResponse::get_ASN1_ITEM(), bp,(unsigned char *)msg)


/*!
  This class is the representation of INTERNAL_CA_CERT
*/
class InternalCaCert : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	InternalCaCert();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	InternalCaCert(const InternalCaCert & other);

	/*! \brief This is the destructor.
	 */
	~InternalCaCert();


	/*! \brief This function converts a INTERNAL_CA_CERT to a InternalCaCert.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const INTERNAL_CA_CERT * Datas);

	/*! \brief This function converts a InternalCaCert to a INTERNAL_CA_CERT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(INTERNAL_CA_CERT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of INTERNAL_CA_CERT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_INTEGER::begin_date.
	 *  \param c_beginDate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_beginDate(unsigned long c_beginDate);

	/*! \brief This function returns ASN1_INTEGER::begin_date.
	 *  \return The value of ASN1_INTEGER::begin_date.
	 */
	unsigned long get_beginDate() const;


	/*! \brief This function sets ASN1_INTEGER::ca_type.
	 *  \param c_caType [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_caType(unsigned long c_caType);

	/*! \brief This function returns ASN1_INTEGER::ca_type.
	 *  \return The value of ASN1_INTEGER::ca_type.
	 */
	unsigned long get_caType() const;


	/*! \brief This function sets X509::cert.
	 *  \param c_cert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cert(const PKI_CERT & c_cert);

	/*! \brief This function returns X509::cert.
	 *  \return The value of X509::cert.
	 */
	const PKI_CERT & get_cert() const;

	/*! \brief This function returns X509::cert.
	 *  \return The value of X509::cert.
	 */
	PKI_CERT & get_cert();


	/*! \brief This function sets ASN1_UTF8STRING::dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const mString & c_dn);

	/*! \brief This function returns ASN1_UTF8STRING::dn.
	 *  \return The value of ASN1_UTF8STRING::dn.
	 */
	const mString & get_dn() const;

	/*! \brief This function returns ASN1_UTF8STRING::dn.
	 *  \return The value of ASN1_UTF8STRING::dn.
	 */
	mString & get_dn();


	/*! \brief This function sets ASN1_INTEGER::end_date.
	 *  \param c_endDate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_endDate(unsigned long c_endDate);

	/*! \brief This function returns ASN1_INTEGER::end_date.
	 *  \return The value of ASN1_INTEGER::end_date.
	 */
	unsigned long get_endDate() const;


	/*! \brief This function sets ASN1_INTEGER::rev_date.
	 *  \param c_revDate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_revDate(unsigned long c_revDate);

	/*! \brief This function returns ASN1_INTEGER::rev_date.
	 *  \return The value of ASN1_INTEGER::rev_date.
	 */
	unsigned long get_revDate() const;


	/*! \brief This function sets ASN1_INTEGER::serial.
	 *  \param c_serial [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_serial(unsigned long c_serial);

	/*! \brief This function returns ASN1_INTEGER::serial.
	 *  \return The value of ASN1_INTEGER::serial.
	 */
	unsigned long get_serial() const;


	/*! \brief This function sets ASN1_INTEGER::state.
	 *  \param c_state [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_state(unsigned long c_state);

	/*! \brief This function returns ASN1_INTEGER::state.
	 *  \return The value of ASN1_INTEGER::state.
	 */
	unsigned long get_state() const;


	/*! \brief This function sets ASN1_INTEGER::susp_date.
	 *  \param c_suspDate [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_suspDate(unsigned long c_suspDate);

	/*! \brief This function returns ASN1_INTEGER::susp_date.
	 *  \return The value of ASN1_INTEGER::susp_date.
	 */
	unsigned long get_suspDate() const;


	/*! \brief This function sets ASN1_UTF8STRING::uid.
	 *  \param c_uid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_uid(const mString & c_uid);

	/*! \brief This function returns ASN1_UTF8STRING::uid.
	 *  \return The value of ASN1_UTF8STRING::uid.
	 */
	const mString & get_uid() const;

	/*! \brief This function returns ASN1_UTF8STRING::uid.
	 *  \return The value of ASN1_UTF8STRING::uid.
	 */
	mString & get_uid();


	/*! \brief This function copies a InternalCaCert to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const InternalCaCert & other);
	/*! \brief This member is an empty instance of InternalCaCert.
	 */
	static InternalCaCert EmptyInstance;

private:
	unsigned long m_beginDate;
	unsigned long m_caType;
	PKI_CERT m_cert;
	mString m_dn;
	unsigned long m_endDate;
	unsigned long m_revDate;
	unsigned long m_serial;
	unsigned long m_state;
	unsigned long m_suspDate;
	mString m_uid;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of LDAP_OBJECT
*/
class LdapObject : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	LdapObject();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	LdapObject(const LdapObject & other);

	/*! \brief This is the destructor.
	 */
	~LdapObject();


	/*! \brief This function converts a LDAP_OBJECT to a LdapObject.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const LDAP_OBJECT * Datas);

	/*! \brief This function converts a LdapObject to a LDAP_OBJECT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(LDAP_OBJECT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of LDAP_OBJECT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ASN1_UTF8STRING::name.
	 *  \param c_name [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_name(const mString & c_name);

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	const mString & get_name() const;

	/*! \brief This function returns ASN1_UTF8STRING::name.
	 *  \return The value of ASN1_UTF8STRING::name.
	 */
	mString & get_name();


	/*! \brief This function sets ASN1_UTF8STRING::value.
	 *  \param c_value [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_value(const mString & c_value);

	/*! \brief This function returns ASN1_UTF8STRING::value.
	 *  \return The value of ASN1_UTF8STRING::value.
	 */
	const mString & get_value() const;

	/*! \brief This function returns ASN1_UTF8STRING::value.
	 *  \return The value of ASN1_UTF8STRING::value.
	 */
	mString & get_value();


	/*! \brief This function copies a LdapObject to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const LdapObject & other);
	/*! \brief This member is an empty instance of LdapObject.
	 */
	static LdapObject EmptyInstance;

private:
	mString m_name;
	mString m_value;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of LDAP_RESULT
*/
class LdapResult : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	LdapResult();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	LdapResult(const LdapResult & other);

	/*! \brief This is the destructor.
	 */
	~LdapResult();


	/*! \brief This function converts a LDAP_RESULT to a LdapResult.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const LDAP_RESULT * Datas);

	/*! \brief This function converts a LdapResult to a LDAP_RESULT.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(LDAP_RESULT ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of LDAP_RESULT.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets LDAP_OBJECT::objects.
	 *  \param c_objects [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_objects(const mVector< LdapObject > & c_objects);

	/*! \brief This function returns LDAP_OBJECT::objects.
	 *  \return The value of LDAP_OBJECT::objects.
	 */
	const mVector< LdapObject > & get_objects() const;

	/*! \brief This function returns LDAP_OBJECT::objects.
	 *  \return The value of LDAP_OBJECT::objects.
	 */
	mVector< LdapObject > & get_objects();


	/*! \brief This function sets ASN1_UTF8STRING::rdn.
	 *  \param c_rdn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_rdn(const mString & c_rdn);

	/*! \brief This function returns ASN1_UTF8STRING::rdn.
	 *  \return The value of ASN1_UTF8STRING::rdn.
	 */
	const mString & get_rdn() const;

	/*! \brief This function returns ASN1_UTF8STRING::rdn.
	 *  \return The value of ASN1_UTF8STRING::rdn.
	 */
	mString & get_rdn();


	/*! \brief This function sets ASN1_UTF8STRING::uid.
	 *  \param c_uid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_uid(const mString & c_uid);

	/*! \brief This function returns ASN1_UTF8STRING::uid.
	 *  \return The value of ASN1_UTF8STRING::uid.
	 */
	const mString & get_uid() const;

	/*! \brief This function returns ASN1_UTF8STRING::uid.
	 *  \return The value of ASN1_UTF8STRING::uid.
	 */
	mString & get_uid();


	/*! \brief This function copies a LdapResult to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const LdapResult & other);
	/*! \brief This member is an empty instance of LdapResult.
	 */
	static LdapResult EmptyInstance;

private:
	mVector< LdapObject > m_objects;
	mString m_rdn;
	mString m_uid;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ADMIN_RESPONSE_BODY
*/
class AdminResponseBody : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	AdminResponseBody();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	AdminResponseBody(const AdminResponseBody & other);

	/*! \brief This is the destructor.
	 */
	~AdminResponseBody();


	/*! \brief This function converts a ADMIN_RESPONSE_BODY to a AdminResponseBody.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ADMIN_RESPONSE_BODY * Datas);

	/*! \brief This function converts a AdminResponseBody to a ADMIN_RESPONSE_BODY.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ADMIN_RESPONSE_BODY ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ADMIN_RESPONSE_BODY.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets the CHOICE type, it allocates the right type.
	 *  \param c_type [IN] the type.
	 *  \return true on success, false on failure.
	 */
	bool set_type(int c_type);

	/*! \brief This function returns the CHOICE type.
	 *  \return the type, -1 when no type was set.
	 */
	int get_type() const;

	/*! \brief This function sets X509_ACL::d.acls_conf.
	 *  \param c_aclsConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_aclsConf(const X509Acl & c_aclsConf);

	/*! \brief This function returns X509_ACL::d.acls_conf.
	 *  \return The value of X509_ACL::d.acls_conf.
	 */
	const X509Acl & get_aclsConf() const;

	/*! \brief This function returns X509_ACL::d.acls_conf.
	 *  \return The value of X509_ACL::d.acls_conf.
	 */
	X509Acl & get_aclsConf();


	/*! \brief This function sets ENTITY_AUDIT_ENTRY::d.audits.
	 *  \param c_audits [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_audits(const mVector< EntityAuditEntry > & c_audits);

	/*! \brief This function returns ENTITY_AUDIT_ENTRY::d.audits.
	 *  \return The value of ENTITY_AUDIT_ENTRY::d.audits.
	 */
	const mVector< EntityAuditEntry > & get_audits() const;

	/*! \brief This function returns ENTITY_AUDIT_ENTRY::d.audits.
	 *  \return The value of ENTITY_AUDIT_ENTRY::d.audits.
	 */
	mVector< EntityAuditEntry > & get_audits();


	/*! \brief This function sets X509::d.cert.
	 *  \param c_cert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_cert(const PKI_CERT & c_cert);

	/*! \brief This function returns X509::d.cert.
	 *  \return The value of X509::d.cert.
	 */
	const PKI_CERT & get_cert() const;

	/*! \brief This function returns X509::d.cert.
	 *  \return The value of X509::d.cert.
	 */
	PKI_CERT & get_cert();


	/*! \brief This function sets INTERNAL_CA_CERT::d.certs.
	 *  \param c_certs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_certs(const mVector< InternalCaCert > & c_certs);

	/*! \brief This function returns INTERNAL_CA_CERT::d.certs.
	 *  \return The value of INTERNAL_CA_CERT::d.certs.
	 */
	const mVector< InternalCaCert > & get_certs() const;

	/*! \brief This function returns INTERNAL_CA_CERT::d.certs.
	 *  \return The value of INTERNAL_CA_CERT::d.certs.
	 */
	mVector< InternalCaCert > & get_certs();


	/*! \brief This function sets EXPORTED_PKI_CONF::d.conf.
	 *  \param c_conf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_conf(const ExportedPkiConf & c_conf);

	/*! \brief This function returns EXPORTED_PKI_CONF::d.conf.
	 *  \return The value of EXPORTED_PKI_CONF::d.conf.
	 */
	const ExportedPkiConf & get_conf() const;

	/*! \brief This function returns EXPORTED_PKI_CONF::d.conf.
	 *  \return The value of EXPORTED_PKI_CONF::d.conf.
	 */
	ExportedPkiConf & get_conf();


	/*! \brief This function sets ENTITY_CREATION_RESP::d.cre_entity.
	 *  \param c_creEntity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_creEntity(const EntityCreationResp & c_creEntity);

	/*! \brief This function returns ENTITY_CREATION_RESP::d.cre_entity.
	 *  \return The value of ENTITY_CREATION_RESP::d.cre_entity.
	 */
	const EntityCreationResp & get_creEntity() const;

	/*! \brief This function returns ENTITY_CREATION_RESP::d.cre_entity.
	 *  \return The value of ENTITY_CREATION_RESP::d.cre_entity.
	 */
	EntityCreationResp & get_creEntity();


	/*! \brief This function sets CREATE_PKI_USER_RESPONSE::d.create_pki_user.
	 *  \param c_createPkiUser [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_createPkiUser(const CreatePkiUserResponse & c_createPkiUser);

	/*! \brief This function returns CREATE_PKI_USER_RESPONSE::d.create_pki_user.
	 *  \return The value of CREATE_PKI_USER_RESPONSE::d.create_pki_user.
	 */
	const CreatePkiUserResponse & get_createPkiUser() const;

	/*! \brief This function returns CREATE_PKI_USER_RESPONSE::d.create_pki_user.
	 *  \return The value of CREATE_PKI_USER_RESPONSE::d.create_pki_user.
	 */
	CreatePkiUserResponse & get_createPkiUser();


	/*! \brief This function sets X509_CRL::d.crls.
	 *  \param c_crls [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_crls(const mVector< PKI_CRL > & c_crls);

	/*! \brief This function returns X509_CRL::d.crls.
	 *  \return The value of X509_CRL::d.crls.
	 */
	const mVector< PKI_CRL > & get_crls() const;

	/*! \brief This function returns X509_CRL::d.crls.
	 *  \return The value of X509_CRL::d.crls.
	 */
	mVector< PKI_CRL > & get_crls();


	/*! \brief This function sets X509_REQ::d.csr.
	 *  \param c_csr [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_csr(const PKI_CSR & c_csr);

	/*! \brief This function returns X509_REQ::d.csr.
	 *  \return The value of X509_REQ::d.csr.
	 */
	const PKI_CSR & get_csr() const;

	/*! \brief This function returns X509_REQ::d.csr.
	 *  \return The value of X509_REQ::d.csr.
	 */
	PKI_CSR & get_csr();


	/*! \brief This function sets X509_NAME::d.dn.
	 *  \param c_dn [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_dn(const X509_NAME * c_dn);

	/*! \brief This function returns X509_NAME::d.dn.
	 *  \return The value of X509_NAME::d.dn.
	 */
	const X509_NAME * get_dn() const;

	/*! \brief This function returns X509_NAME::d.dn.
	 *  \return The value of X509_NAME::d.dn.
	 */
	X509_NAME * get_dn();


	/*! \brief This function sets NEWPKI_EE_USER::d.ee_users.
	 *  \param c_eeUsers [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_eeUsers(const mVector< NewpkiEeUser > & c_eeUsers);

	/*! \brief This function returns NEWPKI_EE_USER::d.ee_users.
	 *  \return The value of NEWPKI_EE_USER::d.ee_users.
	 */
	const mVector< NewpkiEeUser > & get_eeUsers() const;

	/*! \brief This function returns NEWPKI_EE_USER::d.ee_users.
	 *  \return The value of NEWPKI_EE_USER::d.ee_users.
	 */
	mVector< NewpkiEeUser > & get_eeUsers();


	/*! \brief This function sets ENTITY_ENTRY_INFO::d.entities.
	 *  \param c_entities [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entities(const mVector< EntityEntryInfo > & c_entities);

	/*! \brief This function returns ENTITY_ENTRY_INFO::d.entities.
	 *  \return The value of ENTITY_ENTRY_INFO::d.entities.
	 */
	const mVector< EntityEntryInfo > & get_entities() const;

	/*! \brief This function returns ENTITY_ENTRY_INFO::d.entities.
	 *  \return The value of ENTITY_ENTRY_INFO::d.entities.
	 */
	mVector< EntityEntryInfo > & get_entities();


	/*! \brief This function sets ENTITY_LINKS::d.entities_links.
	 *  \param c_entitiesLinks [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entitiesLinks(const mVector< EntityLinks > & c_entitiesLinks);

	/*! \brief This function returns ENTITY_LINKS::d.entities_links.
	 *  \return The value of ENTITY_LINKS::d.entities_links.
	 */
	const mVector< EntityLinks > & get_entitiesLinks() const;

	/*! \brief This function returns ENTITY_LINKS::d.entities_links.
	 *  \return The value of ENTITY_LINKS::d.entities_links.
	 */
	mVector< EntityLinks > & get_entitiesLinks();


	/*! \brief This function sets ENTITY_CONF_BODY::d.entity_conf.
	 *  \param c_entityConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_entityConf(const EntityConfBody & c_entityConf);

	/*! \brief This function returns ENTITY_CONF_BODY::d.entity_conf.
	 *  \return The value of ENTITY_CONF_BODY::d.entity_conf.
	 */
	const EntityConfBody & get_entityConf() const;

	/*! \brief This function returns ENTITY_CONF_BODY::d.entity_conf.
	 *  \return The value of ENTITY_CONF_BODY::d.entity_conf.
	 */
	EntityConfBody & get_entityConf();


	/*! \brief This function sets ERROR_ENTRY::d.errors.
	 *  \param c_errors [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_errors(const mVector< ErrorEntry > & c_errors);

	/*! \brief This function returns ERROR_ENTRY::d.errors.
	 *  \return The value of ERROR_ENTRY::d.errors.
	 */
	const mVector< ErrorEntry > & get_errors() const;

	/*! \brief This function returns ERROR_ENTRY::d.errors.
	 *  \return The value of ERROR_ENTRY::d.errors.
	 */
	mVector< ErrorEntry > & get_errors();


	/*! \brief This function sets USERS_GROUP::d.groups.
	 *  \param c_groups [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_groups(const mVector< UsersGroup > & c_groups);

	/*! \brief This function returns USERS_GROUP::d.groups.
	 *  \return The value of USERS_GROUP::d.groups.
	 */
	const mVector< UsersGroup > & get_groups() const;

	/*! \brief This function returns USERS_GROUP::d.groups.
	 *  \return The value of USERS_GROUP::d.groups.
	 */
	mVector< UsersGroup > & get_groups();


	/*! \brief This function sets ASN1_INTEGER::d.id.
	 *  \param c_id [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_id(unsigned long c_id);

	/*! \brief This function returns ASN1_INTEGER::d.id.
	 *  \return The value of ASN1_INTEGER::d.id.
	 */
	unsigned long get_id() const;


	/*! \brief This function sets OBJECTS_LIST_SYNCHRO::d.known_objects.
	 *  \param c_knownObjects [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_knownObjects(const ObjectsListSynchro & c_knownObjects);

	/*! \brief This function returns OBJECTS_LIST_SYNCHRO::d.known_objects.
	 *  \return The value of OBJECTS_LIST_SYNCHRO::d.known_objects.
	 */
	const ObjectsListSynchro & get_knownObjects() const;

	/*! \brief This function returns OBJECTS_LIST_SYNCHRO::d.known_objects.
	 *  \return The value of OBJECTS_LIST_SYNCHRO::d.known_objects.
	 */
	ObjectsListSynchro & get_knownObjects();


	/*! \brief This function sets LDAP_RESULT::d.ldap_results.
	 *  \param c_ldapResults [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_ldapResults(const mVector< LdapResult > & c_ldapResults);

	/*! \brief This function returns LDAP_RESULT::d.ldap_results.
	 *  \return The value of LDAP_RESULT::d.ldap_results.
	 */
	const mVector< LdapResult > & get_ldapResults() const;

	/*! \brief This function returns LDAP_RESULT::d.ldap_results.
	 *  \return The value of LDAP_RESULT::d.ldap_results.
	 */
	mVector< LdapResult > & get_ldapResults();


	/*! \brief This function sets ENTITY_CONF::d.local_entity_conf.
	 *  \param c_localEntityConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_localEntityConf(const EntityConf & c_localEntityConf);

	/*! \brief This function returns ENTITY_CONF::d.local_entity_conf.
	 *  \return The value of ENTITY_CONF::d.local_entity_conf.
	 */
	const EntityConf & get_localEntityConf() const;

	/*! \brief This function returns ENTITY_CONF::d.local_entity_conf.
	 *  \return The value of ENTITY_CONF::d.local_entity_conf.
	 */
	EntityConf & get_localEntityConf();


	/*! \brief This function sets LOG_ENTRY::d.logs.
	 *  \param c_logs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_logs(const mVector< LogEntry > & c_logs);

	/*! \brief This function returns LOG_ENTRY::d.logs.
	 *  \return The value of LOG_ENTRY::d.logs.
	 */
	const mVector< LogEntry > & get_logs() const;

	/*! \brief This function returns LOG_ENTRY::d.logs.
	 *  \return The value of LOG_ENTRY::d.logs.
	 */
	mVector< LogEntry > & get_logs();


	/*! \brief This function sets ASN1_INTEGER::d.logs_type.
	 *  \param c_logsType [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_logsType(const mVector< unsigned long > & c_logsType);

	/*! \brief This function returns ASN1_INTEGER::d.logs_type.
	 *  \return The value of ASN1_INTEGER::d.logs_type.
	 */
	const mVector< unsigned long > & get_logsType() const;

	/*! \brief This function returns ASN1_INTEGER::d.logs_type.
	 *  \return The value of ASN1_INTEGER::d.logs_type.
	 */
	mVector< unsigned long > & get_logsType();


	/*! \brief This function sets EMAIL_CONF::d.mail_conf.
	 *  \param c_mailConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_mailConf(const EmailConf & c_mailConf);

	/*! \brief This function returns EMAIL_CONF::d.mail_conf.
	 *  \return The value of EMAIL_CONF::d.mail_conf.
	 */
	const EmailConf & get_mailConf() const;

	/*! \brief This function returns EMAIL_CONF::d.mail_conf.
	 *  \return The value of EMAIL_CONF::d.mail_conf.
	 */
	EmailConf & get_mailConf();


	/*! \brief This function sets ASN1_BIT_STRING::d.my_acl.
	 *  \param c_myAcl [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_myAcl(const ASN1_BIT_STRING * c_myAcl);

	/*! \brief This function returns ASN1_BIT_STRING::d.my_acl.
	 *  \return The value of ASN1_BIT_STRING::d.my_acl.
	 */
	const ASN1_BIT_STRING * get_myAcl() const;

	/*! \brief This function returns ASN1_BIT_STRING::d.my_acl.
	 *  \return The value of ASN1_BIT_STRING::d.my_acl.
	 */
	ASN1_BIT_STRING * get_myAcl();


	/*! \brief This function sets ENTITY_CONF_CRYPTED::d.my_conf.
	 *  \param c_myConf [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_myConf(const EntityConfCrypted & c_myConf);

	/*! \brief This function returns ENTITY_CONF_CRYPTED::d.my_conf.
	 *  \return The value of ENTITY_CONF_CRYPTED::d.my_conf.
	 */
	const EntityConfCrypted & get_myConf() const;

	/*! \brief This function returns ENTITY_CONF_CRYPTED::d.my_conf.
	 *  \return The value of ENTITY_CONF_CRYPTED::d.my_conf.
	 */
	EntityConfCrypted & get_myConf();


	/*! \brief This function sets CRYPTED_NEWPKI_REQUESTS::d.object_reqs.
	 *  \param c_objectReqs [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_objectReqs(const CryptedNewpkiRequests & c_objectReqs);

	/*! \brief This function returns CRYPTED_NEWPKI_REQUESTS::d.object_reqs.
	 *  \return The value of CRYPTED_NEWPKI_REQUESTS::d.object_reqs.
	 */
	const CryptedNewpkiRequests & get_objectReqs() const;

	/*! \brief This function returns CRYPTED_NEWPKI_REQUESTS::d.object_reqs.
	 *  \return The value of CRYPTED_NEWPKI_REQUESTS::d.object_reqs.
	 */
	CryptedNewpkiRequests & get_objectReqs();


	/*! \brief This function sets CRYPTED_NEWPKI_RESPONSES::d.object_resps.
	 *  \param c_objectResps [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_objectResps(const CryptedNewpkiResponses & c_objectResps);

	/*! \brief This function returns CRYPTED_NEWPKI_RESPONSES::d.object_resps.
	 *  \return The value of CRYPTED_NEWPKI_RESPONSES::d.object_resps.
	 */
	const CryptedNewpkiResponses & get_objectResps() const;

	/*! \brief This function returns CRYPTED_NEWPKI_RESPONSES::d.object_resps.
	 *  \return The value of CRYPTED_NEWPKI_RESPONSES::d.object_resps.
	 */
	CryptedNewpkiResponses & get_objectResps();


	/*! \brief This function sets ASN1_NULL::d.other.
	 *  \param c_other [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_other(const ASN1_NULL * c_other);

	/*! \brief This function returns ASN1_NULL::d.other.
	 *  \return The value of ASN1_NULL::d.other.
	 */
	const ASN1_NULL * get_other() const;

	/*! \brief This function returns ASN1_NULL::d.other.
	 *  \return The value of ASN1_NULL::d.other.
	 */
	ASN1_NULL * get_other();


	/*! \brief This function sets PKCS7::d.p7b.
	 *  \param c_p7b [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_p7b(const PKI_P7B & c_p7b);

	/*! \brief This function returns PKCS7::d.p7b.
	 *  \return The value of PKCS7::d.p7b.
	 */
	const PKI_P7B & get_p7b() const;

	/*! \brief This function returns PKCS7::d.p7b.
	 *  \return The value of PKCS7::d.p7b.
	 */
	PKI_P7B & get_p7b();


	/*! \brief This function sets NEWPKI_PROFILE_DATAS::d.profiles.
	 *  \param c_profiles [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_profiles(const mVector< NewpkiProfileDatas > & c_profiles);

	/*! \brief This function returns NEWPKI_PROFILE_DATAS::d.profiles.
	 *  \return The value of NEWPKI_PROFILE_DATAS::d.profiles.
	 */
	const mVector< NewpkiProfileDatas > & get_profiles() const;

	/*! \brief This function returns NEWPKI_PROFILE_DATAS::d.profiles.
	 *  \return The value of NEWPKI_PROFILE_DATAS::d.profiles.
	 */
	mVector< NewpkiProfileDatas > & get_profiles();


	/*! \brief This function sets REP_ENTRY_INFO::d.repositories.
	 *  \param c_repositories [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_repositories(const mVector< RepEntryInfo > & c_repositories);

	/*! \brief This function returns REP_ENTRY_INFO::d.repositories.
	 *  \return The value of REP_ENTRY_INFO::d.repositories.
	 */
	const mVector< RepEntryInfo > & get_repositories() const;

	/*! \brief This function returns REP_ENTRY_INFO::d.repositories.
	 *  \return The value of REP_ENTRY_INFO::d.repositories.
	 */
	mVector< RepEntryInfo > & get_repositories();


	/*! \brief This function sets ENTITY_SIGNATURE_RESP::d.sign_entity.
	 *  \param c_signEntity [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signEntity(const EntitySignatureResp & c_signEntity);

	/*! \brief This function returns ENTITY_SIGNATURE_RESP::d.sign_entity.
	 *  \return The value of ENTITY_SIGNATURE_RESP::d.sign_entity.
	 */
	const EntitySignatureResp & get_signEntity() const;

	/*! \brief This function returns ENTITY_SIGNATURE_RESP::d.sign_entity.
	 *  \return The value of ENTITY_SIGNATURE_RESP::d.sign_entity.
	 */
	EntitySignatureResp & get_signEntity();


	/*! \brief This function sets ASN1_INTEGER::d.status.
	 *  \param c_status [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_status(unsigned long c_status);

	/*! \brief This function returns ASN1_INTEGER::d.status.
	 *  \return The value of ASN1_INTEGER::d.status.
	 */
	unsigned long get_status() const;


	/*! \brief This function sets ASN1_OCTET_STRING::d.transactionID.
	 *  \param c_transactionid [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_transactionid(const Asn1OctetString & c_transactionid);

	/*! \brief This function returns ASN1_OCTET_STRING::d.transactionID.
	 *  \return The value of ASN1_OCTET_STRING::d.transactionID.
	 */
	const Asn1OctetString & get_transactionid() const;

	/*! \brief This function returns ASN1_OCTET_STRING::d.transactionID.
	 *  \return The value of ASN1_OCTET_STRING::d.transactionID.
	 */
	Asn1OctetString & get_transactionid();


	/*! \brief This function sets USER_ENTRY::d.users.
	 *  \param c_users [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_users(const mVector< UserEntry > & c_users);

	/*! \brief This function returns USER_ENTRY::d.users.
	 *  \return The value of USER_ENTRY::d.users.
	 */
	const mVector< UserEntry > & get_users() const;

	/*! \brief This function returns USER_ENTRY::d.users.
	 *  \return The value of USER_ENTRY::d.users.
	 */
	mVector< UserEntry > & get_users();


	/*! \brief This function sets USER_ENTRY_CERT::d.users_cert.
	 *  \param c_usersCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_usersCert(const mVector< UserEntryCert > & c_usersCert);

	/*! \brief This function returns USER_ENTRY_CERT::d.users_cert.
	 *  \return The value of USER_ENTRY_CERT::d.users_cert.
	 */
	const mVector< UserEntryCert > & get_usersCert() const;

	/*! \brief This function returns USER_ENTRY_CERT::d.users_cert.
	 *  \return The value of USER_ENTRY_CERT::d.users_cert.
	 */
	mVector< UserEntryCert > & get_usersCert();


	/*! \brief This function sets ASN1_INTEGER::d.usertype.
	 *  \param c_usertype [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_usertype(unsigned long c_usertype);

	/*! \brief This function returns ASN1_INTEGER::d.usertype.
	 *  \return The value of ASN1_INTEGER::d.usertype.
	 */
	unsigned long get_usertype() const;


	/*! \brief This function sets NEWPKI_PROFILE_DATAS_CERT::d.webuser_certs.
	 *  \param c_webuserCerts [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_webuserCerts(const mVector< NewpkiProfileDatasCert > & c_webuserCerts);

	/*! \brief This function returns NEWPKI_PROFILE_DATAS_CERT::d.webuser_certs.
	 *  \return The value of NEWPKI_PROFILE_DATAS_CERT::d.webuser_certs.
	 */
	const mVector< NewpkiProfileDatasCert > & get_webuserCerts() const;

	/*! \brief This function returns NEWPKI_PROFILE_DATAS_CERT::d.webuser_certs.
	 *  \return The value of NEWPKI_PROFILE_DATAS_CERT::d.webuser_certs.
	 */
	mVector< NewpkiProfileDatasCert > & get_webuserCerts();


	/*! \brief This function copies a AdminResponseBody to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const AdminResponseBody & other);
	/*! \brief This member is an empty instance of AdminResponseBody.
	 */
	static AdminResponseBody EmptyInstance;

private:
	int m_type;
	bool malloc_byType(int c_type);
	X509Acl * m_aclsConf;
	mVector< EntityAuditEntry > * m_audits;
	PKI_CERT * m_cert;
	mVector< InternalCaCert > * m_certs;
	ExportedPkiConf * m_conf;
	EntityCreationResp * m_creEntity;
	CreatePkiUserResponse * m_createPkiUser;
	mVector< PKI_CRL > * m_crls;
	PKI_CSR * m_csr;
	X509_NAME * m_dn;
	mVector< NewpkiEeUser > * m_eeUsers;
	mVector< EntityEntryInfo > * m_entities;
	mVector< EntityLinks > * m_entitiesLinks;
	EntityConfBody * m_entityConf;
	mVector< ErrorEntry > * m_errors;
	mVector< UsersGroup > * m_groups;
	unsigned long * m_id;
	ObjectsListSynchro * m_knownObjects;
	mVector< LdapResult > * m_ldapResults;
	EntityConf * m_localEntityConf;
	mVector< LogEntry > * m_logs;
	mVector< unsigned long > * m_logsType;
	EmailConf * m_mailConf;
	ASN1_BIT_STRING * m_myAcl;
	EntityConfCrypted * m_myConf;
	CryptedNewpkiRequests * m_objectReqs;
	CryptedNewpkiResponses * m_objectResps;
	ASN1_NULL * m_other;
	PKI_P7B * m_p7b;
	mVector< NewpkiProfileDatas > * m_profiles;
	mVector< RepEntryInfo > * m_repositories;
	EntitySignatureResp * m_signEntity;
	unsigned long * m_status;
	Asn1OctetString * m_transactionid;
	mVector< UserEntry > * m_users;
	mVector< UserEntryCert > * m_usersCert;
	unsigned long * m_usertype;
	mVector< NewpkiProfileDatasCert > * m_webuserCerts;
	void resetAll();
	void freeAll();
protected:
};


/*!
  This class is the representation of ADMIN_RESPONSE
*/
class AdminResponse : public NewPKIObject
{
public:
	/*! \brief This is the constructor.
	 */
	AdminResponse();

	/*! \brief This is the constructor.
	 *  \param other [IN] the datas to copy from.
	 */
	AdminResponse(const AdminResponse & other);

	/*! \brief This is the destructor.
	 */
	~AdminResponse();


	/*! \brief This function converts a ADMIN_RESPONSE to a AdminResponse.
	 *  \param Datas [IN] the datas to load from.
	 *  \return true on success, false on failure.
	 */
	bool load_Datas(const ADMIN_RESPONSE * Datas);

	/*! \brief This function converts a AdminResponse to a ADMIN_RESPONSE.
	 *  \param Datas [OUT] the datas to copy to.
	 *  \return true on success, false on failure.
	 */
	bool give_Datas(ADMIN_RESPONSE ** Datas) const;

	/*! \brief This function clears all the internals.
	 */
	void Clear();

	/*! \brief This function returns the ASN1_ITEM of ADMIN_RESPONSE.
	 *  \return The ASN1_ITEM.
	 */
	static const ASN1_ITEM * get_ASN1_ITEM();

	/*! \brief This function sets ADMIN_RESPONSE_BODY::body.
	 *  \param c_body [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_body(const AdminResponseBody & c_body);

	/*! \brief This function returns ADMIN_RESPONSE_BODY::body.
	 *  \return The value of ADMIN_RESPONSE_BODY::body.
	 */
	const AdminResponseBody & get_body() const;

	/*! \brief This function returns ADMIN_RESPONSE_BODY::body.
	 *  \return The value of ADMIN_RESPONSE_BODY::body.
	 */
	AdminResponseBody & get_body();


	/*! \brief This function sets X509_ALGOR::sig_algo.
	 *  \param c_sigAlgo [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_sigAlgo(const X509_ALGOR * c_sigAlgo);

	/*! \brief This function returns X509_ALGOR::sig_algo.
	 *  \return The value of X509_ALGOR::sig_algo.
	 */
	const X509_ALGOR * get_sigAlgo() const;

	/*! \brief This function returns X509_ALGOR::sig_algo.
	 *  \return The value of X509_ALGOR::sig_algo.
	 */
	X509_ALGOR * get_sigAlgo();


	/*! \brief This function sets ASN1_BIT_STRING::signature.
	 *  \param c_signature [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signature(const ASN1_BIT_STRING * c_signature);

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	const ASN1_BIT_STRING * get_signature() const;

	/*! \brief This function returns ASN1_BIT_STRING::signature.
	 *  \return The value of ASN1_BIT_STRING::signature.
	 */
	ASN1_BIT_STRING * get_signature();


	/*! \brief This function sets X509::signing_cert.
	 *  \param c_signingCert [IN] the datas.
	 *  \return true on success, false on failure.
	 */
	bool set_signingCert(const PKI_CERT & c_signingCert);

	/*! \brief This function returns X509::signing_cert.
	 *  \return The value of X509::signing_cert.
	 */
	const PKI_CERT & get_signingCert() const;

	/*! \brief This function returns X509::signing_cert.
	 *  \return The value of X509::signing_cert.
	 */
	PKI_CERT & get_signingCert();


	/*! \brief This function copies a AdminResponse to another.
	 *  \param other [IN] the datas to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=(const AdminResponse & other);
	/*! \brief This member is an empty instance of AdminResponse.
	 */
	static AdminResponse EmptyInstance;

private:
	AdminResponseBody m_body;
	X509_ALGOR * m_sigAlgo;
	ASN1_BIT_STRING * m_signature;
	PKI_CERT m_signingCert;
	void resetAll();
	void freeAll();
protected:
};


#endif // #ifndef ASN1RESP_H
