/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PKI_EXT.h: interface for the CERT class.
//
//////////////////////////////////////////////////////////////////////

#ifndef PKI_EXT_H
#define PKI_EXT_H


#include <stdio.h>
#include <string.h>

#include <HashTable/HashTable_String.h>
#include <openssl/x509v3.h>

/*!
  This class is the representation of extensions
*/
class PKI_EXT
{
public:
	/*! \brief This is the constructor.
	 */
	PKI_EXT();

	/*! \brief This is the destructor.
	 */
	virtual ~PKI_EXT();

	/*! \brief This function clears the internal datas.
	 */
	void Clear();

	/*! \brief This function verifies that an extensions syntaxis valid.
	 *  \param FieldName [IN] The extension name.
	 *  \param FieldValue [IN] The extension value.
	 *  \return true on success, false on failure.
	 */
	static bool VerifyExtensionValue(const char *FieldName, const char *FieldValue);

	/*! \brief This function adds extensions to a certificate.
	 *  \param Ext [IN] The extensions list.
	 *  \param ctx [IN] The extensions context.
	 *  \param keep_csr_exts [IN] Whether to keep the CSR's extensions.
	 *  \param csr_exts_overwrite [IN] Whether to overwrite the CSR's extensions.
	 *  \param cert [IN] The certificate.
	 *  \return true on success, false on failure.
	 */
	static bool Add_CertExtensions(const HashTable_String * Ext, X509V3_CTX * ctx, bool keep_csr_exts, bool csr_exts_overwrite, X509 * cert);

	/*! \brief This function adds extensions to a CRL.
	 *  \param Ext [IN] The extensions list.
	 *  \param ctx [IN] The extensions context.
	 *  \param crl [IN] The CRL.
	 *  \return true on success, false on failure.
	 */
	static bool Add_CrlExtensions(const HashTable_String * Ext, X509V3_CTX *ctx, X509_CRL *crl);

	/*! \brief This function returns the extensions.
	 *  \return The extensions.
	 */
	const HashTable_String & GetExts() const;

	/*! \brief This function loads the extensions.
	 *  \param Exts [IN] The list of extensions.
	 *  \return true on success, false on failure.
	 */
	bool Load(const STACK_OF(X509_EXTENSION) * Exts);
	
	/*! \brief This operator copies one PKI_EXT into another.
	 *  \param other [IN] The other PKI_EXT to copy from.
	 *  \return true on success, false on failure.
	 */
	bool operator=( const PKI_EXT &other );

private:
	#define DN_EXT_RE "^(.*)<dn>(.*):(move|copy)</dn>"
	static char * newpki_strip_spaces(char *name);
	static STACK_OF(CONF_VALUE) * newpki_X509V3_parse_list(const char *line);
	static X509_EXTENSION * newpki_do_ext_i2d(X509V3_EXT_METHOD *method, int ext_nid,  int crit, void *ext_struc);
	static X509_EXTENSION * newpki_do_ext(X509V3_CTX *ctx, int ext_nid, int crit, char *value);
	static int newpki_v3_check_critical(char **value);
	static X509_EXTENSION * newpki_v3_generic_extension(const char *ext, char *value, int crit, int type);
	static int newpki_v3_check_generic(char **value);
	static X509_EXTENSION * GetExtensionValue(X509V3_CTX *ctx, const char * Name, const char * Value);
	static bool VerifyExtension(const char *FieldName, const char *FieldValue);
	static const char * GetExtensionName(ASN1_OBJECT *obj);
	static X509_EXTENSION * GetExtension(const STACK_OF(X509_EXTENSION) * Exts, const char * name);
	HashTable_String m_Exts;
};

#endif // PKI_EXT_H

