/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/



// CA_Handler.cpp: implementation of the CA_Handler class.
//
//////////////////////////////////////////////////////////////////////

#include "CA_Handler.h"
#include "svintl.h"
#include <openssl/ocsp.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

const char * CA_Handler::m_Version = NULL;

CA_Handler::CA_Handler(const mString & EntityName, int CaType)
{
	m_EntityName = EntityName;
	m_DbConn = NULL;
	m_CaType = CaType;
}

CA_Handler::~CA_Handler()
{
	if(m_DbConn)
		delete m_DbConn;
	Reset();
}

bool CA_Handler::CreateTables(SQL_Connection * DbConn)
{
	SQL sql(DbConn, SQL_ACCESS_WRITE);
	long i;
	char * CommonCreates[] = {CA_HANDLER_CREATE_1, CA_HANDLER_CREATE_2, CA_HANDLER_CREATE_3, NULL};

	//We execute each request
	for(i=0; CommonCreates[i]; i++)
	{
		if(!sql.Execute(CommonCreates[i]))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}

PKI_CERT * CA_Handler::get_CaCert()
{
	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		return NULL;
	}
	return &m_CaCert;
}

bool CA_Handler::Create(const PKI_CERT & cert, const InternalCaKey & privkey)
{
	Reset();

	m_CaKey = cert.GetPrivateKey();
	
	if(!m_CaKey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	
	if(!m_InternCa.set_caKey(privkey))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!m_InternCa.set_caCert(cert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	m_InternCaBody.set_caType(m_CaType);
	m_InternCaBody.set_caSerial(1);
	m_InternCaBody.set_isOK();
	m_InternCa.set_isOK();

	//Flush new datas into database	
	if(!Flush())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Load datas
	if(!Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool CA_Handler::Flush()
{
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	mString ca_pem;
	
	if(!m_InternCaBody || !m_InternCa)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}

	//Encrypt the body and generate signature
	if(!m_InternCaBody.to_SignEncrypt(m_InternCa.get_cryptedBody(), m_CaKey.GetRsaKey(), m_CaKey.GetRsaKey(), EVP_sha1(), EVP_des_ede3_cbc())) 
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	//Convert to PEM	
	if(!m_InternCa.to_PEM(ca_pem))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CleanMemory();
		return false;
	}
	//Save some memory
	CleanMemory();
	
	if(req.sprintf(CA_HANDLER_INSERT_CA, m_CaType, ca_pem.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	return true;
}

bool CA_Handler::Load()
{
	Reset();
	long NumRows;
	
	if(!m_DbConn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	
	//We load the CA datas	
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	mString datas;

	if(req.sprintf(CA_HANDLER_GET_CA, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}	
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	// It can be loaded latter
	if(!NumRows)
	{
		return true;
	}
	
	if(!sql.Value(0, "datas", datas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	//We convert the datas from the PEM blob
	if(!m_InternCa.from_PEM(datas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	
	//Setting the CA cert
	if( !(m_CaCert = m_InternCa.get_caCert()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Setting the CA key
	if(!INTERNAL_CA_KEY_load(m_InternCa.get_caKey(), m_CaKey, m_Engine))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	//Associate private key and cert
	if(!m_CaCert.SetPrivateKey(m_CaKey))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!m_InternCaBody.from_SignEncrypt(m_InternCa.get_cryptedBody(), m_CaKey.GetRsaKey(), m_CaKey.GetRsaKey()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	//Now we verify that the two ca_types match
	if((int)m_InternCaBody.get_caType() != m_CaType)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	// We can now do the upgrade
	if(m_Version && !Private_Upgrade(m_Version))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}


void CA_Handler::Reset()
{
	CaLock.EnterCS();
	m_InternCaBody.Clear();
	m_InternCa.Clear();
	CaLock.LeaveCS();
}

void CA_Handler::set_ENGINE(ENGINE *e)
{
	CaLock.EnterCS();
	m_Engine = e;
	CaLock.LeaveCS();
}

ENGINE * CA_Handler::get_ENGINE()
{
	return m_Engine;
}

bool CA_Handler::Sign(const PKI_CSR & csr, PKI_CERT & ResultCert, const mString & uid, int Days, const HashTable_String *Exts, bool keepCsrExts, bool csrExtsOverwrite, bool check_sig)
{
	InternalCaCert currCert;
	time_t currTime;
	X509_REQ * reqcsr;
	char * strDn;	
	mString dn;
	time_t begin_date;
	time_t end_date;
	unsigned long serial;
	mString strUID;
	mString strDnHash;
	
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	mString req;
	long NumRows;
	mString signature;
	time_t startTime;
	
	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		return false;
	}
	
	//Search if we have a certificate with same DN !
	reqcsr = csr.GetX509_REQ();
	time_gmt(&currTime);

	strDn = X509_NAME_oneline(reqcsr->req_info->subject, NULL, 0);
	if(!strDn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	currCert.set_dn(strDn);
	dn = sql.FormatString(strDn);
	free(strDn);
	
	if(!dn.size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(uid.size())
	{
		strUID = sql.FormatString(uid);
		if(!strUID.size())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	if(!NewPKIStore::X509_NAMEtoHash(reqcsr->req_info->subject, strDnHash))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(req.sprintf(CA_HANDLER_GET_CERT_UNIQUE, strDnHash.c_str(), CERT_STATE_VALID, currTime, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	
	CaLock.EnterCS();
	
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CaLock.LeaveCS();
		return false;
	}
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CaLock.LeaveCS();
		return false;
	}
	if(NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CERT_ALREADY_SIGNED_DN);
		CaLock.LeaveCS();
		return false;
	}


	
	serial = m_InternCaBody.get_caSerial();
	//We sign the request
	if(!m_CaCert.SignCSR(ResultCert, csr, Exts, Days, serial, "sha1", 
						 keepCsrExts, csrExtsOverwrite, check_sig))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CaLock.LeaveCS();
		return false;
	}
	
	
	//Preparing signature
	begin_date = ResultCert.GetStartDate();
	end_date = ResultCert.GetEndDate();

	currCert.set_caType(m_CaType);
	currCert.set_serial(serial);
	currCert.set_state(CERT_STATE_VALID);
	currCert.set_uid(uid);
	currCert.set_beginDate(begin_date);
	currCert.set_endDate(end_date);
	currCert.set_revDate(0);
	if(!currCert.set_cert(ResultCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CaLock.LeaveCS();
		return false;
	}
	if(!get_InternalCertSig(currCert, signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CaLock.LeaveCS();
		return false;
	}
	 
	//Ok let's do the insertion in database
	if(req.sprintf(CA_HANDLER_INSERT_CERT, m_CaType, serial, CERT_STATE_VALID, strDnHash.c_str(), dn.c_str(), strUID.c_str(), begin_date, end_date, ResultCert.GetCertPEM().c_str(), signature.c_str()) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		CaLock.LeaveCS();
		return false;
	}
	dn = "";

	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CaLock.LeaveCS();
		return false;
	}
	

	//Increment serial
	serial++;
	m_InternCaBody.set_caSerial(serial);
	Flush();

	// Every 5000 certificates we request the table to be optimized
	if( (serial % 5000) == 0 )
	{
		NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimizing certificates table..."));
		time(&startTime);
		if(!sql.OptimizeTable(CA_HANDLER_CERTS_TABLE))
		{
			req = "";
			ERR_to_mstring(req);
			NewpkiDebug(LOG_LEVEL_WARNING, m_EntityName.c_str(), _sv("Failed to optimize certificates table - Reason: %s"), req.c_str());
			ERR_clear_error();
		}
		else
		{
			NewpkiDebug(LOG_LEVEL_INFO, m_EntityName.c_str(), _sv("Optimized certificates table in %ld secondes"), time(NULL) - startTime);
		}
	}

	CaLock.LeaveCS();
	return true;
}

CA_Handler::operator int()
{
	if(!m_InternCaBody || !m_CaCert || !m_CaKey || !m_DbConn)
		return 0;
	else
		return 1;
}

void CA_Handler::CleanMemory()
{
	if(!m_InternCa) return;
	CaLock.EnterCS();
	m_InternCa.get_cryptedBody().Clear();
	CaLock.LeaveCS();
}

bool CA_Handler::get_Certs(mVector<InternalCaCert> & Certs, CERT_STATE state, long index, long num)
{
	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		return false;
	}
	InternalCaCert res;
	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long i;

	
	if(state)
	{
		if(req.sprintf(CA_HANDLER_GET_CERTS_BY_STATE, state, m_CaType) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
	}
	else
	{
		if(req.sprintf(CA_HANDLER_GET_CERTS, m_CaType) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
	}
	if(num)
	{
		req += " LIMIT ";
		req += index;
		req += ",";
		req += num;
	}
	req += ";";
	
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	for(i=0; i<NumRows; i++)
	{
		//Convert the sql result to the internal struct	
		Certs.insert(Certs.begin()+i);
		ERR_clear_error();
		if(!SQL2InternalCert(&sql, i, Certs[i]))
		{
			ERR_to_mstring(req);
			Certs[i].set_caType(m_CaType);
			Certs[i].set_serial(0);
			Certs[i].set_state(CERT_STATE_UNKNOWN);
			Certs[i].set_uid("");
			Certs[i].set_beginDate(0);
			Certs[i].set_endDate(0);
			Certs[i].set_revDate(0);
			Certs[i].set_dn(req);
		}
	}
	
	return true;
}

bool CA_Handler::get_Crls(mVector<PKI_CRL> & Crls, long index, long num)
{
	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		return false;
	}

	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	long i;
	int v_index;

	
	if(req.sprintf(CA_HANDLER_GET_CRLS, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(num)
	{
		req += " LIMIT ";
		req += index;
		req += ",";
		req += num;
	}
	req += ";";
	
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	v_index = 0;
	for(i=0; i<NumRows; i++)
	{
		//Convert the sql result to the internal struct	
		if(!sql.Value(i, "crl", req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		Crls.insert(Crls.begin() + v_index);
		if(!Crls[v_index].SetCRL(req.c_str())) continue;
		v_index++;
	}
	
	return true;
}

bool CA_Handler::get_Cert(unsigned long serial, InternalCaCert & cert)
{
	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		return false;
	}
	
	long NumRows;
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	mString req;
	

	//Get cert
	if(req.sprintf(CA_HANDLER_GET_CERT, serial, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!NumRows)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NO_CERT);
		return false;
	}

	//Convert the sql result to the internal struct	
	if(!SQL2InternalCert(&sql, 0, cert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}	
	return true;
}

bool CA_Handler::Revoke(unsigned long serial, PKI_CERT & Cert, CERT_STATE & CertStatus, time_t & rev_date, mString & uid)
{
	InternalCaCert currCert;
	mString signature;
	mString req;
	int tmp_enum;
	time_t revdate;

	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}
	
	//Might be a, initial CRL generation
	if(!serial)
	{
		CertStatus = CERT_STATE_UNKNOWN;
		return true;
	}
	
	if(!get_Cert(serial, currCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}

	if(! (Cert = currCert.get_cert()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}
	
	//We verify that the certificate hasn't already been revoked
	if(currCert.get_state() != CERT_STATE_VALID && 
		currCert.get_state() != CERT_STATE_SUSPENDED)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_CERT_STATE);
		CertStatus = (CERT_STATE)currCert.get_state();
		return false;
	}
	
	//We update its status
	currCert.set_state(CERT_STATE_REVOKED);
	time_gmt(&revdate);
	rev_date = revdate;
	uid = currCert.get_uid();
	currCert.set_revDate(rev_date);
	
	if(!get_InternalCertSig(currCert, signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_VALID;
		return false;
	}
	
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
		
	//We update it in the database	
	tmp_enum = (int)CERT_STATE_REVOKED;
	if(req.sprintf(CA_HANDLER_REV_CERT, tmp_enum, signature.c_str(), rev_date, serial, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		CertStatus = CERT_STATE_VALID;
		return false;
	}

	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_VALID;
		return false;
	}
	CertStatus = CERT_STATE_REVOKED;
	return true;
}

bool CA_Handler::Suspend(unsigned long serial, PKI_CERT & Cert, CERT_STATE & CertStatus, time_t & susp_date, mString & uid)
{
	InternalCaCert currCert;
	mString signature;
	mString req;
	int tmp_enum;
	time_t suspdate;

	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}
	
	//Might be a, initial CRL generation
	if(!serial)
	{
		CertStatus = CERT_STATE_UNKNOWN;
		return true;
	}
	
	if(!get_Cert(serial, currCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}

	if(! (Cert = currCert.get_cert()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}
	
	//We verify that the certificate hasn't already been revoked
	if(currCert.get_state() != CERT_STATE_VALID)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_CERT_STATE);
		CertStatus = (CERT_STATE)currCert.get_state();
		return false;
	}
	
	//We update its status
	currCert.set_state(CERT_STATE_SUSPENDED);
	time_gmt(&suspdate);
	susp_date = suspdate;
	uid = currCert.get_uid();
	currCert.set_suspDate(susp_date);
	
	if(!get_InternalCertSig(currCert, signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_VALID;
		return false;
	}
	
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
		
	//We update it in the database	
	tmp_enum = (int)CERT_STATE_SUSPENDED;
	if(req.sprintf(CA_HANDLER_SUSP_CERT, tmp_enum, signature.c_str(), susp_date, serial, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		CertStatus = CERT_STATE_VALID;
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_VALID;
		return false;
	}
	CertStatus = CERT_STATE_SUSPENDED;
	return true;
}

bool CA_Handler::Unsuspend(unsigned long serial, PKI_CERT & Cert, CERT_STATE & CertStatus, mString & uid)
{
	InternalCaCert currCert;
	mString signature;
	mString req;
	int tmp_enum;

	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}
	
	//Might be a, initial CRL generation
	if(!serial)
	{
		CertStatus = CERT_STATE_UNKNOWN;
		return true;
	}
	
	if(!get_Cert(serial, currCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}

	if(! (Cert = currCert.get_cert()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_UNKNOWN;
		return false;
	}
	
	//We verify that the certificate is suspended
	if(currCert.get_state() != CERT_STATE_SUSPENDED)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_CERT_STATE);
		CertStatus = (CERT_STATE)currCert.get_state();
		return false;
	}
	
	//We update its status
	currCert.set_state(CERT_STATE_VALID);
	uid = currCert.get_uid();
	currCert.set_suspDate(0);
	
	if(!get_InternalCertSig(currCert, signature))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_SUSPENDED;
		return false;
	}
	
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
		
	//We update it in the database	
	tmp_enum = (int)CERT_STATE_VALID;
	if(req.sprintf(CA_HANDLER_UNSUSP_CERT, tmp_enum, signature.c_str(), serial, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		CertStatus = CERT_STATE_SUSPENDED;
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CertStatus = CERT_STATE_SUSPENDED;
		return false;
	}
	CertStatus = CERT_STATE_VALID;
	return true;
}

bool CA_Handler::Generate_CRL(PKI_CRL & resCrl, int Days, int Hours, const HashTable_String * Exts, const char * md, bool InsertCRL)
{
	mString tmpdatas;
	mString rev_date;
	REVOCATION_INFO rev_inf;
	SQL sql(m_DbConn, SQL_ACCESS_WRITE);
	long NumRows;
	long i;
	mString req;

	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		return false;
	}
	if((!Days && !Hours) || ! md)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	
	resCrl.Clear();

	
	//Get revoked certs
	if(req.sprintf(CA_HANDLER_GET_CERT_BY_STATE, CERT_STATE_REVOKED, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	//Add them to the CRL
	for(i=0; i < NumRows; i++)
	{
		if(!sql.Value(i, "serial", tmpdatas))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		rev_inf.serial = tmpdatas.c_ulng();

		if(!sql.Value(i, "rev_date", tmpdatas))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		rev_inf.rev_date = tmpdatas.c_ulng();
		rev_inf.reason = OCSP_REVOKED_STATUS_NOSTATUS;
		
		resCrl.AddRevokedCert(&rev_inf);
	}


	//Get suspended certs
	if(req.sprintf(CA_HANDLER_GET_CERT_BY_STATE, CERT_STATE_SUSPENDED, m_CaType) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	if(!sql.Execute(req))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!sql.NumRows(&NumRows))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	//Add them to the CRL
	for(i=0; i < NumRows; i++)
	{
		if(!sql.Value(i, "serial", tmpdatas))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		rev_inf.serial = tmpdatas.c_ulng();

		if(!sql.Value(i, "susp_date", tmpdatas))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		rev_inf.rev_date = tmpdatas.c_ulng();
		rev_inf.reason = OCSP_REVOKED_STATUS_CERTIFICATEHOLD;
		
		resCrl.AddRevokedCert(&rev_inf);
	}

	CaLock.EnterCS();
	if(!resCrl.Generate(m_CaCert, md, (Days * 24) + Hours, Exts))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		CaLock.LeaveCS();
		return false;
	}
	if(InsertCRL)
	{
		//Save last crl
		if(!m_InternCaBody.set_lastcrl(resCrl))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			CaLock.LeaveCS();
			return false;
		}
		if(!Flush())
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			CaLock.LeaveCS();
			return false;
		}
	}
	CaLock.LeaveCS();
	
	
	if(InsertCRL)
	{
		//We insert the new CRL in the database
		if(req.sprintf(CA_HANDLER_INSERT_CRL, m_CaType, resCrl.GetPemCRL().c_str()) <= 0)
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
		}
		if(!sql.Execute(req))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}

	return true;
}

bool CA_Handler::get_InternalCertSig(const InternalCaCert & cert, mString & sig_pem)
{
	InternalCaCertSig sig;
	const EVP_PKEY * privateKey;
	INTERNAL_CA_CERT * lCert;
	
	privateKey = m_CaKey.GetRsaKey();
	if(!privateKey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}

	lCert=NULL;
	if(!cert.give_Datas(&lCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		if(lCert) ASN1_item_free((ASN1_VALUE*)lCert, InternalCaCert::get_ASN1_ITEM());
		return false;
	}
	
	//We calculate the cert signature
	if(INTERNAL_CA_CERT_sign(lCert, sig, (EVP_PKEY*)privateKey) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)lCert, InternalCaCert::get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)lCert, InternalCaCert::get_ASN1_ITEM());
	
	//We convert the signature to PEM
	if(!sig.to_PEM(sig_pem))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool CA_Handler::verify_InternalCertSig(const InternalCaCert & cert, const char * sig_pem)
{
	InternalCaCertSig sig;
	const EVP_PKEY * privateKey;
	INTERNAL_CA_CERT * lCert;


	privateKey = m_CaKey.GetRsaKey();
	if(!privateKey)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		return false;
	}
		
	//We convert the signature from PEM
	if(!sig.from_PEM(sig_pem))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	lCert=NULL;
	if(!cert.give_Datas(&lCert))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		if(lCert) ASN1_item_free((ASN1_VALUE*)lCert, InternalCaCert::get_ASN1_ITEM());
		return false;
	}
	
	//We verify the cert signature
	if(INTERNAL_CA_CERT_verify(lCert, sig, (EVP_PKEY *)privateKey) <= 0)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ASN1_item_free((ASN1_VALUE*)lCert, InternalCaCert::get_ASN1_ITEM());
		return false;
	}
	ASN1_item_free((ASN1_VALUE*)lCert, InternalCaCert::get_ASN1_ITEM());
	
	return true;
}



bool CA_Handler::SQL2InternalCert(SQL * sql, long i, InternalCaCert & cert, bool checkSig)
{
	mString strdatas;
	
	if(!sql->Value(i, "ca_type", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_caType(strdatas.c_ulng());

	if(!sql->Value(i, "serial", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_serial(strdatas.c_ulng());

	if(!sql->Value(i, "state", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_state(strdatas.c_ulng());

	if(!sql->Value(i, "dn", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_dn(strdatas);

	if(!sql->Value(i, "uid", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_uid(strdatas);

	if(!sql->Value(i, "begin_date", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_beginDate(strdatas.c_ulng());

	if(!sql->Value(i, "end_date", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_endDate(strdatas.c_ulng());
	
	if(!sql->Value(i, "rev_date", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_revDate(strdatas.c_ulng());
	
	if(!sql->Value(i, "susp_date", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	cert.set_suspDate(strdatas.c_ulng());
	
	if(!sql->Value(i, "cert", strdatas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!cert.get_cert().SetCert(strdatas.c_str()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(checkSig)
	{
		//now we verify the signature of the thing !
		if(!sql->Value(i, "signature", strdatas))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
		if(!verify_InternalCertSig(cert, strdatas.c_str()))
		{
			NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
			return false;
		}
	}
	return true;
}

bool CA_Handler::GEN_PRIVATE_KEY_load(const GenPrivateKey & KeyGen, PKI_RSA & Key, InternalCaKey &InternalKey, ENGINE * e)
{
	mString keyid;
	int rsalen;

	switch(KeyGen.get_type())
	{
		case GEN_PRIVATE_KEY_TYPE_ENGINE:
			keyid = KeyGen.get_keyid();
			if(!keyid.size())
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!Key.SetKey(keyid, e))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!InternalKey.set_type(INTERNAL_CA_KEY_TYPE_ENGINE))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			InternalKey.set_keyid(keyid);
			break;
	
		case GEN_PRIVATE_KEY_TYPE_KEYLEN:
			rsalen = (int)KeyGen.get_keylen();
			if(!rsalen)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!Key.GenerateKey(rsalen, e))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!InternalKey.set_type(INTERNAL_CA_KEY_TYPE_KEY))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			if(!InternalKey.set_privkey(Key.GetRSA()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;

		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
			return false;
			break;
	}

	return true;
}

bool CA_Handler::INTERNAL_CA_KEY_set(InternalCaKey & InternalKey, const PKI_RSA & Key)
{
	if(!InternalKey.set_type(INTERNAL_CA_KEY_TYPE_KEY))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!InternalKey.set_privkey(Key.GetRSA()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool CA_Handler::INTERNAL_CA_KEY_load(const InternalCaKey & ca_key, PKI_RSA & key, ENGINE * e)
{
	switch(ca_key.get_type())
	{
		case INTERNAL_CA_KEY_TYPE_KEY:
			if(!key.SetKey(ca_key.get_privkey()))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		case INTERNAL_CA_KEY_TYPE_ENGINE:
			if(!key.SetKey(ca_key.get_keyid(), e))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			break;
		default:
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			return false;
	}
	return true;
}

bool CA_Handler::SetDbConn(const SQL_Connection *DbConn)
{
	if(m_DbConn)
	{
		delete m_DbConn;
		m_DbConn = NULL;
	}

	try
	{
		m_DbConn = DbConn->Clone();
	}
	catch(ExceptionNewPKI e)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_DbConn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		return false;
	}
	return true;
}

bool CA_Handler::GetLastCrl(PKI_CRL & crl)
{
	bool res;
	
	if(!*this)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_CA_NOT_CONFIGURED);
		return false;
	}

	CaLock.EnterCS();
	if(m_InternCaBody.get_lastcrl())
	{
		if(!(crl = m_InternCaBody.get_lastcrl()))
			res = false;
		else
			res = true;
	}
	else
	{
		res = false;
	}
	CaLock.LeaveCS();
	return res;
}

bool CA_Handler::Upgrade(const char * Version, SQL_Connection* DbConn)
{
	SQL sql(DbConn, SQL_ACCESS_WRITE);

	m_Version = Version;
	while(strcmp(Version, NEWPKI_VERSION) != 0)
	{
		if(strcmp(Version, "2.0.0-beta4") == 0)
		{
			if(!sql.Execute("ALTER TABLE `internal_certs` ADD `susp_date` INT UNSIGNED NOT NULL AFTER `rev_date`;"))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			Version = "2.0.0-rc1";
		}
	}
	return true;
}

bool CA_Handler::Private_Upgrade(const char * Version)
{
	SQL sql(m_DbConn, SQL_ACCESS_READ);
	SQL sql2(m_DbConn, SQL_ACCESS_WRITE);
	InternalCaCert res;
	long NumRows;
	mString req;
	long i;
	unsigned long id;
	mString signature;

	while(strcmp(Version, NEWPKI_VERSION) != 0)
	{
		if(strcmp(Version, "2.0.0-beta4") == 0)
		{
			if(req.sprintf("SELECT * FROM internal_certs WHERE ca_type='%d';", m_CaType) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
			// Upgrade the signature off all the entries
			if(!sql.Execute(req))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			
			if(!sql.NumRows(&NumRows))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				return false;
			}
			for(i=0; i<NumRows; i++)
			{
				if(!SQL2InternalCert(&sql, i, res, false))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				if(!sql.Value(i, "unique_id", req))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
				id = req.c_ulng();
				if(!get_InternalCertSig(res, signature))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				} 
				//Ok let's do the insertion in database
				if(req.sprintf("UPDATE internal_certs SET signature='%s' WHERE unique_id='%ld';", signature.c_str(), id) <= 0)
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					return false;
				}
				if(!sql2.Execute(req))
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
					return false;
				}
			}
			Version = "2.0.0-rc1";
		}
	}
	return true;
}
