/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/

 // Entity_EE.cpp: implementation of the Entity_EE class.
//
//////////////////////////////////////////////////////////////////////

#include "Entity_EE.h"
#include "svintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

Entity_EE::Entity_EE(ENTITY_CONSTRUCTOR_PARAMETERS):
			Entity(ENTITY_CONSTRUCTOR_PARAM_PASSTHRU, &myConf, &lStore,
				ASYNCHMSGS_TYPE_REQUESTER | ASYNCHMSGS_TYPE_RESPONDER | ASYNCHMSGS_RESPONDER_ASYNCH),
			lStore(EntityName, e)
{
	hThreadNotifyUsers.Create(ThreadNotifyUsers, this);
}

Entity_EE::~Entity_EE()
{
	hThreadNotifyUsers.Stop();
	m_Jobs.StopAll();
}

bool Entity_EE::Create(const EntityCreationDatas & Params, AdminResponseBody & response)
{
	if(!Params)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(Params.get_type() != ENTITY_TYPE_EE)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	if(!Params.get_entityKey())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}

	//We create the database
	if(!Common_Create(Params.get_entityKey(), NULL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.get_creEntity().set_type(ENTITY_TYPE_EE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Destroy();
		return false;
	}
	if(!response.get_creEntity().set_entityPubKey(m_EntityKey.GetPublicKey()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_MALLOC);
		Destroy();
		return false;
	}

	return true;
}

bool Entity_EE::Load()
{
	if(!Common_Load())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!hThreadNotifyUsers.Start())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!IsFullyInit())
	{
		return true;
	}
	return true;
}

bool Entity_EE::Init(const EntitySignatureResp & init_datas)
{
	if(IsFullyInit())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!init_datas)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(init_datas.get_body().get_type() != ENTITY_TYPE_EE)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PARAM);
		return false;
	}
	if(!Common_Init(init_datas.get_body().get_entitycert(), init_datas))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_EE::LoginUser(UserHandle & hUser, int & UserType)
{
	if( hUser.GetUserName() != "webuser" && 
		!AclValidator.CanUserPerform(hUser.GetUserCert(), ACL_TYPE_AUTHENTICATE_ON_ENTITY))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	UserType = USER_TYPE_EE;
	return true;
}

void Entity_EE::LogoutUser(const UserHandle & hUser)
{
}

bool Entity_EE::ParseNewConf()
{
	bool ret;
	mString Err;

	if(!m_Policies.From_POLICY_VALUE(myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_policies()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	lStore.set_SiteName(myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_siteName());

	ret = lStore.SetOptions(
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_dnSpecs(),
				m_Policies,
				(ASN1_BIT_STRING_get_bit(myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_LDAP_AUTHENTICATE) == 1),
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapServer(),
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapPort(),
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapUsername(),
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapPassword(),
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapBase(),
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapAttrName(),
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapFilters(),
				myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapUtf8());

	if(ASN1_BIT_STRING_get_bit(myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_LDAP_AUTHENTICATE) == 1)
	{
		if(!ret)
		{
			ERR_to_mstring(Err);
			m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LOG_MESSAGE_TYPE_LDAP_CONNECTION, 0, myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapUsername().c_str(), LOG_NO_OBJECTID, myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapServer().c_str(), Err.c_str());
		}
		else
		{
			m_Logging->LogMessage(LOG_STATUS_TYPE_SUCCESS, LOG_MESSAGE_TYPE_LDAP_CONNECTION, 0, myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapUsername().c_str(), LOG_NO_OBJECTID, myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_ldapServer().c_str());
		}
	}

	return true;
}

bool Entity_EE::Upgrade(const char * Version)
{
	if(!Entity::Common_Upgrade(Version))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}


void Entity_EE::GetACL_List(mVector<unsigned long> & acl_list)
{
	int i;
	static ACL_TYPE list_acls[] =
	{
		ACL_TYPE_VIEW_LOGS,
		ACL_TYPE_EE_MANAGE_PROFILES,
		(ACL_TYPE)0
	};
	for(i=0; list_acls[i]; i++)
	{
		acl_list.push_back(list_acls[i]);
	}	
}


bool Entity_EE::ParseAdminCommand(AdminResponseBody & response, const PKI_CERT & ClientCert, const AdminRequest & AdminRequest)
{
	return Private_ParseAdminCommand(true, this, LogsType, response, ClientCert, AdminRequest, GetUserHandle());
}

bool Entity_EE::Private_ParseAdminCommand(bool ExecuteCmd, Entity * me_this, mVector<unsigned long> &  mLogsType, AdminResponseBody & response, const PKI_CERT & ClientCert, const AdminRequest & AdminRequest, UserHandle & hUser)
{	
	PARSER_COMMAND_BEGIN(Entity_EE, response, 0, AdminRequest, ClientCert, hUser, ExecuteCmd, me_this, mLogsType)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_ENTITY_GET_MY_CONF)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_SEND_MAIL)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_EE_DN_REMOVE)
		PARSER_ADD_LOG_ENTRY(LOG_MESSAGE_TYPE_EE_VAL_REQUEST)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_LOGIN,					Entity_EE::UserLogin,				LOG_MESSAGE_TYPE_USER_LOGIN, (ClientCert)?(char*)ClientCert.GetStringName():AdminRequest.get_body().get_login().get_username().c_str(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_ENUM_LOGS,				Entity_EE::EnumLogs)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_LOGS_COUNT,			Entity_EE::GetLogsCount)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_MY_ACL,				Entity_EE::GetMyACL)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_LOGS_TYPE,			Entity_EE::GetLogsType)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_CHECK_LOGS,				Entity_EE::CheckLogsIntegrity)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_DN_VALIDATION,			Entity_EE::ValidateDn,				LOG_MESSAGE_TYPE_DN_VALIDATION,	_sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_GET_DN_VALIDATION,		Entity_EE::GetValidateDn)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_GET_LOCAL_CONF,			Entity_EE::GetLocalConf,			LOG_MESSAGE_TYPE_GET_LOCAL_CONF,	_sv("none"), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_WEBUSER_CREATE,			Entity_EE::WebUserCreate,			LOG_MESSAGE_TYPE_WEB_USER_CREATE,	body.get_webcreate().get_email().c_str(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_WEBUSER_LOGIN,			Entity_EE::WebUserLogin,			LOG_MESSAGE_TYPE_WEB_USER_LOGIN,	body.get_weblogin().get_email().c_str(), LOG_NO_OBJECTID)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_WEBUSER_ACTIVATE,		Entity_EE::WebUserActivate)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_EE_REQUEST_CERT,			Entity_EE::WebUserRequestCert,		LOG_MESSAGE_TYPE_CERT_REQUEST,		"", body.get_eeCertRequest().get_profileId())
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_WEBUSER_ENUM_CERTS,		Entity_EE::WebUserEnumCerts)
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_WEBUSER_GET_DN,			Entity_EE::WebUserGetDn)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_EE_REVOKE_CERT,			Entity_EE::WebUserRevokeCert,		LOG_MESSAGE_TYPE_CERT_REVOCATION,	"", body.get_eeCertRevoke().get_certId())
		PARSER_COMMAND_ENTRY(		ADMIN_REQ_TYPE_ENUM_EE_USERS,			Entity_EE::EnumProfiles)
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_DELETE_PKCS12,			Entity_EE::DeletePkcs12,			LOG_MESSAGE_TYPE_DELETE_PKCS12,	"", body.get_serial())
		PARSER_COMMAND_ENTRY_LOG(	ADMIN_REQ_TYPE_WEBUSER_CHANGE_PASSWD,	Entity_EE::WebUserChangePassword,	LOG_MESSAGE_TYPE_PWD_MOD,	"", body.get_webChgPasswd().get_userId())
	PARSER_COMMAND_END(Entity_EE)
}

void Entity_EE::LogsTypeGet(mVector<unsigned long> &  cLogsType)
{
	Private_ParseAdminCommand(false, NULL, cLogsType, AdminResponseBody::EmptyInstance, PKI_CERT::EmptyInstance, AdminRequest::EmptyInstance, UserHandle::EmptyInstance);
}

bool Entity_EE::ValidateDn(COMMAND_PARAMETERS)
{
	unsigned long ra_id;
	NewpkiResponse resp;
	Asn1OctetString id;
	HashTable_Dn hDn;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if( hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// We now make sure the DN respects the policies
	if(!hDn.From_X509_NAME(body.get_dnVal().get_dn()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.LockRead();
	if(!hDn.ValidateAgainstPolicy(m_Policies))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!NewPKIStore::StringtoTransactionID(body.get_dnVal().get_id(), id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.GetDnValidationRequestRaId(id, ra_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!resp.set_type(NEWPKI_RESPONSE_TYPE_EE_VAL))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!resp.get_eeValResponse().set_dn(body.get_dnVal().get_dn()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!resp.get_eeValResponse().set_raId(ra_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	if(!InsertResponse(id, resp, 
		myConf.get_conf().get_ras().get_list()[0].get_rassl().GetX509_PUBKEY()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();
	lStore.DeleteDnValidationRequest(id);

	return true;
}

bool Entity_EE::GetValidateDn(COMMAND_PARAMETERS)
{
	X509_NAME * dn;
	Asn1OctetString id;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if( hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_DN_VALIDATION))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!NewPKIStore::StringtoTransactionID(body.get_dnId(), id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	dn = lStore.GetDnValidationRequest(id);
	if(!dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!response.set_dn(dn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_EE::GetLocalConf(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(hUser.GetUserName() != "webuser" &&
		AclValidator.ValidateCert(UserCert) != INTERNAL_CA_TYPE_USER)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_LOCAL_ENTITY_CONF))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	if( !response.set_localEntityConf(myConf.get_conf()) )
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();

	// We don't send the LDAP info, those are very private datas
	response.get_localEntityConf().get_body().get_eeConf().EE_CONF_PTR.set_ldapServer("");
	response.get_localEntityConf().get_body().get_eeConf().EE_CONF_PTR.set_ldapUsername("");
	response.get_localEntityConf().get_body().get_eeConf().EE_CONF_PTR.set_ldapPassword("");
	response.get_localEntityConf().get_body().get_eeConf().EE_CONF_PTR.set_ldapBase("");
	response.get_localEntityConf().get_body().get_eeConf().EE_CONF_PTR.set_ldapPort(0);
	response.get_localEntityConf().get_body().get_eeConf().EE_CONF_PTR.set_ldapFilters("");
	response.get_localEntityConf().get_body().get_eeConf().EE_CONF_PTR.set_ldapAttrName("");
	return true;
}

bool Entity_EE::WebUserCreate(COMMAND_PARAMETERS)
{
	unsigned long user_id;

	ConfAccessLock.LockRead();
	if(ASN1_BIT_STRING_get_bit(myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_LDAP_AUTHENTICATE) == 1)
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	ConfAccessLock.UnlockRead();

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	user_id = lStore.WebUserCreate(body.get_webcreate().get_email(), 
								   body.get_webcreate().get_password(),
								   body.get_webcreate().get_dn());
	if(!user_id)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}

bool Entity_EE::WebUserLogin(COMMAND_PARAMETERS)
{
	unsigned long user_id;
	unsigned long created_user_id = 0;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_WEBUSER_ID))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(ASN1_BIT_STRING_get_bit((ASN1_BIT_STRING*)myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_flags(), 
								EE_LDAP_AUTHENTICATE))
	{
		// If user if not known it we be created
	}

	user_id = lStore.WebUserLogin(body.get_weblogin().get_email(), 
							 body.get_weblogin().get_password(), 
							 created_user_id);
	if(!user_id)
	{
		// Was the user created ?
		if(created_user_id)
		{
			if(!WebUserSendDnToRa(created_user_id))
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
				lStore.WebUserDelete(created_user_id);
				return false;
			}
		}
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	response.set_id(user_id);

	return true;
}

bool Entity_EE::WebUserSendDnToRa(unsigned long user_id)
{
	const X509_PUBKEY * recipient=NULL;
	X509_NAME * dn;
	NewpkiRequest Request;

	// Generate the request
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_EE_DN_PUBLISH))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	if(!myConf.get_conf().get_ras().get_list().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_add_error_data(1, _sv("No RA has been associated to this EE"));
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();

	dn = lStore.WebUserGetUserDn(user_id);
	if(!dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	ConfAccessLock.LockRead();
	recipient = myConf.get_conf().get_ras().get_list()[0].get_rassl().GetX509_PUBKEY();
	if(!recipient)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		X509_NAME_free(dn);
		return false;
	}
	Request.get_eeDnPublish().set_eeId(user_id);
	if(!Request.get_eeDnPublish().set_dn(dn))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		X509_NAME_free(dn);
		return false;
	}
	X509_NAME_free(dn);
	if(!InsertRequest(user_id, Request, recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();
	return true;
}

bool Entity_EE::WebUserActivate(COMMAND_PARAMETERS)
{
	unsigned long user_id;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.WebUserGetUserId(body.get_webactivate(), user_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.WebUserActivate(body.get_webactivate()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!WebUserSendDnToRa(user_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		lStore.WebUserDeactivate(body.get_webactivate());
		return false;
	}
	return true;
}

bool Entity_EE::WebUserEnumCerts(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_WEBUSER_CERTS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.WebUserEnumCerts(body.get_profileId(), response.get_webuserCerts()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_EE::WebUserRequestCert(COMMAND_PARAMETERS)
{
	NewpkiRequest Request;
	size_t i;
	const X509_PUBKEY * recipient=NULL;
	unsigned long CertReqId;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	// Is the type set ?
	if(!body.get_eeCertRequest().get_type())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}
	ConfAccessLock.LockRead();
	if(!myConf.get_conf().get_ras().get_list().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_add_error_data(1, _sv("No RA has been associated to this EE"));
		ConfAccessLock.UnlockRead();
		return false;
	}

	// Are we allowed to request certs from this CA ?
	for(i=0; i < myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_cas().size(); i++)
	{
		if(myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_cas()[i] == body.get_eeCertRequest().get_caName())
		{
			break;
		}
	}
	// Did we find the CA ?
	if(i == myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_cas().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}


	// We now verify if the request respects the policy constraints
	if(!ValidateRequestConstraints(ENTITY_TYPE_EE, body.get_eeCertRequest().get_type(),
									body.get_eeCertRequest().get_request(), 
									myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_minpwdlen(),
									myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_minkeylen(),
									myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_flags()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();


	// Generate the requests
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_EE_CERT))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!Request.set_eeCertRequest(body.get_eeCertRequest()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

		

	if(!lStore.WebUserInsertCertReq(body.get_eeCertRequest(), CertReqId))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	ConfAccessLock.LockRead();
	recipient = myConf.get_conf().get_ras().get_list()[0].get_rassl().GetX509_PUBKEY();
	if(!recipient)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		lStore.WebUserDeleteCertReq(CertReqId);
		return false;
	}
	
	if(!InsertRequest(CertReqId, Request, recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		lStore.WebUserDeleteCertReq(CertReqId);
		return false;
	}
	ConfAccessLock.UnlockRead();

	return true;
}

bool Entity_EE::WebUserRevokeCert(COMMAND_PARAMETERS)
{
	NewpkiRequest Request;
	const X509_PUBKEY * recipient=NULL;
	unsigned long ra_id;

	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	ConfAccessLock.LockRead();
	if(!myConf.get_conf().get_ras().get_list().size())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ERR_add_error_data(1, _sv("No RA has been associated to this EE"));
		ConfAccessLock.UnlockRead();
		return false;
	}
	ConfAccessLock.UnlockRead();


	// Generate the requests
	if(!Request.set_type(NEWPKI_REQUEST_TYPE_EE_REV))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}	

	if(!lStore.WebUserMarkCertRev(body.get_eeCertRevoke().get_userId(), 
								  body.get_eeCertRevoke().get_certId(), ra_id))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	Request.get_eeCertRevoke().set_certId(ra_id);

	ConfAccessLock.LockRead();
	recipient = myConf.get_conf().get_ras().get_list()[0].get_rassl().GetX509_PUBKEY();
	if(!recipient)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		lStore.WebUserUnmarkCertRev(body.get_eeCertRevoke().get_certId());
		return false;
	}
	
	if(!InsertRequest(body.get_eeCertRevoke().get_certId(), Request, recipient))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		ConfAccessLock.UnlockRead();
		lStore.WebUserUnmarkCertRev(body.get_eeCertRevoke().get_certId());
		return false;
	}
	ConfAccessLock.UnlockRead();
	return true;
}

bool Entity_EE::WebUserGetDn(COMMAND_PARAMETERS)
{
	X509_NAME * dn;
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(hUser.GetUserName() != "webuser")
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!response.set_type(ADMIN_RESP_TYPE_DN))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	dn = lStore.WebUserGetDn(body.get_profileId());
	if(!dn)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!response.set_dn(dn))
	{
		X509_NAME_free(dn);
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	X509_NAME_free(dn);
	return true;
}

bool Entity_EE::WebUserChangePassword(COMMAND_PARAMETERS)
{
	bool check_pwd;
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	ConfAccessLock.LockRead();
	if(ASN1_BIT_STRING_get_bit(myConf.get_conf().get_body().get_eeConf().EE_CONF_PTR.get_flags(), EE_LDAP_AUTHENTICATE) == 1)
	{
		ConfAccessLock.UnlockRead();
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	ConfAccessLock.UnlockRead();

	if(hUser.GetUserName() == "webuser")
	{
		check_pwd = true;
	}
	else if(AclValidator.CanUserPerform(UserCert, ACL_TYPE_EE_MANAGE_PROFILES))
	{
		check_pwd = false;
	}
	else
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}
	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!lStore.WebUserChangePassword(body.get_webChgPasswd(), check_pwd))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_EE::EnumProfiles(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!AclValidator.CanUserPerform(UserCert, ACL_TYPE_EE_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_EE_USERS))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	
	if(!lStore.EnumWebUsers(response.get_eeUsers(), 
							body.get_enumObjects().get_index(), 
							body.get_enumObjects().get_num(), 
							body.get_enumObjects().get_state(), 
							body.get_enumObjects().get_filter()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	return true;
}

bool Entity_EE::DeletePkcs12(COMMAND_PARAMETERS)
{
	if(!hUser)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(hUser.GetUserName() != "webuser" &&
		!AclValidator.CanUserPerform(UserCert, ACL_TYPE_EE_MANAGE_PROFILES))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		return false;
	}

	if(!response.set_type(ADMIN_RESP_TYPE_NONE))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!lStore.DeletePKCS12(body.get_serial()))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	return true;
}








bool Entity_EE::Requester_OnNewResponse(const Asn1OctetString & transactionID, const X509_PUBKEY * sender, const NewpkiResponse & Response)
{
	mString sender_name;
	size_t i;
	mVector< ErrorEntry > errors;
	unsigned long priv_attr;
	unsigned long user_id;
	mString strError;
	MailInfo NoticeMail;
	mString body;
	mString email;
	int Type;

	if(!GetPrivAttr(transactionID, priv_attr))
	{
		return false;
	}
	if(!GetRequestType(transactionID, Type))
	{
		return false;
	}

	ConfAccessLock.LockRead();
	// Now search the RA, to make sure it is the real signer !
	for(i=0; i < myConf.get_conf().get_ras().get_list().size(); i++)
	{
		// Are the public keys the same ?
		if(myConf.get_conf().get_ras().get_list()[i].get_rassl() == sender)
		{
			sender_name = myConf.get_conf().get_ras().get_list()[i].get_name();
			break;
		}
		
	}
	// We didn't find the RA
	if(i == myConf.get_conf().get_ras().get_list().size())
	{
		ConfAccessLock.UnlockRead();
		return true;
	}
	ConfAccessLock.UnlockRead();

	if(Response.get_type() == NEWPKI_RESPONSE_TYPE_EE_DN_PUBLISH)
	{
		errors = Response.get_eeDnPublishResponse().get_errors();
		if(!errors.size())
		{
			// We can now activate the account
			if(!lStore.WebUserRaValidated(priv_attr))
			{
				return false;
			}
			if(lStore.WebUserGetEmail(priv_attr, email))
			{
				body = _sv("Your profile was validated by the RA, you may now login.");
				NoticeMail.set_MailTo(email);
				NoticeMail.set_Subject(_sv("Profile accepted"));
				NoticeMail.set_Body(body);
				NoticeMail.set_SignMail(false);
				m_Jobs.SendMail("EE Verification", NoticeMail, false);
			}
			return true;
		}
	}
	else if(Response.get_type() == NEWPKI_RESPONSE_TYPE_EE_CERT)
	{
		if(Response.get_eeCertResponse().get_status() == CERT_RESPONSE_ERROR)
		{
			if(!lStore.WebUserImportCertResponse(priv_attr, Response.get_eeCertResponse(), mVector<ErrorEntry>::EmptyInstance, user_id))
				return true;
			errors = Response.get_eeCertResponse().get_errors();
		}
		else
		{
			if(!lStore.WebUserImportCertResponse(priv_attr, Response.get_eeCertResponse(), mVector<ErrorEntry>::EmptyInstance, user_id))
				return true;

			if(lStore.WebUserGetEmail(user_id, email))
			{
				body = _sv("Your certificate is available on the site.");
				NoticeMail.set_MailTo(email);
				NoticeMail.set_Subject(_sv("Certificate signed"));
				NoticeMail.set_Body(body);
				NoticeMail.set_SignMail(false);
				m_Jobs.SendMail("EE Verification", NoticeMail, false);
			}
			return true;
		}
	}
	else if(Response.get_type() == NEWPKI_RESPONSE_TYPE_EE_REV)
	{
		if(!lStore.WebUserImportRevResponse(priv_attr, Response.get_eeRevResponse(), mVector<ErrorEntry>::EmptyInstance, user_id))
			return true;
		return true;
	}
	else if(Response.get_type() == NEWPKI_RESPONSE_TYPE_ERR)
	{
		errors = Response.get_errors();
	}

	// If we have an error
	if(errors.size())
	{
		if(Type == NEWPKI_REQUEST_TYPE_EE_DN_PUBLISH)
		{
			// we remove the entry from the DB
			// and we send the error to the user by mail
			if(lStore.WebUserGetEmail(priv_attr, email))
			{
				ERROR_ENTRIES_to_string(errors, strError);

				if(body.sprintf(_sv("Your profile was rejected by the RA for the following reason(s):\n%s"), 
							strError.c_str()) > 0)
				{
					NoticeMail.set_MailTo(email);
					NoticeMail.set_Subject(_sv("Profile was rejected"));
					NoticeMail.set_Body(body);
					NoticeMail.set_SignMail(false);
					m_Jobs.SendMail("EE Verification", NoticeMail, false);
				}
			}
			lStore.WebUserDelete(priv_attr);
		}
		else if(Type == NEWPKI_REQUEST_TYPE_EE_CERT)
		{
			// Errors have already been imported
			if(Response.get_eeCertResponse().get_status() != CERT_RESPONSE_ERROR)
			{
				if(!lStore.WebUserImportCertResponse(priv_attr, NewpkiEeCertResponse::EmptyInstance, errors, user_id))
					return true;
			}

			if(lStore.WebUserGetEmail(user_id, email))
			{
				ERROR_ENTRIES_to_string(errors, strError);

				if(body.sprintf(_sv("Your certificate wasn't signed for the following reason(s):\n%s"), 
							strError.c_str()) > 0)
				{
					NoticeMail.set_MailTo(email);
					NoticeMail.set_Subject(_sv("Certificate not generated"));
					NoticeMail.set_Body(body);
					NoticeMail.set_SignMail(false);
					m_Jobs.SendMail("EE Verification", NoticeMail, false);
				}
			}
		}
		else if(Type == NEWPKI_REQUEST_TYPE_EE_REV)
		{
			lStore.WebUserImportRevResponse(priv_attr, NewpkiEeRevResponse::EmptyInstance, errors, user_id);
		}
	}

	return true;
}

bool Entity_EE::Responder_TreatRequestAsynch(const NewpkiRequest & Request, const Asn1OctetString & transactionId, const mString & SenderName)
{
	mString StrErr;
	mString ObjectName;
	LOG_MESSAGE_TYPE LogType;
	
	StrErr = "";
	ObjectName = _sv("Unknown");

	if(ProceedWithRequest(Request, transactionId, LogType, ObjectName, StrErr))
	{
		if(StrErr.size())
			m_Logging->LogMessage(LOG_STATUS_TYPE_FAILURE, LogType, 0, SenderName.c_str(), LOG_NO_OBJECTID, ObjectName.c_str(), StrErr.c_str());
		else
			m_Logging->LogMessage(LOG_STATUS_TYPE_SUCCESS, LogType, 0, SenderName.c_str(), LOG_NO_OBJECTID, ObjectName.c_str());

		return true;
	}
	else
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

}

bool Entity_EE::ProceedWithRequest(const NewpkiRequest & req, const Asn1OctetString & transactionId, LOG_MESSAGE_TYPE & LogType, mString & ObjectName, mString & Err)
{
	char * dn_line;
	NewpkiResponse Response;

	//Proceed with the request
	switch(req.get_type())
	{
		case NEWPKI_REQUEST_TYPE_EE_DN_REMOVE:
			LogType = LOG_MESSAGE_TYPE_EE_DN_REMOVE;
			if(ObjectName.sprintf("ID: %ld", req.get_eeDnRemove().get_eeId()) <= 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
			if(!Response.set_type(NEWPKI_RESPONSE_TYPE_EE_DN_REMOVE))
			{
				ERR_to_mstring(Err);
				return false;
			}
			if(!lStore.WebUserDelete(req.get_eeDnRemove().get_eeId()))
			{
				ERR_to_ERROR_ENTRIES(Response.get_eeDnRemoveResponse().get_errors());
			}
			ConfAccessLock.LockRead();
			if(!InsertResponse(transactionId, Response, 
						myConf.get_conf().get_ras().get_list()[0].get_rassl().GetX509_PUBKEY()))
			{
				ERR_to_mstring(Err);
				ConfAccessLock.UnlockRead();
				return false;
			}
			ConfAccessLock.UnlockRead();
			break;
		case NEWPKI_REQUEST_TYPE_EE_UPDATE_CERT_STATUS:
			LogType = LOG_MESSAGE_TYPE_EE_UPD_CERT_STATE;
			if(ObjectName.sprintf("RA ID: %ld", req.get_eeUpdCertStatus().get_raId()) < 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
			if(!Response.set_type(NEWPKI_RESPONSE_TYPE_EE_UPDATE_CERT_STATUS))
			{
				ERR_to_mstring(Err);
				return false;
			}
			if(!lStore.WebUserSetStatusByRaId(req.get_eeUpdCertStatus().get_raId(),
											  req.get_eeUpdCertStatus().get_status(),
											  req.get_eeUpdCertStatus().get_lastCrl()))
			{
				ERR_to_ERROR_ENTRIES(Response.get_eeUpdCertStatusResponse().get_errors());
			}
			ConfAccessLock.LockRead();
			if(!InsertResponse(transactionId, Response, 
						myConf.get_conf().get_ras().get_list()[0].get_rassl().GetX509_PUBKEY()))
			{
				ERR_to_mstring(Err);
				ConfAccessLock.UnlockRead();
				return false;
			}
			ConfAccessLock.UnlockRead();
			break;
		case NEWPKI_REQUEST_TYPE_EE_CERT_PUBLISH:
			LogType = LOG_MESSAGE_TYPE_EE_CERT_PUBLISH;
			if(ObjectName.sprintf("RA ID: %ld - Profile ID: %ld", 
								req.get_eeCertPublish().get_raId(),
								req.get_eeCertPublish().get_profileId()) < 0)
			{
				NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
				return false;
			}
			if(!Response.set_type(NEWPKI_RESPONSE_TYPE_EE_CERT_PUBLISH))
			{
				ERR_to_mstring(Err);
				return false;
			}
			if(!lStore.WebUserPublishCert(req.get_eeCertPublish()))
			{
				ERR_to_ERROR_ENTRIES(Response.get_eeCertPublishResponse().get_errors());
			}
			ConfAccessLock.LockRead();
			if(!InsertResponse(transactionId, Response, 
						myConf.get_conf().get_ras().get_list()[0].get_rassl().GetX509_PUBKEY()))
			{
				ERR_to_mstring(Err);
				ConfAccessLock.UnlockRead();
				return false;
			}
			ConfAccessLock.UnlockRead();
			break;
		case NEWPKI_REQUEST_TYPE_EE_VAL:
			LogType = LOG_MESSAGE_TYPE_EE_VAL_REQUEST;

			dn_line = X509_NAME_oneline((X509_NAME*)req.get_eeValRequest().get_dn(), NULL, 0);
			if(dn_line)
			{
				if(ObjectName.sprintf("DN: %s", dn_line) <= 0)
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					return false;
				}
				free(dn_line);
			}
			else
			{
				if(ObjectName.sprintf("RA ID: %ld", req.get_eeValRequest().get_raId()) <= 0)
				{
					NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
					return false;
				}
			}
			if(!lStore.InsertDnValidationRequest(transactionId, req.get_eeValRequest().get_raId(), req.get_eeValRequest().get_dn()))
			{
				ERR_to_mstring(Err);
				return false;
			}
			break;
		default:
			LogType = LOG_MESSAGE_TYPE_EE_VAL_REQUEST;
			NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
			ERR_to_mstring(Err);
			return false;
			break;
	}
	return true;
}


bool Entity_EE::Responder_ValidateRequest(const NewpkiRequest & Request, const X509_PUBKEY * Requester, mString & SenderName)
{
	ConfAccessLock.LockRead();

	// We need to search that the RA has the right 
	// to access us
	if(myConf.get_conf().get_ras().get_list().size() != 1)
	{
		//This RA is not allowed to access us
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}

	//Is it the same public key
	if(!(myConf.get_conf().get_ras().get_list()[0].get_rassl() ==
		Requester))
	{
		//This RA is not allowed to access us
		NEWPKIerr(PKI_ERROR_TXT, ERROR_NOT_ALLOWED);
		ConfAccessLock.UnlockRead();
		return false;
	}

	SenderName = myConf.get_conf().get_ras().get_list()[0].get_name();
	ConfAccessLock.UnlockRead();

	// Is it the right type of request ?
	if(Request.get_type() != NEWPKI_REQUEST_TYPE_EE_VAL &&
		Request.get_type() != NEWPKI_REQUEST_TYPE_EE_DN_REMOVE &&
		Request.get_type() != NEWPKI_REQUEST_TYPE_EE_UPDATE_CERT_STATUS &&
		Request.get_type() != NEWPKI_REQUEST_TYPE_EE_CERT_PUBLISH)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_DATAS);
		return false;
	}		

	return true;
}

bool Entity_EE::PrepareConfToWrite()
{
	return true;
}

void Entity_EE::PrintInfo(FILE *out)
{

}

void Entity_EE::ThreadNotifyUsers(const NewpkiThread * Thread, void * param)
{
	Entity_EE * me_this = (Entity_EE*)param;

	while(!Thread->ShouldStop())
	{
		me_this->lStore.ResendNotificationMails();
		if(!Thread->SleepInterrupt(3600))
			break;
	}
}

