%{
/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include <string.h>
#include <stdio.h>
#include "Symbol.h"
#include "RCX_Constants.h"
#include "JumpStmt.h"
#include "Condition.h"
#include "Buffer.h"
#include "parser.h"
#include "Compiler.h"
#include "Error.h"


#define kMaxFileDepth	16

typedef struct InputFile
{
	struct InputFile	*fNext;
	Buffer*	fSourceBuffer;
	YY_BUFFER_STATE	fBufferState;
	int		fSavedLineNumber;
	int		fSavedTokenEnd;
	const char*	fDataPtr;
	int			fDataRemain;
} InputFile;

static int sFileDepth = 0;
static InputFile *sCurrentInputFile = 0;
static Buffer*	sLastBuffer = 0;
static int sLineNumber;
static int sTokenEnd = 0;
static int sReturnWhitespace = 0;
static int sReturnNL = 0;

static int FillBuffer(char *ptr, int max);

#define YY_DECL int yylex(YYSTYPE &yylval)
#define YY_USER_ACTION	sTokenEnd += yyleng;
#define YY_INPUT(buf,res,max)	(res = FillBuffer(buf, max))

#define Return(tok, val) do { yylval.fInt = val; return tok; } while(0)


%}

%option outfile="lexer.cpp"

id0			[a-zA-Z_]
idn			[a-zA-Z_0-9]
digit		[0-9]
hex			[0-9a-fA-F]
any			[^\r\n]
ws			[ \t]
nl			[\r\n]

%x COMMENT PREPROC

%%

"/*"         BEGIN(COMMENT);

"//"{any}*	;

{nl}				{ sLineNumber++; sTokenEnd = 0;  if (sReturnNL) { sReturnNL = 0; return NL; } }
\\{nl}				{ sLineNumber++; sTokenEnd = 0; }


{ws}*#{ws}*			{ if (sTokenEnd == yyleng) { BEGIN(PREPROC); sReturnNL = 1; } else return '#'; }
<PREPROC>"include"	{ BEGIN(INITIAL); return PP_INCLUDE; }
<PREPROC>"define"	{ BEGIN(INITIAL); return PP_DEFINE; }
<PREPROC>"ifdef"	{ BEGIN(INITIAL); Return(PP_IFDEF, true); }
<PREPROC>"ifndef"	{ BEGIN(INITIAL); Return(PP_IFDEF, false); }
<PREPROC>"if"		{ BEGIN(INITIAL); return PP_IF; }
<PREPROC>"else"		{ BEGIN(INITIAL); return PP_ELSE; }
<PREPROC>"elif"		{ BEGIN(INITIAL); return PP_ELIF; }
<PREPROC>"endif"	{ BEGIN(INITIAL); return PP_ENDIF; }
<PREPROC>"undef"	{ BEGIN(INITIAL); return PP_UNDEF; }
<PREPROC>"pragma"	{ BEGIN(INITIAL); return PP_PRAGMA; }
<PREPROC>{nl}		{ BEGIN(INITIAL); yyless(yyleng-1); return PP_ERROR; }
<PREPROC>.			{ BEGIN(INITIAL); return PP_ERROR; }

"if"			{ return IF; }
"else"			{ return ELSE; }
"while"			{ return WHILE; }
"do"			{ return DO; }
"repeat"		{ return REPEAT; }
"break"			{ yylval.fInt = kJump_Break; return JUMP; }
"continue"		{ yylval.fInt = kJump_Continue; return JUMP; }
"return"		{ return RETURN; }

"int"			{ return INT; }
"void"			{ return T_VOID; }
"const"			{ return T_CONST; }
"__sensor"		{ return SENSOR; }
"asm"			{ return ASM; }
"task"			{ return TASK; }
"sub"			{ return SUB; }
"inline"		{ return INLINE; }
"stop"			{ Return( TASKOP, kRCX_StopTaskOp); }
"start"			{ Return( TASKOP, kRCX_StartTaskOp); }
"abs"			{ return ABS; }
"sign"			{ return SIGN; }

"+="			{ Return( ASSIGN, kRCX_AddVar); }
"-="			{ Return( ASSIGN, kRCX_SubVar); }
"*="			{ Return( ASSIGN, kRCX_MulVar); }
"/="			{ Return( ASSIGN, kRCX_DivVar); }
"&="			{ Return( ASSIGN, kRCX_AndVar); }
"|="			{ Return( ASSIGN, kRCX_OrVar); }
"||="			{ Return( ASSIGN, kRCX_AbsVar); }
"+-="			{ Return( ASSIGN, kRCX_SgnVar); }

"=="			{ return REL_EQ; }
"!="			{ return REL_NE; }
"<="			{ return REL_LE; }
">="			{ return REL_GE; }

"&&"			{ return AND; }
"||"			{ return OR; }

"++"			{ Return( INCDEC, 1); }
"--"			{ Return( INCDEC, 0); }

"true"			{ return CTRUE; }
"false"			{ return CFALSE; }

"<<"			{ return LEFT; }
">>"			{ return RIGHT; }

{id0}{idn}*		{ yylval.fSymbol = Symbol::Get(yytext); return ID; }
0[xX]{hex}+		{ char*ptr; yylval.fInt = strtol(yytext, &ptr, 0); return NUMBER; }
{digit}+		{ yylval.fInt = (int)atof(yytext); return NUMBER; }

"\""[^\"]*"\""		{ yytext[yyleng-1]=0; yylval.fString = yytext+1; return STRING; }

{ws}+			{ if (sReturnWhitespace) return WS; }


.				{ return yytext[0]; }

<COMMENT>[^*\n\r]*        /* eat anything that's not a '*' */
<COMMENT>"*"+[^*/\n\r]*   /* eat up '*'s not followed by '/'s */
<COMMENT>{nl}           { sTokenEnd = 0; sLineNumber++; }
<COMMENT>"*"+"/"        BEGIN(INITIAL);

%%


void LexCurrentLocation(LexLocation &loc)
{
	loc.fLine = sLineNumber;
	loc.fCol = sTokenEnd - yyleng;
	loc.fLength = yyleng;
	loc.fBuffer = sCurrentInputFile ? sCurrentInputFile->fSourceBuffer : sLastBuffer;
}

void LexReturnWhitespace(int mode)
{
	sReturnWhitespace = mode;
}


void LexReturnNL(int mode)
{
	sReturnNL = mode;
}


int LexFindAndPushFile(const char *name)
{
	Buffer *b = Compiler::Get()->GetBuffer(name);
	if (!b)
	{
		Error(kErr_FileOpen, name).RaiseLex();
		return 0;
	}
	return LexPush(b);
}


int LexPush(Buffer *b)
{
	InputFile *inputFile;
	
	if (sFileDepth == kMaxFileDepth)
	{
		Compiler::Get()->ReturnBuffer(b);
		return 0;
	}
	

	// save line number and buffer in previous file
	if (sCurrentInputFile)
	{
		sCurrentInputFile->fSavedLineNumber = sLineNumber;
		sCurrentInputFile->fSavedTokenEnd = sTokenEnd;
		sCurrentInputFile->fBufferState = YY_CURRENT_BUFFER;
	}
	
	inputFile = (InputFile *)malloc(sizeof(InputFile));	
	inputFile->fBufferState = yy_create_buffer(0, YY_BUF_SIZE);
	inputFile->fDataPtr = b->GetData();
	inputFile->fDataRemain = b->GetLength();
	inputFile->fSourceBuffer = b;
	
	// link into list
	inputFile->fNext = sCurrentInputFile;
	sCurrentInputFile = inputFile;
	sFileDepth++;
	
	// switch to new buffer
	sLineNumber = 1;
	sTokenEnd = 0;
	yy_switch_to_buffer(inputFile->fBufferState);
	
	return 1;
}


int yywrap()
{
	InputFile *inputFile;

	// if no input files, just return
	if (sCurrentInputFile == 0) return 1;

	// pop an input file off the list
	sLastBuffer = sCurrentInputFile->fSourceBuffer;
	inputFile = sCurrentInputFile;
	sCurrentInputFile = inputFile->fNext;
	sFileDepth--;

	// cleanup the input file
	yy_delete_buffer(YY_CURRENT_BUFFER);
	Compiler::Get()->ReturnBuffer(inputFile->fSourceBuffer);
	free(inputFile);

	// if no more files, just return 1
	if (!sCurrentInputFile) return 1;

	// switch to next file
	yy_switch_to_buffer(sCurrentInputFile->fBufferState);
	sLineNumber = sCurrentInputFile->fSavedLineNumber;
	sTokenEnd = sCurrentInputFile->fSavedTokenEnd;

	// tell yylex() to continue
	return 0;
}


void LexReset()
{
	while(yywrap() == 0)
		;
}

int FillBuffer(char *buf, int max_size)
{
	int result;
	// if no files are pending, return 0 (EOF)
	if (!sCurrentInputFile) return 0;
	
	int n = sCurrentInputFile->fDataRemain;
	if (n > max_size)
		n = max_size;
	
	memcpy(buf, sCurrentInputFile->fDataPtr, n);
	sCurrentInputFile->fDataPtr += n;
	sCurrentInputFile->fDataRemain -= n;
	
	result = n;
	return result;
}
