// nss-multidomain: NSS module for multidomain authentification
// Copyright (C) 2001, 2002 LibertySurf Telecom, Antoniu-George Savu
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307, USA.

#include <config.h>

#if defined(HAVE_BERKELEY_DB)
#include <db.h>
#else
#error Needs Berkeley DB to compile. See http://www.sleepycat.com/
#endif /* defined(HAVE_BERKELEY_DB) */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <nss.h>
#include <paths.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/types.h>


#if defined(DB_VERSION_MAJOR)

#if (DB_VERSION_MAJOR == 3) || (DB_VERSION_MAJOR == 4)
#define DB3
#endif

#if (DB_VERSION_MAJOR == 2)
#define DB2
#endif
#else
#error Cannot determine libdb header version.
#endif

#define LOCAL_AUTH	1

/* DB variables */

#define DATABASE	"passwd"
#define LOCAL_DBFILE	"/var/db/" DATABASE ".db"

/* DB file treatement variables */
DB 			*db;
static int 		keep_db;
char			path_db_file[256];

/* buffer used to print errors with syslog */
char 			syslog_buffer[256];

char *ptr_ip;


/* Open database file if not already opened.  */
static enum nss_status internal_setent (int stayopen) {
	enum nss_status status = NSS_STATUS_SUCCESS;
        if ( getenv("VIRTUAL_DOMAIN") == (char*) NULL) {
#ifdef DEBUG
                syslog(LOG_ERR, "nss-multidomain error. VIRTUAL_DOMAIN variable not set.");
#endif /*DEBUG*/                
#ifdef LOCAL_AUTH
		strcpy(path_db_file, LOCAL_DBFILE);
#else
                status = NSS_STATUS_NOTFOUND;
#endif /* LOCAL_AUTH */
        }
        else {
		/* Warning!
		** Security consideration:
		**
		** If the $VIRTUAL_DOMAIN is something like
		** "../../../../../../../home/luser/passwd.db" any
		** user can override the passw.db file used to
		** authenticate himself. Thus, by default the module
		** refuses to return passwd structures filled with
		** uid < _MIN_UID and/or gid < _MIN_GID
		*/ 
                strcpy(path_db_file, _PATH_DB);
                strcat(path_db_file, "/");
                strcat(path_db_file, getenv("VIRTUAL_DOMAIN"));
                strcat(path_db_file, "/passwd.db");
        }

	if ((db == NULL) && (status == NSS_STATUS_SUCCESS)) { 
		int err;
#ifdef DEBUG
		sprintf(syslog_buffer, "DB file used is: %s", path_db_file);
		syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */			

		/* Prepare to open DB file */
#if defined(DB3) /* DB version 3 */
		if (db_create(&db, NULL, 0) == 0) {
			err = db->open(db, path_db_file, NULL, _DB_TYPE, DB_RDONLY, 0664);
		}
		else {
			return NSS_STATUS_UNAVAIL;
		}
#elif defined(DB2) /* DB version 2 */
		err = db_open (path_db_file, _DB_TYPE, DB_RDONLY, 0, NULL, NULL, &db);
#else /* DB version 1 */
		err = db = dbopen(path_db_file, O_RDONLY, 0664, _DB_TYPE, NULL);
#endif /* DB API compat */

		if (err != 0) {
#ifdef DEBUG
			sprintf(syslog_buffer, "failed to open %s file: %s", (_DB_TYPE==DB_BTREE ? "btree":"hash"), path_db_file);
			syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */					
			status = err == EAGAIN ? NSS_STATUS_TRYAGAIN : NSS_STATUS_UNAVAIL;
		}
		else {
			/* We have to make sure the file is Closed on exec'.  */
			int fd;
			int result;
			err = db->fd (db, &fd);
			if (err != 0) {
				result = -1;		
			}
			else {
				int flags = result = fcntl (fd, F_GETFD, 0);
				if (result >= 0) {
					flags |= FD_CLOEXEC;
					result = fcntl (fd, F_SETFD, flags);
				}
			}
			if (result < 0) {
				/* Something went wrong */
#if defined(DB2) || defined(DB3)
			    db->close(db, 0);  
#else
			    db->close(db);
#endif
				db = NULL;
				status = NSS_STATUS_UNAVAIL;
			}
		}
	}
	/* Remember STAYOPEN flag.  */
	if (db != NULL)
		keep_db |= stayopen;
	return status;
}

/* Close the database file.  */
static void internal_endent (void) {  
	if (db != NULL)    {      
#if defined(DB2) || defined(DB3)
        db->close(db, 0);
#else
     	db->close(db);
#endif
		db = NULL;    
	}
}                                                                                                                                                                                                      

/* Do a database lookup for KEY.  */
static enum nss_status lookup (DBT *key, struct passwd *result,
		        void *buffer, size_t buflen)
{
	char *p, *ptr;
	enum nss_status status;
	int err;
	DBT value;

	status = internal_setent (keep_db);

	if (status != NSS_STATUS_SUCCESS)
		return status;

	value.flags = 0;
#if defined(DB2) || defined(DB3)
	err = db->get (db, NULL, key, &value, 0);
#else 
	err = db->get(db, key, &value, 0);
#endif
	if (err != 0) {
              if (err == DB_NOTFOUND) {
#ifdef DEBUG
		sprintf(syslog_buffer, "key not found. error was: %s", strerror(errno));
		syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */
              	status = NSS_STATUS_NOTFOUND;
              }
              else
		status = NSS_STATUS_UNAVAIL;
	}
	else {
	     if (buflen < value.size)
			status = NSS_STATUS_TRYAGAIN;
	     else {
			/* Copy the result to a safe place. */	
			buffer = (char*) malloc(sizeof(char)*value.size+1);
			p = (char *) memcpy (buffer, value.data, value.size);
#ifdef DEBUG
			sprintf(syslog_buffer, "value found: %s", p);
			syslog(LOG_ERR,syslog_buffer);
#endif /* DEBUG */						
			if ( (ptr = strsep(&p, ":")) != NULL) {
				if (ptr_ip) {
					result->pw_name = (char*) malloc(sizeof(char)*(strlen(ptr)+strlen(ptr_ip))+2);
					strcpy(result->pw_name, ptr);
					strcat(result->pw_name, "@");
					strcat(result->pw_name, ptr_ip);
				}
				else {
					result->pw_name = (char*) malloc(sizeof(char)*strlen(ptr)+1);
					strcpy(result->pw_name, ptr);
				}
			}
			else return NSS_STATUS_UNAVAIL;
#ifdef DEBUG
			sprintf (syslog_buffer, "pwd->pw_name: %s", result->pw_name);
			syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */			
			if ( (ptr = strsep(&p, ":")) != NULL) {	
				result->pw_passwd = (char*) malloc(sizeof(char)*strlen(ptr)+1);
				strcpy(result->pw_passwd, ptr);
			}
			else return NSS_STATUS_UNAVAIL;
#ifdef DEBUG
			sprintf(syslog_buffer, "pwd->pw_passwd: %s", result->pw_passwd);
			syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */		
			if ( (ptr = strsep(&p, ":")) != NULL) 
				result->pw_uid = atoi(ptr);
			else return NSS_STATUS_UNAVAIL;
#ifndef BIG_SECURITY_HOLE
			if(result->pw_uid < _MIN_UID) {
#ifdef DEBUG
				sprintf(syslog_buffer, "user with UID=%d denied while MIN_UID is %d", result->pw_uid, _MIN_UID);
				syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */
				return NSS_STATUS_NOTFOUND;
			}
#endif /* BIG_SECURITY_HOLE */
#ifdef DEBUG
			sprintf(syslog_buffer, "pwd->pw_uid: %d", result->pw_uid);
			syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */
			if ( (ptr = strsep(&p, ":")) != NULL)
				result->pw_gid = atoi(ptr);
			else return NSS_STATUS_UNAVAIL;
#ifndef BIG_SECURITY_HOLE
			if (result->pw_gid < _MIN_GID) {
#ifdef DEBUG
				sprintf(syslog_buffer, "user with GID=%d denied while MIN_GID is %d", result->pw_gid, _MIN_GID);
				syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */
				return NSS_STATUS_NOTFOUND;
			}
#endif /* BIG_SECURITY_HOLE */
#ifdef DEBUG                        
			sprintf(syslog_buffer, "pwd->pw_gid: %d", result->pw_gid);                        
			syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */      
			if ( (ptr = strsep(&p, ":")) != NULL) {
				result->pw_gecos = (char*) malloc(sizeof(char)*strlen(ptr)+1);
				strcpy(result->pw_gecos, ptr);
			}
			else return NSS_STATUS_UNAVAIL;
#ifdef DEBUG			
			sprintf(syslog_buffer, "pwd->pw_gecos: %s", result->pw_gecos);
			syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */			
			if ( (ptr = strsep(&p, ":")) != NULL) {
				result->pw_dir = (char*) malloc(sizeof(char)*strlen(ptr)+1);
				strcpy(result->pw_dir, ptr);
			}
			else return NSS_STATUS_UNAVAIL;
#ifdef DEBUG
			sprintf(syslog_buffer, "pwd->pw_dir: %s", result->pw_dir);
			syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */					
			if ( (ptr = strsep(&p, ":")) != NULL) {
				if (!strlen(ptr)) { /* No shell */
					result->pw_shell = (char*) malloc(sizeof(char)*strlen(_DEFAULT_SHELL)+1);
					strcpy(result->pw_shell, _DEFAULT_SHELL);
				}
				else {
					result->pw_shell = (char*) malloc(sizeof(char)*strlen(ptr)+1);
					strcpy(result->pw_shell, ptr);
				}
			}		
			else return NSS_STATUS_UNAVAIL;
#ifdef DEBUG			
			sprintf(syslog_buffer, "pwd->pw_shell: %s", result->pw_shell);
			syslog(LOG_ERR, syslog_buffer);
#endif /* DEBUG */
			return NSS_STATUS_SUCCESS;		
	     }
	}     
	if (! keep_db)
		internal_endent ();
	return status;
}
	  	
enum nss_status _nss_multidomain_getpwent_r (void) {
#ifdef DEBUG 
	syslog(LOG_ERR, "nss-multidomain getpwent() request");
#endif /* DEBUG */	
	return NSS_STATUS_SUCCESS;
}

enum nss_status _nss_multidomain_getpwnam_r (char* name, 
				struct passwd *pwd, char *buffer, 
				size_t buflen, int *errnop) 
{
	DBT key;
	enum nss_status status;
	char *ptr;

	ptr = index(name, '@');
	if (ptr) {
		name = strtok(name,"@");
		ptr_ip = strtok(NULL, "@");
	}
	key.flags = 0;
#ifdef PREFIX_USER_WITH_DOT
	key.size = strlen(name)+1;
#else
	key.size = strlen(name);
#endif /* PREFIX_USER_WITH_DOT */
	key.data = (char *) malloc(sizeof(char)*key.size+1);
#ifdef PREFIX_USER_WITH_DOT
	strcpy(key.data, ".");
	strcat(key.data, name);
#else
	strcpy(key.data, name);
#endif /* PREFIX_USER_WITH_DOT */

#ifdef DEBUG	
	sprintf(syslog_buffer, "key is: %s", (char*) key.data);
	syslog(LOG_ERR, syslog_buffer);
#endif /*DEBUG*/
	status = lookup (&key, pwd, buffer, buflen);
	return status;
}
