/* upsmon - monitor power status over the 'net (talks to upsd via TCP)

   Copyright (C) 1998  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "common.h"

#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/socket.h>

#include "upsfetch.h"
#include "upsmon.h"
#include "parseconf.h"
#include "timehead.h"
#
#ifdef HAVE_STDARG_H
#include <stdarg.h>
#endif

static	char	*shutdowncmd = NULL, *notifycmd = NULL;
static	char	*powerdownflag = NULL;

static	int	minsupplies = 1, sleepval = 5, deadtime = 15;

	/* default polling interval = 5 sec */
static	int	pollfreq = 5, pollfreqalert = 5;

	/* slave hosts are given 15 sec by default to logout from upsd */
static	int	hostsync = 15;  

	/* sum of all power values from config file */
static	int	totalpv = 0;

	/* should we go into an infinite loop upon shutdown? */
static	int	playdead = 0;

	/* default replace battery warning interval (seconds) */
static	int	rbwarntime = 43200;

	/* default "all communications down" warning interval (seconds) */
static	int	nocommwarntime = 300;

	/* default interval between the shutdown warning and the shutdown */
static	int	finaldelay = 5;

	/* set by SIGHUP handler, cleared after reload finishes */
static	int	reload = 0;

	/* userid for unprivileged process when using fork mode */
static	char	*run_as_user;

static	int	debuglevel = 0, userfsd = 0, use_pipe = 1, pipefd[2];

static	utype	*firstups = NULL;

	/* signal handling things */
static	struct sigaction sa;
static	sigset_t sigmask;

#ifdef SHUT_RDWR
#define	shutdown_how SHUT_RDWR
#else
#define	shutdown_how 2
#endif

void debug(char *format, ...)
{
#ifdef HAVE_STDARG_H
	va_list	args;

	if (debuglevel < 1)
		return;

	va_start(args, format);
	vprintf(format, args);
	va_end(args);
#endif

	return;
}	

void setflag(int *val, int flag)
{
	*val = (*val |= flag);
}
        
void clearflag(int *val, int flag)  
{
	*val = (*val ^= (*val & flag));
}

int isset(int num, int flag)
{
	return ((num & flag) == flag);
}

void wall(const char *text)
{
	FILE	*wf;

	wf = popen("wall", "w");

	if (!wf) {
		upslog(LOG_NOTICE, "Can't invoke wall");
		return;
	}

	fprintf(wf, "%s\n", text);
	pclose(wf);
} 

void notify(const char *notice, int flags, const char *ntype, const char *upsname)
{
	char	exec[LARGEBUF];
	int	ret;

	if (isset(flags, NOTIFY_SYSLOG))
		upslogx(LOG_NOTICE, "%s", notice);

	/* fork here so upsmon doesn't get wedged if the notifier is slow */
	ret = fork();

	if (ret < 0) {
		upslog(LOG_ERR, "Can't fork to notify");
		return;
	}

	if (ret != 0)	/* parent */
		return;

	/* child continues and does all the work */

	if (isset(flags, NOTIFY_WALL))
		wall(notice);

	if (isset(flags, NOTIFY_EXEC)) {
		if (notifycmd != NULL) {
			snprintf(exec, sizeof(exec), "%s \"%s\"", notifycmd, notice);

			if (upsname)
				setenv("UPSNAME", upsname, 1);
			else
				setenv("UPSNAME", "", 1);

			setenv("NOTIFYTYPE", ntype, 1);
			system(exec);
		}
	}

	exit(0);
}

void do_notify(const utype *ups, int ntype)
{
	int	i;
	char	msg[SMALLBUF], *upsname = NULL;

	/* grab this for later */
	if (ups)
		upsname = ups->sys;

	for (i = 0; notifylist[i].name != NULL; i++) {
		if (notifylist[i].type == ntype) {
			debug("do_notify: ntype 0x%04x (%s)\n", ntype, notifylist[i].name);

			snprintf(msg, sizeof(msg), notifylist[i].msg, ups ? ups->sys : "");

			notify(msg, notifylist[i].flags, notifylist[i].name, upsname);
			return;
		}
	}

	/* not found ?! */
}

/* check for master permissions on the server for this ups */
void checkmaster(utype *ups)
{
	char	buf[SMALLBUF];

	/* don't bother if we're not configured as a master for this ups */
	if (!isset(ups->status, ST_MASTER))
		return;

	if (ups->upsname)
		snprintf(buf, sizeof(buf), "MASTER %s\n", ups->upsname);
	else
		snprintf(buf, sizeof(buf), "MASTER\n");

	if (upssendraw(ups->fd, buf) < 0) {
		upslogx(LOG_ALERT, "Can't set master mode on UPS [%s] - %s",
			ups->sys, upsstrerror(upserror));
		return;
	}

	if (upsreadraw(ups->fd, buf, sizeof(buf)) > 0) {
		if (!strncmp(buf, "OK", 2))
			return;

		/* not ERR, but not caught by readraw either? */

		upslogx(LOG_ALERT, "Master privileges unavailable on UPS [%s]", ups->sys);
		upslogx(LOG_ALERT, "Response: [%s]", buf);
	}
	else {	/* something caught by readraw's parsing call */
		upslogx(LOG_ALERT, "Master privileges unavailable on UPS [%s]", ups->sys);
		upslogx(LOG_ALERT, "Reason: %s", upsstrerror(upserror));
	}
}

/* authenticate to upsd and login */
void do_login(utype *ups)
{
	char	buf[SMALLBUF];

	/* authenticate first */
	snprintf(buf, sizeof(buf), "PASSWORD %s\n", ups->pw);

	if (upssendraw(ups->fd, buf) < 0) {
		upslogx(LOG_ERR, "Can't set password on [%s]: %s",
			ups->sys, upsstrerror(upserror));
			return;
	}

	if (upsreadraw(ups->fd, buf, sizeof(buf)) < 0) {
		upslogx(LOG_ERR, "Set password on [%s] failed: %s",
			ups->sys, upsstrerror(upserror));
		return;
	}

	/* catch insanity from the server - not ERR and not OK either */
	if (strncmp(buf, "OK", 2) != 0) {
		upslogx(LOG_ERR, "Set password on [%s] failed - got [%s]",
			ups->sys, buf);
		return;
	}

	/* password is set, let's login */

	if ((ups->upsname == NULL) || (strlen(ups->upsname) == 0))
		snprintf(buf, sizeof(buf), "LOGIN\n");
	else
		snprintf(buf, sizeof(buf), "LOGIN %s\n", ups->upsname);

	if (upssendraw(ups->fd, buf) < 0) {
		upslogx(LOG_ERR, "Login to UPS [%s] failed: %s",
			ups->sys, upsstrerror(upserror));
		return;
	}

	if (upsreadraw(ups->fd, buf, sizeof(buf)) < 0) {
		upslogx(LOG_ERR, "Can't login to UPS [%s]: %s",
			ups->sys, upsstrerror(upserror));
		return;
	}

	/* catch insanity from the server - not ERR and not OK either */
	if (strncmp(buf, "OK", 2) != 0) {
		upslogx(LOG_ERR, "Login on UPS [%s] failed - got [%s]",
			ups->sys, buf);
		return;
	}

	/* finally - everything is OK */
	debug("Logged into UPS %s\n", ups->sys);
	setflag(&ups->status, ST_LOGIN);

	/* now see if we also need to test master permissions */
	checkmaster(ups);
}

/* set flags and make announcements when a UPS has been checked successfully */
void heardups(utype *ups)
{
	time_t	now;

	time(&now);
	ups->lastpoll = now;

	if (isset(ups->status, ST_ALIVE))
		return;

	setflag(&ups->status, ST_ALIVE);

	/* only notify on subsequent reconnects */
	if (!isset(ups->status, ST_FIRST))
		setflag(&ups->status, ST_FIRST);
	else
		do_notify(ups, NOTIFY_COMMOK);

	if (ups->pv == 0)	/* monitor only, no need to login */
		return;

	/* if not already logged in, do it now */
	if (!isset(ups->status, ST_LOGIN))
		do_login(ups);
}

void upsgone(utype *ups)
{
	/* if the UPS is already known to be gone, then do nothing here */
	if (!isset(ups->status, ST_ALIVE))
		return;

	/* don't clear OL/OB/LB flags since we may use them later */
	clearflag(&ups->status, ST_ALIVE);
	clearflag(&ups->status, ST_LOGIN);

	do_notify(ups, NOTIFY_COMMBAD);
}

void upsonbatt(utype *ups)
{
	sleepval = pollfreqalert;	/* bump up polling frequency */

	if (!isset(ups->status, ST_ALIVE)) {
		debug("upsonbatt: %s (alive out)\n", ups->sys);
		heardups(ups);
		do_notify(ups, NOTIFY_ONBATT);
		setflag(&ups->status, ST_ONBATT);
		clearflag(&ups->status, ST_ONLINE);
		return;
	}

	debug("upsonbatt: %s (normal)\n", ups->sys);
	
	heardups(ups);

	if (isset(ups->status, ST_ONBATT)) 	/* no change */
		return;

	/* must have changed from OL to OB, so notify */

	do_notify(ups, NOTIFY_ONBATT);
	setflag(&ups->status, ST_ONBATT);
	clearflag(&ups->status, ST_ONLINE);
}

void upsonline(utype *ups)
{
	if (!isset(ups->status, ST_ALIVE)) {
		debug("upsonline: %s (alive out)\n", ups->sys);
		heardups(ups);
		setflag(&ups->status, ST_ONLINE);
		clearflag(&ups->status, ST_ONBATT);
		return;
	}

	debug("upsonline: %s (normal)\n", ups->sys);

	heardups(ups);

	if (isset(ups->status, ST_ONLINE)) 	/* no change */
		return;

	/* must have changed from OB to OL, so notify */

	do_notify(ups, NOTIFY_ONLINE);
	setflag(&ups->status, ST_ONLINE);
	clearflag(&ups->status, ST_ONBATT);
}

/* create the flag file if necessary */
void setpdflag(void)
{
	FILE	*pdf;

	if (powerdownflag != NULL) {
		pdf = fopen(powerdownflag, "w");
		if (!pdf) {
			upslogx(LOG_ERR, "Failed to create power down flag!");
			return;
		}
		fprintf(pdf, "%s", SDMAGIC);
		fclose(pdf);
	}
}

/* the actual shutdown procedure */
void doshutdown(void)
{
	int	ret;

	/* this should probably go away at some point */
	upslogx(LOG_CRIT, "Executing automatic power-fail shutdown");
	wall("Executing automatic power-fail shutdown\n");

	do_notify(NULL, NOTIFY_SHUTDOWN);

	sleep(finaldelay);

	/* in the pipe model, we let the parent do this for us */
	if (use_pipe) {
		char	ch;

		ch = 1;
		ret = write(pipefd[1], &ch, 1);
	} else {
		/* one process model = we do all the work here */

		if (geteuid() != 0)
			upslogx(LOG_WARNING, "Not root, shutdown may fail");

		setpdflag();

		ret = system(shutdowncmd);

		if (ret != 0)
			upslogx(LOG_ERR, "Unable to call shutdown command: %s\n",
		        	shutdowncmd);
	}

	/* if instructed to go into the infinite loop, then do so */
	if (playdead == 1)
		for (;;)
			sleep(100);

	/* hopefully not reached */
	exit(1);
}

/* set forced shutdown flag so other upsmons know what's going on here */
void setfsd(utype *ups)
{
	char	buf[SMALLBUF];
	int	ret;

	debug("Setting FSD on UPS %s (fd %d)\n", ups->sys, ups->fd);

	if (ups->upsname)
		snprintf(buf, sizeof(buf), "FSD %s\n", ups->upsname);
	else
		snprintf(buf, sizeof(buf), "FSD\n");

	ret = upssendraw(ups->fd, buf);

	if (ret < 0) {
		upslogx(LOG_ERR, "FSD set on UPS %s failed: %s", ups->sys,
		        upsstrerror(upserror));
		return;
	}

	ret = upsreadraw(ups->fd, buf, sizeof(buf));

	if (!strncmp(buf, "OK", 2))
		return;

	if (ret < 0)
		upslogx(LOG_ERR, "FSD set on UPS %s failed: %s", ups->sys,
		        upsstrerror(upserror));
	else
		upslogx(LOG_ERR, "FSD set on UPS %s failed: %s", ups->sys,
		        buf);
}

void slavesync(void)
{
	utype	*ups;
	char	temp[SMALLBUF];
	time_t	start, now;
	int	maxlogins, logins;

	time(&start);

	for (;;) {
		maxlogins = 0;

		for (ups = firstups; ups != NULL; ups = ups->next) {

			/* only check login count on our master(s) */
			if (!isset(ups->status, ST_MASTER))
				continue;

			if (getupsvarfd(ups->fd, ups->upsname, "numlogins", 
			                temp, sizeof(temp)) >= 0) {
				logins = strtol(temp, (char **)NULL, 10);

				if (logins > maxlogins)
					maxlogins = logins;
			}
		}

		/* if no UPS has more than 1 login (us), then slaves are gone */
		if (maxlogins <= 1)
			return;

		/* after HOSTSYNC seconds, assume slaves are stuck and bail */
		time(&now);

		if ((now - start) > hostsync) {
			upslogx(LOG_INFO, "Host sync timer expired, forcing shutdown");
			return;
		}

		usleep(250000);
	}
}

void forceshutdown(void)
{
	utype	*ups;
	int	isamaster = 0;

	debug("Shutting down any UPSes in MASTER mode...\n");

	/* set FSD on any "master" UPS entries (forced shutdown in progress) */
	for (ups = firstups; ups != NULL; ups = ups->next)
		if (isset(ups->status, ST_MASTER)) {
			isamaster = 1;
			setfsd (ups);
		}

	/* if we're not a master on anything, we should shut down now */
	if (!isamaster)
		doshutdown();

	/* must be the master now */
	debug("This system is a master... waiting for slave logout...\n");

	/* wait up to HOSTSYNC seconds for slaves to logout */
	slavesync();

	/* time expired or all the slaves are gone, so shutdown */
	doshutdown();
}

int is_ups_critical(utype *ups)
{
	time_t	now;

	/* FSD = the master is forcing a shutdown */
	if (isset(ups->status, ST_FSD))
		return 1;

	/* not OB or not LB = not critical yet */
	if ((!isset(ups->status, ST_ONBATT)) ||
		(!isset(ups->status, ST_LOWBATT)))
		return 0;

	/* must be OB+LB now */

	/* if we're a master, declare it critical so we set FSD on it */
	if (isset(ups->status, ST_MASTER))
		return 1;

	/* must be a slave now */

	/* FSD isn't set, so the master hasn't seen it yet */

	time(&now);

	/* give the master up to HOSTSYNC seconds before shutting down */
	if ((now - ups->lastnoncrit) > hostsync) {
		upslogx(LOG_WARNING, "Giving up on the master for UPS [%s]",
			ups->sys);
		return 1;
	}

	/* there's still time left */
	return 0;
}

/* recalculate the online power value and see if things are still OK */
void recalc(void)
{
	utype	*ups;
	int	val_ol = 0;
	time_t	now;

	time(&now);
	ups = firstups;
	while (ups != NULL) {
		/* promote dead UPSes that were last known OB to OB+LB */
		if ((now - ups->lastpoll) > deadtime)
			if (isset(ups->status, ST_ONBATT)) {
				debug ("Promoting dead UPS: %s\n", ups->sys);
				setflag(&ups->status, ST_LOWBATT);
			}

		/* note: we assume that a UPS that isn't critical must be OK *
		 *							     *
		 * this means a UPS we've never heard from is assumed OL     *
		 * whether this is really the best thing to do is undecided  */

		/* crit = (FSD) || (OB & LB) > HOSTSYNC seconds */
		if (is_ups_critical(ups))
			debug("Critical UPS: %s\n", ups->sys);
		else
			val_ol += ups->pv;

		ups = ups->next;
	}

	/* debug("Current power value: %d\n", val_ol);
	debug("Minimum power value: %d\n", minsupplies); */

	if (val_ol < minsupplies)
		forceshutdown();
}		

void upslowbatt(utype *ups)
{
	if (!isset(ups->status, ST_ALIVE)) {
		heardups(ups);
		setflag(&ups->status, ST_LOWBATT);
		return;
	}

	heardups(ups);

	if (isset(ups->status, ST_LOWBATT)) 	/* no change */
		return;

	/* must have changed from !LB to LB, so notify */

	do_notify(ups, NOTIFY_LOWBATT);
	setflag(&ups->status, ST_LOWBATT);
}

void upsreplbatt(utype *ups)
{
	time_t	now;

	time(&now);

	if ((now - ups->lastrbwarn) > rbwarntime) {
		do_notify(ups, NOTIFY_REPLBATT);
		ups->lastrbwarn = now;
	}
}

/* deal with a UPS that stays unavailable for awhile */
void upsnocomm(utype *ups)
{
	time_t	now;

	time(&now);

	/* first only act if we're <nocommtime> seconds past the last poll */
	if ((now - ups->lastpoll) < nocommwarntime)
		return;

	/* now only complain if we haven't lately */
	if ((now - ups->lastncwarn) > nocommwarntime) {
		do_notify(ups, NOTIFY_NOCOMM);
		ups->lastncwarn = now;
	}
}

void upsfsd(utype *ups)
{
	if (!isset(ups->status, ST_ALIVE)) {
		heardups(ups);
		setflag(&ups->status, ST_FSD);
		return;
	}

	heardups(ups);

	if (isset(ups->status, ST_FSD)) 	/* no change */
		return;

	/* must have changed from !FSD to FSD, so notify */

	do_notify(ups, NOTIFY_FSD);
	setflag(&ups->status, ST_FSD);

	/* TODO: if we're a master, log something about it */
}

/* cleanly close the connection to a given UPS */
void drop_connection(utype *ups)
{
	debug("Dropping connection to UPS [%s]\n", ups->sys);

	clearflag(&ups->status, ST_ALIVE);
	clearflag(&ups->status, ST_LOGIN);
	shutdown(ups->fd, shutdown_how);
	close(ups->fd);
	ups->fd = -1;
}

/* change some UPS parameters during reloading */
void redefine_ups(utype *ups, int pv, const char *pw, const char *master)
{
	ups->retain = 1;

	if (ups->pv != pv) {
		upslogx(LOG_INFO, "UPS [%s]: redefined power value to %d", 
			ups->sys, pv);
		ups->pv = pv;
	}

	totalpv += ups->pv;

	/* paranoia */
	if (!ups->pw)
		ups->pw = xstrdup("");	/* give it a bogus, but non-NULL one */

	/* obviously don't put the new password in the syslog... */
	if (strcmp(ups->pw, pw) != 0) {
		upslogx(LOG_INFO, "UPS [%s]: redefined password", ups->sys);

		if (ups->pw)
			free(ups->pw);

		ups->pw = xstrdup(pw);

		/* if not logged in, force a reconnection -		*
		 * they may have redefined this to make the login work	*/
		if (!isset(ups->status, ST_LOGIN)) {
			upslogx(LOG_INFO, "UPS [%s]: retrying connection\n",
				ups->sys);

			drop_connection(ups);
		}
	}

	/* slave -> master */
	if ((!strcasecmp(master, "master")) && (!isset(ups->status, ST_MASTER))) {
		upslogx(LOG_INFO, "UPS [%s]: redefined as master", ups->sys);
		setflag(&ups->status, ST_MASTER);

		/* reset connection to ensure master mode gets checked */
		drop_connection(ups);
		return;
	}

	/* master -> slave */
	if ((!strcasecmp(master, "slave")) && (isset(ups->status, ST_MASTER))) {
		upslogx(LOG_INFO, "UPS [%s]: redefined as slave", ups->sys);
		clearflag(&ups->status, ST_MASTER);
		return;
	}
}

void addups(int reloading, const char *sys, const char *pvs, const char *pw, 
		const char *master)
{
	utype	*tmp, *last;
	char	*upsname, *ptr;
	int	pv;

	if ((!sys) || (!pvs) || (!pw) || (!master)) {
		upslogx(LOG_WARNING, "Ignoring invalid MONITOR line in upsmon.conf!");
		upslogx(LOG_WARNING, "MONITOR configuration directives require four arguments.");
		return;
	}

	pv = strtol(pvs, (char **) NULL, 10);

	if (pv < 0) {
		upslogx(LOG_WARNING, "UPS [%s]: ignoring invalid power value [%s]", 
			sys, pvs);
		return;
	}

	last = tmp = firstups;

	while (tmp) {
		last = tmp;

		/* check for duplicates */
		if (!strcmp(tmp->sys, sys)) {
			if (reloading)
				redefine_ups(tmp, pv, pw, master);
			else
				upslogx(LOG_WARNING, "Warning: ignoring duplicate"
					" UPS [%s]", sys);
			return;
		}

		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(utype));
	tmp->fd = -1;
	tmp->sys = xstrdup(sys);

	/* parse out <upsname>@<host> for later */
	upsname = xstrdup(sys);
	ptr = strchr(upsname, '@');
	if (ptr) {
		*ptr = '\0';
		tmp->upsname = xstrdup(upsname);
		tmp->host = xstrdup(++ptr);
	}
	else {
		tmp->upsname = NULL;
		tmp->host = xstrdup(sys);
	}

	free(upsname);

	tmp->pv = pv;

	/* build this up so the user doesn't run with bad settings */
	totalpv += tmp->pv;

	tmp->pw = xstrdup(pw);
	tmp->status = 0;
	tmp->retain = 1;
	tmp->lastpoll = 0;
	tmp->lastnoncrit = 0;
	tmp->lastrbwarn = 0;
	tmp->lastncwarn = 0;

	if (!strcasecmp(master, "master"))
		setflag(&tmp->status, ST_MASTER);

	tmp->next = NULL;

	if (last)
		last->next = tmp;
	else
		firstups = tmp;

	if (tmp->pv)
		upslogx(LOG_INFO, "UPS: %s (%s) (power value %d)", tmp->sys, 
		        isset(tmp->status, ST_MASTER) ? "master" : "slave",
			tmp->pv);
	else
		upslogx(LOG_INFO, "UPS: %s (monitoring only)", tmp->sys);

	tmp->fd = upsconnect(tmp->host);
}		

void set_notifymsg(const char *name, const char *msg)
{
	int	i;

	for (i = 0; notifylist[i].name != NULL; i++) {
		if (!strcasecmp(notifylist[i].name, name)) {

			/* only free if it's not the stock msg */
			if (notifylist[i].msg != notifylist[i].stockmsg)
				free(notifylist[i].msg);

			notifylist[i].msg = xstrdup(msg);
			return;
		}
	}

	upslogx(LOG_WARNING, "'%s' is not a valid notify event name\n", name);
}

void set_notifyflag(const char *ntype, char *flags)
{
	int	i, pos;
	char	*ptr, *tmp;

	/* find ntype */

	pos = -1;
	for (i = 0; notifylist[i].name != NULL; i++) {
		if (!strcasecmp(notifylist[i].name, ntype)) {
			pos = i;
			break;
		}
	}

	if (pos == -1) {
		upslogx(LOG_WARNING, "Warning: invalid notify type [%s]\n", ntype);
		return;
	}

	ptr = flags;

	/* zero existing flags */
	notifylist[pos].flags = 0;

	while (ptr) {
		int	newflag;

		tmp = strchr(ptr, '+');
		if (tmp)
			*tmp++ = '\0';

		newflag = 0;

		if (!strcmp(ptr, "SYSLOG"))
			newflag = NOTIFY_SYSLOG;
		if (!strcmp(ptr, "WALL"))
			newflag = NOTIFY_WALL;
		if (!strcmp(ptr, "EXEC"))
			newflag = NOTIFY_EXEC;
		if (!strcmp(ptr, "IGNORE"))
			newflag = NOTIFY_IGNORE;

		if (newflag)
			notifylist[i].flags |= newflag;
		else
			upslogx(LOG_WARNING, "Invalid notify flag: [%s]\n", ptr);

		ptr = tmp;
	}
}

/* in split mode, the parent doesn't hear about reloads */
void checkmode(char *cfgentry, char *oldvalue, char *newvalue, int reloading)
{
	/* nothing to do if in "all as root" mode */
	if (use_pipe == 0)
		return;

	/* it's ok if we're not reloading yet */
	if (reloading == 0)
		return;

	/* also nothing to do if it didn't change */
	if ((oldvalue) && (newvalue)) {
		if (!strcmp(oldvalue, newvalue))
			return;
	}

	/* otherwise, yell at them */
	upslogx(LOG_WARNING, "Warning: %s redefined in split-process mode!",
		cfgentry);
	upslogx(LOG_WARNING, "You must restart upsmon for this change to work");
}

void conf_arg(int numargs, char **arg)
{
	/* using up to arg[1] below */
	if (numargs < 2)
		return;

	/* SHUTDOWNCMD <cmd> */
	if (!strcmp(arg[0], "SHUTDOWNCMD")) {
		checkmode(arg[0], shutdowncmd, arg[1], reload);

		if (shutdowncmd)
			free(shutdowncmd);
		shutdowncmd = xstrdup(arg[1]);
		return;
	}

	/* POWERDOWNFLAG <fn> */
	if (!strcmp(arg[0], "POWERDOWNFLAG")) {
		checkmode(arg[0], powerdownflag, arg[1], reload);

		if (powerdownflag)
			free(powerdownflag);

		powerdownflag = xstrdup(arg[1]);

		if (!reload)
			upslogx(LOG_INFO, "Using power down flag file %s\n",
				arg[1]);

		return;
	}		

	/* NOTIFYCMD <cmd> */
	if (!strcmp(arg[0], "NOTIFYCMD")) {
		if (notifycmd)
			free(notifycmd);

		notifycmd = xstrdup(arg[1]);
		return;
	}

	/* POLLFREQ <num> */
	if (!strcmp(arg[0], "POLLFREQ")) {
		pollfreq = atoi(arg[1]);
		return;
	}

	/* POLLFREQALERT <num> */
	if (!strcmp(arg[0], "POLLFREQALERT")) {
		pollfreqalert = atoi(arg[1]);
		return;
	}

	/* HOSTSYNC <num> */
	if (!strcmp(arg[0], "HOSTSYNC")) {
		hostsync = atoi(arg[1]);
		return;
	}

	/* MINSUPPLIES <num> */
	if (!strcmp(arg[0], "DEADTIME")) {
		deadtime = atoi(arg[1]);
		return;
	}

	/* RBWARNTIME <num> */
	if (!strcmp(arg[0], "RBWARNTIME")) {
		rbwarntime = atoi(arg[1]);
		return;
	}

	/* NOCOMMWARNTIME <num> */
	if (!strcmp(arg[0], "NOCOMMWARNTIME")) {
		nocommwarntime = atoi(arg[1]);
		return;
	}

	/* FINALDELAY <num> */
	if (!strcmp(arg[0], "FINALDELAY")) {
		finaldelay = atoi(arg[1]);
		return;
	}

	/* RUN_AS_USER <userid> */
 	if (!strcmp(arg[0], "RUN_AS_USER")) {
		if (run_as_user)
			free(run_as_user);

		run_as_user = xstrdup(arg[1]);
		return;
	}

	/* using up to arg[2] below */
	if (numargs < 3)
		return;

	/* NOTIFYMSG <notify type> <replacement message> */
	if (!strcmp(arg[0], "NOTIFYMSG")) {
		set_notifymsg(arg[1], arg[2]);
		return;
	}

	/* NOTIFYFLAG <notify type> <flags> */
	if (!strcmp(arg[0], "NOTIFYFLAG")) {
		set_notifyflag(arg[1], arg[2]);
		return;
	}	

	/* using up to arg[4] below */
	if (numargs < 5)
		return;

	/* MONITOR <system> <powervalue> <password> ("master"|"slave") */
	if (!strcmp(arg[0], "MONITOR")) {
		addups(reload, arg[1], arg[2], arg[3], arg[4]);
		return;
	}
}

void conf_err(int linenum, char *errtext)
{
	if (linenum == 0)
		upslogx(LOG_ERR, "upsmon.conf: %s", errtext);
	else
		upslogx(LOG_ERR, "upsmon.conf line %d: %s", linenum, errtext);
}

void loadconfig(void)
{
	int	ret;
	char	cfn[SMALLBUF];

	snprintf(cfn, sizeof(cfn), "%s/upsmon.conf", CONFPATH);

	ret = parseconf(cfn, conf_arg, conf_err);

	if (ret != 0) {
		if (reload == 1) {
			upslog(LOG_ERR, "Reload failed: can't open %s", cfn);
			return;
		}

		fatal("Can't parse %s", cfn);
	}
}

/* SIGPIPE handler */
void sigpipe(int sig)
{
	debug("SIGPIPE: dazed and confused, but continuing...\n");
}

/* SIGQUIT, SIGTERM handler */
void shut_myself_down(int sig)
{
	utype	*ups;

	upslogx(LOG_INFO, "Signal %d: exiting", sig);

	/* close all fds */
	for (ups = firstups; ups != NULL; ups = ups->next)
		if (ups->fd != -1) {
			shutdown(ups->fd, shutdown_how);
			close(ups->fd);
		}

	exit(0);
}

void user_fsd(int sig)
{
	upslogx(LOG_INFO, "User requested FSD!");

	userfsd = 1;
}

void set_reload_flag(int sig)
{
	reload = 1;
}

/* install handlers for a few signals */
void setupsignals(void)
{
	sigemptyset(&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;

	sa.sa_handler = sigpipe;
	sigaction(SIGPIPE, &sa, NULL);

	sa.sa_handler = shut_myself_down;
	sigaction(SIGINT, &sa, NULL);
	sigaction(SIGQUIT, &sa, NULL);
	sigaction(SIGTERM, &sa, NULL);

	/* deal with the ones from userspace as well */

	sa.sa_handler = user_fsd;
	sigaction(SIGCMD_FSD, &sa, NULL);

	sa.sa_handler = set_reload_flag;
	sigaction(SIGCMD_RELOAD, &sa, NULL);
}

/* handler for alarm when getupsvarfd times out */
void timeout(int sig)
{
	/* don't do anything here, just return */
}

/* remember the last time the ups was not critical (OB + LB) */
void update_crittimer(utype *ups)
{
	/* if !OB or !LB, then it's not critical, so log the time */
	if ((!isset(ups->status, ST_ONBATT)) || 
		(!isset(ups->status, ST_LOWBATT))) {

		time(&ups->lastnoncrit);
		return;
	}

	/* fallthrough: let the timer age */
}

/* see what the status of the UPS is and handle any changes */
void pollups(utype *ups)
{
	char	status[SMALLBUF], *stat, *ptr;
	struct	sigaction sa;
	sigset_t sigmask;

	debug("pollups: %s (fd %d)\n", ups->sys, ups->fd);

	/* try a reconnect here */
	if (ups->fd < 0) {
		ups->fd = upsconnect(ups->host);

		debug("pollups: upsconnect: got fd %d\n", ups->fd);

		/* if it didn't work, don't try to poll it now */
		if (ups->fd < 0) {
			upslogx(LOG_ERR, "UPS [%s]: connect failed: %s",
				ups->sys, upsstrerror(upserror));
			upsgone(ups);
			upsnocomm(ups);
			return;
		}
	}

	sa.sa_handler = timeout;
	sigemptyset(&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction(SIGALRM, &sa, NULL);

	alarm(10);

	if (getupsvarfd(ups->fd, ups->upsname, "status", status, sizeof(status)) >= 0) {
		signal(SIGALRM, SIG_IGN);
		alarm(0);

		debug(" - status [%s] : ", status);

		/* empty response is the same as a dead ups */
		if (!strcmp(status, "")) {
			upsgone(ups);
			upsnocomm(ups);
			return;
		}

		/* clear these out early if they disappear */
		if (!strstr(status, "LB"))
			clearflag(&ups->status, ST_LOWBATT);
		if (!strstr(status, "FSD"))
			clearflag(&ups->status, ST_FSD);

		stat = status;

		while (stat != NULL) {
			ptr = strchr(stat, ' ');
			if (ptr)
				*ptr++ = '\0';

			debug("{%s} ", stat);

			if (!strcasecmp(stat, "OL"))
				upsonline(ups);
			if (!strcasecmp(stat, "OB"))
				upsonbatt(ups);
			if (!strcasecmp(stat, "LB"))
				upslowbatt(ups);
			if (!strcasecmp(stat, "RB"))
				upsreplbatt(ups);

			/* do it last to override any possible OL */
			if (!strcasecmp(stat, "FSD"))
				upsfsd(ups);

			update_crittimer(ups);

			stat = ptr;
		} 

		debug("\n");
		return;
	}

	/* fallthrough: no communications */
	signal(SIGALRM, SIG_IGN);
	alarm(0);

	/* try to make some of these a little friendlier */
	switch(upserror) {
		case UPSF_UNKNOWNUPS:
			if (!ups->upsname)
				upslogx(LOG_ERR, "Poll UPS [%s] failed - no "
				"UPS defined on server %s", ups->host);
			else
				upslogx(LOG_ERR, "Poll UPS [%s] failed - [%s] "
				"does not exist on server %s", 
				ups->sys, ups->upsname,	ups->host);
			break;
		default:
			upslogx(LOG_ERR, "Poll UPS [%s] failed - %s", 
				ups->sys, upsstrerror(upserror));
			break;
	}

	/* restart the connection if we had ever polled it */
	if (isset(ups->status, ST_ALIVE)) {
		drop_connection(ups);
		upsgone(ups);
		upsnocomm(ups);
		return;
	}

	/* if anything worse than staleness/not defined, restart the link */
	if ((upserror != UPSF_DATASTALE) && (upserror != UPSF_UNKNOWNUPS))
		drop_connection(ups);

	upsgone(ups);
	upsnocomm(ups);
}

/* remove the power down flag if it exists and is the proper form */
void clearpdf(void)
{
	FILE	*pdf;
	char	buf[SMALLBUF];

	pdf = fopen(powerdownflag, "r");

	if (pdf == NULL)	/* no such file, nothing to do */
		return;

	/* if it exists, see if it has the right text in it */

	fgets(buf, sizeof(buf), pdf);
	fclose(pdf);

	/* reasoning: say upsmon.conf is world-writable (!) and some nasty
	 * user puts something "important" as the power flag file.  This 
	 * keeps upsmon from utterly trashing it when starting up or powering
	 * down at the expense of not shutting down the UPS.
	 *
	 * solution: don't let mere mortals edit that configuration file.
	 */

	if (!strncmp(buf, SDMAGIC, strlen(SDMAGIC)))	/* ok, it's from us */
		unlink(powerdownflag);
	else {
		upslogx(LOG_INFO, "%s doesn't seem to be a proper flag file.  Disabling.",
			  powerdownflag);
		powerdownflag = NULL;
	}
}

void help(const char *progname)
{
	printf("Monitor UPS servers and possibly shutdown if necessary.\n\n");
	printf("usage: %s [-h] [-d] [-i] [-p] [-u <user>] [-c <cmd>]\n\n", progname);
	printf("\n");	
	printf("  -c <cmd>	send command to running process\n");	
	printf("		commands:\n");
	printf("		 - fsd: shutdown all master UPSes (use with caution)\n");
	printf("		 - reload: reread configuration\n");
	printf("		 - stop: stop monitoring and exit\n");
	printf("  -d		enable debugging code\n");
	printf("  -i		go into an infinite loop after calling shutdown\n");
	printf("  -p		always run privileged (disable privileged parent)\n");
	printf("  -u <user>	run child as user <user> (ignored when using -p)\n");
	exit(0);
}

/* set all the notify values to a default */
void initnotify(void)
{
	int	i;

	for (i = 0; notifylist[i].name != NULL; i++) {
		notifylist[i].flags = NOTIFY_SYSLOG | NOTIFY_WALL;
		notifylist[i].msg = notifylist[i].stockmsg;
	}
}

void runparent(int pipe)
{
	int	ret;
	char	ch;

	/* handling signals is the child's job */
	signal(SIGHUP, SIG_IGN);
	signal(SIGUSR1, SIG_IGN);
	signal(SIGUSR2, SIG_IGN);

	ret = read(pipe, &ch, 1);

	if (ret < 1) {
		if (errno == ENOENT)
			fatalx("upsmon parent: exiting (child exited)");

		fatal("upsmon parent: read");
	}

	if (ch != 1)
		fatalx("upsmon parent: got bogus pipe command %c", ch);

	/* have to do this here - child is unprivileged */
	setpdflag();

	ret = system(shutdowncmd);

	if (ret != 0)
		upslogx(LOG_ERR, "parent: Unable to call shutdown command: %s\n",
	        	shutdowncmd);

	close(pipe);
	exit(0);
}

/* fire up the split parent/child scheme */
void start_pipe(char *user)
{
	int	ret;

	ret = pipe(pipefd);

	if (ret)
		fatal("pipe creation failed");

	ret = fork();

	if (ret < 0)
		fatal("fork failed");

	/* start the privileged parent */
	if (ret != 0) {
		close(pipefd[1]);
		runparent(pipefd[0]);

		exit(1);	/* NOTREACHED */
	}

	close(pipefd[0]);

	/* write the pid file now, as we will soon lose root */
	writepid("upsmon");

	/* continuing from here is the unprivileged child, so let's drop root */
	if (user)
		switch_to_user(user, NULL);
	else
		droproot();
}

void delete_ups(utype *target)
{
	utype	*ptr, *last;

	if (!target)
		return;

	ptr = last = firstups;

	while (ptr) {
		if (ptr == target) {
			upslogx(LOG_NOTICE, "No longer monitoring UPS [%s]\n",
				target->sys);

			/* disconnect cleanly */
			drop_connection(ptr);

			/* about to delete the first ups? */
			if (ptr == last)
				firstups = ptr->next;
			else
				last->next = ptr->next;

			/* release memory */

			if (ptr->sys)
				free(ptr->sys);
			if (ptr->upsname)
				free(ptr->upsname);
			if (ptr->host)
				free(ptr->host);
			if (ptr->pw)
				free(ptr->pw);
			free(ptr);

			return;
		}

		last = ptr;
		ptr = ptr->next;
	}

	/* shouldn't happen */
	upslogx(LOG_ERR, "delete_ups: UPS not found\n");
}	

/* see if we can open a file */
static int check_file(char *fn)
{
	char	chkfn[SMALLBUF];
	FILE	*f;

	snprintf(chkfn, sizeof(chkfn), "%s/%s", CONFPATH, fn);

	f = fopen(chkfn, "r");

	if (!f) {
		upslog(LOG_ERR, "Reload failed: can't open %s", chkfn);
		return 0;	/* failed */
	}

	fclose(f);
	return 1;	/* OK */
}

void reload_conf(void)
{
	utype	*tmp, *next;

	upslogx(LOG_INFO, "Reloading configuration");

	/* sanity check */
	if (!check_file("upsmon.conf")) {
		reload = 0;
		return;
	}

	/* flip through ups list, clear retain value */
	tmp = firstups;

	while (tmp) {
		tmp->retain = 0;
		tmp = tmp->next;
	}

	/* reset paranoia checker */
	totalpv = 0;

	/* reread upsmon.conf */
	loadconfig();

	/* go through the utype struct again */
	tmp = firstups;

	while (tmp) {
		next = tmp->next;

		/* !retain means it wasn't in the .conf this time around */
		if (tmp->retain == 0)
			delete_ups(tmp);

		tmp = next;
	}

	/* see if the user just blew off a foot */
	if (totalpv < minsupplies) {
		upslogx(LOG_CRIT, "Fatal error: total power value (%d) less "
			"than MINSUPPLIES (%d)", totalpv, minsupplies);

		fatalx("Impossible power configuation, unable to continue");
	}

	/* finally clear the flag */
	reload = 0;
}

/* make sure the parent is still alive */
void check_parent(void)
{
	int	ret;
	fd_set	rfds;
	struct	timeval	tv;
	time_t	now;
	static	time_t	lastwarn = 0;

	FD_ZERO(&rfds);
	FD_SET(pipefd[1], &rfds);

	tv.tv_sec = 0;
	tv.tv_usec = 0;

	ret = select(pipefd[1] + 1, &rfds, NULL, NULL, &tv);

	if (ret == 0)
		return;

	/* this should never happen, but we MUST KNOW if it ever does */

	time(&now);

	/* complain every 2 minutes */
	if ((now - lastwarn) < 120) 
		return;

	lastwarn = now;
	do_notify(NULL, NOTIFY_NOPARENT);

	/* also do this in case the notifier isn't being effective */
	upslogx(LOG_ALERT, "Parent died - shutdown impossible");
}

int main(int argc, char *argv[])  
{
	int	i, cmd;

	cmd = 0;

	printf("Network UPS Tools upsmon %s\n", UPS_VERSION);

	while ((i = getopt(argc, argv, "+dhic:pu:")) != EOF) {
		switch (i) {
			case 'd':
				debuglevel = 1;
				break;
			case 'h':
				help(argv[0]);
				break;
			case 'i':
				playdead = 1;
				break;
			case 'p':
				use_pipe = 0;
				break;
			case 'c':
				if (!strncmp(optarg, "fsd", strlen(optarg)))
					cmd = SIGCMD_FSD;
				if (!strncmp(optarg, "stop", strlen(optarg)))
					cmd = SIGCMD_STOP;
				if (!strncmp(optarg, "reload", strlen(optarg)))
					cmd = SIGCMD_RELOAD;

				/* bad command name given */
				if (cmd == 0)
					help(argv[0]);
				break;
			case 'u':
				run_as_user = optarg;
				break;
			default:
				help(argv[0]);
				break;
		}
	}

	if (cmd) {
		sendsignal("upsmon", cmd);
		exit(0);
	}

	argc -= optind;
	argv += optind;

	openlog("upsmon", LOG_PID, LOG_FACILITY);

	initnotify();
	loadconfig();

	if (shutdowncmd == NULL)
		printf("Warning: no shutdown command defined!\n");

	/* we may need to get rid of a flag from a previous shutdown */
	if (powerdownflag != NULL)
		clearpdf();

	if (totalpv < minsupplies) {
		printf("\nFatal error: insufficient power configured!\n\n");

		printf("Sum of power values........: %d\n", totalpv);
		printf("Minimum value (MINSUPPLIES): %d\n", minsupplies);

		printf("\nEdit your upsmon.conf and change the values.\n");
		exit(1);
	}

	if (debuglevel < 1)
		background();

	/* === root parent and unprivileged child split here === */

	/* only do the pipe stuff if the user hasn't disabled it */
	if (use_pipe)
		start_pipe(run_as_user);
	else {
		upslog(LOG_INFO, "Not using split process model.");
		writepid("upsmon");
	}

	/* ignore sigpipes before the connection opens */
	setupsignals();

	/* reopen the log for the child process */
	closelog();
	openlog("upsmon", LOG_PID, LOG_FACILITY);

	for (;;) {
		utype	*ups;

		/* check flags from signal handlers */
		if (userfsd)
			forceshutdown();

		if (reload)
			reload_conf();

		sleepval = pollfreq;
		for (ups = firstups; ups != NULL; ups = ups->next)
			pollups(ups);

		recalc();

		/* make sure the parent hasn't died */
		if (use_pipe)
			check_parent();

		/* reap children that have exited */
		waitpid(-1, NULL, WNOHANG);

		sleep(sleepval);
	}

	/* NOTREACHED */
}
