/* NVTV Gui -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This is open software protected by the GPL. See GPL.txt for details.
 *
 * The GTK graphical user interface. Brooktree part.
 */

#include <gtk/gtk.h>

#include "debug.h"
#include "gui.h"
#include "gui_bt.h"
#include "backend.h"
#include "calc_bt.h"
#include "data_bt.h"

/* Automatic minimum distance between overscan percentages */

#define AUTO_DIST 10.0

static GtkAdjustment *update_bt;

static NVBtRegs calc_bt;
static NVCrtRegs calc_crt;

static NVMode calc_mode =
  {TV_SYSTEM_NONE, 0, 0, "Custom", 0.0, 0.0, 
   &calc_crt, (NVTvRegs *) &calc_bt};

#define FIELD(b,m) addr:&(b.m), size:sizeof(b.m)
#define FIELD_BT(m) FIELD(gui_tv.bt,m)

/* -------- GUI Masks -------- */

struct mask_calc {
  GtkAdjustment *hres, *vres;
  GtkAdjustment *hoc_min, *hoc_max, *voc_min, *voc_max;
  GtkLabel *hoc_sel, *voc_sel;
  GtkCList *list;
  GtkToggleButton *dist;
  GtkButton *calc;
};

struct mask_calc gui_mask_calc;

struct mask_bt_freq {
  GtkLabel *clk, *hsyn, *vsyn, *hover, *vover, *aspect;
};

struct mask_bt_freq gui_mask_bt_freq;

struct mask_status {
  GtkToggleButton *tvon, *pll, *over, *under, *pal, *busy;
};

struct mask_status gui_mask_status;

static struct reg_mask bt_mask [] = {
  {label:"hsynoffset:",    bits:-10,tick:1, FIELD_BT(hsynoffset)},
  {label:"vsynoffset:",    bits:11, tick:1, FIELD_BT(vsynoffset)},
  {label:"hsynwidth:",     bits:6,  tick:1, FIELD_BT(hsynwidth)},
  {label:"vsynwidth:",     bits:3,  tick:1, FIELD_BT(vsynwidth)},
  {label:"h_clko:",        bits:12, tick:1, FIELD_BT(h_clko)},
  {label:"h_active:",      bits:10, tick:1, FIELD_BT(h_active)},
  {label:"hsync_width:",   bits:8,  tick:1, FIELD_BT(hsync_width)},
  {label:"hburst_begin:",  bits:8,  tick:1, FIELD_BT(hburst_begin)},
  {label:"hburst_end:",    bits:8,  tick:1, FIELD_BT(hburst_end)},
  {label:"h_blanko:",      bits:11, tick:1, FIELD_BT(h_blanko)},
  {label:"v_blanko:",      bits:10, tick:1, FIELD_BT(v_blanko)},
  {label:"v_activeo:",     bits:9,  tick:1, FIELD_BT(v_activeo)},
  {label:"h_fract:",       bits:8,  tick:1, FIELD_BT(h_fract)},
  {label:"h_clki:",        bits:11, tick:1, FIELD_BT(h_clki)},
  {label:"h_blanki:",      bits:9,  tick:1, FIELD_BT(h_blanki)},
  {label:"v_blank_dly:",   bits:1,  tick:1, FIELD_BT(v_blank_dly)},
  {label:"v_linesi:",      bits:10, tick:1, FIELD_BT(v_linesi)},
  {label:"v_blanki:",      bits:8,  tick:1, FIELD_BT(v_blanki)},
  {label:"v_activei:",     bits:10, tick:1, FIELD_BT(v_activei)},
  {label:"v_scale:",       bits:14, tick:1, FIELD_BT(v_scale)},
  {label:"pll_fract:",     bits:16, tick:1, FIELD_BT(pll_fract)},
  {label:"pll_int:",       bits:6,  tick:1, FIELD_BT(pll_int)},
  {label:"sync_amp:",      bits:8,  tick:1, FIELD_BT(sync_amp)},
  {label:"bst_amp:",       bits:8,  tick:1, FIELD_BT(bst_amp)},
  {label:"mcr:",           bits:8,  tick:1, FIELD_BT(mcr)},
  {label:"mcb:",           bits:8,  tick:1, FIELD_BT(mcb)},
  {label:"my:",            bits:8,  tick:1, FIELD_BT(my)},
  {label:NULL}
};				    

static struct reg_mask bt_mask_ext [] = {
  {label:"f_sely:",        bits:3,  tick:1, FIELD_BT(f_sely)},
  {label:"f_selc:",        bits:3,  tick:1, FIELD_BT(f_selc)},
  {label:"ylpf:",          bits:2,  tick:1, FIELD_BT(ylpf)},
  {label:"clpf:",          bits:2,  tick:1, FIELD_BT(clpf)},
  {label:"ycoring:",       bits:3,  tick:1, FIELD_BT(ycoring)},
  {label:"ccoring:",       bits:3,  tick:1, FIELD_BT(ccoring)},
  {label:"yattenuate:",    bits:3,  tick:1, FIELD_BT(yattenuate)},
  {label:"cattenuate:",    bits:3,  tick:1, FIELD_BT(cattenuate)},
  {label:"out_muxa:",      bits:2,  tick:1, FIELD_BT(out_muxa)},
  {label:"out_muxb:",      bits:2,  tick:1, FIELD_BT(out_muxb)},
  {label:"out_muxc:",      bits:2,  tick:1, FIELD_BT(out_muxc)},
  {label:NULL}
};				    

static struct reg_mask bt_mask_msc = 
  {label:"msc:",           bits:32, tick:1, FIELD_BT(msc)};

static struct reg_mask bt_mask_macro = 
  {label:"macro:",         bits:3,  tick:1, FIELD_BT(macro)};

static struct flag_mask bt_mask_flags [] = {
  {label:"ni_out",      mask:BT_FLAG1_NI_OUT,      FIELD_BT(flags1)}, 
  {label:"setup",       mask:BT_FLAG1_SETUP,       FIELD_BT(flags1)}, 
  {label:"625line",     mask:BT_FLAG1_625LINE,     FIELD_BT(flags1)}, 
  {label:"vsync_dur",   mask:BT_FLAG1_VSYNC_DUR,   FIELD_BT(flags1)}, 
  {label:"dis_screset", mask:BT_FLAG1_DIS_SCRESET, FIELD_BT(flags1)}, 
  {label:"pal_md",      mask:BT_FLAG1_PAL_MD,      FIELD_BT(flags1)}, 
  {label:"dis_ffilt",   mask:BT_FLAG2_DIS_FFILT,   FIELD_BT(flags2)}, 
  {label:"dis_yflpf",   mask:BT_FLAG2_DIS_YFLPF,   FIELD_BT(flags2)}, 
  {label:"dis_gmshy",   mask:BT_FLAG2_DIS_GMSHY,   FIELD_BT(flags2)}, 
  {label:"dis_gmushy",  mask:BT_FLAG2_DIS_GMUSHY,  FIELD_BT(flags2)}, 
  {label:"dis_gmshc",   mask:BT_FLAG2_DIS_GMSHC,   FIELD_BT(flags2)}, 
  {label:"dis_gmushc",  mask:BT_FLAG2_DIS_GMUSHC,  FIELD_BT(flags2)}, 
  {label:"dis_chroma",  mask:BT_FLAG2_DIS_CHROMA,  FIELD_BT(flags2)}, 
  {label:"dacdisa",     mask:BT_FLAG4_DACDISA,     FIELD_BT(flags4)}, 
  {label:"dacdisb",     mask:BT_FLAG4_DACDISB,     FIELD_BT(flags4)}, 
  {label:"dacdisc",     mask:BT_FLAG4_DACDISC,     FIELD_BT(flags4)}, 
  {label:"eclip",       mask:BT_FLAG1_ECLIP,	   FIELD_BT(flags1)}, 
  {label:NULL }
};

void bt_freq_calc_cb (GtkObject *obj, struct mask_bt_freq *m)
{
  double Fxtal = 13500000; 
  double Fclk; 
  double Fhsyn;
  double Fvsyn;
  double dALO, dTLO, dATO, dTTO, dVACTIVEO;
  double dAspect, dHOC, dVOC;
  char s [20];

  if (gui_tv_chip != NV_BROOKTREE && gui_tv_chip != NV_CONEXANT) return;

  Fclk = Fxtal * (gui_tv.bt.pll_int + 
		  gui_tv.bt.pll_fract / 65536.0) / 6.0;
  sprintf (s, "%3.2f MHz", Fclk / 1e6);
  gtk_label_set_text (m->clk, s);

  Fhsyn = Fclk / gui_tv.bt.h_clki;
  snprintf (s, 20, "%3.2f kHz", Fhsyn / 1e3);
  gtk_label_set_text (m->hsyn, s);

  Fvsyn = Fhsyn / gui_tv.bt.v_linesi;
  snprintf (s, 20, "%3.2f  Hz", Fvsyn);
  gtk_label_set_text (m->vsyn, s);

  if (gui_tv.bt.flags1 & BT_FLAG1_625LINE) {
    dALO = 288.0;
    dTLO = 312.0;
    dATO = .000052;
    dTTO = .000064;
  } else {
    dALO = 243.0;
    dTLO = 262.0;
    dATO = .00005265556;
    dTTO = .00006355556;
  }
  /* if interlace then dTLO += 0.5 */
  dVACTIVEO = (int) (((gui_tv.bt.v_activei * dTLO) + 
		      (gui_tv.bt.v_linesi - 1.0)) / 
		     gui_tv.bt.v_linesi);
  /* normally, dVACTIVEO = v_activeo - 2 */
  /* These two euqations are also possible, but that's not what happens
   *  in recalc.c, so don't use them.
   * dVOC = 1.0 - (gui_tv.bt.v_activeo - 2) / dALO;
   * dVOC = 1.0 - ((gui_tv.bt.v_activei / gui_tv.bt.v_linesi) / 
   *              (dALO / dTLO));
   */
  dVOC = 1.0 - dVACTIVEO / dALO;
  dHOC = 1.0 - ((2.0 * gui_tv.bt.h_active / gui_tv.bt.h_clko) / 
               (dATO / dTTO));
  dAspect = (1.0 - dHOC) / (1.0 - dVOC);

  snprintf (s, 20, "%06.3f %%", dHOC * 100.0);
  gtk_label_set_text (m->hover, s);
  snprintf (s, 20, "%06.3f %%", dVOC * 100.0);
  gtk_label_set_text (m->vover, s);
  snprintf (s, 20, "%07.5f", dAspect);
  gtk_label_set_text (m->aspect, s);

  /* FIXME: check above, calculate aspect */
}

gint check_fifo_cb (struct mask_status *m)
{
  int status;

  if (gui_tv_chip != NV_BROOKTREE && gui_tv_chip != NV_CONEXANT) return TRUE;
  status = backend->getStatus (2);
  if (status < 0) return TRUE;
  if (! (status & 0x100)) status = 0;
  gtk_toggle_button_set_active (m->tvon,  (status & 0x100) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->pll,   (status & 0x010) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->over,  (status & 0x008) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->under, (status & 0x004) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->pal,   (status & 0x002) ? TRUE : FALSE);
  gtk_toggle_button_set_active (m->busy,  (status & 0x001) ? TRUE : FALSE);
  return TRUE;
}

/* TODO:
 * Saturation -> cattenuate
 * Brightness -> yattenuate
 * Luminance  -> my
 * Color      -> mcb, mcr
 * Flicker/Text -> various regs 
 */

void calc_update_cb (GtkWidget *widget, struct mask_calc *m)
{
  gtk_widget_set_sensitive (GTK_WIDGET (m->calc),
			    (m->list->selection) ? TRUE : FALSE);
}

void calc_dist_min_cb (GtkAdjustment *min, GtkAdjustment *max)
{
  if (gui_mask_calc.dist->active && max->value - min->value < AUTO_DIST)
    gtk_adjustment_set_value (max, min->value + AUTO_DIST);
  if (min->value > max->value)
    gtk_adjustment_set_value (max, min->value);
}

void calc_dist_max_cb (GtkAdjustment *max, GtkAdjustment *min)
{
  if (gui_mask_calc.dist->active && max->value - min->value < AUTO_DIST)
    gtk_adjustment_set_value (min, max->value - AUTO_DIST);
  if (min->value > max->value)
    gtk_adjustment_set_value (min, max->value);
}

void calc_result_cb (double hoc, double voc, double badness,
  double aspect)
{
  char sh[10], sv[10], sb[10], sa[10];
  char *text [4];

  snprintf (sh, 10, "%06.3f %%", hoc * 100.0);
  snprintf (sv, 10, "%06.3f %%", voc * 100.0);
  snprintf (sb, 10, "%9.6f", badness);
  snprintf (sa, 10, "%7.5f", aspect);
  text [0] = sh;
  text [1] = sv;
  text [2] = sb;
  text [3] = sa;
  gtk_clist_append (gui_mask_calc.list, text);
}

void calc_list_cb (GtkWidget *widget, struct mask_calc *m)
{
  int hres, vres;
  double hoc_min, hoc_max, voc_min, voc_max;

  gtk_clist_freeze (m->list);
  gtk_clist_clear (m->list);
  hres = (int) m->hres->value;
  vres = (int) m->vres->value;
  hoc_min = m->hoc_min->value / 100.0;
  voc_min = m->voc_min->value / 100.0;
  hoc_max = m->hoc_max->value / 100.0;
  voc_max = m->voc_max->value / 100.0;
  recalc_find (gui_system, hres, vres, hoc_min, hoc_max, voc_min, voc_max, 
	       &calc_result_cb);
  gtk_clist_sort (m->list);
  gtk_clist_select_row (m->list, 0, 0);
  gtk_clist_thaw (m->list);
  calc_update_cb (NULL, m);
}

void calc_reset_cb (GtkWidget *widget, struct mask_calc *m)
{
  if (gui_tv_chip != NV_BROOKTREE && gui_tv_chip != NV_CONEXANT) return;
  if (gui_list_mode) {
    gtk_adjustment_set_value (m->hres, gui_list_mode->res_x);
    gtk_adjustment_set_value (m->vres, gui_list_mode->res_y);
    gtk_adjustment_set_value (m->hoc_min, gui_list_mode->hoc - AUTO_DIST/2.0);
    gtk_adjustment_set_value (m->hoc_max, gui_list_mode->hoc + AUTO_DIST/2.0);
    gtk_adjustment_set_value (m->voc_min, gui_list_mode->voc - AUTO_DIST/2.0);
    gtk_adjustment_set_value (m->voc_max, gui_list_mode->voc + AUTO_DIST/2.0);
    gtk_clist_clear (m->list);
    calc_update_cb (NULL, m);
  }
}

void calc_column_cb (GtkCList *clist, gint column, struct mask_calc *m)
{
  gtk_clist_set_sort_column (m->list, column); 
  gtk_clist_sort (m->list);
  gtk_clist_set_sort_column (m->list, 2); 
}

void calc_select_cb (GtkCList *clist, gint row, gint column,
  GdkEventButton *event, struct mask_calc *m)
{
  gchar *text;

  g_return_if_fail (GTK_IS_CLIST (clist));
  gtk_clist_get_text (clist, row, 0, &text);
  gtk_label_set_text (m->hoc_sel, text);
  gtk_clist_get_text (clist, row, 1, &text);
  gtk_label_set_text (m->voc_sel, text);
}

void calc_calc_cb (GtkButton *button, struct mask_calc *m)
{
  int hres, vres;
  double hoc, voc;

  calc_mode.res_x = hres = (int) m->hres->value;
  calc_mode.res_y = vres = (int) m->vres->value;
  sscanf (m->hoc_sel->label, "%lf", &hoc); 
  sscanf (m->voc_sel->label, "%lf", &voc); 
  calc_mode.hoc = hoc;
  calc_mode.voc = voc;
  hoc /= 100.0;
  voc /= 100.0;

  recalc_custom (gui_system, hres, vres, hoc, voc, &calc_bt, &calc_crt);
  gui_act_mode_set (&calc_mode);
}

void reset_bt_cb (GtkObject *obj, gpointer data)
{
  if (gui_tv_chip != NV_BROOKTREE && gui_tv_chip != NV_CONEXANT) return;
  if (gui_act_mode) {
    gui_tv.bt = gui_act_mode->tv->bt;
    gui_tv_set ();
    gtk_signal_emit_by_name (GTK_OBJECT (update_bt), "changed");
    gtk_signal_emit_by_name (GTK_OBJECT (changed_all), "changed");
  }
}

void update_bt_cb (GtkObject *obj, gpointer data)
{
  if (gui_tv_chip != NV_BROOKTREE && gui_tv_chip != NV_CONEXANT) return;
  gtk_signal_emit_by_name (GTK_OBJECT (update_bt), "changed");
}

/* -------- GUI Pages -------- */

GtkWidget *gui_bt_flag_page (void)
{
  GtkWidget *frame;
  GtkWidget *page;
  GtkWidget *table;
  GtkWidget *button;
  int i;
  int lines;
  GtkAccelGroup *gui_bt_flag_accel_group;

  gui_bt_flag_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_bt_flag_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_bt_flag_accel_group);

  frame = gtk_frame_new ("BT 868/869 Flag Values");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(page), frame, 0,5,0,1);
  
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  lines = 11;
  for (i = 0; TRUE; i++) {
    struct reg_mask *m = bt_mask_ext + i;
    int y =  i % lines;
    int x = (i / lines) * 2;

    if (!(m->label)) break;
    gui_mask_entry (table, GTK_OBJECT (update_bt), m, x, x+1, x+2, y);
  }
  lines = 8; 
  for (i = 0; TRUE; i++) {
    struct flag_mask *m = bt_mask_flags + i;
    int y =  i % lines;
    int x = (i / lines) + 6;

    if (!(m->label)) break;
    gui_mask_checkbutton (table, GTK_OBJECT (update_bt), m, x, x+1, y);
  }

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), (gpointer) PRINT_CHIP_REGS);
  gtk_table_attach(GTK_TABLE(page), button, 3,4,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (reset_bt_cb), NULL); 
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_bt_flag_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}

GtkWidget *gui_bt_reg_page (void)
{
  GtkWidget *frame;
  GtkWidget *page;
  GtkWidget *table;
  GtkWidget *button;
  int i;
  int lines = 9;
  GtkAccelGroup *gui_bt_reg_accel_group;

  gui_bt_reg_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (1,1,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_bt_reg_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_bt_reg_accel_group);

  frame = gtk_frame_new ("BT 868/869 Register Values");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(page), frame, 0,5,0,1);
  
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  for (i = 0; TRUE; i++) {
    struct reg_mask *m = bt_mask + i;
    int y =  i % lines;
    int x = (i / lines) * 3;

    if (!(m->label)) break;
    gui_mask_entry (table, GTK_OBJECT (update_bt), m, x, x+2, x+3, y);
  }
  gui_mask_twin_entry (table, GTK_OBJECT (update_bt), &bt_mask_msc, 
    0, 2, 3, 4, lines);
  gui_mask_entry (table, GTK_OBJECT (update_bt), &bt_mask_macro, 
    6, 8, 9, lines);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);
  gtk_table_set_row_spacings (GTK_TABLE(table), 5);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), (gpointer) PRINT_CHIP_REGS);
  gtk_table_attach(GTK_TABLE(page), button, 3,4,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (reset_bt_cb), NULL); 
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_bt_reg_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}

GtkWidget *gui_bt_status_page (void)
{
  GtkWidget *page;
  GtkWidget *bbox;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *button;

  page = gtk_hbox_new (FALSE, 5);
  gtk_container_set_border_width (GTK_CONTAINER (page), 5);

  frame = gtk_frame_new ("Frequencies");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  table = gtk_table_new (3, 4, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  /* Frequencies CRT / TV */

  label = gtk_label_new ("Monitor dot clock");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,1,2, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("---.-- MHz");
  gui_mask_bt_freq.clk = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,1,2, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("Monitor horiz. sync");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,2,3, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("---.-- kHz");
  gui_mask_bt_freq.hsyn = (GtkLabel *) label; 
  gtk_table_attach (GTK_TABLE(table), label, 1,2,2,3, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("Monitor vert. sync");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,3,4, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("---.--  Hz");
  gui_mask_bt_freq.vsyn = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,3,4, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("TV horiz. overscan");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,4,5, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("--.--  %");
  gui_mask_bt_freq.hover = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,4,5, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("TV vert. overscan");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,5,6, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("--.--  %");
  gui_mask_bt_freq.vover = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,5,6, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("TV aspect ratio");
  gtk_table_attach (GTK_TABLE(table), label, 0,1,6,7, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("-.---  ");
  gui_mask_bt_freq.aspect = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 1,2,6,7, GTK_FILL,GTK_FILL,0,0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  gtk_table_set_col_spacings (GTK_TABLE(table), 10);
  gtk_table_set_row_spacings (GTK_TABLE(table), 10);

  /* BT Fifo overrun/underrun */

  frame = gtk_frame_new ("TV Status");
  gtk_box_pack_start (GTK_BOX (page), frame, TRUE, TRUE, 0);

  bbox = gtk_vbutton_box_new ();
  gtk_container_set_border_width (GTK_CONTAINER (bbox), 10);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing (GTK_BUTTON_BOX (bbox), 5);
  gtk_container_add (GTK_CONTAINER (frame), bbox);

  button = gtk_check_button_new_with_label ("TV on");
  gui_mask_status.tvon = (GtkToggleButton *) button;
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("PLL locked");
  gui_mask_status.pll = (GtkToggleButton *) button;
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("FIFO overrun");
  gui_mask_status.over = (GtkToggleButton *) button;
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("FIFO underrun");
  gui_mask_status.under = (GtkToggleButton *) button;
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("PAL pin");
  gui_mask_status.pal = (GtkToggleButton *) button;
  gtk_container_add (GTK_CONTAINER(bbox), button);

  button = gtk_check_button_new_with_label ("Busy");
  gui_mask_status.busy = (GtkToggleButton *) button;
  gtk_container_add (GTK_CONTAINER(bbox), button);

  return page;
}
  
GtkWidget *gui_bt_calc_page (void)
{
  GtkWidget *page;
  GtkWidget *frame;
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *spin;   
  GtkWidget *button;
  GtkWidget *scroller;
  GtkWidget *clist;
  GtkAdjustment *adj;
  GtkAccelGroup *gui_bt_calc_accel_group;

  gui_bt_calc_accel_group = gtk_accel_group_new ();

  page = gtk_table_new (5,2,FALSE);
  gtk_signal_connect (GTK_OBJECT (page), "map",
    GTK_SIGNAL_FUNC (gui_map_cb), (gpointer) gui_bt_calc_accel_group);
  gtk_signal_connect (GTK_OBJECT (page), "unmap",
    GTK_SIGNAL_FUNC (gui_unmap_cb), (gpointer) gui_bt_calc_accel_group);

  frame = gtk_frame_new ("Calculate registers");
  gtk_container_set_border_width (GTK_CONTAINER (frame), 5);
  gtk_table_attach_defaults(GTK_TABLE(page), frame, 0,5,0,1);
  
  table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new ("Horizontal:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  label = gtk_label_new ("Vertical:");
  gtk_table_attach (GTK_TABLE(table), label, 0,1, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);

  label = gtk_label_new ("Resolution");
  gtk_table_attach (GTK_TABLE(table), label, 1,2, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0, 
          1024.0, 1.0, 10.0, 0.0);
  gui_mask_calc.hres = adj;
  spin = gtk_spin_button_new (adj, 0.0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 1,2, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.0, 
          1024.0, 1.0, 10.0, 0.0);
  gui_mask_calc.vres = adj;
  spin = gtk_spin_button_new (adj, 0.0, 0);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 1,2, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  label = gtk_label_new ("Overscan range (%)");
  gtk_table_attach (GTK_TABLE(table), label, 2,4, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.00, 
          50.00, 0.1, 1.0, 0.0);
  gui_mask_calc.hoc_min = adj;
  spin = gtk_spin_button_new (adj, 0.0, 2);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 2,3, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.00, 
          50.00, 0.1, 1.0, 0.0);
  gui_mask_calc.hoc_max = adj;
  spin = gtk_spin_button_new (adj, 0.0, 2);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 3,4, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  gtk_signal_connect (GTK_OBJECT (gui_mask_calc.hoc_min), "value-changed",
    GTK_SIGNAL_FUNC (calc_dist_min_cb), (gpointer) gui_mask_calc.hoc_max);
  gtk_signal_connect (GTK_OBJECT (gui_mask_calc.hoc_max), "value-changed",
    GTK_SIGNAL_FUNC (calc_dist_max_cb), (gpointer) gui_mask_calc.hoc_min);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.00, 
          50.00, 0.1, 1.0, 0.0);
  gui_mask_calc.voc_min = adj;
  spin = gtk_spin_button_new (adj, 0.0, 2);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 2,3, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  adj = (GtkAdjustment *) gtk_adjustment_new (0.0, 0.00, 
          50.00, 0.1, 1.0, 0.0);
  gui_mask_calc.voc_max = adj;
  spin = gtk_spin_button_new (adj, 0.0, 2);
  gtk_spin_button_set_numeric (GTK_SPIN_BUTTON (spin), TRUE);
  gtk_spin_button_set_update_policy (GTK_SPIN_BUTTON (spin), 
				     GTK_UPDATE_IF_VALID);
  gtk_table_attach (GTK_TABLE(table), spin, 3,4, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);

  gtk_signal_connect (GTK_OBJECT (gui_mask_calc.voc_min), "value-changed",
    GTK_SIGNAL_FUNC (calc_dist_min_cb), (gpointer) gui_mask_calc.voc_max);
  gtk_signal_connect (GTK_OBJECT (gui_mask_calc.voc_max), "value-changed",
    GTK_SIGNAL_FUNC (calc_dist_max_cb), (gpointer) gui_mask_calc.voc_min);

  label = gtk_label_new ("Selected");
  gtk_table_attach (GTK_TABLE(table), label, 4,5, 0,1,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

  label = gtk_label_new ("--.--- %");
  gui_mask_calc.hoc_sel = (GtkLabel *) label;
  gtk_table_attach (GTK_TABLE(table), label, 4,5, 1,2,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);

  label = gtk_label_new ("--.--- %");
  gui_mask_calc.voc_sel = (GtkLabel *) label;
   gtk_table_attach (GTK_TABLE(table), label, 4,5, 2,3,
    GTK_FILL, GTK_FILL, 0, 0);
  gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.0);

  scroller = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroller),
				  GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
  gtk_table_attach (GTK_TABLE(table), scroller, 0,5, 3,4,
    GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

  clist = gtk_clist_new (4);
  gui_mask_calc.list = (GtkCList *) clist;
  gtk_clist_set_column_title (GTK_CLIST (clist), 0, "Horiz Overscan");
  gtk_clist_set_column_title (GTK_CLIST (clist), 1, "Vert Overscan");
  gtk_clist_set_column_title (GTK_CLIST (clist), 2, "Badness");
  gtk_clist_set_column_title (GTK_CLIST (clist), 3, "Aspect");
  gtk_clist_column_titles_show (GTK_CLIST (clist));
  gtk_clist_column_titles_active (GTK_CLIST (clist)); /* must be after show */
  gtk_clist_set_selection_mode (GTK_CLIST (clist), GTK_SELECTION_BROWSE);
  gtk_clist_set_shadow_type (GTK_CLIST (clist), GTK_SHADOW_IN); 
  gtk_clist_set_column_width (GTK_CLIST (clist), 0, 90); /* some guess */
  gtk_clist_set_column_width (GTK_CLIST (clist), 1, 90); /* some guess */
  gtk_clist_set_column_width (GTK_CLIST (clist), 2, 80); /* some guess */
  gtk_clist_set_sort_column (GTK_CLIST (clist), 2); /* badness */
  gtk_clist_set_sort_type (GTK_CLIST (clist), GTK_SORT_ASCENDING); 
  gtk_signal_connect (GTK_OBJECT (clist), "select_row",
    GTK_SIGNAL_FUNC (calc_select_cb), &gui_mask_calc);
  gtk_signal_connect (GTK_OBJECT (clist), "click_column",
    GTK_SIGNAL_FUNC (calc_column_cb), &gui_mask_calc);
  gtk_container_add(GTK_CONTAINER(scroller), clist);

  gtk_table_set_col_spacings (GTK_TABLE(table), 5);

  button = gtk_button_new_with_label ("List");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (calc_list_cb), &gui_mask_calc);
  gtk_table_attach(GTK_TABLE(page), button, 0,1,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Calc");
  gui_mask_calc.calc = (GtkButton *) button;
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (calc_calc_cb), &gui_mask_calc);
  gtk_table_attach(GTK_TABLE(page), button, 1,2,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_toggle_button_new_with_label ("AutoDist");
  gui_mask_calc.dist = (GtkToggleButton *) button;
  gtk_table_attach(GTK_TABLE(page), button, 2,3,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Print");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (gui_print_cb), 
    (gpointer) (PRINT_CHIP_REGS | PRINT_CRT_REGS));
  gtk_table_attach(GTK_TABLE(page), button, 3,4,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  button = gtk_button_new_with_label ("Reset");
  gtk_signal_connect (GTK_OBJECT (button), "clicked",
    GTK_SIGNAL_FUNC (calc_reset_cb), &gui_mask_calc);
  gtk_widget_add_accelerator (GTK_WIDGET (button), "clicked", 
    gui_bt_calc_accel_group, 
    gui_accel[ACCEL_RESET].key, gui_accel[ACCEL_RESET].mods,
    GTK_ACCEL_VISIBLE);
  gtk_table_attach(GTK_TABLE(page), button, 4,5,1,2,
    GTK_FILL, GTK_FILL, 0, 0);

  return page;
}

void gui_bt_init (void)
{
  update_bt = (GtkAdjustment *) gtk_adjustment_new (0, 0, 0, 0, 0, 0);
  gtk_signal_connect (GTK_OBJECT (update_chip), "changed", 
    GTK_SIGNAL_FUNC (update_bt_cb), NULL);
  gtk_signal_connect (GTK_OBJECT (update_mode), "changed",
    GTK_SIGNAL_FUNC (calc_reset_cb), &gui_mask_calc);
  gtk_signal_connect (GTK_OBJECT (changed_all), "changed",
    GTK_SIGNAL_FUNC (bt_freq_calc_cb), &gui_mask_bt_freq);

  /* FIXME: Only add timeout if status page is mapped */
#ifndef DISABLE_TIMEOUT
  gtk_timeout_add( 500 /* ms */, (GtkFunction) check_fifo_cb, 
		   &gui_mask_status);
#endif
}
