//                                               -*- C++ -*-
/**
 *  @file  WrapperData.cxx
 *  @brief This class declares the wrapper data that are exchanged with the platform
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: WrapperData.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <iostream>               // for std::ostream
#include <cstring>                // for std::strncpy
#include <algorithm>              // for std::copy
#include "OSS.hxx"
#include "Log.hxx"
#include "WrapperCommon.h"
#include "WrapperData.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Func
    {

      using Common::Log;


      WrapperFunctionDescription::WrapperFunctionDescription()
	: name_(),
	  provided_(WrapperSymbolProvided::NO)
      {
	// Nothing to do
      }

      String WrapperFunctionDescription::__repr__() const
      {
	OSS oss;
	oss << "name="      << name_
	    << " provided=" << WrapperProvidedAsString[provided_];
	return oss;
      }




      WrapperDataFile::WrapperDataFile()
	: id_(),
	  path_(),
	  subst_(),
	  type_(WrapperDataFileType::IN)
      {
	// Nothing to do
      }

      Bool WrapperDataFile::operator == (const WrapperDataFile & other) const
      {
	return (id_ == other.id_) && (path_ == other.path_) && (subst_ == other.subst_) && (type_ == other.type_);
      }

      String WrapperDataFile::__repr__() const
      {
	OSS oss;
	oss << "id="     << id_
	    << " name="  << name_
	    << " path="  << path_
	    << " subst=" << subst_
	    << " type="  << WrapperListElementTypeAsString[type_];
	return oss;
      }



      WrapperFrameworkData::WrapperFrameworkData()
	: studyid_(0),
	  studycase_(),
	  componentname_()
      {
	// Nothing to do
      }

      String WrapperFrameworkData::__repr__() const
      {
	OSS oss;
	oss << "studyid=" << studyid_
	    << " studycase=" << studycase_
	    << " componentname=" << componentname_;
	return oss;
      }






      WrapperDataVariable::WrapperDataVariable()
	: id_(),
	  comment_(),
	  unit_(),
	  regexp_(),
	  format_(),
	  type_(WrapperDataVariableType::IN),
	  gradient_(WrapperComputedGradient::NO)
      {
	// Nothing to do
      }


      Bool WrapperDataVariable::operator == (const WrapperDataVariable & other) const
      {
	return (id_ == other.id_) && (comment_ == other.comment_) && (unit_ == other.unit_) && 
	  (regexp_ == other.regexp_) && (format_ == other.format_) && (type_ == other.type_) &&
	  (gradient_ == other.gradient_);
      }

      String WrapperDataVariable::__repr__() const
      {
	OSS oss;
	oss << "id="        << id_
	    << " comment="  << comment_
	    << " unit="     << unit_
	    << " regexp="   << regexp_
	    << " format="   << format_
	    << " type="     << WrapperListElementTypeAsString[type_]
	    << " gradient=" << WrapperProvidedAsString[gradient_];
	return oss;
      }




      WrapperParameter::WrapperParameter()
	: state_(WrapperState::SHARED),
	  mode_(WrapperMode::STATICLINK),
	  in_(WrapperDataTransfer::FILES),
	  out_(WrapperDataTransfer::FILES),
	  command_()
      {
	// Nothing to do
      }


      String WrapperParameter::__repr__() const
      {
	OSS oss;
	oss << "state="    << WrapperConfigurationStateAsString[state_]
	    << " mode="    << WrapperConfigurationModeAsString[mode_]
	    << " in="      << WrapperDataTransferModeAsString[in_]
	    << " out="     << WrapperDataTransferModeAsString[out_]
	    << " command=" << command_;
	return oss;
      }


      CLASSNAMEINIT(WrapperData);

      /* Default constructor */
      WrapperData::WrapperData()
	: Object(),
	  libraryPath_(),
	  function_(),
	  gradient_(),
	  hessian_(),
	  p_fileList_(new WrapperData::FileListType),
	  p_variableList_(new WrapperData::VariableListType),
	  p_parameters_(new WrapperParameter),
	  p_framework_(new WrapperFrameworkData)
      {
	// Nothing to do
      }


      /* Constructor from C structure */
      WrapperData::WrapperData(const struct WrapperExchangedData * p_exchangedData)
	: Object(),
	  libraryPath_(),
	  function_(),
	  gradient_(),
	  hessian_(),
	  p_fileList_(new WrapperData::FileListType),
	  p_variableList_(new WrapperData::VariableListType),
	  p_parameters_(new WrapperParameter),
	  p_framework_(new WrapperFrameworkData)
      {
	if (  p_exchangedData ) {
	  printWrapperExchangedData( "WrapperData::WrapperData(const struct WrapperExchangedData * p_exchangedData)", p_exchangedData );

	  // libraryPath, function, gradient, hessian and platform are not handled by struct WrapperExchangedData
	  // so they can't be set here.

	  // Set file list
	  const struct WrapperFileList *  currentFileElement  = p_exchangedData->fileList_;
	  while ( currentFileElement ) {
	    WrapperDataFile aFile;
	    aFile.id_    = currentFileElement->file_->id_;
	    aFile.name_  = currentFileElement->file_->name_;
	    aFile.path_  = currentFileElement->file_->path_;
	    aFile.subst_ = currentFileElement->file_->subst_;
	    aFile.type_  = currentFileElement->file_->type_;
	    Log::Debug( OSS() << "Add aFile = " << aFile );
	    p_fileList_->add( aFile );

	    currentFileElement = currentFileElement->next_;
	  } // end while (currentFileElement)

	  // Set variable list
	  const struct WrapperVariableList *  currentVariableElement  = p_exchangedData->variableList_;
	  while ( currentVariableElement ) {
	    WrapperDataVariable aVariable;
	    aVariable.id_       = currentVariableElement->variable_->id_;
	    aVariable.comment_  = currentVariableElement->variable_->comment_;
	    aVariable.unit_     = currentVariableElement->variable_->unit_;
	    aVariable.regexp_   = currentVariableElement->variable_->regexp_;
	    aVariable.format_   = currentVariableElement->variable_->format_;
	    aVariable.type_     = currentVariableElement->variable_->type_;
	    aVariable.gradient_ = currentVariableElement->variable_->gradient_;
	    Log::Debug( OSS() << "Add aVariable = " << aVariable );
	    p_variableList_->add( aVariable );

	    currentVariableElement = currentVariableElement->next_;
	  } // end while (currentVariableElement)

	  // Set parameters
	  if ( p_exchangedData->parameters_ ) {
	    p_parameters_->state_   = p_exchangedData->parameters_->state_;
	    p_parameters_->mode_    = p_exchangedData->parameters_->mode_;
	    p_parameters_->in_      = p_exchangedData->parameters_->in_;
	    p_parameters_->out_     = p_exchangedData->parameters_->out_;
	    p_parameters_->command_ = p_exchangedData->parameters_->command_;
	    Log::Debug( OSS() << "Add * p_parameters = " << *p_parameters_ );
	  }
	
	  // Framework isn't copied : not justified for the moment
	}
      }



      /* String converter */
      String WrapperData::__repr__() const
      {	OSS oss;
	oss << "libraryPath=" << libraryPath_
	    << " function={"  << function_
	    << "} gradient={" << gradient_
	    << "} hessian={"  << hessian_
	    << "} fileList=[";

	std::copy( p_fileList_->begin(), p_fileList_->end(), OSS_iterator<WrapperDataFile>(oss, ", ") );

	oss << "] variableList=[";

	std::copy( p_variableList_->begin(), p_variableList_->end(), OSS_iterator<WrapperDataVariable>(oss, ", ") );

	oss << "] parameters={" << *p_parameters_
	    << "} framework={" << *p_framework_
	    << "}";
	return oss;

      }


      /* Library path accessor */
      void WrapperData::setLibraryPath(const FileName & path)
      {
	libraryPath_ = path;
      }

      /* Library path accessor */
      FileName WrapperData::getLibraryPath() const
      {
	return libraryPath_;
      }

      /* Function description accessor */
      void WrapperData::setFunctionDescription(const WrapperFunctionDescription & funcDescription)
      {
	function_ = funcDescription;
      }

      /* Function description accessor */
      WrapperFunctionDescription WrapperData::getFunctionDescription() const
      {
	return function_;
      }

      /* Gradient description accessor */
      void WrapperData::setGradientDescription(const WrapperFunctionDescription & gradDescription)
      {
	gradient_ = gradDescription;
      }

      /* Gradient description accessor */
      WrapperFunctionDescription WrapperData::getGradientDescription() const
      {
	return gradient_;
      }

      /* Hessian description accessor */
      void WrapperData::setHessianDescription(const WrapperFunctionDescription & hessDescription)
      {
	hessian_ = hessDescription;
      }

      /* Hessian description accessor */
      WrapperFunctionDescription WrapperData::getHessianDescription() const
      {
	return hessian_;
      }

      /* Accessor */
      void WrapperData::setFileList(const WrapperData::FileListType & fileList)
      {
	p_fileList_.reset(new WrapperData::FileListType(fileList));
      }



      /* Accessor */
      const WrapperData::FileListType & WrapperData::getFileList() const
      {
	return *p_fileList_;
      }


      /* Conversion method for C interface */
      struct WrapperFileList * WrapperData::getNewFileListForCInterface() const
      {
	struct WrapperFileList * fileList = 0;

	// we convert the data into the exchange format
	struct WrapperFileList ** previousElementPointer = & fileList;
	FileListType::const_iterator fileList_iterator;
	for(fileList_iterator  = getFileList().begin();
	    fileList_iterator != getFileList().end();
	    fileList_iterator++) {
	  WrapperDataFile file = *fileList_iterator;

	  struct WrapperFileList * p_linkOnFile    = new WrapperFileList;
	  p_linkOnFile->next_ = 0;
	  p_linkOnFile->file_ = new WrapperFileListElement;

	  p_linkOnFile->file_->id_ = new char[file.id_.size() + 1];
	  strncpy(p_linkOnFile->file_->id_, file.id_.c_str(), file.id_.size() + 1);

	  p_linkOnFile->file_->name_ = new char[file.name_.size() + 1];
	  strncpy(p_linkOnFile->file_->name_, file.name_.c_str(), file.name_.size() + 1);

	  p_linkOnFile->file_->path_ = new char[file.path_.size() + 1];
	  strncpy(p_linkOnFile->file_->path_, file.path_.c_str(), file.path_.size() + 1);

	  p_linkOnFile->file_->subst_ = new char[file.subst_.size() + 1];
	  strncpy(p_linkOnFile->file_->subst_, file.subst_.c_str(), file.subst_.size() + 1);

	  p_linkOnFile->file_->type_ = file.type_;

	  // we insert the element into the list...
	  *previousElementPointer = p_linkOnFile;
	  
	  // ... then we jump to the next element
	  previousElementPointer = & p_linkOnFile->next_;

	} /* end for */

	return fileList;
      }


      /* Frees the memory allocated by getNewFileListForCInterface() method */
      void WrapperData::FreeFileListForCInterface(const struct WrapperFileList * fileList)
      {
	const struct WrapperFileList * currentFileElement = fileList;
	while (currentFileElement) {
	  const struct WrapperFileList * nextElement = currentFileElement->next_;
	  delete [] currentFileElement->file_->id_;
	  delete [] currentFileElement->file_->name_;
	  delete [] currentFileElement->file_->path_;
	  delete [] currentFileElement->file_->subst_;
	  delete currentFileElement->file_;
	  delete currentFileElement;
	  currentFileElement = nextElement;
	}
      }



      /* Accessor */
      void WrapperData::setVariableList(const WrapperData::VariableListType & variableList)
      {
	p_variableList_.reset(new WrapperData::VariableListType(variableList));
      }



      /* Accessor */
      const WrapperData::VariableListType & WrapperData::getVariableList() const
      {
	return *p_variableList_;
      }


      /* Conversion method for C interface */
      struct WrapperVariableList * WrapperData::getNewVariableListForCInterface() const
      {
	struct WrapperVariableList * variableList = 0;

	// we convert the data into the exchange format
	struct WrapperVariableList ** previousElementPointer = & variableList;
	VariableListType::const_iterator variableList_iterator;
	for(variableList_iterator  = getVariableList().begin();
	    variableList_iterator != getVariableList().end();
	    variableList_iterator++) {
	  WrapperDataVariable variable = *variableList_iterator;

	  struct WrapperVariableList * p_linkOnVariable    = new WrapperVariableList;
	  p_linkOnVariable->next_ = 0;
	  p_linkOnVariable->variable_ = new WrapperVariableListElement;

	  p_linkOnVariable->variable_->id_ = new char[variable.id_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->id_, variable.id_.c_str(), variable.id_.size() + 1);

	  p_linkOnVariable->variable_->comment_ = new char[variable.comment_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->comment_, variable.comment_.c_str(), variable.comment_.size() + 1);

	  p_linkOnVariable->variable_->unit_ = new char[variable.unit_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->unit_, variable.unit_.c_str(), variable.unit_.size() + 1);

	  p_linkOnVariable->variable_->regexp_ = new char[variable.regexp_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->regexp_, variable.regexp_.c_str(), variable.regexp_.size() + 1);

	  p_linkOnVariable->variable_->format_ = new char[variable.format_.size() + 1];
	  strncpy(p_linkOnVariable->variable_->format_, variable.format_.c_str(), variable.format_.size() + 1);

	  p_linkOnVariable->variable_->type_ = variable.type_;

	  p_linkOnVariable->variable_->gradient_ = variable.gradient_;

	  // we insert the element into the list...
	  *previousElementPointer = p_linkOnVariable;
	  
	  // ... then we jump to the next element
	  previousElementPointer = & p_linkOnVariable->next_;

	} /* end for */

	return variableList;
      }


      /* Frees the memory allocated by getNewVariableListForCInterface() method */
      void WrapperData::FreeVariableListForCInterface(const struct WrapperVariableList * variableList)
      {
	const struct WrapperVariableList * currentVariableElement = variableList;
	while (currentVariableElement) {
	  const struct WrapperVariableList * nextElement = currentVariableElement->next_;
	  delete [] currentVariableElement->variable_->id_;
	  delete [] currentVariableElement->variable_->comment_;
	  delete [] currentVariableElement->variable_->unit_;
	  delete [] currentVariableElement->variable_->regexp_;
	  delete [] currentVariableElement->variable_->format_;
	  delete currentVariableElement->variable_;
	  delete currentVariableElement;
	  currentVariableElement = nextElement;
	}
      }



      /* Accessor */
      void WrapperData::setParameters(const WrapperParameter & parameters)
      {
	p_parameters_.reset(new WrapperParameter(parameters));
      }

      const WrapperParameter & WrapperData::getParameters() const
      {
	return *p_parameters_;
      }
      

      /* Conversion method for C interface */
      struct WrapperConfiguration * WrapperData::getNewParametersForCInterface() const
      {
	struct WrapperConfiguration * parameters = new WrapperConfiguration;

	parameters->state_ = p_parameters_->state_;
	parameters->mode_  = p_parameters_->mode_;
	parameters->in_    = p_parameters_->in_;
	parameters->out_   = p_parameters_->out_;

	parameters->command_ = new char[p_parameters_->command_.size() + 1];
	strncpy(parameters->command_, p_parameters_->command_.c_str(), p_parameters_->command_.size() + 1);

	return parameters;
      }
      
      /* Frees the memory allocated by getNewParametersForCInterface() method */
      void WrapperData::FreeParametersForCInterface(const struct WrapperConfiguration * parameters)
      {
	delete [] parameters->command_;
	delete parameters;
      }
      




      /* Accessor */
      void WrapperData::setFrameworkData(const WrapperFrameworkData & framework)
      {
	p_framework_.reset(new WrapperFrameworkData(framework));
      }

      const WrapperFrameworkData & WrapperData::getFrameworkData() const
      {
	return *p_framework_;
      }


      /* Conversion method for C interface */
      struct FrameworkData * WrapperData::getNewFrameworkForCInterface() const
      {
	struct FrameworkData * framework = new FrameworkData;

	framework->studyid_ = p_framework_->studyid_;

	framework->studycase_ = new char[p_framework_->studycase_.size() + 1];
	strncpy(framework->studycase_, p_framework_->studycase_.c_str(), p_framework_->studycase_.size() + 1);

	framework->componentname_ = new char[p_framework_->componentname_.size() + 1];
	strncpy(framework->componentname_, p_framework_->componentname_.c_str(), p_framework_->componentname_.size() + 1);

	return framework;
      }
      
      /* Frees the memory allocated by getNewFrameworkForCInterface() method */
      void WrapperData::FreeFrameworkForCInterface(const struct FrameworkData * framework)
      {
	delete [] framework->studycase_;
	delete [] framework->componentname_;
	delete framework;
      }





      /* Check the correctness of the stored data */
      Bool WrapperData::isValid() const
      {
	Bool valid( true );

	if (libraryPath_.empty()) {
	  Log::Debug( OSS() << "Wrapper data validation failed: library path is empty" );
	  valid = false;
	}

	if ( (function_.provided_ == WrapperSymbolProvided::YES) &&
	     (function_.name_.empty() ) ) {
	  Log::Debug( OSS() << "Wrapper data validation failed: function name is set but empty" );
	  valid = false;
	}

	if ( (gradient_.provided_ == WrapperSymbolProvided::YES) &&
	     (gradient_.name_.empty() ) ) {
	  Log::Debug( OSS() << "Wrapper data validation failed: gradient name is set but empty" );
	  valid = false;
	}

	if ( (hessian_.provided_ == WrapperSymbolProvided::YES) &&
	     (hessian_.name_.empty() ) ) {
	  Log::Debug( OSS() << "Wrapper data validation failed: hessian name is set but empty" );
	  valid = false;
	}

	for (VariableListType::const_iterator vit = p_variableList_->begin() ;
	     vit != p_variableList_->end() ;
	     ++vit ) {
	  if ( (*vit).id_.empty() ) {
	    Log::Debug( OSS() << "Wrapper data validation failed: a variable has no id" );
	    valid = false;
	  }
	}

	for (FileListType::const_iterator fit = p_fileList_->begin() ;
	     fit != p_fileList_->end() ;
	     ++fit ) {
	  if ( (*fit).id_.empty() ) {
	    Log::Debug( OSS() << "Wrapper data validation failed: a file has no id" );
	    valid = false;
	  }
	  if ( (*fit).path_.empty() ) {
	    Log::Debug( OSS() << "Wrapper data validation failed: a file has no path (id='" << (*fit).id_ << "')" );
	    valid = false;
	  }
	}

	return valid;
      }


    } /* namespace Func */
  } /* namespace Base */
} /* namespace OpenTURNS */
