//                                               -*- C++ -*-
/**
 *  @file  KernelSmoothing.cxx
 *  @brief This class acts like a KernelMixture factory, implementing a
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: KernelSmoothing.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "KernelSmoothing.hxx"
#include "Normal.hxx"
#include "KernelMixture.hxx"
#include "TruncatedDistribution.hxx"
#include "PersistentObjectFactory.hxx"
#include "Brent.hxx"
#include "MethodBoundNumericalMathEvaluationImplementation.hxx"
#include "NumericalMathFunction.hxx"
#include "HermiteFactory.hxx"
#include "UniVariatePolynomial.hxx"
#include "SpecFunc.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /**
       * @class KernelSmoothing
       *
       * The class describes the probabilistic concept of KernelSmoothing.
       */

      CLASSNAMEINIT(KernelSmoothing);

      static Base::Common::Factory<KernelSmoothing> RegisteredFactory("KernelSmoothing");

      const UnsignedLong KernelSmoothing::SmallSize = 250;

      typedef Base::Solver::Brent                    Brent;
      typedef Base::Func::UniVariatePolynomial       UniVariatePolynomial;
      typedef Uncertainty::Algorithm::HermiteFactory HermiteFactory;

      /* Default constructor */
      KernelSmoothing::KernelSmoothing(const String & name):
        PersistentObject(name),
        bandwidth_(NumericalPoint(0)),
        kernel_(Normal())
      {
        // Nothing to do
      }

      /* Default constructor */
      KernelSmoothing::KernelSmoothing(const Distribution & kernel, const String & name) /* throw (InvalidArgumentException) */:
        PersistentObject(name),
        bandwidth_(NumericalPoint(0)),
        kernel_(kernel)
      {
        // Only 1D kernel allowed here
        if (kernel.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: only 1D kernel allowed for product kernel smoothing";
      }

      /* Virtual constructor */
      KernelSmoothing * KernelSmoothing::clone() const
      {
        return new KernelSmoothing(*this);
      }

      /* Compute the bandwidth according to Silverman's rule */
      KernelSmoothing::NumericalPoint KernelSmoothing::computeSilvermanBandwidth(const NumericalSample & sample)
      {
        UnsignedLong dimension(sample.getDimension());
        UnsignedLong size(sample.getSize());
        NumericalPoint standardDeviations(sample.computeStandardDeviationPerComponent());
        // Silverman's Normal rule
        NumericalScalar factor(pow(size, -1.0 / (4.0 + dimension)) / kernel_.getStandardDeviation()[0]);
        // Scott's Normal rule
        return factor * standardDeviations;
      }

      struct PluginConstraint
      {
        /** Constructor from a sample and a derivative factor estimate */
        PluginConstraint(const KernelSmoothing::NumericalSample & sample,
                         const NumericalScalar K,
                         const UnsignedLong order):
          sample_(sample),
          K_(K),
          order_(order),
          normalizationFactor_(sqrt(SpecFunc::Gamma(order + 1) / (2.0 * M_PI))),
          hermitePolynomial_(HermiteFactory().build(order) * normalizationFactor_)
        {
          // Nothing to do
        };

        /** Compute the derivative estimate based on the given bandwidth */
        NumericalScalar computePhi(const NumericalScalar h) const
        {
          const UnsignedLong size(sample_.getSize());
          NumericalScalar phi(size * hermitePolynomial_(0.0));
          for (UnsignedLong i = 1; i < size; ++i)
            {
              for (UnsignedLong j = 0; j < i; ++j)
                {
                  const NumericalScalar dx(sample_[i][0] - sample_[j][0]);
                  const NumericalScalar x(dx / h);
                  phi += (hermitePolynomial_(x) + hermitePolynomial_(-x)) * exp(-0.5 * x * x);
                }
            }
          const NumericalScalar res(phi / ((size * (size - 1.0)) * pow(h, order_)));
          return res;
        }

        /** Compute the constraint for the plugin bandwidth */
        KernelSmoothing::NumericalPoint bandwidthConstraint(const KernelSmoothing::NumericalPoint & x) const
        {
          const NumericalScalar h(x[0]);
          const NumericalScalar factor(computePhi(K_ * pow(h, 5.0 / 7.0)));
          const NumericalScalar res(h - pow(2.0 * sqrt(M_PI) * factor * sample_.getSize(), -1.0 / 5.0));
          return KernelSmoothing::NumericalPoint(1, res);
        }

        KernelSmoothing::NumericalSample sample_;
        NumericalScalar K_;
        UnsignedLong order_;
        NumericalScalar normalizationFactor_;
        UniVariatePolynomial hermitePolynomial_;
      };

      /* Compute the bandwidth according to the plugin rule */
      KernelSmoothing::NumericalPoint KernelSmoothing::computePluginBandwidth(const NumericalSample & sample)
      {
        const UnsignedLong dimension(sample.getDimension());
        if (dimension != 1) throw InvalidArgumentException(HERE) << "Error: plugin bandwidth is available only for 1D sample";
        const UnsignedLong size(sample.getSize());
	// Approximate the derivatives by smoothing under the Normal assumption
        const NumericalScalar sd(sample.computeStandardDeviationPerComponent()[0]);
        const NumericalScalar phi6Normal(-15.0 / (16.0 * sqrt(M_PI)) * pow(sd, -7.0));
        const NumericalScalar phi8Normal(105.0 / (32.0 * sqrt(M_PI)) * pow(sd, -9.0));
        const NumericalScalar g1(pow(-6.0 / (sqrt(2.0 * M_PI) * phi6Normal * size), 1.0 / 7.0));
        const NumericalScalar g2(pow(30.0 / (sqrt(2.0 * M_PI) * phi8Normal * size), 1.0 / 9.0));
        const NumericalScalar phi4(PluginConstraint(sample, 1.0, 4).computePhi(g1));
        const NumericalScalar phi6(PluginConstraint(sample, 1.0, 6).computePhi(g2));
        const NumericalScalar K(pow(-6.0 * sqrt(2.0) * phi4 / phi6, 1.0 / 7.0));
        PluginConstraint constraint(sample, K, 4);
        const Base::Func::NumericalMathFunction f(Base::Func::bindMethod<PluginConstraint, NumericalPoint, NumericalPoint>(constraint, &PluginConstraint::bandwidthConstraint, 1, 1));
	// Find a bracketing interval
	NumericalScalar a(g1);
	NumericalScalar b(g1);
	NumericalScalar fA(f(NumericalPoint(1, a))[0]);
	NumericalScalar fB(f(NumericalPoint(1, b))[0]);
	// While f has the same sign at the two bounds, update the interval
	while ((fA * fB > 0.0))
	  {
	    a = 0.5 * a;
	    fA = f(NumericalPoint(1, a))[0];
	    if (fA * fB <= 0.0) break;
	    b = 2.0 * b;
	    fB = f(NumericalPoint(1, b))[0];
	  }
	// Solve loosely the constraint equation
	Brent solver(0.0, 1.0e-3, 50);
        return NumericalPoint(1, solver.solve(f, 0.0, a, b) / kernel_.getStandardDeviation()[0]);
      }

      /* Compute the bandwidth according to a mixed rule:
       * simply use the plugin rule for small sample, and
       * estimate the ratio between the plugin rule and
       * the Silverman rule on a small sample, then
       * scale the Silverman bandwidth computed on the full
       * sample with this ratio
       */
      KernelSmoothing::NumericalPoint KernelSmoothing::computeMixedBandwidth(const NumericalSample & sample)
      {
        const UnsignedLong dimension(sample.getDimension());
        if (dimension != 1) throw InvalidArgumentException(HERE) << "Error: mixed bandwidth is available only for 1D sample";
        const UnsignedLong size(sample.getSize());
        // Small sample, just return the plugin bandwidth
        if (size <= SmallSize) return computePluginBandwidth(sample);
        NumericalSample smallSample(SmallSize, 1);
        for (UnsignedLong i = 0; i < SmallSize; ++i) smallSample[i][0] = sample[i][0];
        const NumericalScalar h1(computePluginBandwidth(smallSample)[0]);
        const NumericalScalar h2(computeSilvermanBandwidth(smallSample)[0]);
        return computeSilvermanBandwidth(sample) * (h1 / h2);
      }

      /* Build a Normal kernel mixture based on the given sample. If no bandwith has already been set, Silverman's rule is used */
      KernelSmoothing::Distribution KernelSmoothing::buildImplementation(const NumericalSample & sample, const Bool boundaryCorrection)
      {
        // For 1D sample, use the rule that give the best tradeoff between speed and precision
        if (sample.getDimension() == 1) return buildImplementation(sample, computeMixedBandwidth(sample), boundaryCorrection);
        // For nD sample, use the only available rule
        return buildImplementation(sample, computeSilvermanBandwidth(sample), boundaryCorrection);
      }

      /* Build a Normal kernel mixture based on the given sample and bandwidth */
      KernelSmoothing::Distribution KernelSmoothing::buildImplementation(const NumericalSample & sample,
                                                                         const NumericalPoint & bandwidth,
                                                                         const Bool boundaryCorrection)
      /* throw(InvalidDimensionException, InvalidArgumentException) */
      {
        const UnsignedLong dimension(sample.getDimension());
        if (bandwidth.getDimension() != dimension) throw InvalidDimensionException(HERE) << "Error: the given bandwidth must have the same dimension as the given sample, here bandwidth dimension=" << bandwidth.getDimension() << " and sample dimension=" << dimension;
        setBandwidth(bandwidth);
        // Make cheap boundary correction by extending the sample. Only valid for 1D sample.
        if (boundaryCorrection && (dimension == 1))
          {
            NumericalScalar min(sample.getMin()[0]);
            NumericalScalar max(sample.getMax()[0]);
            NumericalScalar h(bandwidth[0]);
            // Reflect and add points close to the boundaries to the sample
            NumericalSample newSample(sample);
            const UnsignedLong size(sample.getSize());
            for (UnsignedLong i = 0; i < size; i++)
              {
                NumericalScalar realization(sample[i][0]);
                if (realization <= min + h) newSample.add(NumericalPoint(1, 2.0 * min - realization));
                if (realization >= max - h) newSample.add(NumericalPoint(1, 2.0 * max - realization));
              }
            TruncatedDistribution kernelMixture(KernelMixture(kernel_, bandwidth, newSample), min, max);
            return kernelMixture;
          }
        KernelMixture kernelMixture(kernel_, bandwidth, sample);
        kernelMixture.setName("Kernel smoothing from sample " + sample.getName());
        return kernelMixture;
      }

      /* Bandwidth accessor */
      void KernelSmoothing::setBandwidth(const NumericalPoint & bandwidth)
      /* throw(InvalidArgumentException) */
      {
        // Check the given bandwidth
        for (UnsignedLong i = 0; i < bandwidth.getDimension(); i++)
          {
            if (bandwidth[i] <= 0.0) throw InvalidArgumentException(HERE) << "Error: the bandwidth must be > 0, here bandwith=" << bandwidth;
          }
        bandwidth_ = bandwidth;
      }

      KernelSmoothing::NumericalPoint KernelSmoothing::getBandwidth() const
      {
        return bandwidth_;
      }

      KernelSmoothing::Distribution KernelSmoothing::getKernel() const
      {
        return kernel_;
      }

      /* Method save() stores the object through the StorageManager */
      void KernelSmoothing::save(StorageManager::Advocate & adv) const
      {
        PersistentObject::save(adv);
        adv.saveAttribute( "bandwidth_", bandwidth_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void KernelSmoothing::load(StorageManager::Advocate & adv)
      {
        PersistentObject::load(adv);
        adv.loadAttribute( "bandwidth_", bandwidth_ );
      }

    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
