/*
 * print.c
 *
 * 
 * Author: 
 *  Richard Hult <rhult@hem.passagen.se>
 * 
 *  http://www.dtek.chalmers.se/~d4hult/oregano/ 
 * 
 * Copyright (C) 1999,2000  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <math.h>
#include <gnome.h>
#include <libgnomeprint/gnome-printer.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-printer-dialog.h>
#include "schematic.h"
#include "schematic-view.h"
#include "sheet.h"
#include "node-store.h"
#include "print.h"

#include <libgnomeprint/gnome-print-master.h>
#include <libgnomeprint/gnome-print-master-preview.h>


/*
 * This is stolen from Dia:
 */
#define ELLIPSE_CTRL1 0.26521648984
#define ELLIPSE_CTRL2 0.519570402739
#define ELLIPSE_CTRL3 M_SQRT1_2
#define ELLIPSE_CTRL4 0.894643159635

void
print_draw_ellipse (GnomePrintContext *ctx, ArtPoint *center, gdouble width, gdouble height)
{
	gdouble x1 = center->x - width/2,  x2 = center->x + width/2;
	gdouble y1 = center->y - height/2, y2 = center->y + height/2;
	gdouble cw1 = ELLIPSE_CTRL1 * width / 2, cw2 = ELLIPSE_CTRL2 * width / 2;
	gdouble cw3 = ELLIPSE_CTRL3 * width / 2, cw4 = ELLIPSE_CTRL4 * width / 2;
	gdouble ch1 = ELLIPSE_CTRL1 * height / 2, ch2 = ELLIPSE_CTRL2 * height / 2;
	gdouble ch3 = ELLIPSE_CTRL3 * height / 2, ch4 = ELLIPSE_CTRL4 * height / 2;

	gnome_print_moveto (ctx, x1, center->y);
	gnome_print_curveto (ctx,
			    x1,              center->y - ch1,
			    center->x - cw4, center->y - ch2,
			    center->x - cw3, center->y - ch3);
	gnome_print_curveto (ctx,
			    center->x - cw2, center->y - ch4,
			    center->x - cw1, y1,
			    center->x,       y1);
	gnome_print_curveto (ctx,
			    center->x + cw1, y1,
			    center->x + cw2, center->y - ch4,
			    center->x + cw3, center->y - ch3);
	gnome_print_curveto (ctx,
			    center->x + cw4, center->y - ch2,
			    x2,              center->y - ch1,
			    x2,              center->y);
	gnome_print_curveto (ctx,
			    x2,              center->y + ch1,
			    center->x + cw4, center->y + ch2,
			    center->x + cw3, center->y + ch3);
	gnome_print_curveto (ctx,
			    center->x + cw2, center->y + ch4,
			    center->x + cw1, y2,
			    center->x,       y2);
	gnome_print_curveto (ctx,
			    center->x - cw1, y2,
			    center->x - cw2, center->y + ch4,
			    center->x - cw3, center->y + ch3);
	gnome_print_curveto (ctx,
			    center->x - cw4, center->y + ch2,
			    x1,              center->y + ch1,
			    x1,              center->y);
}

void
print_draw_text (GnomePrintContext *ctx, const gchar *text, ArtPoint *pos)
{
	gnome_print_moveto (ctx, pos->x, pos->y);
	
	gnome_print_gsave (ctx);
	gnome_print_scale (ctx, 1, -1);
	gnome_print_show (ctx, text);
	gnome_print_grestore (ctx);
}

static void
print_page (NodeStore *store, OreganoPrintContext *opc, ArtDRect *bounds,
	    gdouble lmargin, gdouble bmargin, gdouble scale)
{
	if (node_store_count_items (store, bounds) == 0) {
		return;
	}

	gnome_print_beginpage (opc->ctx, "");
	
	gnome_print_gsave (opc->ctx);
	
	gnome_print_scale (opc->ctx, 28.346457 * scale, -28.346457 * scale);
	gnome_print_translate (opc->ctx, lmargin/scale - bounds->x0,
			       -bmargin/scale - bounds->y1);
	
	gnome_print_newpath (opc->ctx);
	gnome_print_moveto (opc->ctx, bounds->x0, bounds->y0);
	gnome_print_lineto (opc->ctx, bounds->x1, bounds->y0);
	gnome_print_lineto (opc->ctx, bounds->x1, bounds->y1);
	gnome_print_lineto (opc->ctx, bounds->x0, bounds->y1);
	gnome_print_lineto (opc->ctx, bounds->x0, bounds->y0);
	gnome_print_clip (opc->ctx);
	
	gnome_print_newpath (opc->ctx);
	node_store_print_items (store, opc, bounds);
	
	gnome_print_grestore (opc->ctx);
	gnome_print_showpage (opc->ctx);
}

static void
paginate (SchematicView *sv, OreganoPrintContext *opc,
	  const GnomePaper *paper, gdouble scale)
{
	ArtDRect rect;
	const GnomeUnit *unit;
	gdouble pswidth, psheight, lmargin, tmargin, rmargin, bmargin;
	gdouble width, height;
	gdouble x, y;
	Schematic *sm;
	NodeStore *store;

	unit = gnome_unit_with_name ("Centimeter");
	pswidth = gnome_paper_convert (gnome_paper_pswidth (paper), unit);
	psheight = gnome_paper_convert (gnome_paper_psheight (paper), unit);
	lmargin = gnome_paper_convert (gnome_paper_lmargin (paper), unit);
	tmargin = gnome_paper_convert (gnome_paper_tmargin (paper), unit);
	rmargin = gnome_paper_convert (gnome_paper_rmargin (paper), unit);
	bmargin = gnome_paper_convert (gnome_paper_bmargin (paper), unit);
	
	/* Printable area. */
	width = pswidth - lmargin - rmargin;
	height = psheight - tmargin - bmargin;
	
	width /= scale;
	height /= scale;

	sm = schematic_view_get_schematic (sv);
	store = schematic_get_store (sm);
	node_store_get_bounds (store, &rect);

	/* The bounds are a bit off so we add a bit for now. */
	rect.x0 -= 20;
	rect.y0 -= 20;
	rect.x1 -= 20;
	rect.y1 -= 20;

	for (y = rect.y0; y < rect.y1; y += height) {
		for (x = rect.x0; x < rect.x1; x += width) {
			ArtDRect page_rect;

			page_rect.x0 = x;
			page_rect.x1 = x + width;
			page_rect.y0 = y;
			page_rect.y1 = y + height;
			
			print_page (store, opc, &page_rect, lmargin, bmargin, scale);
		}
	}	
}

void
print_schematic (GtkWidget *w, gboolean preview_only)
{
	GtkWidget *dialog, *notebook;
	GtkWidget *printersel, *papersel, *scalewid;
	GtkWidget *label, *box, *frame;
	SchematicView *sv;
	GnomePrinter *printer;
	gchar *paper_name;
	gdouble scale;
	GnomePrintMaster *print_master;
	GnomePrintContext *ctx;
	const GnomePaper *paper;
	OreganoPrintContext *opc;
	gboolean preview;
       	int btn;
	
	sv = SCHEMATIC_VIEW (w);

	scale = 1.0;
	dialog = NULL;
	printer = NULL;
	paper_name = NULL;

	if (!preview_only) {
		dialog = gnome_dialog_new (_("Print Schematic"), GNOME_STOCK_BUTTON_OK, _("Print preview"), 
					   GNOME_STOCK_BUTTON_CANCEL, NULL);
		notebook = gtk_notebook_new ();
		gtk_box_pack_start (GTK_BOX(GNOME_DIALOG (dialog)->vbox), notebook,
				    TRUE, TRUE, 0);
		gtk_widget_show (notebook);
		
		box = gtk_vbox_new (FALSE, GNOME_PAD);
		gtk_container_set_border_width (GTK_CONTAINER (box), GNOME_PAD);
		label = gtk_label_new (_("Printer"));
		gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, label);
		gtk_widget_show (box);
		gtk_widget_show (label);
		
		printersel = gnome_printer_widget_new ();
		gtk_box_pack_start (GTK_BOX (box), printersel, FALSE, TRUE, 0);
		gtk_widget_show (printersel);
	
		box = gtk_vbox_new (FALSE, GNOME_PAD);
		gtk_container_set_border_width (GTK_CONTAINER(box), GNOME_PAD);
		label = gtk_label_new (_("Scaling"));
		gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, label);
		gtk_widget_show (box);
		gtk_widget_show (label);
		
		frame = gtk_frame_new (_("Scaling"));
		gtk_box_pack_start (GTK_BOX (box), frame, FALSE, TRUE, 0);
		gtk_widget_show (frame);
		box = gtk_vbox_new (FALSE, GNOME_PAD);
		gtk_container_set_border_width (GTK_CONTAINER (box), GNOME_PAD_SMALL);
		gtk_container_add (GTK_CONTAINER (frame), box);
		gtk_widget_show (box);
		scalewid = gtk_spin_button_new (
			GTK_ADJUSTMENT (gtk_adjustment_new (1.0, 0.1, 10.0, 0.1, 0.1, 1.0)), 0, 3);
		gtk_box_pack_start (GTK_BOX (box), scalewid, FALSE, TRUE, 0);
		gtk_widget_show (scalewid);
		
		box = gtk_vbox_new (FALSE, GNOME_PAD);
		gtk_container_set_border_width (GTK_CONTAINER (box), GNOME_PAD);
		label = gtk_label_new (_("Paper Size"));
		gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, label);
		gtk_widget_show (box);
		gtk_widget_show (label);
		
		frame = gtk_frame_new (_("Paper Size"));
		gtk_box_pack_start (GTK_BOX (box), frame, FALSE, TRUE, 0);
		gtk_widget_show (frame);
		papersel = gnome_paper_selector_new ();
		gtk_container_set_border_width (GTK_CONTAINER (papersel), GNOME_PAD_SMALL);
		gtk_container_add (GTK_CONTAINER (frame), papersel);
		gtk_widget_show (papersel);
		
		gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
		btn = gnome_dialog_run (GNOME_DIALOG (dialog));
		if (btn < 0)
			return;
		
		/* Cancel was pressed. */
		if (btn == 2) {
			gtk_widget_destroy (dialog);
			return;
		}
		
		/* Preview? */
		preview = (btn == 1);
		printer = gnome_printer_widget_get_printer (GNOME_PRINTER_WIDGET (printersel));
		scale = gtk_spin_button_get_value_as_float (GTK_SPIN_BUTTON (scalewid));
		paper_name = gnome_paper_selector_get_name (GNOME_PAPER_SELECTOR (papersel));
	} else {
		preview = TRUE;
	}
		
	print_master = gnome_print_master_new ();

	/* Get the paper metrics. */
	if (paper_name) {
		paper = gnome_paper_with_name (paper_name);
	} else {
		paper = gnome_paper_with_name (gnome_paper_name_default ());
	}

	gnome_print_master_set_paper (print_master, paper);

	if (printer) {
		gnome_print_master_set_printer (print_master, printer);
	}

	/*gnome_print_master_set_copies (print_master, copies, collate);*/
	ctx = gnome_print_master_get_context (print_master);

	opc = g_new0 (OreganoPrintContext, 1);
	opc->ctx = ctx;
	opc->label_font = gnome_font_new_closest ("Helvetica", GNOME_FONT_MEDIUM, FALSE, 8);
	if (!opc->label_font) {
		g_warning (_("Could not create font for printing."));
		goto bail_out;
	} else {
		paginate (sv, opc, paper, 0.03 * scale);
	}

	gnome_print_master_close (print_master);

	if (preview) {
		gboolean landscape = FALSE;
		GnomePrintMasterPreview *preview;
		
		preview = gnome_print_master_preview_new_with_orientation (
			print_master, _("Print Preview"), landscape);
		gtk_widget_show (GTK_WIDGET (preview));
	} else {
		int result = gnome_print_master_print (print_master);

		if (result == -1) {
			g_warning (_("Printing failed."));
		}
	}

 bail_out:
	gtk_object_unref (GTK_OBJECT (opc->label_font));
	gtk_object_unref (GTK_OBJECT (print_master));
	if (!preview_only) {
		gtk_widget_destroy (dialog);
	}
	g_free (opc);
}

