
/****************************************************************************
**
**  Copyright (C) 2002  - the shmoo group -
**
**  This program is free software; you can redistribute it and/or
**  modify it, however, you cannot sell it.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
**  You should have received a copy of the license attached to the
**  use of this software.  If not, visit www.shmoo.com/osiris for
**  details.
**
*****************************************************************************/

/*****************************************************************************
**
**    The Shmoo Group (TSG)
**
**    File:      osirisdb.c
**    Author:    Brian Wotring
**
**    Date:      May 14, 2002
**    Project:   osiris
**
******************************************************************************/


#include "libosiris.h"
#include "libosirism.h"
#include "libosirisdb.h"
#include "libfileapi.h"

int osi_db_create( OSI_DB *db )
{
    int result = OSI_ERROR_UNKNOWN;

    if( db != NULL )
    {
        int db_result = 0;
        u_int32_t flags = ( DB_CREATE );

        /* create the db handle. */

        db_result |= db_create( &( db->records ), NULL, 0 );

        if( db_result != 0 )
        {
            /* something bad happened when initializing one of these DBs. */
            /* we don't care which one it was, we bail here.              */

            return OSI_ERROR_DB_INIT_FAILED;
        }

        db_result |= ( db->records )->open( db->records, NULL, db->path,
                       NULL, OSI_DB_TYPE, flags, OSI_DB_PERMISSIONS );

        if( db_result != 0 )
        {
            /* something bad happened, delete it and bail. */

            ( db->records )->remove( db->records, db->path, NULL, 0 );

            return OSI_ERROR_DB_CREATE_FAILED;
        }

        /* now close the databases in this file. */

        db->error_id = 0;
        osi_db_close( db );

        result = OSI_DB_OK;
    }

    return result;
}

int osi_db_open( OSI_DB *db, int mode )
{
    int result = OSI_ERROR_UNKNOWN;

    if( db != NULL )
    {
        int db_result = 0;
        int flags = 0;

        /* get db handles for all of the databases in the file. */

        db_result |= db_create( &( db->records ), NULL, 0 );

        if( db_result != 0 )
        {
            /* something bad happend when initializing one of these DBs. */
            /* we don't care which one it was, we bail here.             */

            return OSI_ERROR_DB_INIT_FAILED;
        }

        /* now open the database based upon the mode. we never create them. */

        switch( mode )
        {
            case OSI_DB_READ:

                flags = ( flags | DB_RDONLY );

                db_result |= ( db->records )->open( db->records, NULL,
                               db->path, NULL, DB_UNKNOWN,
                               flags, OSI_DB_PERMISSIONS );
                break;

            case OSI_DB_WRITE:

                db_result |= ( db->records )->open( db->records, NULL,
                               db->path, NULL, DB_UNKNOWN,
                               flags, OSI_DB_PERMISSIONS );
                break;

            default:
                break;
        }

        /* now check our result, if we were unable to open any of them */
        /* we close them and bail.                                     */

        if( db_result == 0 )
        {
            db->open = TRUE;
            result   = OSI_DB_OK;
        }

        else
        {
            osi_db_close( db );
            result = OSI_ERROR_DB_OPEN_FAILED;
        }
    }

    return result;
}

int osi_db_close( OSI_DB *db )
{
    int result = OSI_ERROR_UNKNOWN;

    if( db != NULL )
    {
        result = OSI_DB_OK;

        if( db->records != NULL )
        {
            (db->records)->close( db->records, 0 );
            db->records = NULL;
        }

        db->open = FALSE;
    }

    return result;
}

int osi_db_truncate( OSI_DB *db )
{
    int result = OSI_ERROR_UNKNOWN;
    int truncate_count = 0;

    if( db != NULL )
    {
        (db->records)->truncate( db->records, NULL, &truncate_count, 0 );
    }

    return result;
}


int osi_db_delete( OSI_DB *db )
{
    int result = OSI_ERROR_UNKNOWN;

    if( db != NULL )
    {
        osi_db_close( db );
        result = osi_remove_file( db->path );
    }

    return result;
}

long osi_db_get_record_count( OSI_DB *db )
{       
    long count = -1;

    if( ( db != NULL ) && ( db->open ) )
    {
        DB_HASH_STAT *db_stats = NULL;

        if( db->records->stat( db->records, &db_stats, 0 ) == 0 )
        {
            if( db_stats != NULL )
            {
                count = (long)db_stats->hash_ndata;
                osi_free( db_stats );
            }
        }
    }

    return count;
}


int osi_db_get_record_with_name( OSI_DB *db, void *record,
                                 int record_size, char *name )
{
    int size;
    int result = OSI_ERROR_UNKNOWN;

    unsigned char buffer[MAX_SCAN_RECORD_LENGTH];

    if( ( db == NULL ) || ( record == NULL ) || ( name == NULL ) )
    {
        return result;
    }

    result = osi_db_get_data_with_key( ( db->records ), buffer,
                                       sizeof( buffer ), &size, name );

    /* unpack the data into our passed in record buffer. */

    if( result == OSI_DB_OK )
    {
        /* unpack and unwrap before we return the record. */

        memcpy( record, buffer, size );
    }

    return result;
}

int osi_db_store_record( OSI_DB *db, void *record, int record_size )
{
    int result = OSI_ERROR_UNKNOWN;
    int sequence;

    char key[20];

    if( ( db == NULL ) || ( db->records == NULL ) || ( record == NULL ) )
    {
        return result;
    }

    /* create a unique key. */

    sequence = osi_db_get_record_count( db );
    osi_snprintf( key, sizeof( key ), "%d", sequence );

    result = osi_db_store_data_with_key( ( db->records ),
                                         (void *)record, record_size, key );

    return result;
}


int osi_db_delete_record_with_name( OSI_DB *db, char *name )
{
    int result = OSI_ERROR_UNKNOWN;
    DBT key;

    if( ( db == NULL ) || ( db->records == NULL ) || ( name == NULL ) )
    {
        return result;
    }

    memset( &key, 0, sizeof( key ) );

    key.data = name;
    key.size = strlen( name );

    if( ( db->records )->del( (db->records), NULL, &key, 0 ) == 0 )
    {
        result = OSI_DB_OK;
    }

    return result;
}


int osi_db_get_first_record( OSI_DB *db, void *record, int record_size,
                             int *result_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( db != NULL ) && ( record != NULL ) && ( result_size != NULL ) )
    {
        DBT key;                 
        DBT data;

        *result_size = 0;

        result = db->records->cursor( db->records, NULL, &db->r_cursor, 0 );

        /* we can't do much without a cursor. */

        if( ( result != 0 ) || ( db->r_cursor == NULL ) )
        {
            return OSI_ERROR_DB_INVALID_CURSOR;
        }

        /* initialize the key/data pair. */

        memset( &key, 0, sizeof( key ) );
        memset( &data, 0, sizeof( data ) );

        result = db->r_cursor->c_get( db->r_cursor, &key, &data, DB_FIRST );

        /* if we just hit the end, inform caller. */

        if( result == DB_NOTFOUND )
        {
            return OSI_ERROR_DB_LAST_ITEM;
        }

        if( result != 0 )
        {
            /* we failed to retreive with the cursor,  so */
            /* set the error, close cursor, and bail.     */

            db->r_cursor->c_close( db->r_cursor );
            db->r_cursor = NULL;

            return OSI_ERROR_DB_RETRIEVING;
        }

        /* we have the data, set fields, make sure it will fit first. */

        if( data.size > (unsigned int)record_size )
        {
            return OSI_ERROR_DB_DATA_TOO_LARGE;
        }

        memcpy( record, data.data, data.size );
        *result_size = data.size;

        result = OSI_DB_OK;
    }

    return result;
}

int osi_db_get_next_record( OSI_DB *db, void *record, int record_size,
                            int *result_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( db != NULL ) && ( record != NULL ) && ( result_size != NULL ) )
    {
        DBT key;
        DBT data;

        *result_size = 0;

        /* verify our cursor first. */

        if( db->r_cursor == NULL )
        {
            return OSI_ERROR_DB_INVALID_CURSOR;
        }

        /* initialize the key/data pair. */

        memset( &key, 0, sizeof( key ) );
        memset( &data, 0, sizeof( data ) );

        result = db->r_cursor->c_get( db->r_cursor, &key, &data, DB_NEXT );

        /* if we hit the end, close cursor and inform caller. */

        if( result == DB_NOTFOUND )
        {
            db->r_cursor->c_close( db->r_cursor );
            db->r_cursor = NULL;

            return OSI_ERROR_DB_LAST_ITEM;
        }

        if( result != 0 )
        {
            /* we failed to retreive record, bail here. */

            return OSI_ERROR_DB_RETRIEVING;
        }

        /* we have the data, set fields, make sure it will fit first. */

        if( data.size > (unsigned int)record_size )
        {
            return OSI_ERROR_DB_DATA_TOO_LARGE;
        }

        memcpy( record, data.data, data.size );
        *result_size = data.size;

        result = OSI_DB_OK;
    }

    return result;
}


int osi_scan_db_create( OSI_DB *db )
{
    int result = OSI_ERROR_UNKNOWN;

    if( db != NULL )
    {
        int db_result = 0;
        u_int32_t flags = ( DB_CREATE );

        /* create the db handles for the three databases. */

        db_result |= db_create( &( db->header ), NULL, 0 );
        db_result |= db_create( &( db->errors ), NULL, 0 );
        db_result |= db_create( &( db->records ), NULL, 0 );
        db_result |= db_create( &( db->system ), NULL, 0 );

        if( db_result != 0 )
        {
            /* something bad happend when initializing one of these DBs. */
            /* we don't care which one it was, we bail here.             */

            return OSI_ERROR_DB_INIT_FAILED;
        }

        if( strlen( db->passphrase ) != 0 )
        {
            (db->header)->set_encrypt( (db->header), db->passphrase,
                                       DB_ENCRYPT_AES );

            (db->errors)->set_encrypt( (db->errors), db->passphrase,
                                       DB_ENCRYPT_AES );

            (db->records)->set_encrypt( (db->records), db->passphrase,
                                        DB_ENCRYPT_AES );

            (db->system)->set_encrypt( (db->system), db->passphrase,
                                       DB_ENCRYPT_AES );
        }

        /* delete file if already there. */

        if( osi_file_exists( db->path ) )
        {
            osi_remove_file( db->path );
        }

        /* now we actually create the file and the four databases that */
        /* file will hold.                                             */

        db_result |= ( db->header )->open( db->header, NULL, db->path,
                       OSI_DB_NAME_HEADER, OSI_DB_TYPE,
                       flags, OSI_DB_PERMISSIONS );

        db_result |= ( db->errors )->open( db->errors, NULL, db->path,
                       OSI_DB_NAME_ERRORS, OSI_DB_TYPE,
                       flags, OSI_DB_PERMISSIONS );

        db_result |= ( db->records )->open( db->records, NULL, db->path,
                       OSI_DB_NAME_RECORDS, OSI_DB_TYPE,
                       flags, OSI_DB_PERMISSIONS );

        db_result |= ( db->system )->open( db->system, NULL, db->path,
                       OSI_DB_NAME_SYSTEM, OSI_DB_TYPE,
                       flags, OSI_DB_PERMISSIONS );

        if( db_result != 0 )
        {
            /* something bad happend when creating one of these databases. */
            /* we don't care which one it was, delete them all and bail.   */

            ( db->header )->remove( db->header, db->path,
                                    OSI_DB_NAME_HEADER, 0 );

            ( db->errors )->remove( db->errors, db->path,
                                    OSI_DB_NAME_ERRORS, 0 );

            ( db->records )->remove( db->records, db->path,
                                     OSI_DB_NAME_RECORDS, 0 );

            ( db->system )->remove( db->system, db->path,
                                    OSI_DB_NAME_SYSTEM, 0 );

            return OSI_ERROR_DB_CREATE_FAILED;
        }

        /* we now have three databases, so we populate the header as much as */
        /* we can now, first, we create all of the fields, then overwrite    */
        /* with what we have.                                                */
        /* the int values must be completely zeroed out!                     */

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_COMPLETE,
                                       "\0\0\0\0\0\0\0\0", 8 );

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_RECORD_TYPE,
                                       "\0\0\0\0\0\0\0\0", 8 );

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_SCAN_RESULTS,
                                       "\0",1 );

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_CONFIG_DATA,
                                       "\0", 1 );

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_CONFIG_NAME,
                                       "\0", 1 );

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_CONFIG_ID,
                                       "\0", 1 );

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_HOST,
                                       "\0", 1 );

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_NAME,
                                       "\0", 1 );

        osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_NOTES,
                                       "\0", 1 );

        if( db->cfg != NULL )
        {
            /* store config's name if there. */

            if( strlen( db->cfg->name ) > 0 )
            {
                char *name = db->cfg->name;
                osi_scan_db_store_header_item( db,
                                               OSI_DB_HEADER_ITEM_CONFIG_NAME,
                                               name, strlen( name ) );
            }

            /* store config's id if there. */

            if( strlen( db->cfg->id ) > 0 )
            {
                char *id = db->cfg->id;
                osi_scan_db_store_header_item( db, OSI_DB_HEADER_ITEM_CONFIG_ID,
                                               id, strlen( id ) );
            }

            /* store all config file data if there. */

            if( db->cfg->data != NULL )
            {
                string_list *lines = db->cfg->data;

                if( lines->size > 0 )
                {
                    int index;
                    int size;
                    char *data;

                    size = ( ( lines->size ) * MAX_LINE_LENGTH );

                    data = osi_malloc( size );
                    memset( data, 0, size );

                    for( index = 0; (unsigned int)index < lines->size; index++ )
                    {
                        char *line = lines->list[index];
                        osi_strlcat( data, line, size );
                    }

                    osi_scan_db_store_header_item( db,
                                              OSI_DB_HEADER_ITEM_CONFIG_DATA,
                                              data, strlen( data ) );
                    osi_free( data );
                }
            }
        }

        /* now close the databases in this file. */

        db->error_id = 0;
        osi_scan_db_close( db );

        result = OSI_DB_OK;
    }

    return result;
}


int osi_scan_db_open( OSI_DB *db, int mode )
{
    int result = OSI_ERROR_UNKNOWN;

    int db_result = 0;
    int flags = 0;

    if( db == NULL )
    {
        return OSI_ERROR_NULL_ARGUMENT;
    }

    db_result = db_env_create( &db->env, 0 );

    if( db_result != 0 )
    {
        return OSI_ERROR_DB_INIT_FAILED;
    }

    db_result = db->env->open( db->env, NULL, 
                               ( DB_PRIVATE | DB_CREATE | DB_INIT_MPOOL ), 0 );

    if( db_result != 0 )
    {
        return OSI_ERROR_DB_INIT_FAILED;
    }

    /* get db handles for all of the databases in the file. */

    db_result |= db_create( &( db->header ), db->env, 0 );
    db_result |= db_create( &( db->errors ), db->env, 0 );
    db_result |= db_create( &( db->records ), db->env, 0 );
    db_result |= db_create( &( db->system ), db->env, 0 );

    if( db_result != 0 )
    {
        /* something bad happend when initializing one of these DBs. */
        /* we don't care which one it was, we bail here.             */

        db->env->close( db->env, 0 );
        return OSI_ERROR_DB_INIT_FAILED;
    }

    if( strlen( db->passphrase ) != 0 )
    {
        (db->header)->set_encrypt( (db->header), db->passphrase,
                                   DB_ENCRYPT_AES );

        (db->errors)->set_encrypt( (db->errors), db->passphrase,
                                   DB_ENCRYPT_AES );

        (db->records)->set_encrypt( (db->records), db->passphrase,
                                    DB_ENCRYPT_AES );

        (db->system)->set_encrypt( (db->system), db->passphrase,
                                    DB_ENCRYPT_AES );
    }

    /* now open the databases based upon the mode. we never create them. */

    switch( mode )
    {
        case OSI_DB_READ:

            flags = ( flags | DB_RDONLY );

            db_result |= ( db->header )->open( db->header, NULL,
                           db->path, OSI_DB_NAME_HEADER, DB_UNKNOWN,
                           flags, OSI_DB_PERMISSIONS );

            db_result |= ( db->errors )->open( db->errors, NULL,
                           db->path, OSI_DB_NAME_ERRORS, DB_UNKNOWN,
                           flags, OSI_DB_PERMISSIONS );

            db_result |= ( db->records )->open( db->records, NULL,
                           db->path, OSI_DB_NAME_RECORDS, DB_UNKNOWN,
                           flags, OSI_DB_PERMISSIONS );

            db_result |= ( db->system )->open( db->system, NULL,
                           db->path, OSI_DB_NAME_SYSTEM, DB_UNKNOWN,
                           flags, OSI_DB_PERMISSIONS );
            break;

        case OSI_DB_WRITE:

            db_result |= ( db->header )->open( db->header, NULL,
                           db->path, OSI_DB_NAME_HEADER, DB_UNKNOWN,
                           flags, OSI_DB_PERMISSIONS );

            db_result |= ( db->errors )->open( db->errors, NULL,
                           db->path, OSI_DB_NAME_ERRORS, DB_UNKNOWN,
                           flags, OSI_DB_PERMISSIONS );

            db_result |= ( db->records )->open( db->records, NULL,
                           db->path, OSI_DB_NAME_RECORDS, DB_UNKNOWN,
                           flags, OSI_DB_PERMISSIONS );

            db_result |= ( db->system )->open( db->system, NULL,
                           db->path, OSI_DB_NAME_SYSTEM, DB_UNKNOWN,
                           flags, OSI_DB_PERMISSIONS );

            break;

        default:
            break;
    }

    /* now check our result, if we were unable to open any of them */
    /* we close them and bail.                                     */

    if( db_result == 0 )
    {
        db->open = TRUE;
        result   = OSI_DB_OK;
    }

    else
    {
        osi_scan_db_close( db );
        result = OSI_ERROR_DB_OPEN_FAILED;
    }

    
    /* WE NEED TO DELETE THE DB OR TRUNCATE IT HERE! */
    /* IF WE ARE OPENING FOR WRITE.                  */

    return result;
}


int osi_scan_db_close( OSI_DB *db )
{
    int result = OSI_ERROR_UNKNOWN;

    if( db != NULL )
    {
        result = OSI_DB_OK;

        if( db->header != NULL )
        {
            (db->header)->close( db->header, 0 );
            db->header = NULL;
        }

        if( db->errors != NULL )
        {
            (db->errors)->close( db->errors, 0 );
            db->errors = NULL;
        }

        if( db->records != NULL )
        {
            (db->records)->close( db->records, 0 );
            db->records = NULL;
        }

        if( db->system != NULL )
        {
            (db->system)->close( db->system, 0 );
            db->system = NULL;
        }

        if( db->env != NULL )
        {
            db->env->close( db->env, 0 );
        }

        db->open = FALSE;
    }

    return result;
}


int osi_scan_db_delete( OSI_DB *db )
{
    int result = OSI_ERROR_UNKNOWN;

    if( db != NULL )
    {
        osi_scan_db_close( db );
        result = osi_remove_file( db->path );
    }
    
    return result;
}


long osi_scan_db_get_error_count( OSI_DB *db )
{
    long count = -1;

    if( ( db != NULL ) && ( db->open ) )
    {
        DB_HASH_STAT *db_stats = NULL;

        if( db->errors->stat( db->errors, &db_stats, 0 ) == 0 )
        {
            if( db_stats != NULL )
            {
                count = (long)db_stats->hash_ndata;
                osi_free( db_stats );
            }
        }
    }

    return count;
}

long osi_scan_db_get_record_count( OSI_DB *db )
{
    long count = -1;

    if( ( db != NULL ) && ( db->open ) )
    {
        DB_HASH_STAT *db_stats = NULL;

        if( db->records->stat( db->records, &db_stats, 0 ) == 0 )
        {
            if( db_stats != NULL )
            {
                count = (long)db_stats->hash_ndata;
                osi_free( db_stats );
            }
        }
    }

    return count;
}

long osi_scan_db_get_system_count( OSI_DB *db )
{
    long count = -1;

    if( ( db != NULL ) && ( db->open ) )
    {
        DB_HASH_STAT *db_stats = NULL;

        if( db->system->stat( db->system, &db_stats, 0 ) == 0 )
        {
            if( db_stats != NULL )
            {
                count = (long)db_stats->hash_ndata;
                osi_free( db_stats );
            }
        }
    }

    return count;
}


int osi_scan_db_store_record( OSI_DB *db, void *record )
{
    int result = OSI_ERROR_UNKNOWN;
    int size = 0;

    char *name = NULL;
    unsigned char buffer[MAX_SCAN_RECORD_LENGTH];

    if( ( db == NULL ) || ( record == NULL ) )
    {
        return result;
    }

    name = get_scan_record_path( (SCAN_RECORD *)record );

    wrap_scan_record( (SCAN_RECORD *)record );
    size = pack_scan_record( (SCAN_RECORD *)record, buffer, sizeof( buffer ) );

    result = osi_db_store_data_with_key( ( db->records ), 
                                         (void *)buffer, size, name );

    return result;
}

int osi_scan_db_store_system( OSI_DB *db, void *system )
{
    int result = OSI_ERROR_UNKNOWN;
    int size = 0;

    char *name = NULL;
    unsigned char buffer[MAX_SCAN_RECORD_LENGTH];

    if( ( db == NULL ) || ( system == NULL ) )
    {
        return result;
    }

    name = get_scan_record_path( (SCAN_RECORD *)system );

    wrap_scan_record( (SCAN_RECORD *)system );
    size = pack_scan_record( (SCAN_RECORD *)system, buffer, sizeof( buffer ) );

    result = osi_db_store_data_with_key( ( db->system ),    
                                         (void *)buffer, size, name );

    return result;
}


int osi_scan_db_get_record_with_name( OSI_DB *db, void *record,
                                      int record_size, char *name )
{
    int size;
    int result = OSI_ERROR_UNKNOWN;

    unsigned char buffer[MAX_SCAN_RECORD_LENGTH];

    if( ( db == NULL ) || ( record == NULL ) || ( name == NULL ) )
    {
        return result;
    }


    result = osi_db_get_data_with_key( ( db->records ), buffer,
                                       sizeof( buffer ), &size, name );

    /* unpack the data into our passed in record buffer. */

    if( result == OSI_DB_OK )
    {
        /* unpack and unwrap before we return the record. */

        unpack_scan_record( (SCAN_RECORD *)record, buffer, size );
        unwrap_scan_record( (SCAN_RECORD *)record );
    }

    return result;
}

int osi_scan_db_get_system_with_name( OSI_DB *db, void *system, int system_size,
                                      char *name )
{
    int size;
    int result = OSI_ERROR_UNKNOWN;

    unsigned char buffer[MAX_SCAN_RECORD_LENGTH];

    if( ( db == NULL ) || ( system == NULL ) || ( name == NULL ) )
    {
        return result;
    }


    result = osi_db_get_data_with_key( ( db->system ), buffer,
                                       sizeof( buffer ), &size, name );

    /* unpack the data into our passed in record buffer. */

    if( result == OSI_DB_OK )
    {
        /* unpack and unwrap before we return the record. */

        unpack_scan_record( (SCAN_RECORD *)system, buffer, size );
        unwrap_scan_record( (SCAN_RECORD *)system );
    }

    return result;
}

int osi_scan_db_store_header_item( OSI_DB *db, char *item, void *data,
                                   int data_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( data == NULL ) || ( item == NULL ) )
    {
        return OSI_ERROR_NULL_ARGUMENT;
    }

    if( db != NULL )
    {
        
        if( !strcmp( item, OSI_DB_HEADER_ITEM_COMPLETE ) ||
            !strcmp( item, OSI_DB_HEADER_ITEM_RECORD_TYPE ) )
        {
            osi_uint64 value;

            /* must be 64 bit value, or we don't wrap, caller must be smart. */
            
            if( data_size == sizeof( osi_uint64 ) )
            {
                memcpy( &value, data, sizeof( value ) );
                value = OSI_HTONLL( value );
                
                result = osi_db_store_data_with_key( ( db->header ),
                                                     &value,
                                                     sizeof( value ), item );
            }
        }
        
        else if( !strcmp( item, OSI_DB_HEADER_ITEM_SCAN_RESULTS ) )
        {
            OSI_SCAN_RESULTS_1 scan_results;
            
            /* must have actual scan results structure passed in. */
            /* copy it and wrap it for storage.                   */
            
            if( data_size == sizeof( scan_results ) )
            {
                memcpy( &scan_results, data, sizeof( scan_results ) );
                wrap_scan_results( ( (OSI_SCAN_RESULTS_1 *)&scan_results ) );
                
                result = osi_db_store_data_with_key( ( db->header ),
                                                     &scan_results,
                                                     sizeof( scan_results ),
                                                     item );
            }
        }
        
        /* if we were aren't wrapping, we just store the data raw. */
        
        else
        {
            result = osi_db_store_data_with_key( ( db->header ), data,
                                                 data_size, item );
        }
    }

    /* force this write to disk. */

    db->header->sync( db->header, 0 );

    return result;
}

int osi_scan_db_get_header_item( OSI_DB *db, char *item, void *data,
                                 int max_data, int *result_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( data == NULL ) || ( item == NULL ) )
    {
        return OSI_ERROR_NULL_ARGUMENT;
    }
    
    if( db != NULL )
    {
        result = osi_db_get_data_with_key( ( db->header ), data, max_data,
                                           result_size, item );

        if( !strcmp( item, OSI_DB_HEADER_ITEM_COMPLETE ) ||
            !strcmp( item, OSI_DB_HEADER_ITEM_RECORD_TYPE ) )
        {
            osi_uint64 value;

            /* must be 64 bit value, or we don't unwrap, caller beware. */
            
            if( (*result_size) == sizeof( osi_uint64 ) )
            {
                memcpy( &value, data, sizeof( value ) );
                value = OSI_NTOHLL( value );
                memcpy( data, &value, sizeof( value ) );
            }
        }
        
        else if( !strcmp( item, OSI_DB_HEADER_ITEM_SCAN_RESULTS ) )
        {
            /* if the fetched data is the size of a scan result */
            /* structure, we unwrap it before we return.        */
            
            if( (*result_size) == sizeof( OSI_SCAN_RESULTS_1 ) )
            {
                unwrap_scan_results( ( (OSI_SCAN_RESULTS_1 *)data ) );
            }
        }
    }

    return result;
}

int osi_scan_db_get_first_record( OSI_DB *db, void *record, int record_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( db != NULL ) && ( record != NULL ) )
    {
        DBT key;
        DBT data;

        result = db->records->cursor( db->records, NULL, &db->r_cursor, 0 );

        /* we can't do much without a cursor. */

        if( ( result != 0 ) || ( db->r_cursor == NULL ) )
        {
            return OSI_ERROR_DB_INVALID_CURSOR;
        }

	    /* initialize the key/data pair. */

	    memset( &key, 0, sizeof( key ) );
	    memset( &data, 0, sizeof( data ) );

        result = db->r_cursor->c_get( db->r_cursor, &key, &data, DB_FIRST );

        /* if we just hit the end, inform caller. */

        if( result == DB_NOTFOUND )
        {
            return OSI_ERROR_DB_LAST_ITEM;
        }

        if( result != 0 )
        {
            /* we failed to retreive with the cursor,  so */
            /* set the error, close cursor, and bail.     */

            db->r_cursor->c_close( db->r_cursor );
            db->r_cursor = NULL;

            return OSI_ERROR_DB_RETRIEVING;
        }

        /* we have the data, set fields, make sure it will fit first. */

        if( data.size > (unsigned int)record_size )
        {
            return OSI_ERROR_DB_DATA_TOO_LARGE;
        }

        /* unpack the data into our passed in record buffer. */

        unpack_scan_record( (SCAN_RECORD *)record, data.data, data.size );
        
        /* unwrap this record for the local byte order. */
        
        unwrap_scan_record( (SCAN_RECORD *)record );

        /* it will fit, copy it, set the size and return success. */

        result = OSI_DB_OK;
    }

    return result;
}

int osi_scan_db_get_next_record( OSI_DB *db, void *record, int record_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( db != NULL ) && ( record != NULL ) )
    {
        DBT key;
        DBT data;

        /* verify our cursor first. */

        if( db->r_cursor == NULL )
        {
            return OSI_ERROR_DB_INVALID_CURSOR;
        }

	    /* initialize the key/data pair. */

	    memset( &key, 0, sizeof( key ) );
	    memset( &data, 0, sizeof( data ) );

        result = db->r_cursor->c_get( db->r_cursor, &key, &data, DB_NEXT );

        /* if we hit the end, close cursor and inform caller. */

        if( result == DB_NOTFOUND )
        {
            db->r_cursor->c_close( db->r_cursor );
            db->r_cursor = NULL;

            return OSI_ERROR_DB_LAST_ITEM;
        }

        if( result != 0 )
        {
            /* we failed to retreive record, bail here. */

            return OSI_ERROR_DB_RETRIEVING;
        }

        /* we have the data, set fields, make sure it will fit first. */

        if( data.size > (unsigned int)record_size )
        {
            return OSI_ERROR_DB_DATA_TOO_LARGE;
        }

        /* unpack the data into our passed in record buffer. */

        unpack_scan_record( (SCAN_RECORD *)record, data.data, data.size );

        /* unwrap this record for the local byte order. */
        
        unwrap_scan_record( (SCAN_RECORD *)record );

        /* it will fit, copy it, set the size and return success. */

        result = OSI_DB_OK;
    }

    return result;
}


int osi_scan_db_store_error( OSI_DB *db, OSI_ERROR *error )
{
    int result = OSI_ERROR_UNKNOWN;
    char buffer[sizeof(OSI_ERROR)];

    if( ( db != NULL ) && ( error != NULL ) )
    {
        /* get error id. */

        char id[32] = "";
        osi_snprintf( id, sizeof( id ), "%d", db->error_id );

        /* push current time into the error message if it isn't there. */

        if( error->time == 0 )
        {
            error->time = (osi_uint64)osi_get_time();
        }
        
        /* wrap this error for storage, preserve byte order, then store it. */
        
        wrap_error( error );
        pack_error( error, buffer, sizeof(buffer) );

        result = osi_db_store_data_with_key( ( db->errors ), error,
                                             sizeof( OSI_ERROR ), id );

        /* increment error id. */

        if( result == OSI_DB_OK )
        {
            db->error_id += 1;
        }
    }

    return result;
}

int osi_scan_db_get_first_system( OSI_DB *db, void *system, int system_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( db != NULL ) && ( system != NULL ) )
    {
        DBT key;
        DBT data;

        result = db->system->cursor( db->system, NULL, &db->s_cursor, 0 );

        /* we can't do much without a cursor. */

        if( ( result != 0 ) || ( db->s_cursor == NULL ) )
        {
            return OSI_ERROR_DB_INVALID_CURSOR;
        }

	    /* initialize the key/data pair. */

	    memset( &key, 0, sizeof( key ) );
	    memset( &data, 0, sizeof( data ) );

        result = db->s_cursor->c_get( db->s_cursor, &key, &data, DB_FIRST );

        /* if we just hit the end, inform caller. */

        if( result == DB_NOTFOUND )
        {
            return OSI_ERROR_DB_LAST_ITEM;
        }

        if( result != 0 )
        {
            /* we failed to retreive with the cursor,  so */
            /* set the error, close cursor, and bail.     */

            db->s_cursor->c_close( db->s_cursor );
            db->s_cursor = NULL;

            return OSI_ERROR_DB_RETRIEVING;
        }

        /* we have the data, set fields, make sure it will fit first. */

        if( data.size > (unsigned int)system_size )
        {
            return OSI_ERROR_DB_DATA_TOO_LARGE;
        }

        /* unpack the data into our passed in record buffer. */

        unpack_scan_record( (SCAN_RECORD *)system, data.data, data.size );
        
        /* unwrap this record for the local byte order. */
        
        unwrap_scan_record( (SCAN_RECORD *)system );

        /* it will fit, copy it, set the size and return success. */

        result = OSI_DB_OK;
    }

    return result;
}

int osi_scan_db_get_next_system( OSI_DB *db, void *system, int system_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( db != NULL ) && ( system != NULL ) )
    {
        DBT key;
        DBT data;

        /* verify our cursor first. */

        if( db->s_cursor == NULL )
        {
            return OSI_ERROR_DB_INVALID_CURSOR;
        }

	    /* initialize the key/data pair. */

	    memset( &key, 0, sizeof( key ) );
	    memset( &data, 0, sizeof( data ) );

        result = db->s_cursor->c_get( db->s_cursor, &key, &data, DB_NEXT );

        /* if we hit the end, close cursor and inform caller. */

        if( result == DB_NOTFOUND )
        {
            db->s_cursor->c_close( db->s_cursor );
            db->s_cursor = NULL;

            return OSI_ERROR_DB_LAST_ITEM;
        }

        if( result != 0 )
        {
            /* we failed to retreive record, bail here. */

            return OSI_ERROR_DB_RETRIEVING;
        }

        /* we have the data, set fields, make sure it will fit first. */

        if( data.size > (unsigned int)system_size )
        {
            return OSI_ERROR_DB_DATA_TOO_LARGE;
        }

        /* unpack the data into our passed in record buffer. */

        unpack_scan_record( (SCAN_RECORD *)system, data.data, data.size );

        /* unwrap this record for the local byte order. */
        
        unwrap_scan_record( (SCAN_RECORD *)system );

        /* it will fit, copy it, set the size and return success. */

        result = OSI_DB_OK;
    }

    return result;
}


int osi_scan_db_get_first_error( OSI_DB *db, void *error, int max_size,
                                 int *result_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( db != NULL ) && ( error != NULL ) && ( result_size != NULL ) )
    {
        DBT key;
        DBT data;

        result = db->errors->cursor( db->errors, NULL, &db->e_cursor, 0 );

        /* we can't do much without a cursor. */

        if( ( result != 0 ) || ( db->e_cursor == NULL ) )
        {
            result = OSI_ERROR_DB_INVALID_CURSOR;
            return result;
        }

        /* initialize the key/data pair. */

        memset( &key, 0, sizeof( key ) );
        memset( &data, 0, sizeof( data ) );

        result = db->e_cursor->c_get( db->e_cursor, &key, &data, DB_NEXT );

        /* if we just hit the end, inform caller. */

        if( result == DB_NOTFOUND )
        {
            return OSI_ERROR_DB_LAST_ITEM;
        }

        if( result != 0 )
        {
            /* we failed to retreive with the cursor,  so */
            /* set the error, close cursor, and bail.     */

            db->e_cursor->c_close( db->e_cursor );
            db->e_cursor = NULL;

            return OSI_ERROR_DB_RETRIEVING;
        }

        (*result_size) = data.size;

        /* we have the data, set fields, make sure it will fit first. */

        if( data.size > (unsigned int)max_size )
        {
            return OSI_ERROR_DB_DATA_TOO_LARGE;
        }
        
        /* unwrap this error for the local byte order. */
        
        unwrap_error( (OSI_ERROR *)data.data );
        unpack_error( error, data.data, data.size );

        result = OSI_DB_OK;
    }

    return result;
}

int osi_scan_db_get_next_error( OSI_DB *db, void *error, int max_size,
                                int *result_size )
{
    int result = OSI_ERROR_UNKNOWN;

    if( ( db != NULL ) && ( error != NULL ) && ( result_size != NULL ) )
    {
        DBT key;
        DBT data;

        /* verify our cursor first. */

        if( db->e_cursor == NULL )
        {
            return OSI_ERROR_DB_INVALID_CURSOR;
        }

        /* initialize the key/data pair. */

        memset( &key, 0, sizeof( key ) );
        memset( &data, 0, sizeof( data ) );

        result = db->e_cursor->c_get( db->e_cursor, &key, &data, DB_NEXT );

        /* if we hit the end, close cursor and inform caller. */

        if( result == DB_NOTFOUND )
        {
            db->e_cursor->c_close( db->e_cursor );
            db->e_cursor = NULL;

            return OSI_ERROR_DB_LAST_ITEM;
        }

        if( result != 0 )
        {
            /* we failed to retreive record, bail here. */

            (*result_size) = 0;
            return OSI_ERROR_DB_RETRIEVING;
        }

        (*result_size) = data.size;

        /* we have the data, set fields, make sure it will fit first. */

        if( data.size > (unsigned int)max_size )
        {
            return OSI_ERROR_DB_DATA_TOO_LARGE;
        }
        
        /* unwrap this error for the local byte order. */
        
        unwrap_error( (OSI_ERROR *)data.data );
        unpack_error( error, data.data, data.size );

        result = OSI_DB_OK;
    }

    return result;
}


int osi_db_store_data_with_key( DB *db, void *data, int data_size, char *key )
{
    int result = OSI_ERROR_UNKNOWN;

    if( db != NULL )
    {
        if( ( key != NULL ) && ( data != NULL ) )
        {
            DBT db_key;
            DBT db_data;

            memset( &db_key, 0, sizeof( db_key ) );
            memset( &db_data, 0, sizeof( db_data ) );

            db_key.data  = key;
            db_key.size  = strlen( key );

            db_data.data = data;
            db_data.size = data_size;

            result = db->put( db, NULL, &db_key, &db_data, 0 );

            if( result == 0 )
            {
                result = OSI_DB_OK;
            }

            else
            {
                if( result == DB_KEYEXIST )
                {
                    result = OSI_ERROR_DB_ITEM_EXISTS;
                }

                else
                {
                    result = OSI_ERROR_DB_STORING;
                }
            }
        }

        else
        {
            result = OSI_ERROR_NULL_ARGUMENT;
        }
    }

    else
    {
        result = OSI_ERROR_DB_NOT_OPENED;
    }

    return result;
}

int osi_db_get_data_with_key( DB *db, void *data, int max_data,
                              int *result_size, char *key )
{
    int result = OSI_ERROR_UNKNOWN;

    DBT db_key;
    DBT db_data;

    if( db == NULL )
    {
        return OSI_ERROR_DB_NOT_OPENED;
    }
 
    if( ( key == NULL ) || ( data == NULL ) )
    {
        return OSI_ERROR_NULL_ARGUMENT;
    }

    memset( &db_key, 0, sizeof( db_key ) );
    memset( &db_data, 0, sizeof( db_data ) );

    db_key.data  = key;
    db_key.size  = strlen( key );

    result = db->get( db, NULL, &db_key, &db_data, 0 );

    if( result == 0 )
    {
        /* copy data into buffer, as much as we can, and set size. */

        if( db_data.size <= (unsigned int)max_data )
        {
            memcpy( data, db_data.data, db_data.size );
            (*result_size) = db_data.size;

            result = OSI_DB_OK;
        }

        else
        {
            result = OSI_ERROR_DB_DATA_TOO_LARGE;
        }
    }

    else if( result == DB_NOTFOUND )
    {
        result = OSI_ERROR_DB_ITEM_NOT_FOUND;
    }

    else
    {
        result = OSI_ERROR_DB_RETRIEVING;
    }

    return result;
}


int osi_scan_db_get_record_type( OSI_DB *db )
{
    int type = -1;

    if( ( db != NULL ) && ( db->open ) )
    {
        int result;
        int size;
        osi_uint64 record_type;

        result = osi_scan_db_get_header_item( db,
                                         OSI_DB_HEADER_ITEM_RECORD_TYPE,
                                         &record_type, sizeof( record_type ),
                                         &size );

        if( result == OSI_DB_OK )
        {
            type = (int)record_type;
        }
    }

    return type;
}


void osi_wrap_scan_db_header( OSI_DB_HEADER *header )
{
    if( header != NULL )
    {
        header->complete    = OSI_HTONLL( header->complete );
        header->record_type = OSI_HTONLL( header->record_type );
        
        wrap_scan_results( &( header->scan_results ) );
    }
}

void osi_unwrap_scan_db_header( OSI_DB_HEADER *header )
{
    if( header != NULL )
    {
        header->complete    = OSI_NTOHLL( header->complete );
        header->record_type = OSI_NTOHLL( header->record_type );
        
        unwrap_scan_results( &( header->scan_results ) );
    }
} 


void print_database_header( OSI_DB *db )
{
    int length;
    
    long record_count = 0;
    long error_count  = 0;
    long system_count = 0;

    OSI_DB_HEADER header;

    /* get and print db header items. */
    
    osi_scan_db_get_header_item( db, OSI_DB_HEADER_ITEM_COMPLETE,
                            &( header.complete ), sizeof( header.complete ),
                            &length );

    if( length == 0 )
    {
        header.complete = 0;
    }

    osi_scan_db_get_header_item( db, OSI_DB_HEADER_ITEM_RECORD_TYPE,
                            &( header.record_type ),
                            sizeof( header.record_type ), &length );
    
    if( length == 0 )
    {
        header.record_type = 0;
    }

    osi_scan_db_get_header_item( db, OSI_DB_HEADER_ITEM_NAME, header.name,
                            sizeof( header.name ), &length );

    header.name[length] = '\0';

    osi_scan_db_get_header_item( db, OSI_DB_HEADER_ITEM_HOST, header.host,
                            sizeof( header.host ), &length );

    header.host[length] = '\0';

    osi_scan_db_get_header_item( db, OSI_DB_HEADER_ITEM_CONFIG_NAME,
                            header.config_name, sizeof( header.config_name ),
                            &length );

    header.config_name[length] = '\0';

    osi_scan_db_get_header_item( db, OSI_DB_HEADER_ITEM_CONFIG_ID,
                            header.config_id, sizeof( header.config_id ),
                            &length );

    header.config_id[length] = '\0';

    osi_scan_db_get_header_item( db, OSI_DB_HEADER_ITEM_SCAN_RESULTS,
                            &( header.scan_results ),
                            sizeof( header.scan_results ), &length );

    osi_scan_db_get_header_item( db, OSI_DB_HEADER_ITEM_NOTES,
                                 &( header.notes ),
                                 sizeof( header.notes ), &length );

    header.notes[length] = '\0';

    /* get counts for records and errors. */

    error_count  = osi_scan_db_get_error_count( db );
    record_count = osi_scan_db_get_record_count( db );
    system_count = osi_scan_db_get_system_count( db );

    fprintf( stdout, "\n  DATABASE: %s\n\n", header.name );

    /* print status. */

    fprintf( stdout, "         status: " );

    if( header.complete == 1 )
    {
        fprintf( stdout, "complete\n" );
    }

    else
    {
        fprintf( stdout, "incomplete\n" );
    }

    /* print record type, only if incomplete, it is in the scan results. */

    if( header.complete == 0 )
    {
        fprintf( stdout, "    record type: %s\n",
                 get_scan_record_name_from_type( (short)header.record_type ) );
    }

    /* print host. */

    fprintf( stdout, "           host: " );

    if( strlen( header.host ) > 0 )
    {
        fprintf( stdout, "%s", header.host );
    }

    else
    {
        fprintf( stdout, "unknown" );
    }

    fprintf( stdout, "\n" );

    /* print config name and id */

    fprintf( stdout, "         config: " );

    if( strlen( header.config_name ) > 0 )
    {
        fprintf( stdout, "%s", header.config_name );

        if( strlen( header.config_id ) > 0 )
        {
            fprintf( stdout, " (%s)", header.config_id );
        }
    }

    else
    {
        fprintf( stdout, "unknown" );
    }

    fprintf( stdout, "\n\n" );

    fprintf( stdout, "         errors: %d\n", (int)error_count );
    fprintf( stdout, "   file records: %d\n", (int)record_count );
    fprintf( stdout, " system records: %d\n", (int)system_count ); 

    if( header.complete == 1 )
    {
        fprintf( stdout, "\n  SCAN RESULTS:" );
        print_scan_results( &( header.scan_results ) );
    }

    else
    {
        fprintf( stdout, "\n" );
        fprintf( stdout, "  ** database is incomplete, no scan results.\n\n" );
    }

    if( strlen( header.notes ) )
    {
        fprintf( stdout, "\n  NOTES:\n\n" );
        fprintf( stdout, "%s\n", header.notes );
    }
}

void print_database_errors( OSI_DB *db )
{
    int result;
    int size;

    OSI_ERROR error;
    result = osi_scan_db_get_first_error( db, &error, sizeof( error ), &size );

    fprintf( stdout, "\n" );

    if( result == OSI_DB_OK )
    {
        while( result == OSI_DB_OK )
        {
            print_osi_error( &error );
            result = osi_scan_db_get_next_error( db,    
                                                 &error,sizeof( error ),&size );
        }
    }

    else
    {
        fprintf( stdout, "[*] database has no errors.\n" );
    }
}

void print_database_file_names( OSI_DB *db )
{
    int result;

    int record_size;
    int record_type;

    SCAN_RECORD *record;

    record_type = osi_scan_db_get_record_type( db );
    record_size = determine_scan_record_size_from_type( record_type );

    if( record_size <= 0 )
    {
        return;
    }

    fprintf( stdout, "\n" );

    record = osi_malloc( record_size );
    result = osi_scan_db_get_first_record( db, record, record_size );

    while( result == OSI_DB_OK )
    {
        fprintf( stdout, "[%s]\n", get_scan_record_path( record ) );
        result = osi_scan_db_get_next_record( db, record, record_size );
    }

    osi_free( record );
    fprintf( stdout, "\n" );
}

void print_database_file_details( OSI_DB *db, const char *file_path )
{
    int result;

    int record_size;
    int record_type;

    SCAN_RECORD *record;

    if( ( db == NULL ) || ( file_path == NULL ) )
    {
        return;
    }

    record_type = osi_scan_db_get_record_type( db );
    record_size = determine_scan_record_size_from_type( record_type );

    if( record_size <= 0 )
    {
        return;
    }

    record = osi_malloc( record_size );

    result = osi_scan_db_get_record_with_name( db, (void *)record,
                                               record_size, (char *)file_path );

    if( result == OSI_DB_OK )
    {
        print_scan_record( record );        
    }

    else
    {
        fprintf( stderr, "unable to retrive record!\n" ); 
    }

    osi_free( record );
    fprintf( stdout, "\n" );
}

void print_database_system_names( OSI_DB *db )
{
    int result;
    int record_size = MAX_SYSTEM_RECORD_SIZE;

    SCAN_RECORD *record;
    char name[MAX_MODULE_NAME_LENGTH] = "";

    record = osi_malloc( record_size );

    /* print out the system entries. */

    result = osi_scan_db_get_first_system( db, record, record_size );

    while( result == OSI_DB_OK )
    {
        if( record->type == SCAN_RECORD_TYPE_TEXT_1 )
        {
            SCAN_RECORD_TEXT_1 *r = (SCAN_RECORD_TEXT_1 *)record;

            if( strcmp( name, r->module_name ) != 0 )
            {
                fprintf( stdout, "\n[ %s ]\n\n", r->module_name );
                osi_strlcpy( name, r->module_name, sizeof( name ) );
            }

            print_scan_record( record );
        }

        result = osi_scan_db_get_next_system( db, record, record_size );
    }

    osi_free( record );
    fprintf( stdout, "\n" );
}

