//
//
//****************************************************************************************//
// Copyright (c) 2002-2010, The MITRE Corporation
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are
// permitted provided that the following conditions are met:
//
//     * Redistributions of source code must retain the above copyright notice, this list
//       of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above copyright notice, this 
//       list of conditions and the following disclaimer in the documentation and/or other
//       materials provided with the distribution.
//     * Neither the name of The MITRE Corporation nor the names of its contributors may be
//       used to endorse or promote products derived from this software without specific 
//       prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY 
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
// OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT 
// SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT
// OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
// TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
// EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
//****************************************************************************************//

#include "ItemEntity.h"

using namespace std;

//****************************************************************************************//
//								ItemEntity Class										  //	
//****************************************************************************************//
ItemEntity::ItemEntity(string name, string value, OvalEnum::Datatype datatype, bool isObjectEntity, OvalEnum::SCStatus status) {
	this->name = name;
	this->value.push_back(new StringEntityValue(value));
	this->datatype = datatype;
	this->isObjectEntity = isObjectEntity;
	this->scStatus = status;
}

ItemEntity::ItemEntity(string name, AbsEntityValueVector value, OvalEnum::Datatype datatype, bool isObjectEntity, OvalEnum::SCStatus status) {
	this->name = name;
	this->value = value;
	this->datatype = datatype;
	this->isObjectEntity = isObjectEntity;
	this->scStatus = status;
}

ItemEntity::ItemEntity(const ItemEntity& itemEntity){
    this->name = itemEntity.name;
    this->value = itemEntity.value;
    this->datatype = itemEntity.datatype;
    this->isObjectEntity = itemEntity.isObjectEntity;
	this->scStatus = itemEntity.scStatus;
}

ItemEntity::~ItemEntity() {
}

// ***************************************************************************************	//
//								 Public members												//
// ***************************************************************************************	//
string ItemEntity::GetName() {

	return this->name;
}

void ItemEntity::SetName(string name) {

	this->name = name;
}

string ItemEntity::GetValue() {
	if ( this->value.empty() ){
		return "";
	}else{
		return this->value.front()->GetValue();
	}
}

void ItemEntity::SetValue(string value) {
	if ( this->value.empty() ){
		this->value.push_back(new StringEntityValue(value));
	}else{
		this->value.front()->SetValue(value);
	}
}

void ItemEntity::SetValues(AbsEntityValueVector value){
	this->value = value;
}

AbsEntityValueVector ItemEntity::GetValues(){
	return this->value;
}

OvalEnum::Datatype ItemEntity::GetDatatype() {

	return this->datatype;
}

void ItemEntity::SetDatatype(OvalEnum::Datatype datatype) {

	this->datatype = datatype;
}

bool ItemEntity::GetIsObjectEntity() {

	return this->isObjectEntity;
}

void ItemEntity::SetIsObjectEntity(bool isObjectEntity) {

	this->isObjectEntity = isObjectEntity;
}

OvalEnum::SCStatus ItemEntity::GetStatus() {

	return this->scStatus;
}

void ItemEntity::SetStatus(OvalEnum::SCStatus scStatus) {
	
	this->scStatus = scStatus;
}

bool ItemEntity::Equals(ItemEntity* entity) {

	bool isEqual = false;

	if(this->GetDatatype() == entity->GetDatatype()) {
		if(this->GetName().compare(entity->GetName()) == 0) {
			if ( this->GetValues().size() == entity->GetValues().size() ){	
				// Need to check both vectors against each other as item entities may have fields that are not unique.
				// As a result, it is possible to have duplicate fields and this way we can make sure that the duplicate
				// values are not causing false positives.
				// Example (if you were to check just one side):
				// FieldVector 1                                  FieldVector 2
				// <field name="numbers" datatype="int">1</field> <field name="numbers" datatype="int">1</field>
				// <field name="numbers" datatype="int">2</field> <field name="numbers" datatype="int">2</field>
				// <field name="numbers" datatype="int">3</field> <field name="numbers" datatype="int">3</field>
				// <field name="numbers" datatype="int">3</field> <field name="numbers" datatype="int">4</field>
				// for each field in FieldVector 1
				//   if field does not exist in FieldVector 2
				// 		return false
				// return true
				// This will return true because 1,2,3, and 3 all exist in FieldVector 2.
				// Now if you check the other way, you will see that they are in fact not equal as 4 does not exist in FieldVector 1.
					
				for(AbsEntityValueVector::iterator it1 = this->GetValues().begin() ; it1 != this->GetValues().end() ; it1++){
					if(!this->ValueExistsInItemEntity(entity->GetValues(), (*it1))) {
						return false; // Short-circuit out. There is no need to do anything else because they are not equal.
					}	
				}

				for(AbsEntityValueVector::iterator it2 = entity->GetValues().begin() ; it2 != entity->GetValues().end() ; it2++){
					if(!this->ValueExistsInItemEntity(this->GetValues(), (*it2))) {
						return false; // Short-circuit out. There is no need to do anything else because they are not equal.
					}	
				}

				isEqual = true;
			}
		}
	}
	return isEqual;
}

bool ItemEntity::ValueExistsInItemEntity(AbsEntityValueVector entityValueVector, AbsEntityValue* entityValue) {

	bool exists = false;
	
	for(AbsEntityValueVector::iterator iterator = entityValueVector.begin(); iterator != entityValueVector.end(); iterator++) {
		if(entityValue->Equals(*iterator)) {
			exists = true;
			break;
		}
	}	

	return exists;
}

void ItemEntity::Write(XERCES_CPP_NAMESPACE_QUALIFIER DOMDocument* scFile, DOMElement* itemElm) {

	// Create new item element
	DOMElement* newItemEntityElem = scFile->createElement(XMLString::transcode(this->GetName().c_str()));
	itemElm->appendChild(newItemEntityElem);

	// Add the attributes
	// handling defaults in the schema
	string strDatatype = OvalEnum::DatatypeToString(this->GetDatatype());
	if(strDatatype.compare("string") != 0)
		XmlCommon::AddAttribute(newItemEntityElem, "datatype", strDatatype);

	string strStatus = OvalEnum::SCStatusToString(this->GetStatus());
	if(strStatus.compare("exists") != 0)
        XmlCommon::AddAttribute(newItemEntityElem, "status", strStatus);

	// Add the value(s)
	AbsEntityValueVector v = this->GetValues();
	for(AbsEntityValueVector::iterator it = v.begin(); it != v.end();it++){
		if( ((*it)->GetValue()).compare("") != 0) {
			if ( this->GetDatatype() != OvalEnum::DATATYPE_RECORD ){
				((StringEntityValue*)*it)->Write(scFile,newItemEntityElem);
			}else{
				((ItemFieldEntityValue*)*it)->Write(scFile,newItemEntityElem);
			}
		}
	}
}

string ItemEntity::UniqueString() {
	string valueStr = "";
	AbsEntityValueVector allValues = this->GetValues();
	for(AbsEntityValueVector::iterator it = allValues.begin(); it != allValues.end(); it++){
		// If a record append the name of each field before its value
		if ( this->GetDatatype() == OvalEnum::DATATYPE_RECORD ){
			valueStr.append(((ItemFieldEntityValue*)(*it))->GetName());
		}
		valueStr.append((*it)->GetValue());
	}
	return this->GetName() + valueStr;
}

void ItemEntity::Parse(DOMElement* itemEntityElm) {
	
	this->SetName(XmlCommon::GetElementName(itemEntityElm));
	this->SetDatatype(OvalEnum::ToDatatype(XmlCommon::GetAttributeByName(itemEntityElm, "datatype")));
	this->SetStatus(OvalEnum::ToSCStatus(XmlCommon::GetAttributeByName(itemEntityElm, "status")));
	
	// The datatype is not 'record' so we can just grab the string value in the element.
	if ( this->GetDatatype() != OvalEnum::DATATYPE_RECORD ){
		this->SetValue(XmlCommon::GetDataNodeValue(itemEntityElm));
	}else{
		// The datatype is 'record' so we need to loop over all of the elements (which are the fields).
		DOMNodeList *itemChildren = itemEntityElm->getChildNodes();
		unsigned int index = 0;
		AbsEntityValueVector aevv;
		while(index < itemChildren->getLength()) {
			DOMNode* tmpNode = itemChildren->item(index);

			//	Only concerned with ELEMENT_NODEs
			if (tmpNode->getNodeType() == DOMNode::ELEMENT_NODE) {
				DOMElement* itemChild = (DOMElement*)tmpNode;

				//	Get the name of the child
				string childName = XmlCommon::GetElementName(itemChild);
				if ( childName.compare("field") == 0 || childName.compare("oval-sc:field") == 0 ){
					ItemFieldEntityValue* fev = new ItemFieldEntityValue();
					fev->Parse(itemChild);
					aevv.push_back(fev);
				}else{
					StringEntityValue* sev = new StringEntityValue();
					sev->Parse(itemChild);
					aevv.push_back(sev);
				}
			}
			index ++;
		}	
		this->SetValues(aevv);
	}
}
