///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file BezierVertex.h
 * \brief Contains definition of Core::BezierVertex class.
 */

#ifndef __OVITO_BEZIER_VERTEX_H
#define __OVITO_BEZIER_VERTEX_H

#include <core/Core.h>

namespace Core {

/**
 * \brief A bezier vertex is part of a BezierCurve.
 *
 * \author Alexander Stukowski
 * \sa BezierCurve
 */
class CORE_DLLEXPORT BezierVertex
{
public:

	/**
	 * \brief The types of vertices that can be used in a BezierCurve.
	 */
	enum VertexType {
	    VertexCorner,
	    VertexSmooth,
	    VertexBezier,
	    VertexBezierCorner
	};

	/**
	 * \brief The types of segments that can be used in a BezierCurve.
	 */
	enum SegmentType {
	    SegmentBezier,
		SegmentLine
	};

	/// \brief Default constructor.
	BezierVertex() :
		_vertexType(VertexCorner), _segmentType(SegmentLine), _pos(ORIGIN), _in(ORIGIN), _out(ORIGIN) {}

	/// \brief Initializes the vertex with the given values.
	BezierVertex(VertexType vertexType, SegmentType segmentType, const Point3& pos, const Vector3& inVector, const Vector3& outVector) :
		_vertexType(vertexType), _segmentType(segmentType), _pos(pos), _in(pos + inVector), _out(pos + outVector) {}

	/// \brief Initializes the vertex with the given values.
	BezierVertex(VertexType vertexType, SegmentType segmentType, const Point3& pos, const Point3& in, const Point3& out) :
		_vertexType(vertexType), _segmentType(segmentType), _pos(pos), _in(in), _out(out) {}

	/// \brief Initializes the vertex to a be sharp corner vertex at the given position.
	BezierVertex(const Point3& pos) :
		_vertexType(VertexCorner), _segmentType(SegmentLine), _pos(pos), _in(pos), _out(pos) {}

	/// \brief Copy constructor.
	BezierVertex(const BezierVertex& vertex) :
		_vertexType(vertex._vertexType), _segmentType(vertex._segmentType), _pos(vertex._pos), _in(vertex._in), _out(vertex._out) {}


	/// \brief Returns the corner type of this vertex.
	VertexType vertexType() const { return _vertexType; }

	/// \brief Returns the line segment type associated with this vertex.
	SegmentType segmentType() const { return _segmentType; }

	/// \brief Returns the position of this vertex.
	const Point3& pos() const { return _pos; }

	/// \brief Returns the incoming tangent vector for this bezier vertex.
	Vector3 inVector() const { return _in - _pos; }

	/// \brief Returns the outgoing tangent vector for this bezier vertex.
	Vector3 outVector() const { return _out - _pos; }

	/// \brief Returns the point that defines the incoming tangent vector for this bezier vertex.
	const Point3& inPoint() const { return _in; }

	/// \brief Returns the point that defines outgoing tangent vector for this bezier vertex.
	const Point3& outPoint() const { return _out; }

private:

	VertexType _vertexType;
	SegmentType _segmentType;
	Point3 _pos;
	Point3 _in;
	Point3 _out;

	friend class BezierCurve;
};


};	// End of namespace Core

#endif // __OVITO_BEZIER_VERTEX_H
