
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: disc.c,v 1.43 2006/01/21 17:53:37 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <pthread.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <linux/cdrom.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "disc.h"
#include "environment.h"
#include "filelist.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"

typedef struct {
    int type;
    char file[128];
} cdrom_files_t;

static const cdrom_files_t cdrom_files[] = {
    {DISC_VCD, "vcd/info.vcd"},
    {DISC_SVCD, "svcd/info.svd"},
    {DISC_DVD_VIDEO, "video_ts"},
    {DISC_DVD_VIDEO, "VIDEO_TS"},
    {DISC_DVD_AUDIO, "audio_ts"},
    {DISC_DVD_AUDIO, "AUDIO_TS"},
    {0, ""}
};

int
disc_is_mounted (const char *mountpoint)
{
    FILE *f;
    char line[256];
    int mounted = 0;
    f = fopen ("/proc/mounts", "r");
    if (f != NULL) {
        while (fgets (line, 255, f) != NULL) {
            if (strstr (line, mountpoint) != NULL) {
                mounted = 1;
                break;
            }
        }
        fclose (f);
    }
    return mounted;
}

int
disc_mount (const char *mountpoint)
{
    char command[256];

    if (disc_is_mounted (mountpoint)) {
        return 1;
    }

    snprintf (command, 256, "%s %s", BIN_MOUNT, mountpoint);
    debug (command);

    if (system (command)) {
        error (_("Could not mount %s!"), mountpoint);
        return 0;
    }
    return 1;
}

int
disc_umount (const char *mountpoint)
{
    char command[256];

    snprintf (command, 256, "%s %s", BIN_UMOUNT, mountpoint);
    debug (command);

    if (system (command)) {
        error (_("Could not unmount %s!"), mountpoint);
        return 0;
    }

    return 1;
}


int
disc_is_in (const char *device)
{
    int fd = open (device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not open '%s': %s!"), device, strerror (errno));
        return 0;
    }

    int ret = 0;
    int drive_status = ioctl (fd, CDROM_DRIVE_STATUS);
    switch (drive_status) {
        case CDS_NO_INFO:
            debug ("drive reports: no info");
            break;
        case CDS_NO_DISC:
            debug ("drive reports: no disc");
            break;
        case CDS_TRAY_OPEN:
            debug ("drive reports: tray open");
            break;
        case CDS_DRIVE_NOT_READY:
            debug ("drive reports: drive not ready");
            break;
        case CDS_DISC_OK:
            debug ("drive reports: disc ok");
            ret = 1;
            break;
        default:
            debug ("drive reports: unknown value");
            break;
    }
    close (fd);

    return ret;
}

int
disc_eject (const char *mountpoint)
{
    char command[256];

    snprintf (command, 256, "%s %s", BIN_EJECT, mountpoint);
    debug (command);

    if (system (command)) {
        error (_("Could not eject %s!"), mountpoint);
        return 0;
    }

    return 1;
}

disc_type_t
disc_get_type (const char *device, const char *mountpoint)
{
    int fd = open (device, O_RDONLY | O_NONBLOCK);
    if (fd < 0) {
        error (_("Could not open '%s': %s!"), device, strerror (errno));
        return 0;
    }

    disc_type_t type = DISC_NO_INFO;
    int disc_status = ioctl (fd, CDROM_DISC_STATUS);
    switch (disc_status) {
        case CDS_AUDIO:
            type = DISC_AUDIO;
            break;
        case CDS_DATA_1:
            type = DISC_DATA;
            break;
        case CDS_DATA_2:
            type = DISC_DATA;
            break;
        case CDS_XA_2_1:
            break;
        case CDS_XA_2_2:
            break;
        case CDS_MIXED:
            break;
        case CDS_NO_INFO:
            type = DISC_NO_INFO;
            break;
        case CDS_NO_DISC:
            type = DISC_NO_DISC;
            break;
        default:
            debug ("unknown value: %u", disc_status);
    }
    close (fd);

    if (type != DISC_DATA)
        return type;

    if (!mountpoint)
        return type;

    if (!disc_mount (mountpoint))
        return type;

    const cdrom_files_t *files = cdrom_files;

    type = DISC_DATA;

    while (files->type) {
        char filename[1024];
        snprintf (filename, 1023, "%s/%s", mountpoint, files->file);
        if (access (filename, F_OK) == 0) {
            disc_umount (mountpoint);
            return files->type;
        }
        files++;
    }
    disc_umount (mountpoint);

    return type;
}


static void
disc_autoscan (playlist_t * playlist, filelist_t * filelist, const char *mrl)
{
    assert (strncasecmp (mrl, "cdda", 4) == 0);

    xine_t *xine = xine_new ();
    xine_init (xine);

    int num_mrls;
    char **autoplay_mrls = xine_get_autoplay_mrls (xine, "CD", &num_mrls);

    if (autoplay_mrls) {
        int i;
        for (i = 0; i < num_mrls; i++) {
            char title[1024];
            sprintf (title, "%s - %s %02d", _("Audio CD"), _("Title"), i + 1);
            char track_mrl[1024];
            sprintf (track_mrl, "%s/%d", mrl, i + 1);

            if (playlist)
                playlist_add (playlist, title, track_mrl);
            if (filelist)
                filelist_add (filelist, title, track_mrl, FILE_TYPE_REGULAR);
        }
    }

    xine_exit (xine);
}


void
disc_autoscan_load (playlist_t * playlist, const char *mrl)
{
    disc_autoscan (playlist, NULL, mrl);
}


void
disc_autoscan_read (filelist_t * filelist)
{
    disc_autoscan (NULL, filelist, filelist->mrl);
}


#ifdef HAVE_DISC_POLLING

static struct {
    pthread_t thread;
    int continue_running;
    oxine_t *oxine;
} disc_thread_info;

static void
poll_device (disc_entry_t * disc_entry)
{
    filelist_t *removable_discs = disc_thread_info.oxine->removable_discs;

    // ***********************************************************************
    // Its not a CD
    // ***********************************************************************
    if (!disc_entry->is_type_cd) {
        if (!filelist_contains (removable_discs, disc_entry->mountpoint)) {
            char title[64];
            snprintf (title, 63, "[%s: %s]", _("Removable Disc"),
                      disc_entry->mountpoint);
            filelist_add (removable_discs, title, disc_entry->mountpoint,
                          FILE_TYPE_MOUNTPOINT);
            debug ("found new device at %s", disc_entry->device);
        }
        return;
    }
    // ***********************************************************************
    // Its a CD
    // ***********************************************************************
    int disc_is_in = 0;
    int fd = open (disc_entry->device, O_RDONLY | O_NONBLOCK);
    if (fd) {
        if (ioctl (fd, CDROM_DRIVE_STATUS) == CDS_DISC_OK) {
            disc_is_in = 1;
        }
        close (fd);
    } else {
        // if we cannot open the device, 
        // we cannot get a disc status, so we exit
        return;
    }

    char tmp[1024];

    snprintf (tmp, 1023, "cdda:/%s", disc_entry->device);
    char *cdda_mrl = ho_strdup (tmp);
    snprintf (tmp, 1023, "dvd://%s", disc_entry->device);
    char *dvd_mrl = ho_strdup (tmp);
    snprintf (tmp, 1023, "vcd://%s", disc_entry->device);
    char *vcd_mrl = ho_strdup (tmp);

    if (!disc_entry->is_disc_in && disc_is_in) {
        disc_type_t disc_type =
            disc_get_type (disc_entry->device, disc_entry->mountpoint);

        switch (disc_type) {
            case DISC_AUDIO:
                if (!filelist_contains (removable_discs, cdda_mrl)) {
                    char title[64];
                    snprintf (title, 63, "[%s: %s]",
                              _("Removable Disc"), _("Audio CD"));
                    filelist_add (removable_discs, title,
                                  cdda_mrl, FILE_TYPE_AUTODISC);
                    debug ("an Audio CD was inserted at %s",
                           disc_entry->device);
                }
                break;

            case DISC_DVD_VIDEO:
                if (!filelist_contains (removable_discs, dvd_mrl)) {
                    char title[64];
                    snprintf (title, 63, "[%s: %s]",
                              _("Removable Disc"), _("Video DVD"));
                    filelist_add (removable_discs, title,
                                  dvd_mrl, FILE_TYPE_REGULAR);
                    debug ("a Video DVD was inserted at %s",
                           disc_entry->device);
                }
                break;

            case DISC_DVD_AUDIO:
                if (!filelist_contains (removable_discs, dvd_mrl)) {
                    char title[64];
                    snprintf (title, 63, "[%s: %s]",
                              _("Removable Disc"), _("Audio DVD"));
                    filelist_add (removable_discs, title,
                                  dvd_mrl, FILE_TYPE_REGULAR);
                    debug ("an Audio DVD was inserted at %s",
                           disc_entry->device);
                }
                break;

            case DISC_VCD:
                if (!filelist_contains (removable_discs, vcd_mrl)) {
                    char title[64];
                    snprintf (title, 63, "[%s: %s]",
                              _("Removable Disc"), _("Video CD"));
                    filelist_add (removable_discs, title,
                                  vcd_mrl, FILE_TYPE_REGULAR);
                    debug ("a Video CD was inserted at %s",
                           disc_entry->device);
                }

            case DISC_SVCD:
                if (!filelist_contains (removable_discs, vcd_mrl)) {
                    char title[64];
                    snprintf (title, 63, "[%s: %s]",
                              _("Removable Disc"), _("Super Video CD"));
                    filelist_add (removable_discs, title,
                                  vcd_mrl, FILE_TYPE_REGULAR);
                    debug ("a Super Video CD was inserted at %s",
                           disc_entry->device);
                }
                break;

            default:
                if (!filelist_contains
                    (removable_discs, disc_entry->mountpoint)) {
                    char title[64];
                    snprintf (title, 63, "[%s: %s]",
                              _("Removable Disc"), _("CDROM"));
                    filelist_add (removable_discs, title,
                                  disc_entry->mountpoint,
                                  FILE_TYPE_MOUNTPOINT);
                    debug ("a normal CDROM was inserted at %s",
                           disc_entry->device);
                }
                break;
        }

        // repaint the current menu
        assert (disc_thread_info.oxine->repaint_menu);
        disc_thread_info.oxine->repaint_menu (disc_thread_info.oxine);

        disc_entry->is_disc_in = 1;
    }

    else if (disc_entry->is_disc_in && !disc_is_in) {
        fileitem_t *fileitem = filelist_first (removable_discs);
        while (fileitem) {
            if ((strcasecmp (fileitem->mrl, disc_entry->mountpoint) == 0)
                || (strcasecmp (fileitem->mrl, cdda_mrl) == 0)
                || (strcasecmp (fileitem->mrl, vcd_mrl) == 0)
                || (strcasecmp (fileitem->mrl, dvd_mrl) == 0)) {
                filelist_remove (removable_discs, fileitem);
                debug ("removing disc at %s", disc_entry->device);
                break;
            }
            fileitem = filelist_next (removable_discs, fileitem);
        }

        // repaint the current menu
        assert (disc_thread_info.oxine->repaint_menu);
        disc_thread_info.oxine->repaint_menu (disc_thread_info.oxine);

        disc_entry->is_disc_in = 0;
    }

    ho_free (cdda_mrl);
    ho_free (dvd_mrl);
    ho_free (vcd_mrl);
}

static void *
disc_thread (void *data)
{
    filelist_t *removable_discs = disc_thread_info.oxine->removable_discs;

    while (disc_thread_info.continue_running) {
        disc_entry_t *disc_entry = get_first_disc (disc_thread_info.oxine);
        while (disc_entry) {
            // check for the existance of this device file
            if (access (disc_entry->device, F_OK) == 0) {
                // poll all existant devices
                poll_device (disc_entry);
            } else {
                // remove any entries that refer to a device that does not exist.
                fileitem_t *fileitem = filelist_first (removable_discs);
                while (fileitem) {
                    if (strcasecmp (fileitem->mrl, disc_entry->mountpoint) ==
                        0) {
                        filelist_remove (removable_discs, fileitem);
                    }
                    fileitem = filelist_next (removable_discs, fileitem);
                }
            }
            disc_entry = get_next_disc (disc_entry);
        }

        sleep (1);
    }

    pthread_exit (NULL);
    return NULL;
}


void
disc_start_polling (oxine_t * oxine)
{
    disc_thread_info.oxine = oxine;
    disc_thread_info.continue_running = 1;

    if (pthread_create (&disc_thread_info.thread, NULL, disc_thread, NULL) !=
        0) {
        error (_("Could not create disc polling thread!"));
    }
}


void
disc_stop_polling (oxine_t * oxine)
{
    disc_thread_info.continue_running = 0;

    void *ret;
    pthread_join (disc_thread_info.thread, &ret);
}
#endif
