
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: environment.c,v 1.25 2006/01/21 17:53:37 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <errno.h>
#include <fcntl.h>
#include <libgen.h>
#include <mntent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <linux/cdrom.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "environment.h"
#include "heap.h"
#include "i18n.h"
#include "list.h"
#include "logger.h"
#include "oxine.h"

static l_list_t *disc_list;

static char *dir_home = NULL;
static char *dir_oxine = NULL;
static char *dir_oxine_cache = NULL;
static char *dir_oxine_tmp = NULL;

static char *file_config = NULL;
static char *file_mediamarks_music = NULL;
static char *file_mediamarks_video = NULL;
static char *file_mediamarks_image = NULL;
static char *file_lirc_config = NULL;
static char *file_dvb_channels = NULL;
static char *file_favorites = NULL;
static char *file_lastplaylist = NULL;

static int
disc_list_contains (const char *device)
{
    disc_entry_t *entry = l_list_first (disc_list);
    while (entry) {
        if (strcmp (entry->device, device) == 0)
            return 1;
        entry = l_list_next (disc_list, entry);
    }
    return 0;
}

/*
 * Looks for the device in /etc/fstab.
 * Returns 1 if it finds the device.
 */
static int
scan_fstab (const char *device)
{
    FILE *fstab = fopen ("/etc/fstab", "r");
    if (fstab) {
        struct mntent *entry = getmntent (fstab);

        while (entry) {
            if ((strcmp (device, entry->mnt_fsname) == 0)
                && !disc_list_contains (device)) {
                info (_("Found a device we are interested in:"));
                info ("             device: %s", entry->mnt_fsname);
                info ("         mountpoint: %s", entry->mnt_dir);
                debug ("         filesystem: %s", entry->mnt_type);

                // We assume, that nobody has the same mountpoint in fstab more than once.
                disc_entry_t *new_entry = ho_new (disc_entry_t);
                new_entry->device = ho_strdup (entry->mnt_fsname);
                new_entry->mountpoint = ho_strdup (entry->mnt_dir);

                new_entry->is_type_cd = 0;
                new_entry->is_disc_in = 0;
                if (strstr (entry->mnt_type, "iso9660")
                    || strstr (entry->mnt_type, "udf")) {
                    new_entry->is_type_cd = 1;
                }
                debug ("         is type CD: %s",
                       new_entry->is_type_cd ? "true" : "false");

                l_list_append (disc_list, new_entry);

                endmntent (fstab);

                return 1;
            }
            entry = getmntent (fstab);
        }
    }

    endmntent (fstab);

    return 0;
}

static void
evaluate_device (const char *device)
{
    // Check for existance of device file.
    if (access (device, F_OK) == 0) {
        struct stat filestat;
        if (lstat (device, &filestat) == -1) {
            error (_("Unable to get status for file '%s': %s."),
                   device, strerror (errno));
            return;
        }
        if (S_ISLNK (filestat.st_mode)) {
            if (!scan_fstab (device)) {
                char link_target[1024];
                int len = readlink (device, link_target, 1024);
                if (len == -1) {
                    error (_("Unable to read link '%s' file: %s."),
                           device, strerror (errno));
                    return;
                }
                link_target[len] = 0;

                if (link_target[0] != '/') {
                    char *dev = ho_strdup (device);
                    char *link = ho_strdup (link_target);
                    snprintf (link_target, 1023,
                              "%s/%s", dirname (dev), link);
                    ho_free (dev);
                    ho_free (link);
                }

                debug ("found symbolic link: %s -> %s", device, link_target);
                evaluate_device (link_target);
            }
        }

        else if (S_ISBLK (filestat.st_mode)) {
            debug (" found block device: %s", device);
            scan_fstab (device);
        }
    } else {
        scan_fstab (device);
    }

}


static void
autofind_discs (oxine_t * oxine)
{
    static const char *confignames[] = {
        "media.dvd.device",
        "media.vcd.device",
        "media.audio_cd.device",
        "media.photo_cd.device",
        NULL
    };

    static const char *devices[] = {
        "/dev/cdrom", "/dev/cdrom1", "/dev/cdrom2",
        "/dev/cdrw", "/dev/cdrw1", "/dev/cdrw2",
        "/dev/dvd", "/dev/dvd1", "/dev/dvd2",
        "/dev/dvdrw", "/dev/dvdrw1", "/dev/dvdrw2",
        NULL
    };

    disc_list = l_list_new ();

    /* If polling is disabled we add all the devices we can find in the
     * configuration file. */
#ifdef HAVE_DISC_POLLING
    if (!oxine->use_polling)
#endif
    {
        int i = 0;
        while (confignames[i]) {
            xine_cfg_entry_t centry;
            xine_config_lookup_entry (oxine->xine, confignames[i++], &centry);

            if (centry.str_value && (strlen (centry.str_value) > 0)) {
                disc_entry_t *new_entry = ho_new (disc_entry_t);
                new_entry->device = ho_strdup (centry.str_value);
                new_entry->mountpoint = NULL;
                l_list_append (disc_list, new_entry);
            }
        }
    }
#ifdef HAVE_DISC_POLLING
    if (!oxine->use_polling)
        return;

    debug ("Start looking for devices (e.g. discs, cdrom)...");

    /* First we look for our standard list of devices. */
    int i = 0;
    while (devices[i]) {
        evaluate_device (devices[i++]);
    }

    /* Next we look for extra devices supplied by the user. */
    const char *extra_devices =
        xine_config_register_string (oxine->xine, "gui.discs.extra_devices",
                                     "",
                                     _("extra devices you want oxine to"
                                       " know about (separate using ';')"),
                                     _("The path to extra devices. This"
                                       " can be used to let oxine know"
                                       " about extra devices (e.g. a"
                                       " USB-Disc), that are not in the"
                                       " standard list of devices (separate"
                                       " using ';')."), 10, NULL,
                                     NULL);
    if (extra_devices && strlen (extra_devices) > 0) {
        char *devices = ho_strdup (extra_devices);
        char *device = strtok (devices, ";");
        while (device) {
            evaluate_device (device);
            device = strtok (NULL, ";");
        }
        ho_free (devices);
    }

    /* If we were not able to get the mountpoint and device from the static
     * entries, we try again using the configured values. */
    i = 0;
    while ((l_list_length (disc_list) == 0) && confignames[i]) {
        xine_cfg_entry_t centry;
        xine_config_lookup_entry (oxine->xine, confignames[i++], &centry);
        evaluate_device (centry.str_value);
    }

    if (l_list_length (disc_list) == 0) {
        xine_cfg_entry_t centry;
        xine_config_lookup_entry (oxine->xine, "media.dvd.device", &centry);

        error (_("Could not find mountpoint and device for the CDROM!"));
        info (_("I'll use media.dvd.device [%s] as the default device."),
              centry.str_value);
        warn (_("I will not be able to mount CDROMs!"));

        disc_entry_t *new_entry = ho_new (disc_entry_t);
        new_entry->device = ho_strdup (centry.str_value);
        new_entry->mountpoint = NULL;
        l_list_append (disc_list, new_entry);
    }
#endif
}


disc_entry_t *
get_first_disc (oxine_t * oxine)
{
    if (!disc_list) {
        autofind_discs (oxine);
    }

    return (disc_entry_t *) l_list_first (disc_list);
}


disc_entry_t *
get_next_disc (disc_entry_t * disc_entry)
{
    assert (disc_list);
    assert (disc_entry);

    return (disc_entry_t *) l_list_next (disc_list, disc_entry);
}


const char *
get_dir_home (void)
{
    if (!dir_home) {
        char *tmp = getenv ("HOME");
        if (tmp) {
            dir_home = ho_strdup (tmp);
            if (dir_home[strlen (tmp) - 1] == '/')
                dir_home[strlen (tmp) - 1] = '\0';
        } else {
            error (_("Unable to get home directory, setting it to '/tmp'."));
            dir_home = ho_strdup ("/tmp");
        }
    }

    return (const char *) dir_home;
}


const char *
get_dir_oxine (void)
{
    if (!dir_oxine) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/.oxine", get_dir_home ());
        dir_oxine = ho_strdup (tmp);

        mkdir (dir_oxine, 0700);
    }

    return (const char *) dir_oxine;
}


const char *
get_dir_oxine_cache (void)
{
    if (!dir_oxine_cache) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/cache", get_dir_oxine ());
        dir_oxine_cache = ho_strdup (tmp);

        mkdir (dir_oxine_cache, 0700);
    }

    return (const char *) dir_oxine_cache;
}


const char *
get_dir_oxine_tmp (void)
{
    if (!dir_oxine_tmp) {
        dir_oxine_tmp = ho_strdup ("/tmp/.oxine");

        mkdir (dir_oxine_tmp, 0700);
    }

    return (const char *) dir_oxine_tmp;
}


const char *
get_file_config (void)
{
    if (!file_config) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/config", get_dir_oxine ());
        file_config = ho_strdup (tmp);
    }

    return (const char *) file_config;
}


const char *
get_file_mediamarks_music (void)
{
    if (!file_mediamarks_music) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/music_mediamarks.xml", get_dir_oxine ());
        file_mediamarks_music = ho_strdup (tmp);
    }

    return (const char *) file_mediamarks_music;
}


const char *
get_file_mediamarks_video (void)
{
    if (!file_mediamarks_video) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/video_mediamarks.xml", get_dir_oxine ());
        file_mediamarks_video = ho_strdup (tmp);
    }

    return (const char *) file_mediamarks_video;
}


const char *
get_file_mediamarks_image (void)
{
    if (!file_mediamarks_image) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/image_mediamarks.xml", get_dir_oxine ());
        file_mediamarks_image = ho_strdup (tmp);
    }

    return (const char *) file_mediamarks_image;
}


const char *
get_file_lirc_config (void)
{
    if (!file_lirc_config) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/lircrc", get_dir_oxine ());
        file_lirc_config = ho_strdup (tmp);
    }

    return (const char *) file_lirc_config;
}


const char *
get_file_dvb_channels (void)
{
    if (!file_dvb_channels) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/channels.conf", get_dir_oxine ());
        file_dvb_channels = ho_strdup (tmp);
    }

    return (const char *) file_dvb_channels;
}


const char *
get_file_favorites (void)
{
    if (!file_favorites) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/favorites.xml", get_dir_oxine ());
        file_favorites = ho_strdup (tmp);
    }

    return (const char *) file_favorites;
}


const char *
get_file_lastplaylist (void)
{
    if (!file_lastplaylist) {
        char tmp[1024];
        snprintf (tmp, 1023, "%s/lastplaylist.m3u", get_dir_oxine ());
        file_lastplaylist = ho_strdup (tmp);
    }

    return (const char *) file_lastplaylist;
}


int
file_is_newer (const char *mrl1, const char *mrl2)
{
    struct stat stat1;
    if (stat (mrl1, &stat1) == -1) {
        error (_("Unable to get status for file '%s': %s."), mrl1,
               strerror (errno));
        return 0;
    }

    struct stat stat2;
    if (stat (mrl2, &stat2) == -1) {
        error (_("Unable to get status for file '%s': %s."), mrl2,
               strerror (errno));
        return 1;
    }

    return (stat1.st_mtime > stat2.st_mtime);
}


int
mkdir_recursive (const char *mrl, mode_t mode)
{
    if (access (mrl, R_OK) != 0) {
        if (mkdir (mrl, mode) == -1) {
            char *dirn = ho_strdup (mrl);
            mkdir_recursive (dirname (dirn), mode);
            ho_free (dirn);

            if (mkdir (mrl, mode) == -1) {
                error (_("Could not create '%s': %s!"), mrl,
                       strerror (errno));
                return 0;
            }
        }
    }
    return 1;
}


/*
 * This converts a relative path to an absolute one.
 */
static char *
relative_to_absolute (const char *filename)
{
    l_list_t *list = l_list_new ();

    // we add all parts of our current directory to the list
    if (filename[0] != '/') {
        char _directory[1024];
        getcwd (_directory, 1024);

        char *d = strtok (_directory, "/");
        while (d) {
            l_list_append (list, d);
            d = strtok (NULL, "/");
        }
    }
    // we add all parts of the filename to the list
    char _filename[1024];
    strncpy (_filename, filename, 1023);

    char *f = strtok (_filename, "/");
    while (f) {
        l_list_append (list, f);
        f = strtok (NULL, "/");
    }
    // we remove any relative parts
    l_item_t *item = list->first;
    while (item) {
        if (strcmp ("..", (char *) item->data) == 0) {
            l_list_remove (list, item->prev->data);
            l_list_remove (list, item->data);
            item = list->first;
        }

        else if (strcmp (".", (char *) item->data) == 0) {
            l_list_remove (list, item->data);
            item = list->first;
        }
        item = item->next;
    }

    char _result[1024];
    char *p = _result;
    item = list->first;
    while (item) {
        *p++ = '/';
        char *q = (char *) item->data;
        while (*q != '\0')
            *p++ = *q++;
        item = item->next;
    }
    *p = '\0';

    l_list_free (list, NULL);

    return ho_strdup (_result);
}


/*
 * This first converts a relative path to an absolute 
 * one and then prepends 'file://' to it.
 */
static char *
filename_to_uri (const char *filename)
{
    char *_filename = relative_to_absolute (filename);

    char *_result = ho_malloc (strlen (_filename) + 8);
    sprintf (_result, "%s%s", "file://", _filename);

    ho_free (_filename);

    return _result;
}


typedef enum {
    UNSAFE_ALL = 0x1,           /* Escape all unsafe characters   */
    UNSAFE_ALLOW_PLUS = 0x2,    /* Allows '+'  */
    UNSAFE_PATH = 0x8,          /* Allows '/', '&', '=', ':', '@', '+', '$' and ',' */
    UNSAFE_HOST = 0x10,         /* Allows '/' and ':' and '@' */
    UNSAFE_SLASHES = 0x20       /* Allows all characters except for '/' and '%' */
} UnsafeCharacterSet;

static const char acceptable[96] = {
    /* A table of the ASCII chars from space (32) to DEL (127) */
    /*      !    "    #    $    %    &    '    (    )    *    +    ,    -    .    / */
    0x00, 0x3F, 0x20, 0x20, 0x28, 0x00, 0x2C, 0x3F,
    0x3F, 0x3F, 0x3F, 0x2A, 0x28, 0x3F, 0x3F, 0x1C,
    /* 0    1    2    3    4    5    6    7    8    9    :    ;    <    =    >    ? */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x38, 0x20, 0x20, 0x2C, 0x20, 0x20,
    /* @    A    B    C    D    E    F    G    H    I    J    K    L    M    N    O */
    0x38, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    /* P    Q    R    S    T    U    V    W    X    Y    Z    [    \    ]    ^    _ */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x20, 0x20, 0x20, 0x20, 0x3F,
    /* `    a    b    c    d    e    f    g    h    i    j    k    l    m    n    o */
    0x20, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    /* p    q    r    s    t    u    v    w    x    y    z    {    |    }    ~  DEL */
    0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F, 0x3F,
    0x3F, 0x3F, 0x3F, 0x20, 0x20, 0x20, 0x3F, 0x20
};

static const char hex[16] = "0123456789ABCDEF";

/* FIXME: The original also escapes the ? as this can be part of a valid http URL I
 *        decided to leave it. I'm not sure if this is a good idea though. */
//#define ACCEPTABLE_URI(a) ((a)>=32 && (a)<128 && (acceptable[(a)-32] & use_mask))
#define ACCEPTABLE_URI(a) ((a) == '?' || ((a)>=32 && (a)<128 && (acceptable[(a)-32] & use_mask)))

char *
filename_escape_to_uri (const char *filename)
{
    if (!filename)
        return NULL;

    // We convert a normal filename to an URI, everything else we leave as it
    // is. This could be done a lot better as this only converts an existing
    // file to an URI (but we dont care about nonexistant files).
    char *_filename;
    if (access (filename, F_OK) == 0)
        _filename = filename_to_uri (filename);
    else
        _filename = ho_strdup (filename);

    // Watchit:
    // The following code was mostly taken from glib (g_filename_to_uri).
    UnsafeCharacterSet use_mask = UNSAFE_PATH;
    int unacceptable = 0;

    const char *p;
    for (p = _filename; *p != '\0'; p++) {
        int c = (unsigned char) *p;
        if (!ACCEPTABLE_URI (c))
            unacceptable++;
    }

    char *result = ho_malloc (strlen (_filename) + 2 * unacceptable + 1);

    char *q = result;
    for (p = _filename; *p != '\0'; p++) {
        int c = (unsigned char) *p;
        if (!ACCEPTABLE_URI (c)) {
            *q++ = '%';
            *q++ = hex[c >> 4];
            *q++ = hex[c & 15];
        } else {
            *q++ = *p;
        }
    }
    *q = '\0';

    ho_free (_filename);

    //debug ("%s -> %s", filename, result);

    return result;
}

#define UNACCEPTABLE_SHELL(a) ((a == ' ') || (a == '&') \
                              || (a == '(') || (a == ')'))

char *
filename_escape_for_shell (const char *filename)
{
    if (!filename)
        return NULL;

    int unacceptable = 0;

    const char *p;
    for (p = filename; *p != '\0'; p++) {
        char c = (char) *p;
        if (UNACCEPTABLE_SHELL (c))
            unacceptable++;
    }

    char *result = ho_malloc (strlen (filename) + unacceptable + 1);

    char *q = result;
    for (p = filename; *p != '\0'; p++) {
        char c = (char) *p;
        if (UNACCEPTABLE_SHELL (c)) {
            *q++ = '\\';
        }
        *q++ = *p;
    }
    *q = '\0';

    //debug ("%s -> %s", filename, result);

    return result;
}


static void
disc_entry_free (void *entry_p)
{
    disc_entry_t *entry = (disc_entry_t *) entry_p;

    if (!entry)
        return;
    if (entry->device)
        ho_free (entry->device);
    if (entry->mountpoint)
        ho_free (entry->mountpoint);
    ho_free (entry);
}

void
environment_free (void)
{
    if (disc_list)
        l_list_free (disc_list, disc_entry_free);
    if (dir_home)
        ho_free (dir_home);
    if (dir_oxine)
        ho_free (dir_oxine);
    if (dir_oxine_cache)
        ho_free (dir_oxine_cache);
    if (dir_oxine_tmp)
        ho_free (dir_oxine_tmp);
    if (file_config)
        ho_free (file_config);
    if (file_mediamarks_music)
        ho_free (file_mediamarks_music);
    if (file_mediamarks_video)
        ho_free (file_mediamarks_video);
    if (file_mediamarks_image)
        ho_free (file_mediamarks_image);
    if (file_lirc_config)
        ho_free (file_lirc_config);
    if (file_dvb_channels)
        ho_free (file_dvb_channels);
    if (file_favorites)
        ho_free (file_favorites);
    if (file_lastplaylist)
        ho_free (file_lastplaylist);
}
