
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: filelist.c,v 1.27 2006/01/21 17:53:37 mschwerin Exp $
 *
 */
#include "config.h"

#include <errno.h>
#include <libgen.h>
#include <assert.h>
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "filelist.h"
#include "disc.h"
#include "i18n.h"
#include "heap.h"
#include "logger.h"
#include "playlist_m3u.h"
#include "mediamarks.h"

extern oxine_t *oxine;

static char *allowed_audio_suffixes[] = {
    "mp3", "ogg", "wav", "m4a",
    "wma", "aac", "flac", "mka",
    NULL
};

static char *allowed_video_suffixes[] = {
    "avi", "mpg", "mpeg", "rm",
    "divx", "ogm", "asf", "m2v",
    "m2p", "mp4", "mov", "cue",
    "wmv", "wma", "wvx", "wax",
    "vob", "mkv", NULL
};

static char *allowed_image_suffixes[] = {
    "jpg", "jpeg", "png", "gif",
    NULL
};

static char *allowed_spu_suffixes[] = {
    "sub", "srt", "asc", "smi",
    "ssa", "txt", NULL
};

/* 
 * ***************************************************************************
 * Name:            is_file_allowed
 * Access:          private
 *
 * Description:     Determines if the file is to be added depending on the
 *                  currently allowed filetypes. The suffix of the file is
 *                  compared to all suffixes of the allowed type.
 * ***************************************************************************
 */
static int
is_file_allowed (const char *mrl, int allowed_filetypes)
{
    if (allowed_filetypes == ALLOW_FILES_ANY) {
        int allowed = 0;

        allowed |= is_file_allowed (mrl, ALLOW_FILES_MUSIC);
        allowed |= is_file_allowed (mrl, ALLOW_FILES_VIDEO);
        allowed |= is_file_allowed (mrl, ALLOW_FILES_IMAGE);

        return allowed;
    }

    char *suffix = rindex (mrl, '.');

    if (!suffix)
        return 0;

    suffix = (char *) (suffix + 1);

    switch (allowed_filetypes) {
        case ALLOW_FILES_MUSIC:
            {
                int i = 0;
                for (; allowed_audio_suffixes[i]; i++) {
                    if (strcasecmp (allowed_audio_suffixes[i], suffix) == 0) {
                        return 1;
                    }
                }
            }
            break;
        case ALLOW_FILES_VIDEO:
            {
                int i = 0;
                for (; allowed_video_suffixes[i]; i++) {
                    if (strcasecmp (allowed_video_suffixes[i], suffix) == 0) {
                        return 1;
                    }
                }
            }
            break;
#ifdef HAVE_IMAGE_TOOLS
        case ALLOW_FILES_IMAGE:
            {
                int i = 0;
                for (; allowed_image_suffixes[i]; i++) {
                    if (strcasecmp (allowed_image_suffixes[i], suffix) == 0) {
                        return 1;
                    }
                }
            }
            break;
#endif
        case ALLOW_FILES_SUBTITLE:
            {
                int i = 0;
                for (; allowed_spu_suffixes[i]; i++) {
                    if (strcasecmp (allowed_spu_suffixes[i], suffix) == 0) {
                        return 1;
                    }
                }
            }
            break;
    }

    return 0;
}


/* 
 * ***************************************************************************
 * Name:            has_suffix
 * Access:          public
 *
 * Description:     Determines if a file has the suffix passed.
 * ***************************************************************************
 */
static int
has_suffix (const char *mrl, const char *sfx)
{
    char *suffix = rindex (mrl, '.');

    if (!suffix)
        return 0;

    suffix = (char *) (suffix + 1);

    return (strcasecmp (suffix, sfx) == 0) ? 1 : 0;
}


/* 
 * ***************************************************************************
 * Name:            fileitem_free_cb
 * Access:          private
 *
 * Description:     This is the callback passed to l_list_free and
 *                  l_list_clear to free the memory of a fileitem. 
 * ***************************************************************************
 */
static void
fileitem_free_cb (void *data)
{
    fileitem_t *fileitem = (fileitem_t *) data;

    assert (fileitem);

    if (fileitem->sublist && fileitem->type != FILE_TYPE_UPLINK) {
        filelist_free (fileitem->sublist);
    }
    if (fileitem->title)
        ho_free (fileitem->title);
    if (fileitem->mrl)
        ho_free (fileitem->mrl);
    if (fileitem->description)
        ho_free (fileitem->description);
    if (fileitem->thumbnail_mrl)
        ho_free (fileitem->thumbnail_mrl);
    ho_free (fileitem);
}


/* 
 * ***************************************************************************
 * Name:            fileitem_swap_cb
 * Access:          private
 *
 * Description:     This is the callback passed to l_list_sort, to determin,
 *                  if two fileitems have to be reordered.
 * ***************************************************************************
 */
static int
fileitem_swap_cb (void *d1, void *d2)
{
    fileitem_t *f1 = (fileitem_t *) d1;
    fileitem_t *f2 = (fileitem_t *) d2;

    assert (f1);
    assert (f2);

    // uplink is always first
    if (f2->type == FILE_TYPE_UPLINK)
        return 1;
    if (f1->type == FILE_TYPE_UPLINK)
        return 0;

    // mediamark should always be at the top
    if (f2->type == FILE_TYPE_MEDIAMARKS)
        return 1;
    if (f1->type == FILE_TYPE_MEDIAMARKS)
        return 0;

    // playlists should always be near the top
    if ((f2->type == FILE_TYPE_M3U) && (f1->type != FILE_TYPE_M3U))
        return 1;
    if ((f1->type == FILE_TYPE_M3U) && (f2->type != FILE_TYPE_M3U))
        return 0;

    // directories should be before files
    if ((f2->type == FILE_TYPE_DIRECTORY)
        && (f1->type != FILE_TYPE_DIRECTORY))
        return 1;
    if ((f1->type == FILE_TYPE_DIRECTORY)
        && (f2->type != FILE_TYPE_DIRECTORY))
        return 0;

    // directories should be before files
    if ((f2->type == FILE_TYPE_MOUNTPOINT)
        && (f1->type != FILE_TYPE_MOUNTPOINT))
        return 0;
    if ((f1->type == FILE_TYPE_MOUNTPOINT)
        && (f2->type != FILE_TYPE_MOUNTPOINT))
        return 1;

    // directories should be before files
    if ((f2->type == FILE_TYPE_AUTODISC)
        && (f1->type != FILE_TYPE_AUTODISC))
        return 0;
    if ((f1->type == FILE_TYPE_AUTODISC)
        && (f2->type != FILE_TYPE_AUTODISC))
        return 1;

    // sort the rest by name
    if (strcmp (f1->title, f2->title) > 0)
        return 1;

    return 0;
}


/* 
 * ***************************************************************************
 * Name:            filelist_new
 * Access:          public
 *
 * Description:     Creates a new filelist. Only files of the allowed type
 *                  will be added when expanding a directory.
 * ***************************************************************************
 */
filelist_t *
filelist_new (fileitem_allowed_t allowed_filetypes, const char *mrl)
{
    filelist_t *filelist = ho_new (filelist_t);

    filelist->list = l_list_new ();
    filelist->top_position = 0;
    filelist->cur_position = 0;
    filelist->allowed_filetypes = allowed_filetypes;
    filelist->mrl = ho_strdup (mrl);

    return filelist;
}


/* 
 * ***************************************************************************
 * Name:            filelist_clear
 * Access:          public
 *
 * Description:     Removes all fileitems from the list.
 * ***************************************************************************
 */
void
filelist_clear (filelist_t * filelist)
{
    assert (filelist);

    l_list_clear (filelist->list, fileitem_free_cb);
    filelist->top_position = 0;
    filelist->cur_position = 0;
}


/* 
 * ***************************************************************************
 * Name:            filelist_free
 * Access:          public
 *
 * Description:     Removes all fileitems and frees the list.
 * ***************************************************************************
 */
void
filelist_free (filelist_t * filelist)
{
    assert (filelist);

    // This is not really the right place to do this,
    // but it has to be done.
    if (filelist == oxine->toplevel_filelist)
        oxine->toplevel_filelist = NULL;
    if (filelist == oxine->current_filelist)
        oxine->current_filelist = oxine->toplevel_filelist;

    l_list_free (filelist->list, fileitem_free_cb);

    if (filelist->mrl)
        ho_free (filelist->mrl);

    ho_free (filelist);
}


/* 
 * ***************************************************************************
 * Name:            filelist_sort
 * Access:          public
 *
 * Description:     Sorts the list. swap_cb is a callback function, that is
 *                  used to determin if two entries must be swapped.
 * ***************************************************************************
 */
void
filelist_sort (filelist_t * filelist, l_swap_cb_t swap_cb)
{
    assert (filelist);

    if (swap_cb)
        l_list_sort (filelist->list, swap_cb);
    else
        l_list_sort (filelist->list, fileitem_swap_cb);
}


/* 
 * ***************************************************************************
 * Name:            filelist_add
 * Access:          public
 *
 * Description:     Adds a new entry to the list.
 * ***************************************************************************
 */
fileitem_t *
filelist_add (filelist_t * filelist, const char *title, const char *mrl,
              fileitem_type_t type)
{
    assert (filelist);
    assert (title);

    if (type == FILE_TYPE_UNKNOWN) {
        struct stat filestat;
        stat (mrl, &filestat);

        // it's a directory
        if (S_ISDIR (filestat.st_mode) && access (mrl, F_OK) == 0) {
            type = FILE_TYPE_DIRECTORY;
        }
        // it's a playlist
        else if (has_suffix (mrl, "m3u")) {
            type = FILE_TYPE_M3U;
        }
        // we assume its a regular file
        else {
            type = FILE_TYPE_REGULAR;
        }
    }

    fileitem_t *fileitem = ho_new (fileitem_t);

    fileitem->title = ho_strdup (title);
    if (mrl)
        fileitem->mrl = ho_strdup (mrl);
    fileitem->type = type;
    fileitem->sublist = NULL;

    l_list_append (filelist->list, fileitem);

    return fileitem;
}


/* 
 * ***************************************************************************
 * Name:            create_title
 * Access:          public
 *
 * Description:     Creates a title from the MRL. The returned string must be
 *                  freed when not used any more.
 * ***************************************************************************
 */
char *
create_title (const char *mrl)
{
    char *title = NULL;
    if (mrl) {
        // get basename
        char *tmp = ho_strdup (mrl);

        if (strncasecmp (tmp, "dvd:/", 5) == 0) {
            title = ho_strdup (_("DVD"));
        }

        else if (strncasecmp (tmp, "vcd:/", 5) == 0) {
            title = ho_strdup (_("Video CD"));
        }

        else if (strncasecmp (tmp, "cdda:/", 6) == 0) {
            title = ho_strdup (_("Audio CD"));
        }

        else {
            title = ho_strdup (basename (tmp));

            // remove suffix
            char *suffix = rindex (title, '.');
            if (suffix)
                suffix[0] = '\0';

            // replace stuff
            int i = 0;
            for (; i < strlen (title); i++) {
                if (title[i] == '_')
                    title[i] = ' ';
            }
        }
        ho_free (tmp);
    }
    return title;
}


/* 
 * ***************************************************************************
 * Name:            directory_read
 * Access:          static
 *
 * Description:     Reads a directory and adds all entries to the list that
 *                  are of the currently allowed type.
 * ***************************************************************************
 */
static int
directory_read (filelist_t * filelist)
{
    DIR *dirp;
    struct dirent *entp;

    assert (filelist);
    assert (filelist->mrl);

    dirp = opendir (filelist->mrl);
    if (dirp == NULL) {
        error (_("Could not open '%s': %s!"), filelist->mrl,
               strerror (errno));
        return 0;
    }

    while ((entp = readdir (dirp))) {
        struct stat filestat;
        char sub_mrl[1024];

        if (entp->d_name[0] == '.') {
            continue;
        }

        snprintf (sub_mrl, 1023, "%s/%s", filelist->mrl, entp->d_name);
        stat (sub_mrl, &filestat);

        // it's a directory
        if (S_ISDIR (filestat.st_mode)) {
            char title[1024];
            snprintf (title, 1023, "[%s]", entp->d_name);
            filelist_add (filelist, title, sub_mrl, FILE_TYPE_DIRECTORY);
        }
        // it's a regular file 
        else if (S_ISREG (filestat.st_mode)) {
            if (has_suffix (sub_mrl, "m3u")) {
                char title[256];
                snprintf (title, 256, _("[M3U-Playlist: %s]"),
                          basename (entp->d_name));
                filelist_add (filelist, title, sub_mrl, FILE_TYPE_M3U);
            } else if (is_file_allowed (sub_mrl, filelist->allowed_filetypes)) {
                char *title = create_title (entp->d_name);
                filelist_add (filelist, title, sub_mrl, FILE_TYPE_REGULAR);
                ho_free (title);
            }
        }
    }
    closedir (dirp);

    filelist_sort (filelist, NULL);

    return 1;
}


/* 
 * ***************************************************************************
 * Name:            filelist_expand
 * Access:          public
 *
 * Description:     Depending on the type of the fileitem it is expanded.
 *                  Expanding means that any directory, playlist or
 *                  mediamarks-file is read and added to the sublist of the
 *                  fileitem.
 * ***************************************************************************
 */
void
filelist_expand (filelist_t * parent, fileitem_t * fileitem)
{
    assert (parent);
    assert (fileitem);

    if (!fileitem->sublist &&
        (fileitem->type == FILE_TYPE_DIRECTORY ||
         fileitem->type == FILE_TYPE_MOUNTPOINT ||
         fileitem->type == FILE_TYPE_MEDIAMARKS ||
         fileitem->type == FILE_TYPE_M3U ||
         fileitem->type == FILE_TYPE_AUTODISC)) {

        assert (fileitem->mrl);

        // create new sublist
        fileitem->sublist =
            filelist_new (parent->allowed_filetypes, fileitem->mrl);

        // and add the uplink
        fileitem_t *uplink =
            filelist_add (fileitem->sublist, "[..]", parent->mrl,
                          FILE_TYPE_UPLINK);
        uplink->sublist = parent;

        switch (fileitem->type) {
            case FILE_TYPE_MOUNTPOINT:
            case FILE_TYPE_DIRECTORY:
                directory_read (fileitem->sublist);
                break;
            case FILE_TYPE_MEDIAMARKS:
                mediamarks_read (fileitem->sublist);
                break;
            case FILE_TYPE_M3U:
                m3u_read (fileitem->sublist);
                break;
            case FILE_TYPE_AUTODISC:
                disc_autoscan_read (fileitem->sublist);
            default:
                break;
        }
    }
}


/* 
 * ***************************************************************************
 * Name:            filelist_remove
 * Access:          public
 *
 * Description:     Removes the fileitem from the list.
 * ***************************************************************************
 */
void
filelist_remove (filelist_t * filelist, fileitem_t * fileitem)
{
    assert (filelist);
    assert (fileitem);

    l_list_remove (filelist->list, fileitem);

    if (fileitem->sublist && fileitem->type != FILE_TYPE_UPLINK) {
        filelist_free (fileitem->sublist);
    }
    if (fileitem->title) {
        ho_free (fileitem->title);
    }
    if (fileitem->mrl) {
        ho_free (fileitem->mrl);
    }
    ho_free (fileitem);
}


/* 
 * ***************************************************************************
 * Name:            filelist_first
 * Access:          public
 *
 * Description:     Returns the first fileitem.
 * ***************************************************************************
 */
fileitem_t *
filelist_first (filelist_t * filelist)
{
    assert (filelist);

    return (fileitem_t *) l_list_first (filelist->list);
}


/* 
 * ***************************************************************************
 * Name:            filelist_next
 * Access:          public
 *
 * Description:     Returns the next fileitem.
 * ***************************************************************************
 */
fileitem_t *
filelist_next (filelist_t * filelist, fileitem_t * fileitem)
{
    assert (filelist);
    assert (fileitem);

    return (fileitem_t *) l_list_next (filelist->list, fileitem);
}


/* 
 * ***************************************************************************
 * Name:            filelist_contains
 * Access:          public
 *
 * Description:     Determines, if the list contains an entry with the given
 *                  MRL.
 * ***************************************************************************
 */
int
filelist_contains (filelist_t * filelist, const char *mrl)
{
    assert (filelist);

    if (mrl) {
        fileitem_t *fileitem = filelist_first (filelist);
        while (fileitem) {
            if (strcasecmp (fileitem->mrl, mrl) == 0) {
                return 1;
            }
            fileitem = filelist_next (filelist, fileitem);
        }
    }

    return 0;
}
