
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: filelist_menu.c,v 1.61 2006/02/07 07:31:48 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <fcntl.h>
#include <libgen.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "disc.h"
#include "download.h"
#include "environment.h"
#include "filelist.h"
#include "filelist_menu.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "mediamarks.h"
#include "mostlistened.h"
#include "oxine.h"
#include "playlist.h"
#include "playlist_menu.h"

extern oxine_t *oxine;

static char *current_thumbnail;

static otk_widget_t *filelist_menu_list;
static otk_widget_t *filelist_menu_window;

static void update_filelist_menu_list (void *oxine_p);

static void
play_videocd_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    please_wait (oxine);

    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "media.vcd.device", &centry);

    if (disc_is_in (centry.str_value)) {
        char mrl[1024];
        snprintf (mrl, 1024, "vcd://%s", centry.str_value);

        // clear the playlist
        playlist_clear (oxine->playlist);

        // add to playlist
        playlist_add (oxine->playlist, _("Video CD"), mrl);

        // play playlist
        playlist_play_first (oxine);
    } else {
        show_message_ok_cancel (oxine, _("Please insert a Video CD!"),
                                play_videocd_cb, oxine->backto_menu);
    }
}

static void
play_dvd_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    please_wait (oxine);

    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "media.dvd.device", &centry);

    if (disc_is_in (centry.str_value)) {
        char mrl[1024];
        snprintf (mrl, 1024, "dvd://%s", centry.str_value);

        // clear the playlist
        playlist_clear (oxine->playlist);

        // add to playlist
        playlist_add (oxine->playlist, _("DVD"), mrl);

        // play playlist
        playlist_play_first (oxine);
    } else {
        show_message_ok_cancel (oxine, _("Please insert a DVD!"),
                                play_dvd_cb, oxine->backto_menu);
    }
}

static void
play_audiocd_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    please_wait (oxine);

    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "media.audio_cd.device", &centry);

    if (disc_get_type (centry.str_value, NULL) == DISC_AUDIO) {
        char mrl[1024];
        snprintf (mrl, 1024, "cdda:/%s", centry.str_value);

        // clear the playlist
        playlist_clear (oxine->playlist);

        // add all titles to the playlist
        disc_autoscan_load (oxine->playlist, mrl);

        // play playlist
        playlist_play_first (oxine);
    } else {
        show_message_ok_cancel (oxine, _("Please insert a Audio CD!"),
                                play_audiocd_cb, oxine->backto_menu);
    }
}


#ifdef HAVE_IMAGE_TOOLS
static void
play_photocd_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;
    please_wait (oxine);

    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "media.audio_cd.device", &centry);

    if (disc_is_in (centry.str_value)) {
        // search for a mountpoint for this device
        disc_entry_t *disc_entry = get_first_disc (oxine);
        while (disc_entry) {
            if (strcmp (centry.str_value, disc_entry->device) == 0) {
                if (disc_mount (disc_entry->mountpoint)) {
                    // clear the playlist
                    playlist_clear (oxine->playlist);

                    // add all images from the disc
                    filelist_t *list = filelist_new (ALLOW_FILES_IMAGE, "/");
                    fileitem_t *item = filelist_add (list, _("Photo CD"),
                                                     disc_entry->mountpoint,
                                                     FILE_TYPE_MOUNTPOINT);
                    filelist_expand (list, item);
                    playlist_add_fileitem (oxine->playlist, item);
                    filelist_free (list);

                    // play playlist
                    playlist_play_first (oxine);
                } else {
                    show_error (oxine, _("Could not open disc!"),
                                oxine->backto_menu);
                }
                return;
            }
            disc_entry = get_next_disc (disc_entry);
        }
        char msg[512];
        snprintf (msg, 511, _("Could not find mountpoint for device %s!"),
                  centry.str_value);
        show_error (oxine, msg, oxine->backto_menu);
    } else {
        show_message_ok_cancel (oxine, _("Please insert a Photo CD!"),
                                play_photocd_cb, oxine->backto_menu);
    }
}
#endif


static int
selected_add_to_playlist (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    int num_selected;
    int i;
    fileitem_t **fileitems =
        (fileitem_t **) otk_list_get_selected (filelist_menu_list,
                                               &num_selected);
    if (fileitems) {
        for (i = 0; i < num_selected; i++) {
            filelist_expand (oxine->current_filelist, fileitems[i]);
            playlist_add_fileitem (oxine->playlist, fileitems[i]);
            if (oxine->mostlistened_filelist
                && fileitems[i]->type == FILE_TYPE_DIRECTORY) {
                mostlistened_add_directory (oxine, fileitems[i]->title,
                                            fileitems[i]->mrl);
            }
        }
        ho_free (fileitems);
    }

    return num_selected;
}

static void
add_selected_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (otk_list_get_selected_count (filelist_menu_list)) {
        if (selected_add_to_playlist (oxine_p)) {
            show_playlist_menu_cb (oxine);
        }
    }
}

static void
play_selected_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (otk_list_get_selected_count (filelist_menu_list)) {
        playlist_clear (oxine->playlist);
        if (selected_add_to_playlist (oxine_p)) {
            playlist_play_first (oxine);
        }
    }
}

static void
show_thumbnail_image (const char *thumbnail_mrl)
{
    if (thumbnail_mrl) {
        char *mrl = NULL;

        if (is_downloadable (thumbnail_mrl))
            mrl = download_to_cache (thumbnail_mrl);
        else
            mrl = ho_strdup (thumbnail_mrl);

        if (mrl && (access (mrl, R_OK) == 0))
            odk_osd_draw_image (oxine->odk, mrl, 20, -20, 180, -1);
        else
            odk_osd_hide_image (oxine->odk);

        if (mrl)
            ho_free (mrl);
    }

    else {
        odk_osd_hide_image (oxine->odk);
    }
}


static char *
get_thumbnail_image (const char *directory)
{
    char thumbnail_mrl[1024];

    snprintf (thumbnail_mrl, 1023, "%s/title.jpg", directory);
    if (access (thumbnail_mrl, R_OK) == 0)
        return ho_strdup (thumbnail_mrl);

    snprintf (thumbnail_mrl, 1023, "%s/title.jpeg", directory);
    if (access (thumbnail_mrl, R_OK) == 0)
        return ho_strdup (thumbnail_mrl);

    snprintf (thumbnail_mrl, 1023, "%s/title.png", directory);
    if (access (thumbnail_mrl, R_OK) == 0)
        return ho_strdup (thumbnail_mrl);

    snprintf (thumbnail_mrl, 1023, "%s/title.gif", directory);
    if (access (thumbnail_mrl, R_OK) == 0)
        return ho_strdup (thumbnail_mrl);

    return NULL;
}


/*
 * This callback is called when an item of the filelist gets the focus.
 */
static void
filelist_focus_enter_cb (void *entry_cb_data)
{
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    /* If the fileitem is a directory this would point to the standard thumbnail file. */
    char *thumbnail_mrl = get_thumbnail_image (fileitem->mrl);

    /* If the fileitem has its own thumbnail we show it. */
    if (fileitem->thumbnail_mrl) {
        show_thumbnail_image (fileitem->thumbnail_mrl);
    }

    /* Else we show the thumbnail of the directory this fileitem points to. */
    else if ((fileitem->type != FILE_TYPE_UPLINK)
             && thumbnail_mrl) {
        show_thumbnail_image (thumbnail_mrl);
    }

    /* Last we try to show the thumbnail of the current directory. */
    else {
        show_thumbnail_image (current_thumbnail);
    }

    if (thumbnail_mrl)
        ho_free (thumbnail_mrl);
}


/*
 * This callback is called when an item of the filelist looses the focus.
 */
static void
filelist_focus_leave_cb (void *entry_cb_data)
{
    /* If available we show the thumbnail of the current directory. */
    show_thumbnail_image (current_thumbnail);
}


/*
 * This is the callback for a doubleclick on an item in the list.
 */
static void
filelist_select_cb (void *list_cb_data, void *entry_cb_data)
{
    oxine_t *oxine = (oxine_t *) list_cb_data;
    fileitem_t *fileitem = (fileitem_t *) entry_cb_data;

    if (fileitem->type == FILE_TYPE_MOUNTPOINT) {
        if (fileitem->sublist) {
            filelist_free (fileitem->sublist);
            fileitem->sublist = NULL;
        }
        if (!disc_mount (fileitem->mrl)) {
            show_error (oxine, _("Could not open disc!"), oxine->backto_menu);
            return;
        }
    }

    filelist_expand (oxine->current_filelist, fileitem);

    if (fileitem->sublist) {
        oxine->current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        oxine->current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);

        oxine->current_filelist = fileitem->sublist;

        /* If we enter a directory that contains a 
         * file 'title.jpg' we display this. */
        if (current_thumbnail)
            ho_free (current_thumbnail);
        current_thumbnail = get_thumbnail_image (fileitem->mrl);

        update_filelist_menu_list (oxine);
        otk_draw (oxine->otk);
    }

    if (fileitem->type == FILE_TYPE_REGULAR) {
        playlist_clear (oxine->playlist);
        playlist_add_fileitem (oxine->playlist, fileitem);
        playlist_play_first (oxine);
    }
}

static void
update_filelist_menu_list (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (filelist_menu_list) {
        otk_clear_list (filelist_menu_list);

        // first we add all the entries from the current filelist
        fileitem_t *fileitem = filelist_first (oxine->current_filelist);
        while (fileitem) {
            char *title = fileitem->title;

            otk_widget_t *w = otk_listentry_new (filelist_menu_list, title,
                                                 filelist_select_cb,
                                                 fileitem, NULL, NULL);
            otk_widget_set_focus_callbacks (w, filelist_focus_enter_cb,
                                            fileitem,
                                            filelist_focus_leave_cb,
                                            fileitem);

            fileitem = filelist_next (oxine->current_filelist, fileitem);
        }

        // next we add all removable drives currently available
        if (oxine->current_filelist == oxine->toplevel_filelist) {
            fileitem_t *removable = filelist_first (oxine->removable_discs);
            while (removable) {
                otk_listentry_new (filelist_menu_list, removable->title,
                                   filelist_select_cb, removable, NULL, NULL);

                removable = filelist_next (oxine->removable_discs, removable);
            }
        }

        otk_list_set_pos (filelist_menu_list,
                          oxine->current_filelist->top_position);
        otk_list_set_focus (filelist_menu_list,
                            oxine->current_filelist->cur_position);
    }
}

/**
 * This shows only the GUI
 */
static void
show_music_menu_gui (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    clean_otk_stuff (oxine);
    show_background (oxine, OXINE_DATADIR "/musicmenu.mpg");

    oxine->filelist_menu = show_music_menu_gui;
    oxine->repaint_menu = show_music_menu_gui;
    oxine->playback_ended_menu = show_music_menu_gui;
    oxine->backto_menu = show_main_menu_cb;

    if (filelist_menu_window) {
        otk_window_set_current (oxine->otk, filelist_menu_window);
        oxine->current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        oxine->current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);
        update_filelist_menu_list (oxine);
        otk_draw (oxine->otk);
        return;
    }
    filelist_menu_window = otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    otk_window_keep (filelist_menu_window, 1);

    show_clock (oxine);

    int y = 100;
    int x = 20;
    otk_button_new (oxine->otk, x, y, 180, 35,
                    _("Play"), play_selected_cb, oxine);
    y += 40;
    otk_button_new (oxine->otk, x, y, 180, 35,
                    _("Add"), add_selected_cb, oxine);
    y += 40;
    otk_button_new (oxine->otk, x, y, 180, 35,
                    _("Playlist"), show_playlist_menu_cb, oxine);

#ifdef HAVE_DISC_POLLING
    if (!oxine->use_polling)
#endif
    {
        y += 40;
        otk_button_new (oxine->otk, x, y, 180, 35,
                        _("Audio CD"), play_audiocd_cb, oxine);
    }

    y += 40;
    otk_button_new (oxine->otk, x, y, 180, 35, _("Eject"), eject_cb, oxine);
    y += 40;
    otk_button_new (oxine->otk, x, y, 180, 35,
                    _("Mainmenu"), show_main_menu_cb, oxine);

    filelist_menu_list = otk_list_new (oxine->otk, 220, 100, 560, 480,
                                       OTK_LIST_MULTIPLE_SELECTION, FALSE,
                                       oxine);
    update_filelist_menu_list (oxine);

    otk_draw (oxine->otk);
}


/**
 * This shows only the GUI
 */
static void
show_video_menu_gui (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    clean_otk_stuff (oxine);
    show_background (oxine, OXINE_DATADIR "/videomenu.mpg");

    oxine->filelist_menu = show_video_menu_gui;
    oxine->repaint_menu = show_video_menu_gui;
    oxine->playback_ended_menu = show_video_menu_gui;
    oxine->backto_menu = show_main_menu_cb;

    if (filelist_menu_window) {
        otk_window_set_current (oxine->otk, filelist_menu_window);
        oxine->current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        oxine->current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);
        update_filelist_menu_list (oxine);
        otk_draw (oxine->otk);
        return;
    }
    filelist_menu_window = otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    otk_window_keep (filelist_menu_window, 1);

    show_clock (oxine);

    int y = 100;
    int x = 20;
    otk_button_new (oxine->otk, x, y, 180, 35,
                    _("Play"), play_selected_cb, oxine);

#ifdef HAVE_DISC_POLLING
    if (!oxine->use_polling)
#endif
    {
        y += 40;
        otk_button_new (oxine->otk, x, y, 180, 35, _("DVD"), play_dvd_cb,
                        oxine);
        y += 40;
        otk_button_new (oxine->otk, x, y, 180, 35,
                        _("Video CD"), play_videocd_cb, oxine);
    }

    y += 40;
    otk_button_new (oxine->otk, x, y, 180, 35, _("Eject"), eject_cb, oxine);
    y += 40;
    otk_button_new (oxine->otk, x, y, 180, 35,
                    _("Mainmenu"), show_main_menu_cb, oxine);

    filelist_menu_list =
        otk_list_new (oxine->otk, 220, 100, 560, 480,
                      OTK_LIST_MULTIPLE_SELECTION, FALSE, oxine);
    update_filelist_menu_list (oxine);

    otk_draw (oxine->otk);
}

/**
 * This shows only the GUI
 */
#ifdef HAVE_IMAGE_TOOLS
static void
show_image_menu_gui (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    clean_otk_stuff (oxine);
    show_background (oxine, OXINE_DATADIR "/imagemenu.mpg");

    oxine->filelist_menu = show_image_menu_gui;
    oxine->repaint_menu = show_image_menu_gui;
    oxine->playback_ended_menu = show_image_menu_gui;
    oxine->backto_menu = show_main_menu_cb;

    if (filelist_menu_window) {
        otk_window_set_current (oxine->otk, filelist_menu_window);
        oxine->current_filelist->top_position =
            otk_list_get_pos (filelist_menu_list);
        oxine->current_filelist->cur_position =
            otk_list_get_focus (filelist_menu_list);
        update_filelist_menu_list (oxine);
        otk_draw (oxine->otk);
        return;
    }
    filelist_menu_window = otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    otk_window_keep (filelist_menu_window, 1);

    show_clock (oxine);

    int y = 100;
    int x = 20;
    otk_button_new (oxine->otk, x, y, 180, 35,
                    _("Slideshow"), play_selected_cb, oxine);

#ifdef HAVE_DISC_POLLING
    if (!oxine->use_polling)
#endif
    {
        y += 40;
        otk_button_new (oxine->otk, x, y, 180, 35,
                        _("Photo CD"), play_photocd_cb, oxine);
    }

    y += 40;
    otk_button_new (oxine->otk, x, y, 180, 35, _("Eject"), eject_cb, oxine);
    y += 40;
    otk_button_new (oxine->otk, x, y, 180, 35,
                    _("Mainmenu"), show_main_menu_cb, oxine);

    filelist_menu_list =
        otk_list_new (oxine->otk, 220, 100, 560, 480,
                      OTK_LIST_MULTIPLE_SELECTION, FALSE, oxine);
    update_filelist_menu_list (oxine);

    otk_draw (oxine->otk);
}
#endif

static void
enter_filelist_menu (oxine_t * oxine, char *mediamarks, int allowed_filetypes)
{
    if (oxine->toplevel_filelist)
        filelist_free (oxine->toplevel_filelist);
    oxine->toplevel_filelist = filelist_new (allowed_filetypes, mediamarks);
    oxine->current_filelist = oxine->toplevel_filelist;
    oxine->mostlistened_filelist = NULL;

    if (!mediamarks_read (oxine->toplevel_filelist)) {
        error (_("Could not load mediamarks."));
        info (_("You should create a file containing mediamarks at %s!"),
              mediamarks);
        char tmp[64];
        snprintf (tmp, 63, "[%s]", _("Your Home"));
        filelist_add (oxine->toplevel_filelist, tmp, get_dir_home (),
                      FILE_TYPE_DIRECTORY);
        snprintf (tmp, 63, "[%s]", _("Filesystem"));
        filelist_add (oxine->toplevel_filelist, tmp, "/",
                      FILE_TYPE_DIRECTORY);
    }

    if (current_thumbnail)
        ho_free (current_thumbnail);
    current_thumbnail = NULL;
}

int
filelist_menu_is_current_menu (oxine_t * oxine)
{
    int ret = 0;
    ret |= (oxine->repaint_menu == show_music_menu_gui);
    ret |= (oxine->repaint_menu == show_video_menu_gui);
#ifdef HAVE_IMAGE_TOOLS
    ret |= (oxine->repaint_menu == show_image_menu_gui);
#endif

    return ret;
}

void
filelist_menu_level_up (oxine_t * oxine)
{
    assert (filelist_menu_is_current_menu (oxine));

    if (oxine->current_filelist == oxine->toplevel_filelist)
        return;

    fileitem_t *uplink = filelist_first (oxine->current_filelist);
    oxine->current_filelist = uplink->sublist;

    /* If we enter a directory that contains a 
     * file 'title.jpg' we display this. */
    if (current_thumbnail)
        ho_free (current_thumbnail);
    current_thumbnail = NULL;

    char thumbnail_mrl[1024];
    snprintf (thumbnail_mrl, 1023, "%s/title.jpg", uplink->mrl);
    if (access (thumbnail_mrl, R_OK) == 0)
        current_thumbnail = ho_strdup (thumbnail_mrl);

    update_filelist_menu_list (oxine);
    otk_draw (oxine->otk);
}

void
show_music_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    enter_filelist_menu (oxine, (char *) get_file_mediamarks_music (),
                         ALLOW_FILES_MUSIC);
    mostlistened_new (oxine);

    otk_window_destroy (filelist_menu_window);
    filelist_menu_window = NULL;
    show_music_menu_gui (oxine_p);
}

void
show_video_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    enter_filelist_menu (oxine, (char *) get_file_mediamarks_video (),
                         ALLOW_FILES_VIDEO);

    otk_window_destroy (filelist_menu_window);
    filelist_menu_window = NULL;
    show_video_menu_gui (oxine_p);
}

#ifdef HAVE_IMAGE_TOOLS
void
show_image_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    enter_filelist_menu (oxine, (char *) get_file_mediamarks_image (),
                         ALLOW_FILES_IMAGE);

    otk_window_destroy (filelist_menu_window);
    filelist_menu_window = NULL;
    show_image_menu_gui (oxine_p);
}
#endif
