
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: list.h,v 1.10 2006/01/19 09:31:32 mschwerin Exp $
 *
 */

#ifndef HAVE_LIST_H
#define HAVE_LIST_H

typedef int (*l_swap_cb_t) (void *data1, void *data2);
typedef void (*l_free_cb_t) (void *data);

typedef struct l_list_s l_list_t;
typedef struct l_item_s l_item_t;

struct l_item_s {
    void *data;
    l_item_t *next;
    l_item_t *prev;
};

struct l_list_s {
    l_item_t *first;
    l_item_t *last;

    l_item_t *current;

    int length;
};

/* 
 * ***************************************************************************
 * Name:            l_list_new
 * Access:          public
 *
 * Description:     Creates a new empty list.
 * ***************************************************************************
 */
l_list_t *l_list_new (void);


/* 
 * ***************************************************************************
 * Name:            l_list_free
 * Access:          public
 *
 * Description:     Clears and frees the list. free_data_cb is a callback that
 *                  can be used to free the memory used by data.
 * ***************************************************************************
 */
void l_list_free (l_list_t * list, l_free_cb_t cb);


/* 
 * ***************************************************************************
 * Name:            l_list_clear
 * Access:          public
 *
 * Description:     Removes all entries from the list. free_data_cb is a
 *                  callback that can be used to free the memory used by data.
 * ***************************************************************************
 */
void l_list_clear (l_list_t * list, l_free_cb_t cb);


/* 
 * ***************************************************************************
 * Name:            l_list_sort
 * Access:          public
 *
 * Description:     Sorts the list. swap_cb is a callback function, that is
 *                  used to determin if two entries must be swapped.
 * ***************************************************************************
 */
void l_list_sort (l_list_t * list, l_swap_cb_t swap_cb);


/* 
 * ***************************************************************************
 * Name:            l_list_concat
 * Access:          public
 *
 * Description:     Adds list l2 to the end of list l1 and frees l2.
 * ***************************************************************************
 */
void l_list_concat (l_list_t * l1, l_list_t * l2);


/* 
 * ***************************************************************************
 * Name:            l_list_insert
 * Access:          public
 *
 * Description:     Inserts a new entry with data at the given position.
 * ***************************************************************************
 */
void *l_list_insert (l_list_t * list, int position, void *data);


/* 
 * ***************************************************************************
 * Name:            l_list_prepend
 * Access:          public
 *
 * Description:     Prepends a new entry to the beginning of the list.
 * ***************************************************************************
 */
void *l_list_prepend (l_list_t * list, void *data);


/* 
 * ***************************************************************************
 * Name:            l_list_append
 * Access:          public
 *
 * Description:     Appends a new entry to the end of the list.
 * ***************************************************************************
 */
void *l_list_append (l_list_t * list, void *data);


/* 
 * ***************************************************************************
 * Name:            l_list_remove
 * Access:          public
 *
 * Description:     Removes the entry identified by data from the list.
 * ***************************************************************************
 */
void *l_list_remove (l_list_t * list, void *data);


/* 
 * ***************************************************************************
 * Name:            l_list_first
 * Access:          public
 *
 * Description:     Returns the first entry of the list.
 * ***************************************************************************
 */
void *l_list_first (l_list_t * list);


/* 
 * ***************************************************************************
 * Name:            l_list_last
 * Access:          public
 *
 * Description:     Returns the last entry of the list.
 * ***************************************************************************
 */
void *l_list_last (l_list_t * list);


/* 
 * ***************************************************************************
 * Name:            l_list_next
 * Access:          public
 *
 * Description:     Returns the next entry. The current entry is identified by
 *                  data.
 * ***************************************************************************
 */
void *l_list_next (l_list_t * list, void *data);


/* 
 * ***************************************************************************
 * Name:            l_list_prev
 * Access:          public
 *
 * Description:     Returns the previous entry. The current entry is
 *                  identified by data.
 * ***************************************************************************
 */
void *l_list_prev (l_list_t * list, void *data);


/* 
 * ***************************************************************************
 * Name:            l_list_length
 * Access:          public
 *
 * Description:     Returns the number of entries in the list.
 * ***************************************************************************
 */
int l_list_length (l_list_t * list);


/* 
 * ***************************************************************************
 * Name:            l_list_move_up
 * Access:          public
 *
 * Description:     Moves the entry identified by data one position towards
 *                  the front of the list.
 * ***************************************************************************
 */
void l_list_move_up (l_list_t * list, void *data);


/* 
 * ***************************************************************************
 * Name:            l_list_move_down
 * Access:          public
 *
 * Description:     Moves the entry identified by data one position towards
 *                  the end of the list.
 * ***************************************************************************
 */
void l_list_move_down (l_list_t * list, void *data);
#endif
