
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: main_menu.c,v 1.45 2006/01/18 11:56:19 mschwerin Exp $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "disc.h"
#include "environment.h"
#include "filelist_menu.h"
#include "gui_utils.h"
#include "heap.h"
#include "help_menu.h"
#include "i18n.h"
#include "logger.h"
#include "main_menu.h"
#include "oxine.h"
#include "playing_menu.h"
#include "playlist_menu.h"

#ifdef HAVE_DVB
void
play_dvb_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine->backto_menu = show_main_menu_cb;
    oxine->repaint_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait (oxine);

    int default_channel = 0;
    playitem_t *default_item = NULL;

    /* Look up the last channel. */
    xine_cfg_entry_t lastchannel;
    xine_config_lookup_entry (oxine->xine, "media.dvb.last_channel",
                              &lastchannel);
    default_channel = lastchannel.num_value;

    /* We add all available channels to the playlist. */
    int num_mrls;
    char **channels = xine_get_autoplay_mrls (oxine->xine, "dvb", &num_mrls);
    if (channels) {
        playlist_clear (oxine->playlist);
        int i;
        for (i = 1; i < num_mrls; i++) {
            if (channels[i]) {
                char *title = (channels[i] + 6);
                char mrl[16];
                snprintf (mrl, 15, "dvb://%d", (i - 1));

                playitem_t *item = playlist_add (oxine->playlist, title, mrl);

                if (default_channel == i)
                    default_item = item;
            }
        }
    }

    /* If we found the last channel we play that. */
    if (default_item)
        playlist_play_item (oxine, default_item);
    else
        playlist_play_first (oxine);
}
#endif

#ifdef HAVE_V4L
void
play_v4l_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    oxine->backto_menu = show_main_menu_cb;
    oxine->repaint_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait (oxine);

    playlist_clear (oxine->playlist);
    playlist_add (oxine->playlist, "Analog Television", "v4l://");
    playlist_play_first (oxine);
}
#endif


static void
execute_shutdown_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    // what do we execute on shutdown
    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "misc.shutdown_command", &centry);
    const char *cmd = centry.str_value;

    if (cmd && strlen (cmd) > 0) {
        debug ("executing shutdown command %s", cmd);
        system (cmd);
    }

    odk_exit (oxine->odk);
}

void
shutdown_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (filelist_menu_is_current_menu (oxine))
        oxine->backto_menu = oxine->repaint_menu;

    show_message_ok_cancel (oxine, _("Are you sure you want to shut down?"),
                            execute_shutdown_cb, oxine->backto_menu);

    odk_forward_events_to_xine (oxine->odk, 0);
    oxine->repaint_menu = shutdown_cb;
}

static otk_widget_t *eject_list;

static void
eject_device_button_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    int p = otk_list_get_pos (eject_list);
    int i = 0;
    disc_entry_t *entry = get_first_disc (oxine);
    while (entry) {
        if (i == p) {

            if (odk_current_is_normal_mode (oxine->odk)) {
                char *current_mrl = odk_get_current_mrl (oxine->odk);
                // are we playing from the disc we're about to eject?
                if ((entry->mountpoint
                     && strncmp (current_mrl, entry->mountpoint,
                                 strlen (entry->mountpoint)) == 0)
                    || (odk_current_is_dvd (oxine->odk)
                        && (strstr (current_mrl, entry->device) != NULL))
                    || (odk_current_is_vcd (oxine->odk)
                        && (strstr (current_mrl, entry->device) != NULL))
                    || (odk_current_is_cdda (oxine->odk)
                        && (strstr (current_mrl, entry->device) != NULL))) {

                    // we stop the stream
                    odk_stop (oxine->odk);
                }
            }

            please_wait (oxine);

            if (entry->mountpoint)
                disc_umount (entry->mountpoint);

            if (!disc_eject (entry->device))
                show_error (oxine, _("Could not eject device!"),
                            oxine->backto_menu);
            else {
                assert (oxine->backto_menu);
                oxine->backto_menu (oxine);
            }

            break;
        }
        i += 1;
        entry = get_next_disc (entry);
    }
}

static void
eject_device_list_cb (void *not_used_p, void *oxine_p)
{
    eject_device_button_cb (oxine_p);
}

void
eject_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    if (filelist_menu_is_current_menu (oxine))
        oxine->backto_menu = oxine->repaint_menu;

    clean_otk_stuff (oxine);

    oxine->repaint_menu = eject_cb;
    oxine->user_interface_is_visible = 1;

    otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);

    show_clock (oxine);

    otk_border_new (oxine->otk, 100, 200, 600, 200);

    otk_label_new (oxine->otk, 120, 220, OTK_ALIGN_LEFT | OTK_ALIGN_TOP,
                   _("Select device to eject:"));

    eject_list = otk_selector_new (oxine->otk, 120, 270, 560, NULL, NULL);
    disc_entry_t *entry = get_first_disc (oxine);
    while (entry) {
        otk_widget_t *lentry = otk_listentry_new (eject_list, entry->device,
                                                  eject_device_list_cb,
                                                  oxine, NULL, NULL);
        otk_widget_set_focus (lentry);

        entry = get_next_disc (entry);
    }

    otk_widget_t *button;
    button = otk_button_new (oxine->otk, 230, 340, 160, 40,
                             _("Eject"), eject_device_button_cb, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    button = otk_button_new (oxine->otk, 410, 340, 160, 40,
                             _("Cancel"), oxine->backto_menu, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    otk_draw (oxine->otk);

    odk_forward_events_to_xine (oxine->odk, 0);
}

#define STD_BG OXINE_DATADIR "/background.mpg"

void
show_background (oxine_t * oxine, const char *mrl)
{
    char *current = odk_get_current_mrl (oxine->odk);
    if (!current || (strcmp (mrl, current) != 0)) {
        odk_set_stream_param (oxine->odk, ODK_PARAM_VO_ASPECT_RATIO,
                              ODK_VO_ASPECT_AUTO);
        if (odk_play (oxine->odk, mrl, NULL, ODK_MODE_LOGO)) {
            return;
        }
        if (odk_play (oxine->odk, STD_BG, NULL, ODK_MODE_LOGO)) {
            return;
        }
        fatal ("Could not open or play background, leaving oxine!");
        abort ();
    }
}

void
show_clock (oxine_t * oxine)
{
    otk_widget_t *time = otk_label_new (oxine->otk, 780, 27,
                                        OTK_ALIGN_RIGHT | OTK_ALIGN_TOP,
                                        "");
    set_time_string (oxine, time);
    otk_label_set_upcall (time, set_time_string, oxine);
    otk_widget_set_update (time, 1);
}


void
show_main_menu_cb (void *oxine_p)
{
    oxine_t *oxine = (oxine_t *) oxine_p;

    clean_otk_stuff (oxine);
    show_background (oxine, OXINE_DATADIR "/mainmenu.mpg");

    oxine->backto_menu = show_main_menu_cb;
    oxine->repaint_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    static otk_widget_t *main_menu_window = NULL;
    if (main_menu_window) {
        otk_window_set_current (oxine->otk, main_menu_window);
        otk_draw (oxine->otk);
        return;
    }
    main_menu_window = otk_window_new (oxine->otk, 0, 0, 800, 600, 0, 0);
    otk_window_keep (main_menu_window, 1);

    show_clock (oxine);

    int n = 4;
#ifdef HAVE_IMAGE_TOOLS
    n += 1;
#endif
#ifdef HAVE_DVB
    n += 1;
#endif
#ifdef HAVE_V4L
    n += 1;
#endif
    int spacing = 60;
    int w = 360;
    int h = 45;
    int x = (800 - w) / 2;
    int y = ((520 - ((n - 1) * spacing) - h) / 2) + 80;

    otk_widget_t *b;
    b = otk_button_new (oxine->otk, x, y, w, h,
                        _("Listen to Music"), show_music_menu_cb, oxine);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    otk_widget_set_font (b, "sans", 24);
    otk_widget_set_focus (b);
    y += spacing;

    b = otk_button_new (oxine->otk, x, y, w, h,
                        _("Watch Films"), show_video_menu_cb, oxine);
    otk_widget_set_font (b, "sans", 24);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    y += spacing;

#ifdef HAVE_DVB
    b = otk_button_new (oxine->otk, x, y, w, h,
                        _("Digital Television"), play_dvb_cb, oxine);
    otk_widget_set_font (b, "sans", 24);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    y += spacing;
#endif

#ifdef HAVE_V4L
    b = otk_button_new (oxine->otk, x, y, w, h,
                        _("Analog Television"), play_v4l_cb, oxine);
    otk_widget_set_font (b, "sans", 24);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    y += spacing;
#endif

#ifdef HAVE_IMAGE_TOOLS
    b = otk_button_new (oxine->otk, x, y, w, h,
                        _("View Photos"), show_image_menu_cb, oxine);
    otk_widget_set_font (b, "sans", 24);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    y += spacing;
#endif

    b = otk_button_new (oxine->otk, x, y, w, h,
                        _("Show Help"), show_help_menu_cb, oxine);
    otk_widget_set_font (b, "sans", 24);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);
    y += spacing;

    b = otk_button_new (oxine->otk, x, y, w, h,
                        _("Shutdown"), shutdown_cb, oxine);
    otk_widget_set_font (b, "sans", 24);
    otk_widget_set_alignment (b, OTK_ALIGN_CENTER);

    otk_draw (oxine->otk);
}
