
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: playlist.h,v 1.15 2006/01/19 09:31:32 mschwerin Exp $
 *
 */

#ifndef HAVE_PLAYLIST_H
#define HAVE_PLAYLIST_H

#ifndef FALSE
#define FALSE 0
#endif
#ifndef TRUE
#define TRUE 1
#endif

#include "list.h"
#include "filelist.h"

typedef enum {
    PLAYLIST_MODE_NORMAL,
    PLAYLIST_MODE_REPEAT,
    PLAYLIST_MODE_RANDOM
} playlist_mode_t;

typedef struct playlist_s playlist_t;
typedef struct playitem_s playitem_t;

struct playitem_s {
    char *title;
    char *mrl;
    char *subtitle_mrl;
    int played;
};

struct playlist_s {
    l_list_t *list;
    playitem_t *current;
    playlist_mode_t playmode;
};


/* 
 * ***************************************************************************
 * Name:            playlist_new
 * Access:          public
 *
 * Description:     Creates a new playlist.
 * ***************************************************************************
 */
playlist_t *playlist_new (void);


/* 
 * ***************************************************************************
 * Name:            playlist_free
 * Access:          public
 *
 * Description:     Removes all playitems and frees the list.
 * ***************************************************************************
 */
void playlist_free (playlist_t * playlist);


/* 
 * ***************************************************************************
 * Name:            playlist_clear
 * Access:          public
 *
 * Description:     Removes all playitems.
 * ***************************************************************************
 */
void playlist_clear (playlist_t * playlist);


/* 
 * ***************************************************************************
 * Name:            playlist_insert
 * Access:          public
 *
 * Description:     Adds a new item to the list.
 * ***************************************************************************
 */
playitem_t *playlist_insert (playlist_t * playlist, int position,
                             const char *title, const char *mrl);


/* 
 * ***************************************************************************
 * Name:            playlist_add
 * Access:          public
 *
 * Description:     Adds a new item to the list.
 * ***************************************************************************
 */
playitem_t *playlist_add (playlist_t * playlist,
                          const char *title, const char *mrl);


/* 
 * ***************************************************************************
 * Name:            playlist_add_fileitem
 * Access:          public
 *
 * Description:     Adds a fileitem to the playlist.
 * ***************************************************************************
 */
playitem_t *playlist_add_fileitem (playlist_t * playlist,
                                   fileitem_t * fileitem);


/* 
 * ***************************************************************************
 * Name:            playlist_remove
 * Access:          public
 *
 * Description:     Removes the item from the list.
 * ***************************************************************************
 */
void playlist_remove (playlist_t * playlist, playitem_t * playitem);


/* 
 * ***************************************************************************
 * Name:            playlist_get_first
 * Access:          public
 *
 * Description:     Returns the first entry to play. If we're currently in
 *                  random mode it may not be the first entry in the list.
 * ***************************************************************************
 */
playitem_t *playlist_get_first (playlist_t * playlist);


/* 
 * ***************************************************************************
 * Name:            playlist_get_prev
 * Access:          public
 *
 * Description:     Get previous entry to play.
 * ***************************************************************************
 */
playitem_t *playlist_get_prev (playlist_t * playlist);


/* 
 * ***************************************************************************
 * Name:            playlist_get_next
 * Access:          public
 *
 * Description:     Get next entry to play. In random mode this may not be the
 *                  next entry in the list.
 * ***************************************************************************
 */
playitem_t *playlist_get_next (playlist_t * playlist);


/* 
 * ***************************************************************************
 * Name:            playlist_set_current
 * Access:          public
 *
 * Description:     Sets the current pointer.
 * ***************************************************************************
 */
playitem_t *playlist_set_current (playlist_t * playlist,
                                  playitem_t * current);


/* 
 * ***************************************************************************
 * Name:            playlist_set_current_pos
 * Access:          public
 *
 * Description:     Sets the current pointer.
 * ***************************************************************************
 */
playitem_t *playlist_set_current_pos (playlist_t * playlist, int pos);


/* 
 * ***************************************************************************
 * Name:            playlist_current_pos
 * Access:          public
 *
 * Description:     Returns the position of the currently playing entry.
 * ***************************************************************************
 */
int playlist_current_pos (playlist_t * playlist);


/* 
 * ***************************************************************************
 * Name:            playlist_first
 * Access:          public
 *
 * Description:     Returns the real first entry.
 * ***************************************************************************
 */
playitem_t *playlist_first (playlist_t * playlist);


/* 
 * ***************************************************************************
 * Name:            playlist_next
 * Access:          public
 *
 * Description:     Returns the real next entry.
 * ***************************************************************************
 */
playitem_t *playlist_next (playlist_t * playlist, playitem_t * current);


/* 
 * ***************************************************************************
 * Name:            playlist_move_up
 * Access:          public
 *
 * Description:     Moves the item one position towards the top of list.
 * ***************************************************************************
 */
void playlist_move_up (playlist_t * playlist, playitem_t * playitem);


/* 
 * ***************************************************************************
 * Name:            playlist_move_down
 * Access:          public
 *
 * Description:     Moves the item one position towards the end of list.
 * ***************************************************************************
 */
void playlist_move_down (playlist_t * playlist, playitem_t * playitem);


/* 
 * ***************************************************************************
 * Name:            playlist_length
 * Access:          public
 *
 * Description:     Returns the number of entries in the list.
 * ***************************************************************************
 */
int playlist_length (playlist_t * playlist);
#endif
