
/*
 * Copyright (C) 2002-2003 Stefan Holst
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 * Some of the code in this file was copied from the xine-ui project.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: oxine.c 2661 2007-08-22 15:08:37Z mschwerin $
 *
 */

#include "config.h"

#include <assert.h>
#include <dirent.h>
#include <errno.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <sys/stat.h>
#include <sys/types.h>

#ifdef HAVE_LIBX11
#include <X11/Xlib.h>
#endif

#include "aex.h"
#include "codeset.h"
#include "disc.h"
#include "download.h"
#include "environment.h"
#include "event.h"
#include "extractor.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "meta_info.h"
#include "oxine.h"
#include "playlist_m3u.h"
#include "scheduler.h"
#include "utils.h"
#include "vdr.h"
#include "weather.h"

#include "menu_base.h"
#include "menu_filelist.h"
#include "menu_main.h"
#include "menu_password.h"
#include "menu_playback.h"
#include "menu_playlist.h"
#include "menu_settings.h"
#include "menu_weather.h"

/* Our only global variable. */
oxine_t *oxine;

#define OPTION_VERBOSE      (1000)
#define OPTION_SKINDIR      (1001)

static const char *short_options = "?c:hvFV:A:l:"
#ifdef HAVE_HTTP
    "N"
#endif
#ifdef HAVE_LIRC
    "L"
#endif
#ifdef HAVE_JOYSTICK
    "J"
#endif
#ifdef HAVE_WEATHER
    "W"
#endif
#ifdef HAVE_HAL
    "H"
#endif
#ifdef HAVE_XINERAMA
    "X:"
#endif
#ifdef HAVE_LIBX11
    "B"
    "T"
    "G:"
#endif
    "";

static struct option long_options[] = {
    {"config", required_argument, 0, 'c'},
    {"help", no_argument, 0, 'h'},
    {"version", no_argument, 0, 'v'},
    {"video-driver", required_argument, 0, 'V'},
    {"audio-driver", required_argument, 0, 'A'},
    {"logfile", required_argument, 0, 'l'},
#ifdef HAVE_HTTP
    {"no-http", no_argument, 0, 'N'},
#endif
#ifdef HAVE_LIRC
    {"no-lirc", no_argument, 0, 'L'},
#endif
#ifdef HAVE_JOYSTICK
    {"no-joystick", no_argument, 0, 'J'},
#endif
#ifdef HAVE_WEATHER
    {"no-weather", no_argument, 0, 'W'},
#endif
#ifdef HAVE_HAL
    {"no-hal-dbus", no_argument, 0, 'H'},
#endif
#ifdef HAVE_XINERAMA
    {"xineramascreen", required_argument, 0, 'X'},
#endif
#ifdef HAVE_LIBX11
    {"geometry", required_argument, 0, 'G'},
    {"stay-on-top", no_argument, 0, 'T'},
    {"no-border", no_argument, 0, 'B'},
    {"no-fullscreen", no_argument, 0, 'F'},
#endif
    {"verbose", optional_argument, 0, OPTION_VERBOSE},
    {"skindir", no_argument, 0, OPTION_SKINDIR},
    {0, no_argument, 0, 0}
};


static void
show_usage (void)
{
    xine_t *xine = xine_new ();
    xine_init (xine);

    printf ("\n");
    printf (_("Usage: oxine [OPTIONS] [MRL]\n"));
    printf ("\n");
    printf (_("OPTIONS are:\n"));
    printf (_("  -c, --config <filename>      Use non-standard "
              "configuration file.\n"));
    printf (_("  -v, --version                Display version.\n"));
    printf (_("      --verbose=<level>        "
              "Set verbosity level. Default is 1.\n"));
    printf (_("  -l, --logfile <filename>     Set the log filename.\n"));
    printf (_("  -V, --video-driver <drv>     Select video driver by ID.\n"));
    printf (_("                               Available drivers:\n"));
    printf ("                               auto ");

    const char *const *driver_ids;
    if ((driver_ids = xine_list_video_output_plugins (xine))) {
        const char *driver_id = *driver_ids++;
        while (driver_id) {
            if (odk_is_supported_video_driver (driver_id))
                printf ("%s ", driver_id);
            driver_id = *driver_ids++;
        }
    }
    printf ("\n");
    printf (_("  -A, --audio-driver <drv>     Select audio driver by ID.\n"));
    printf (_("                               Available drivers:\n"));
    printf ("                               ");
    if ((driver_ids = xine_list_audio_output_plugins (xine))) {
        const char *driver_id = *driver_ids++;
        while (driver_id) {
            printf ("%s ", driver_id);
            driver_id = *driver_ids++;
        }
    }
    printf ("\n");
#ifdef HAVE_XINERAMA
    printf (_("  -X, --xineramascreen=<num>   "
              "Sets the screen on which the fullscreen window\n"
              "                               will be shown.\n"));
#endif
#ifdef HAVE_LIBX11
    printf (_("  -G, --geometry <WxH[+X+Y]>   "
              "Sets output window geometry.\n"));
    printf (_("  -T, --stay-on-top            "
              "Sets the output window to stay on top.\n"));
    printf (_("  -B, --no-border              "
              "Do not show window border.\n"));
    printf (_("  -F, --no-fullscreen          "
              "Do no start in fullscreen mode.\n"));
#endif
#ifdef HAVE_HTTP
    printf (_("  -N, --no-http                "
              "Do not use remote control (HTTP).\n"));
#endif
#ifdef HAVE_LIRC
    printf (_("  -L, --no-lirc                "
              "Do not use remote control (LIRC).\n"));
#endif
#ifdef HAVE_JOYSTICK
    printf (_("  -J, --no-joystick            "
              "Do not use joystick support.\n"));
#endif
#ifdef HAVE_WEATHER
    printf (_("  -W, --no-weather             "
              "Do not automaticaly retrieve weather information.\n"));
#endif
#ifdef HAVE_HAL
    printf (_("  -H, --no-hal-dbus            "
              "Do not use HAL/ DBUS to monitor removable drives.\n"));
#endif
    printf ("\n");
    printf (_("Examples for valid MRLs (media resource locator):\n"
              "  File:  'path/foo.vob'\n"
              "         '/path/foo.vob'\n"
              "         'file://path/foo.vob'\n"
              "         'fifo://[[mpeg1|mpeg2]:/]path/foo'\n"
              "         'stdin://[mpeg1|mpeg2]'\n"
              "  DVD:   'dvd://VTS_01_2.VOB'\n"
              "         'dvd://path/foo.iso'\n"
              "         'dvd://<track number>'\n"
              "  VCD:   'vcd://<track number>'\n"));
    printf ("\n");

    xine_exit (xine);
}


static inline void
oxine_register_config (void)
{
    /* Drivers */
    const char *const *vid;
    if ((vid = xine_list_video_output_plugins (oxine->xine))) {
        int i = 0;
        for (i = 0; vid[i]; i++);

        char **drivers = (char **) ho_malloc (sizeof (char **) * (i + 2));
        drivers[0] = ho_strdup ("auto");
        for (i = 0; vid[i]; i++) {
            drivers[i + 1] = ho_strdup (vid[i]);
        }
        drivers[i + 1] = NULL;

        config_register_enum ("video.driver", 0, drivers,
                              _("video driver to use"));

        for (i = 0; drivers[i]; i++) {
            ho_free (drivers[i]);
        }
        ho_free (drivers);
    }

    const char *const *aud;
    if ((aud = xine_list_audio_output_plugins (oxine->xine))) {
        int i = 0;
        for (i = 0; aud[i]; i++);

        char **drivers = (char **) ho_malloc (sizeof (char **) * (i + 1));
        for (i = 0; aud[i]; i++) {
            drivers[i] = ho_strdup (aud[i]);
        }
        drivers[i] = NULL;

        config_register_enum ("audio.driver", 0, drivers,
                              _("audio driver to use"));

        for (i = 0; drivers[i]; i++) {
            ho_free (drivers[i]);
        }
        ho_free (drivers);
    }

    /* Search for available skins. */
    {
        DIR *dirp = opendir (OXINE_SKINDIR);
        if (dirp == NULL) {
            fatal (_("Unable to list contents of directory '%s': %s!"),
                   OXINE_SKINDIR, strerror (errno));
            abort ();
        }

        int d = 0;
        int s = 0;
        char **skins = NULL;
        struct dirent *entp;
        while ((entp = readdir (dirp))) {
            if ((strcmp (entp->d_name, ".") == 0)
                || (strcmp (entp->d_name, "..") == 0)) {
                /* Do nothing */
            }
            else {
                char *main_png = ho_strdup_printf ("%s/%s/%s/menu_main.png",
                                                   OXINE_SKINDIR,
                                                   entp->d_name,
                                                   "backgrounds");
                char *skin_xml = ho_strdup_printf ("%s/%s/skin.xml",
                                                   OXINE_SKINDIR,
                                                   entp->d_name);
                if (file_exists (skin_xml) && file_exists (main_png)) {
                    if (strcmp (entp->d_name, "default") == 0) {
                        d = s;
                    }
                    skins = (char **) ho_realloc (skins,
                                                  (s + 2) * sizeof (char **));
                    skins[s++] = ho_strdup (entp->d_name);
                    skins[s] = NULL;
                }
                ho_free (skin_xml);
                ho_free (main_png);
            }
        }
        closedir (dirp);

        config_register_enum ("gui.skin", d, skins, _("Skin to use"));

        for (; s >= 0; s--) {
            ho_free (skins[s]);
        }
        ho_free (skins);
    }

    /* Apple Airport Express */
#ifdef HAVE_AEX
    config_register_bool ("audio.driver.use_aex", false,
                          _("use Apple Airport Express instead "
                            "of selected audio driver"));
#endif

    /* Mediamarks */
    config_register_bool ("misc.ask_missing_mediamarks", true,
                          _("ask if there is no mediamarks file."));

    /* GUI */
    config_register_number ("gui.x11.xinerama_screen", 0,
                            _("Xinerama screen for fullscreen window"));

    config_register_number ("gui.x11.window_width", 800,
                            _("non-fullscreen window width"));

    config_register_number ("gui.x11.window_height", 600,
                            _("non-fullscreen window height"));

    config_register_number ("gui.x11.monitor_pixel_aspect", 0,
                            _("aspect ratio of one monitor pixel"));

    config_register_string ("gui.clock_format", _("%a, %d. %b, %H:%M"),
                            _("format for clock (see 'man strftime')"));

    config_register_string ("gui.date_format", _("%a, %d. %B %Y"),
                            _("format for date (see 'man of strftime')"));

    config_register_string ("gui.time_format", _("%H:%M"),
                            _("format for time (see 'man of strftime')"));

    /* Play-all metatrack */
    config_register_bool ("misc.play_all_metatrack.use", true,
                          _("show \"Play all titles in this folder...\" "
                            "metatrack"));

    config_register_bool ("misc.list_select_ctrl_shift", true,
                          _("use Ctrl/ Shift for list selection"));

    /* Eject */
    config_register_bool ("misc.eject.one_device.confirm", true,
                          _("show eject confirmation dialog "
                            "for only one device"));

    /* Shutdown */
    config_register_bool ("shutdown.ask", true,
                          _("ask before leaving oxine"));

    config_register_number ("shutdown.last", 0,
                            _("command that was selected the last time"));

    config_register_string ("shutdown.shutdown_command", "/sbin/poweroff",
                            _("shell command (power off)"));

    config_register_string ("shutdown.reboot_command", "/sbin/reboot",
                            _("shell command (reboot)"));

    config_register_string ("shutdown.standby_command", "",
                            _("shell command (standby)"));

    /* Favorites */
    config_register_bool ("misc.favorites.enable", true,
                          _("enable tracking of often listened to albums "
                            "(favorites)"));

    /* Weather */
#ifdef HAVE_WEATHER
    config_register_string ("weather.metar_url",
                            "ftp://tgftp.nws.noaa.gov/data/"
                            "observations/metar/stations",
                            _("URL of METAR information"));
    config_register_string ("weather.station_id", "EDDI",
                            _("METAR ICAO location indicator"));
#endif

    /* Shoutcast */
#ifdef HAVE_SHOUTCAST
    config_register_bool ("streaming.shoutcast.tv.enable", true,
                          _("enable SHOUTcast television"));
    config_register_bool ("streaming.shoutcast.radio.enable", true,
                          _("enable SHOUTcast radio"));

    config_register_string ("streaming.shoutcast.tv.location",
                            "http://www.shoutcast.com/sbin/"
                            "newtvlister.phtml?alltv=1",
                            _("SHOUTcast television stationlist URL"));
    config_register_string ("streaming.shoutcast.radio.location",
                            "http://www.shoutcast.com/sbin/"
                            "newxml.phtml?genre=Top500",
                            _("SHOUTcast radio stationlist URL"));

    char *ratings[] = {
        "G (general audiences)",
        "PG (parental guidance)",
        "PG-13 (parents cautioned)",
        "R (restricted)",
        "NC-17 (must be 18)",
        NULL
    };
    config_register_enum ("streaming.shoutcast.allowed_rating", 0, ratings,
                          _("MPAA movie rating to allow"));
#endif

#ifdef HAVE_YOUTUBE
    config_register_bool ("streaming.youtube.enable", true,
                          _("enable Youtube"));
    config_register_bool ("streaming.youtube.download_to_cache", false,
                          _("download YouTube videos before playing"));
#endif

#ifdef HAVE_TVLINKS
    config_register_bool ("streaming.tvlinks.enable", true,
                          _("enable TV Links"));
    config_register_bool ("streaming.tvlinks.download_to_cache", false,
                          _("download TV Links videos before playing"));
#endif

    /* Playlist */
    config_register_bool ("playlist.auto_reload", true,
                          _("reload playlist on start"));

    config_register_bool ("playlist.auto_play", false,
                          _("play reloaded playlist on start"));

    config_register_bool ("playlist.metafolder.show", true,
                          _("show \"Playlists\" metafolder"));

    /* Subtitles */
    config_register_bool ("subtitles.autoload", true,
                          _("subtitle autoloading"));

    /* Television */
    char *tv_types[] = {
        "analogue television",
        "digital television",
#ifdef HAVE_VDR
        "video disc recorder",
#endif
        NULL
    };
    config_register_enum ("television.type", 1, tv_types,
                          _("type of television"));
    config_register_bool ("television.enable", true,
                          _("activate television"));
#ifdef HAVE_VDR
    config_register_bool ("television.recordings_show", true,
                          _("show television recordings"));
    config_register_bool ("television.timers_show", true,
                          _("show television timers"));
    config_register_number ("television.vdr.svdrp.port", 2001,
                            _("port used for connecting to VDR"));
    config_register_string ("television.vdr.svdrp.host", "localhost",
                            _("host used for connecting to VDR"));
#endif

#ifdef HAVE_HAL
    /* Removable media */
    char *policies[] = {
        "do nothing",
        "replace playlist",
        "append to playlist",
        NULL
    };

    config_register_enum ("hal.auto.policy", 0, policies,
                          _("if playback is running"));

    char *actions[] = {
        "do nothing",
        "play disc",
#ifdef HAVE_EXTRACTOR
        "extract disc",
#endif
        NULL
    };

    config_register_enum ("hal.auto.action.cdda", 1, actions,
                          _("On insertion of an audio CD"));
    config_register_enum ("hal.auto.action.dvd", 1, actions,
                          _("On insertion of a video DVD"));
    config_register_enum ("hal.auto.action.vcd", 1, actions,
                          _("On insertion of a video CD"));
    actions[1] = "play files on disc";
    actions[2] = NULL;
    config_register_enum ("hal.auto.action.volume.mountable", 1, actions,
                          _("On insertion of a CDROM"));
#endif /* HAVE_HAL */

    /* We want to make sure that all devices defined by the different xine-lib
     * input plugins are set to something sensible. */
    xine_cfg_entry_t config;
    config_lookup_entry ("media.dvd.device", &config);
    if (!config.str_value || (strlen (config.str_value) == 0)) {
        config.str_value = "/dev/dvd";
        xine_config_update_entry (oxine->xine, &config);
    }

    config_lookup_entry ("media.audio_cd.device", &config);
    if (!config.str_value || (strlen (config.str_value) == 0)) {
        config.str_value = "/dev/cdrom";
        xine_config_update_entry (oxine->xine, &config);
    }

    config_lookup_entry ("media.vcd.device", &config);
    if (!config.str_value || (strlen (config.str_value) == 0)) {
        config.str_value = "/dev/cdrom";
        xine_config_update_entry (oxine->xine, &config);
    }
}


int
main (int argc, char **argv)
{
    xine_cfg_entry_t config;

    bool restart = false;

    int c;
    int option_index;

    int geom_x = 0;
    int geom_y = 0;
    int geom_w = 0;
    int geom_h = 0;

    char *video_driver_id = NULL;
    char *audio_driver_id = NULL;

    char *log_filename = NULL;
    int verbosity = XINE_VERBOSITY_NONE;
    int xinerama_screen = -1;

    bool fullscreen = true;
    bool show_border = true;
    bool stay_on_top = false;

#ifdef HAVE_HTTP
    bool use_http = true;
#endif
#ifdef HAVE_LIRC
    bool use_lirc = true;
#endif
#ifdef HAVE_JOYSTICK
    bool use_joystick = true;
#endif
#ifdef HAVE_WEATHER
    bool use_weather = true;
#endif
#ifdef HAVE_HAL
    bool use_hal_monitor = true;
#endif
#ifdef HAVE_AEX
    bool use_aex_monitor = true;
#endif

    /* Set up internationalization. */
#ifdef HAVE_SETLOCALE
    if (!setlocale (LC_ALL, "")) {
        error ("The current locale is not supported by the C library!");
    }
#endif
#ifdef ENABLE_NLS
    bindtextdomain (PACKAGE, OXINE_LOCALEDIR);
    textdomain (PACKAGE);
#endif

    /* Parse command line arguments. */
    while ((c = getopt_long (argc, argv, short_options,
                             long_options, &option_index)) != EOF) {
        switch (c) {
        case OPTION_SKINDIR:
            printf ("%s\n", OXINE_SKINDIR);
            exit (0);
            break;
        case OPTION_VERBOSE:
            if (optarg != NULL) {
                verbosity = strtol (optarg, &optarg, 10);
            }
            else {
                verbosity = 1;
            }
            break;
        case 'c':
            set_file_config (optarg);
            break;
#ifdef HAVE_XINERAMA
        case 'X':
            xinerama_screen = strtol (optarg, &optarg, 10);
            break;
#endif
#ifdef HAVE_LIBX11
        case 'G':
            {
                XParseGeometry (optarg, &geom_x, &geom_y,
                                ((unsigned int *) &geom_w),
                                ((unsigned int *) &geom_h));
            }
            break;
        case 'B':
            show_border = false;
            break;
        case 'T':
            stay_on_top = true;
            break;
        case 'F':
            fullscreen = false;
            break;
#endif
        case 'A':
            audio_driver_id = ho_strdup (optarg);
            break;
        case 'V':
            video_driver_id = ho_strdup (optarg);
            break;
        case 'l':
            log_filename = ho_strdup (optarg);
            break;
#ifdef HAVE_HTTP
        case 'N':
            use_http = false;
            break;
#endif
#ifdef HAVE_LIRC
        case 'L':
            use_lirc = false;
            break;
#endif
#ifdef HAVE_JOYSTICK
        case 'J':
            use_joystick = false;
            break;
#endif
#ifdef HAVE_WEATHER
        case 'W':
            use_weather = false;
            break;
#endif
#ifdef HAVE_HAL
        case 'H':
            use_hal_monitor = false;
            break;
#endif
        case 'v':
            printf (_("This is %s - a free video-player - version %s.\n"),
                    PACKAGE_NAME, PACKAGE_VERSION);
            printf (_("Copyright (c) 2005 by the oxine project.\n"));
            exit (1);
            break;
        case 'h':
        case '?':
            show_usage ();
            exit (0);
            break;
        default:
            error (_("Invalid argument %d!"));
            show_usage ();
            exit (1);
            break;
        }
    }

    logger_init (log_filename);
    ho_free (log_filename);

    /* Print some information about the current system. */
    char *codeset = get_system_encoding ();
    info (_("    oxine version: %s"), PACKAGE_VERSION);
    info (_(" xine-lib version: %s"), xine_get_version_string ());
    info (_("         language: %s"), getenv ("LANG"));
    info (_("          codeset: %s"), codeset);
    ho_free (codeset);
#ifdef DEBUG
    debug ("   home directory: %s", get_dir_home ());
    debug (" locale directory: %s", OXINE_LOCALEDIR);
    debug ("   data directory: %s", OXINE_DATADIR);
    debug ("visuals directory: %s", OXINE_VISUALDIR);
    debug ("   skin directory: %s", OXINE_SKINDIR);
#endif

    /* Check for files we cannot live without. */
    if (!file_exists (OXINE_DATADIR "/mainmenu.xml")) {
        error (_("Could not find menu definition for main menu."));
        fatal (_("Please make sure that you installed oxine correctly!"));
        abort ();
    }

    /* Create our central data object. */
    oxine = (oxine_t *) ho_new (oxine_t);
    oxine->xine = xine_new ();
    oxine->playback_menu_job = 0;
    oxine->restart = false;

#ifdef HAVE_HAL
    filelist_ref_set (&oxine->hal_volume_list,
                      filelist_new (NULL, NULL, NULL,
                                    ALLOW_FILES_MULTIMEDIA));
#endif

#ifdef HAVE_WEATHER
    oxine->weather = NULL;
    pthread_mutex_init (&oxine->weather_mutex, NULL);
    weather_locations_init ();
#endif

    oxine->user_interface_is_visible = false;
    oxine->playback_controls_are_visible = false;
    oxine->stream_parameter_is_visible = false;
    oxine->rw_playlist = playlist_new (get_file_playlist_rw (),
                                       playlist_rw_change_cb);
    oxine->ro_playlist = playlist_new (NULL, NULL);
    oxine->current_playlist = oxine->ro_playlist;
    oxine->current_dialog_data.background_mrl = NULL;
    oxine->current_dialog_data.msg = NULL;
    oxine->shutdown_command = NULL;

    /* Load and init the xine configuration. */
    debug ("reading configuration from file %s", get_file_config ());
    xine_config_load (oxine->xine, get_file_config ());

    /* Initialize the xine engine. */
    xine_engine_set_param (oxine->xine, XINE_ENGINE_PARAM_VERBOSITY,
                           verbosity);
    xine_init (oxine->xine);

    /* Register our xine config items. */
    oxine_register_config ();

    /* Initialize our downloader. */
    downloader_init ();

    /* Initialize meta info. */
    meta_info_init (oxine->xine);
    playlist_set_length_cb (meta_info_get_playback_length);

    /* Initialize the filelist extensions. */
    filelist_extensions_init ();

    /* Register the password callback. */
    filelist_set_pwd_callback (check_password_for_mrl);

#ifdef HAVE_EXTRACTOR
    /* Initialize the extractor. */
    extractor_init ();
#endif

    /* Screen to use when in xinerama fullscreen mode. */
    if (xinerama_screen != -1) {
        config_lookup_entry ("gui.x11.xinerama_screen", &config);
        config.num_value = xinerama_screen;
        xine_config_update_entry (oxine->xine, &config);
    }

    /* Video driver to use. */
    if (video_driver_id) {
        int i = 0;
        config_lookup_entry ("video.driver", &config);
        for (i = 0; config.enum_values[i]; i++) {
            if (strcasecmp (config.enum_values[i], video_driver_id) == 0) {
                config.num_value = i;
                xine_config_update_entry (oxine->xine, &config);
                break;
            }
        }
        ho_free (video_driver_id);
    }

    /* Audio driver to use. */
    if (audio_driver_id) {
        int i = 0;
        config_lookup_entry ("audio.driver", &config);
        for (i = 0; config.enum_values[i]; i++) {
            if (strcasecmp (config.enum_values[i], audio_driver_id) == 0) {
                config.num_value = i;
                xine_config_update_entry (oxine->xine, &config);
                break;
            }
        }
        ho_free (audio_driver_id);
    }

    /* Start the scheduler. */
    start_scheduler ();

    /* Initialize ODK. */
    if (!(oxine->odk = odk_init (oxine->xine))) {
        fatal (_("Could not initialize xine-lib wrapper!"));
        abort ();
    }

    /* Register our main event handler with ODK. */
    odk_add_event_handler (oxine->odk, oxine_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_LOW);

    /* Initialize OTK (widget toolkit). */
    char *skin_xml = ho_strdup_printf ("%s/skin.xml", get_dir_oxine_skin ());
    if (!(oxine->otk = otk_init (oxine->odk, skin_xml))) {
        fatal (_("Could not inizialize graphics toolkit!"));
        abort ();
    }
    ho_free (skin_xml);

#ifdef HAVE_HTTP
    /* Start HTTP input plugin. */
    if (use_http) {
        odk_listen_to (oxine->odk, "http");
    }
#endif
#ifdef HAVE_LIRC
    /* Start LIRC input plugin. */
    if (use_lirc) {
        odk_listen_to (oxine->odk, "lirc");
    }
#endif
#ifdef HAVE_JOYSTICK
    /* Start joystick input plugin. */
    if (use_joystick) {
        odk_listen_to (oxine->odk, "joystick");
    }
#endif

    /* Add anything that was not recognized to the playlist. */
    int i = optind;
    filelist_t *arglist = NULL;
    filelist_ref_set (&arglist, filelist_new (NULL, NULL, "/",
                                              ALLOW_FILES_MULTIMEDIA));
    for (; i < argc; i++) {
        char *mrl = argv[i];
        char *title = create_title (mrl);
        playlist_add_fileitem (oxine->ro_playlist,
                               filelist_add (arglist, title, mrl,
                                             FILE_TYPE_UNKNOWN));
        ho_free (title);
    }
    filelist_ref_set (&arglist, NULL);

    /* If the user wants this, we reload the last playlist. */
    if (config_get_bool ("playlist.auto_reload")) {
        playlist_load (oxine->rw_playlist, NULL);
    }

    /* Decide what to play (if anything). If there is anything in the general
     * playlist (the user added files via the command line) this is played. */
    playitem_t *item2play = NULL;
    if (playlist_length (oxine->ro_playlist) > 0) {
        oxine->current_playlist = oxine->ro_playlist;
        item2play = playlist_get_first (oxine->current_playlist);
    }

    /* If the general playlist is empty we have a look at the music and video
     * lists to see if there is something to be played in them. */
    else if (config_get_bool ("playlist.auto_play")) {
        oxine->current_playlist = oxine->rw_playlist;
        item2play = playlist_get_current (oxine->current_playlist);
    }

    /* Load main menu. */
    init_menu_main ();

    /* Show main window. */
    if (geom_w == 0) {
        geom_w = config_get_number ("gui.x11.window_width");
    }
    if (geom_h == 0) {
        geom_h = config_get_number ("gui.x11.window_height");
    }
    odk_show_window (oxine->odk, fullscreen,
                     show_border, stay_on_top,
                     geom_x, geom_y, geom_w, geom_h);
    set_backto_menu (hide_user_interface, NULL);
    set_playback_ended_menu (show_menu_main, NULL);

    /* Start the weather thread. */
#ifdef HAVE_WEATHER
    if (use_weather) {
        start_weather_thread ();
    }
#endif

    /* Start the volume monitor. */
#ifdef HAVE_HAL
    if (use_hal_monitor) {
        hal_monitor_start ();
    }
#endif

    /* Start the AEX monitor. */
#ifdef HAVE_AEX
    if (use_aex_monitor) {
        aex_monitor_start ();
    }
#endif

    /* If there is an item to start playing at once, we play that. Otherwise
     * we show the main menu. */
    if (item2play) {
        playlist_play_item (oxine->current_playlist, item2play);
    }
    else {
        oxine_event_t ev;
        ev.type = OXINE_EVENT_KEY;
        ev.source.key = OXINE_KEY_MENU_MAIN;
        odk_oxine_event_send (oxine->odk, &ev);
    }

    /* Start the event loop. */
    odk_run (oxine->odk);

    /* Stop the scheduler. */
    stop_scheduler ();

    /* Free the extractor. */
#ifdef HAVE_EXTRACTOR
    extractor_free ();
#endif

    /* Stop the weather thread. */
#ifdef HAVE_WEATHER
    stop_weather_thread ();
#endif

    if (oxine->otk) {
        otk_free (oxine->otk);
    }
    if (oxine->odk) {
        odk_free (oxine->odk);
    }

    /* Free the HAL monitor. */
#ifdef HAVE_HAL
    if (use_hal_monitor) {
        hal_monitor_free ();
    }

    /* Free the list containing the removable discs. */
    filelist_ref_set (&oxine->hal_volume_list, NULL);
#endif

    /* Free the AEX monitor. */
#ifdef HAVE_AEX
    if (use_aex_monitor) {
        aex_monitor_free ();
    }
#endif

    /* Free the weather data. */
#ifdef HAVE_WEATHER
    weather_free (oxine->weather);
    pthread_mutex_destroy (&oxine->weather_mutex);
    weather_locations_free ();
#endif

    /* Free the menu items of the main menu. */
    free_menu_main ();

    /* Free the playlists. */
    playlist_free (oxine->rw_playlist);
    playlist_free (oxine->ro_playlist);

    /* Free the dialog data. */
    ho_free (oxine->current_dialog_data.background_mrl);
    ho_free (oxine->current_dialog_data.msg);

    /* Free the meta info stuff. */
    meta_info_free ();

    /* Free our downloader. */
    downloader_free ();

    /* Save xine config and free xine. */
    xine_config_save (oxine->xine, get_file_config ());
    xine_exit (oxine->xine);

    /* We copy the shutdown command so we can free oxine. */
    char shutdown_command[1024];
    if (oxine->shutdown_command) {
        strncpy (shutdown_command, oxine->shutdown_command, 1024);
    }
    else {
        shutdown_command[0] = '\0';
    }

    /* We copy the restart value so we can free oxine. */
    restart = oxine->restart;

    /* Free oxine. */
    ho_free (oxine->shutdown_command);
    ho_free (oxine);

    filelist_extensions_free ();

    free_menu_filelist ();
    free_menu_settings ();
    free_menu_playlist ();
    free_menu_password ();

    environment_free ();

    logger_free ();

    heapstat ();

    /* If the user selected a shutdown command we run it now. */
    if (strlen (shutdown_command)) {
        info (_("Running command: '%s'"), shutdown_command);
        while (!execute_shell (shutdown_command, 0));
    }

    /* If the user chose to restart oxine we do that now. */
    if (restart) {
        info (_("Restarting oxine..."));
        if (execv (argv[0], argv) == -1) {
            error (_("Restarting oxine failed: %s!"), strerror (errno));
        }
    }

    return 0;
}
