#ifndef WIN32
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <netdb.h>
#include <unistd.h>
#else
#include <winsock2.h>
#include <io.h>
#endif
#include <errno.h>
#include <stdio.h>
#include <errno.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>

#if defined(sun)
typedef unsigned int socklen_t;
#elif defined(__APPLE__)
typedef int socklen_t;
#endif

#include "PapayaList.h"
#include "Prefs.h"
#include "libusers.h"
#include "Message.h"

#include "mudclient.h"

// Define this as 1 if you use a word based host address below.  I strongly
// recommend using an IP address as this doesn't halt the client while it's
// doing an address lookup
#define RESOLVE 0
// The server to send the "hello" and "bye" packets to.  See notes on RESOLVE
// above before setting.
#define SERVER "203.79.110.222"
// The port the server is running on.  For papaya this is port 5000.
#define PORT 5000
// How long do we wait for a response from the server before closing the socket
#define TIMEOUT_PERIOD 10

extern Prefs * globalPreferences;

/**
 * This simply sends a UDP packet to a user-count server running on
 * www.gtk-papaya.org.  The packet contains either a 1 or a 0.  0 means
 * the client has started and 1 means it has finished.
 *
 * The contents of the UDP packet looks like:
 *   <command: get_number_users, get_motd>
 *   <client major version number> <client minor version number>
 *
 * No other data is sent.  It is possible to disable the sending of the
 * major and minor version numbers in the client preferences.
 *
 * In return, the server informs Papaya of the latest version and the
 * estimated number of current users.
 */

void getCurrentVersion(int * major, int * minor) {
  char * version = strdup(VERSION);
  char * pc;

  pc = strchr(version, '.');
  if (pc) {
    *pc = '\0';
    *minor = atoi(pc + 1);
  } else {
    *minor = 0;
  }

  *major = atoi(version);
  free(version);
}

int create_socket() {
  int fd;

  fd = socket(PF_INET, SOCK_DGRAM, 0);
  if (fd == -1) {
    perror("socket");
    return 0;
  }

#ifdef WIN32  
  unsigned long enable = 1;
  int res = ioctlsocket(fd, FIONBIO, &enable);
#else
  int res = fcntl(fd, F_SETFL, O_NONBLOCK);
#endif
  if (res == -1) {
    perror("fcntl");
    close(fd);
    return 0;
  }

  return fd;
}

int disconnect_from_server(int fd) {
  struct sockaddr_in addr;
  struct in_addr inet_address;
  struct client_command cmd;

#if RESOLVE
  struct hostent * hp;

  if (isalpha(*host)) {
    hp = gethostbyname(host);
    if (!hp) {
      printf(_("Unknown host.\n"));
      return 0;
    }

    memcpy((char *)&inet_address, hp->h_addr, sizeof(struct in_addr));
  }
  else
    inet_address.s_addr = inet_addr(host);
#else
  inet_address.s_addr = inet_addr(SERVER);
#endif // RESOLVE

  memset(&addr, 0, sizeof(struct sockaddr_in));
  addr.sin_port = htons(PORT);
  addr.sin_addr = inet_address;
  addr.sin_family = AF_INET;

  cmd.command = htons(1);

  int major, minor;
  getCurrentVersion(&major, &minor);

  cmd.v_major = htons((unsigned short)major);
  cmd.v_minor = htons((unsigned short)minor);

  if (globalPreferences->getSendVersion())
    if (sendto(fd, (const char *)&cmd, sizeof(struct client_command), 0, (struct sockaddr *)&addr, sizeof(struct sockaddr_in)) == -1) {
      perror("sendto");
      return 0;
    }
  else
    if (sendto(fd, (const char *)&(cmd.command), sizeof(unsigned short), 0, (struct sockaddr *)&addr, sizeof(struct sockaddr_in)) == -1) {
      perror("sendto");
      return 0;
    }

  return 1;
}

int connect_to_server(int fd) {
  struct sockaddr_in addr;
  struct in_addr inet_address;
  struct client_command cmd;

#if RESOLVE
  struct hostent * hp;

  if (isalpha(*host)) {
    hp = gethostbyname(host);
    if (!hp) {
      printf(_("Unknown host.\n"));
      return 0;
    }

    memcpy((char *)&inet_address, hp->h_addr, sizeof(struct in_addr));
  }
  else
    inet_address.s_addr = inet_addr(host);
#else
  inet_address.s_addr = inet_addr(SERVER);
#endif // RESOLVE

  memset(&addr, 0, sizeof(struct sockaddr_in));
  addr.sin_port = htons(PORT);
  addr.sin_addr = inet_address;
  addr.sin_family = AF_INET;

  cmd.command = htons(0);
  int major, minor;
  getCurrentVersion(&major, &minor);

  cmd.v_major = htons((unsigned short)major);
  cmd.v_minor = htons((unsigned short)minor);

  if (!globalPreferences || globalPreferences->getSendVersion())
    if (sendto(fd, (const char *)&cmd, sizeof(struct client_command), 0, (struct sockaddr *)&addr, sizeof(struct sockaddr_in)) == -1) {
      perror("sendto");
      return 0;
    }
  else
    if (sendto(fd, (const char *)&(cmd.command), sizeof(unsigned short), 0, (struct sockaddr *)&addr, sizeof(struct sockaddr_in)) == -1) {
      perror("sendto");
      return 0;
    }

  return 1;
}

int receive_message(int fd) {
  struct sockaddr_in addr;
  int addr_len = sizeof(struct sockaddr_in);

  struct message msg;
  memset(&msg, 0, sizeof(struct message));

#ifdef WIN32
  if (recvfrom(fd, (char *)&msg, sizeof(struct message), 0, (struct sockaddr *)&addr, &addr_len) == -1) {
#else
  if (recvfrom(fd, (char *)&msg, sizeof(struct message), 0, (struct sockaddr *)&addr, (socklen_t *)&addr_len) == -1) {
#endif
    perror("recvfrom");
    return 0;
  }

  msg.users = ntohl(msg.users);
  msg.v_major = ntohs(msg.v_major);
  msg.v_minor = ntohs(msg.v_minor);

  printf(_("Current client version is %d.%d"), msg.v_major, msg.v_minor);
  printf(CRLF);
  printf(_("Number of users is %d."), msg.users);
  printf(CRLF);
  printf("%s\n", msg.msg);
  return 1;
}

static PapayaList cb_list;
static gint timeout = 0;

struct cb_data {

  callback cb;
  void * data;
  int fd;
  time_t timestamp;

};

int count_timeout(void * blah) {
  time_t current_time;
  time(&current_time);

  PapayaListElement * tmp_next;
  for (PapayaListElement * tmp = cb_list.firstElement; tmp; tmp = tmp_next) {
    tmp_next = tmp->getNext();
    struct cb_data * cbd = (struct cb_data *)tmp->getData();
    struct message msg;
    struct sockaddr_in addr;
    int addr_len = sizeof(struct sockaddr_in);
    int res;

    if (cbd == NULL)
      continue;

    /* Check for a response on this socket. */

#ifdef WIN32
    WSASetLastError(0);
    res = recvfrom(cbd->fd, (char *)&msg, sizeof(struct message), 0, (struct sockaddr *)&addr, &addr_len);
#else
    res = recvfrom(cbd->fd, (char *)&msg, sizeof(struct message), 0, (struct sockaddr *)&addr, (socklen_t *)&addr_len);
#endif
    if (res == -1) {
#ifdef WIN32
      errno = WSAGetLastError();
      if (errno == WSAEWOULDBLOCK) {
#else
      if (errno == EAGAIN) {
#endif

	if (cbd->timestamp + TIMEOUT_PERIOD >= current_time) {
	  // Timeout this callback, it's taken too long to get a response.
	  close(cbd->fd);
	  cb_list.deleteEntry(tmp);
	}

	continue;
      }

      if (cbd->cb)
	cbd->cb(-1, (void *)cbd->data);
      close(cbd->fd);
      cb_list.deleteEntry(tmp);
      return 1;

    }

    if (cbd->cb)
      cbd->cb(ntohl(msg.users), (void *)cbd->data);

    // Find out if this is a new version we're not currently running.
    int major;
    int minor;
    int current_ver;
    int new_ver;

    msg.v_major = ntohs(msg.v_major);
    msg.v_minor = ntohs(msg.v_minor);

    getCurrentVersion(&major, &minor);
    
    current_ver = major * 100 + minor;
    new_ver = msg.v_major * 100 + msg.v_minor;

    if (!globalPreferences->getPreference("version") || strcmp(VERSION, globalPreferences->getPreference("version"))) {
      if (current_ver % 2 == new_ver % 2 && new_ver > current_ver) {
	char buf[16384];
	snprintf(buf, 16384, _("Version %d.%d of Papaya is now available.  This is newer than your version."), msg.v_major, msg.v_minor);
	new Message("Information", buf, false);
	globalPreferences->setPreference("version", VERSION);
      }
    }

    close(cbd->fd);
    cb_list.deleteEntry(tmp);
    return 1;
  }
  return 1;
}

int count_number_of_users(callback cb, void * data) {

  struct cb_data * cbd = (struct cb_data *)malloc(sizeof(struct cb_data));

  memset(cbd, 0, sizeof(struct cb_data));

  cbd->fd = create_socket();
  if (cbd->fd == -1) {
    printf (_("Unable to create UDP socket to get number of users from server.\n"));
    free(cbd);
    return 0;
  }

  if (connect_to_server(cbd->fd) == 0) {
    printf (_("Unable to connect to remote count server to get number of users.\n"));
    close(cbd->fd);
    free(cbd);
    return 0;
  }

  cbd->cb = cb;
  cbd->data = data;
  time(&cbd->timestamp);

  cb_list.newEntry(cbd);

  if (!timeout)
    timeout = gtk_timeout_add(1000*1, count_timeout, NULL);
  return 1;
}

int delete_count() {
   if (timeout)
     gtk_timeout_remove(timeout);
   return 0;
}
