#include <time.h>
#include <string.h>
#include <stdio.h>

#include "ConnectionTimer.h"
#include "Win32PluginAPI.cpp"

extern PluginHandler * phandler;

static ConnectionTimer * ctimer;

#define MAJOR "1"
#define MINOR "1"

extern "C" G_MODULE_EXPORT char * plugin_query_name() {
  return "ConnectionTimer";
}

extern "C" G_MODULE_EXPORT char * plugin_query_description() {
  return _("Displays how long this connection has been active in the plugin tray.");
}

extern "C" G_MODULE_EXPORT char * plugin_query_major() {
  return MAJOR;
}

extern "C" G_MODULE_EXPORT char * plugin_query_minor() {
  return MINOR;
}

extern "C" G_MODULE_EXPORT void plugin_init(plugin_address_table_t * pat) {
  plugin_address_table_init(pat);
  ctimer = new ConnectionTimer();
}

extern "C" G_MODULE_EXPORT void plugin_cleanup(void) {
  delete ctimer;
}

static int TimerCmp(struct ctdata * c1, struct ctdata * c2) {
  return (c1 < c2);
}

ConnectionTimer::ConnectionTimer() {
  version = 1.0;
  name = strdup("Connection Timer");

  register_plugin(this, VERSION);
}

ConnectionTimer::~ConnectionTimer() {
  free(name);
  name = NULL;

  // Go through each entry, removing the timer widget from the tray.

  for (TimerDataList::iterator i = timerList.begin(); i != timerList.end(); i++) {
    struct ctdata * data = (struct ctdata *) (*i);

    vt_remove_from_tray(connection_get_vt(data->connection), data->event_box, data->frame);
    gtk_timeout_remove(data->timeout);
    free(data);
  }

  unregister_plugin(this);
}

gint conn_timer_timeout(gpointer data) {
  char buf[1024];

  struct ctdata * ct = (struct ctdata *) data;

  time_t currtime;
  time(&currtime);

  int seconds = currtime - ct->start;
  int minutes = seconds / 60;
  int hours = minutes / 60;
  minutes %= 60;
  seconds %= 60;

  sprintf(buf, "%.2d:%.2d:%.2ds", hours, minutes, seconds);
  gtk_label_set_text(GTK_LABEL(ct->widget), buf);
  return 1;
}

void ConnectionTimer::onEvent(Event * e, Connection * c) {

  if (event_get_type(e) == EvConnect) {
    struct ctdata * data = (struct ctdata *)malloc(sizeof(struct ctdata));
    data->connection = c;

    TimerDataList::iterator i = std::lower_bound(timerList.begin(),
						 timerList.end(),
						 data,
						 TimerCmp);
    
    timerList.insert(i, data);
    
    // Configure the connection time.
    time(&data->start);
    // Setup the timeout
    data->timeout = gtk_timeout_add(1000, conn_timer_timeout, (gpointer)data);

    data->event_box = gtk_event_box_new();

    // Configure the output Widget.
    data->widget = gtk_label_new(_("NOT CONNECTED"));
    gtk_widget_show(data->widget);
    gtk_container_add(GTK_CONTAINER(data->event_box), data->widget);

    data->tooltips = gtk_tooltips_new();
    gtk_tooltips_set_tip(data->tooltips, data->event_box, "ConnectionTimer: displays the length of time connected to a MUD.", NULL);


    vt_add_to_tray(connection_get_vt(c), data->event_box, &data->frame);
    return;
  }

  if (event_get_type(e) == EvDisconnect) {

    struct ctdata * d = find_data(c);
    // @@ Make removeFromTray delete the widget in future

    vt_remove_from_tray(connection_get_vt(c), d->event_box, d->frame);
    gtk_timeout_remove(d->timeout);
    g_free(d->tooltips);
    remove_data(d->connection);
    free(d);
    return;
  }

}

char * ConnectionTimer::getDescription() {
  return "Displays the length of time a connection has been active.";
};

struct ctdata * ConnectionTimer::find_data(Connection * conn) {
  for (TimerDataList::iterator i = timerList.begin(); i != timerList.end(); i++)
    if ((*i)->connection == conn)
      return (*i);
  return NULL;
}

void ConnectionTimer::remove_data(Connection * conn) {
  struct ctdata * data = find_data(conn);
  if (!data)
    return;

  TimerDataList::iterator i = std::lower_bound(timerList.begin(),
					       timerList.end(),
					       data,
					       TimerCmp);
  if (i == timerList.end() || (*i) != data)
    return;

  timerList.erase(i);
}
