/* $Id$ */

/* ---------------------------------------------------------------- */
/* Base Python object for all PETSc objects                         */
/* ---------------------------------------------------------------- */

%header %{
EXTERN_C_BEGIN

typedef struct PyPetscObjectObject {
  PyObject_HEAD
  PetscObject obj;
  PyObject*   own;
  PyObject*   swig;
} PyPetscObjectObject;

#define PyPetscObject(op) \
        ((PyPetscObjectObject*)(op))
#define PyPetscObject_OBJ(op) \
        (PyPetscObject(op)->obj)
#define PyPetscObject_OWN(op) \
        (PyPetscObject(op)->own)
#define PyPetscObject_SWIG(op) \
        (PyPetscObject(op)->swig)

#define PyPetscObject_CLEAN(op) \
        do { \
	  PyPetscObject_OBJ(op) = PETSC_NULL ; \
	  PyPetscObject_OWN(op) = Py_True; \
	} while(0)

#if !defined(PyPetscObject_Valid)
SWIGINTERNINLINE int
_py_petscobj_valid(PetscObject obj)
{
  return ((obj != PETSC_NULL) &&
	  (obj->cookie != PETSCFREEDHEADER) &&
	  (obj->cookie >= PETSC_SMALLEST_COOKIE) &&
	  (obj->cookie <= PETSC_LARGEST_COOKIE));
}
#define PyPetscObject_Valid _py_petscobj_valid
#endif

#if !defined(PyPetscObject_Destroy)
SWIGINTERNINLINE PetscErrorCode
_py_petscobj_destroy(PetscObject obj)
{
  if (obj == PETSC_NULL)                    return 0;
  if (obj->cookie == PETSCFREEDHEADER)      return 0;
  if (obj->cookie < PETSC_SMALLEST_COOKIE)  return 0;
  if (obj->cookie > PETSC_LARGEST_COOKIE)   return 0;
  if (!obj->bops->destroy)                  return 0;
  if (PetscFinalizeCalled)                  return 0;
  return PetscObjectDestroy(obj);
}
#define PyPetscObject_Destroy _py_petscobj_destroy
#endif

EXTERN_C_END
%}

/* ---------------------------------------------------------------- */


/* ---------------------------------------------------------------- */
/* Object C API                                                     */
/* ---------------------------------------------------------------- */

%header %{

#define PY_PETSC_OBJTPREG(PYTYPE, CTYPE)               \
EXTERN_C_BEGIN					       \
SWIGINTERN PyObject*				       \
PYTYPE##TypeRegister(PyObject* type) {		       \
  if (Py##CTYPE##_Type == NULL) {		       \
    if (!PyType_Check(type)) {			       \
      PyErr_SetString(PyExc_RuntimeError, 	       \
		      "expecting a type object");      \
      return NULL;				       \
    }						       \
    Py##CTYPE##_Type = (PyTypeObject*) type;	       \
  } else {					       \
    PyErr_SetString(PyExc_RuntimeError,                \
		    "type already registered");	       \
    return NULL;				       \
  }						       \
  Py_RETURN_NONE;				       \
}						       \
EXTERN_C_END

#define PY_PETSC_OBJASVAL(PYTYPE, CTYPE)               \
EXTERN_C_BEGIN 					       \
SWIGINTERN CTYPE				       \
Py##CTYPE##_AsVal(PyObject* op) { 		       \
  CTYPE obj;                                           \
  if (!Py##CTYPE##_Check(op)) { 		       \
    PyErr_SetString(PyExc_TypeError, 		       \
                    "expecting a '"#PYTYPE"' object"); \
    return PETSC_NULL; 				       \
  } 						       \
  obj = Py##CTYPE##_VAL(op); 			       \
  if (PyPetscObject_Valid((PetscObject)obj))	       \
    return obj;					       \
  PyPetscObject_CLEAN(op);			       \
  return PETSC_NULL;                                   \
} 						       \
EXTERN_C_END					       

#define PY_PETSC_OBJASPTR(PYTYPE, CTYPE)               \
EXTERN_C_BEGIN 					       \
SWIGINTERN CTYPE*				       \
Py##CTYPE##_AsPtr(PyObject* op) { 		       \
  CTYPE* ptr;                                          \
  if (!Py##CTYPE##_Check(op)) { 		       \
    PyErr_SetString(PyExc_TypeError, 		       \
		    "expecting a "#PYTYPE" object");   \
    return NULL;				       \
  } 						       \
  ptr = Py##CTYPE##_PTR(op); 			       \
  if (!PyPetscObject_Valid((PetscObject)*ptr))	       \
    PyPetscObject_CLEAN(op);			       \
  return ptr;                                          \
} 						       \
EXTERN_C_END					       

%}
	

%define %PETSC_OBJTYPEREG(PYTYPE, CTYPE)

%header %{

/* Python type object for PYTYPE */
static PyTypeObject* Py##CTYPE##_Type = NULL;

/* macros to check the type of PYTYPE objects */
#define Py##CTYPE##_Check(op) \
        PyObject_TypeCheck(op, Py##CTYPE##_Type)
#define Py##CTYPE##_CheckExact(op) \
        ((op)->ob_type == Py##CTYPE##_Type)

/* macros to extact the underlying CTYPE, no type checking*/
#define Py##CTYPE##_VAL(op) ((CTYPE)   PyPetscObject_OBJ(op))
#define Py##CTYPE##_PTR(op) ((CTYPE *)&PyPetscObject_OBJ(op))

/* functs to extract the underlying CTYPE, does type checking */
PY_PETSC_OBJTPREG(PYTYPE, CTYPE)
PY_PETSC_OBJASVAL(PYTYPE, CTYPE)
PY_PETSC_OBJASPTR(PYTYPE, CTYPE)
%}

static PyObject* PYTYPE##TypeRegister(PyObject*);

%enddef

/* ---------------------------------------------------------------- */



/* ---------------------------------------------------------------- */
/* Factory functions, PetscObject -> Python object                  */
/* ---------------------------------------------------------------- */

%header %{
#define PY_PETSC_OBJFACTORY(PYTYPE, CTYPE, COOKIE)       	    \
EXTERN_C_BEGIN                                                      \
SWIGINTERN PyObject*                                                \
Py##CTYPE##_New(CTYPE obj)                                          \
{                                                                   \
  PyTypeObject* type;                                               \
  PyObject*     self;                                               \
  /* check object type */                                           \
  if ((type = Py##CTYPE##_Type) == NULL) {                          \
    PyErr_SetString(PyExc_RuntimeError,                             \
		    "type object for "#CTYPE" not registered");     \
    return NULL;                                                    \
  }                                                                 \
  /* check input object */                                          \
  if (obj && !PETSC_chkobj(obj, COOKIE)) return NULL;		    \
  /* allocate a new object */                                       \
  self = (PyObject*) type->tp_alloc(type, 0);                       \
  /* fill attributes of allocated object */                         \
  if (self != NULL) {                                               \
    PyPetscObject(self)->obj  = (PetscObject) obj;                  \
    PyPetscObject(self)->own  = Py_True;                            \
    PyPetscObject(self)->swig = NULL;                               \
  }                                                                 \
  return self;                                                      \
}                                                                   \
EXTERN_C_END                                                        \
EXTERN_C_BEGIN 							    \
SWIGINTERN PyObject* 						    \
Py##CTYPE##_Ref(CTYPE obj) 					    \
{ 								    \
  PyTypeObject* type; 						    \
  PyObject*     self; 						    \
  /* check object type */ 					    \
  if ((type = Py##CTYPE##_Type) == NULL) { 			    \
    PyErr_SetString(PyExc_RuntimeError,  			    \
		    "type object for "#CTYPE" not registered");     \
    return NULL; 						    \
  } 								    \
  /* check input object */                                          \
  if (obj && !PETSC_chkobj(obj, COOKIE)) return NULL;		    \
  /* allocate a new  object */ 					    \
  self = (PyObject*) type->tp_alloc(type, 0); 			    \
  /* fill allocated object */ 					    \
  if (self != NULL) { 						    \
    if (obj) PetscObjectReference((PetscObject)obj);		    \
    PyPetscObject(self)->obj  = (PetscObject) obj;                  \
    PyPetscObject(self)->own  = Py_True;                            \
    PyPetscObject(self)->swig = NULL;                               \
  } 								    \
  return self; 							    \
} 								    \
EXTERN_C_END                                                        \
%}

%define %PETSC_OBJFACTORY(PYTYPE, CTYPE, COOKIE)
%header %{PY_PETSC_OBJFACTORY(PYTYPE, CTYPE, COOKIE)%}
%enddef

/* ---------------------------------------------------------------- */


/* ---------------------------------------------------------------- */
/* SWIG pointer                                                     */
/* ---------------------------------------------------------------- */

%define %PETSC_OBJSWIGPTR(PYTYPE, CTYPE)
/* SWIG pointer getter */
%wrapper %{
#define PYTYPE##__swig_this__(obj) Py##CTYPE##_AsPtr((obj))
%}
%exception PYTYPE##__swig_this__ 
{ $action if (result == NULL) SWIG_fail; }
CTYPE* PYTYPE##__swig_this__(PyObject*);
%enddef

/* ---------------------------------------------------------------- */


/* ---------------------------------------------------------------- */
/* This macro applies all previous macros                           */
/* ---------------------------------------------------------------- */

%define %PETSC_OBJALLDECL(PYTYPE, CTYPE, COOKIE)
%PETSC_OBJTYPEREG(PYTYPE, CTYPE)
%PETSC_OBJFACTORY(PYTYPE, CTYPE, COOKIE)
%PETSC_OBJSWIGPTR(PYTYPE, CTYPE)
%enddef

/* ---------------------------------------------------------------- */


/*
 * Local Variables:
 * mode: C
 * End:
 */
