/* shellspawn.c -- Daemon for spawning shells on VTs 2 - 6. */

static char ident[] = "$Progeny: shellspawn.c,v 1.3 2002/02/07 00:13:21 epg Exp $";

/* Copyright (C) 2002 Progeny Linux Systems, Inc.
 * AUTHORS: Eric Gillespie <epg@progeny.com>

 * Mostly  stolen from Busybox init which says:
 * Copyright (C) 1995, 1996 by Bruce Perens <bruce@pixar.com>.
 * Adjusted by so many folks, it's impossible to keep track.

 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.

 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

#include <sys/types.h>
#include <sys/fcntl.h>
#include <sys/ioctl.h>
#include <sys/mount.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <errno.h>
#include <limits.h>
#include <paths.h>
#include <signal.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sysexits.h>
#include <termios.h>
#include <unistd.h>

#define TRUE 1
#define FALSE 0

/* From <linux/vt.h> */
struct vt_stat {
	unsigned short v_active;        /* active vt */
	unsigned short v_signal;        /* signal to send */
	unsigned short v_state;         /* vt bitmask */
};
static const int VT_GETSTATE = 0x5603;  /* get global vt state info */

/* From <linux/serial.h> */
struct serial_struct {
	int     type;
	int     line;
	int     port;
	int     irq;
	int     flags;
	int     xmit_fifo_size;
	int     custom_divisor;
	int     baud_base;
	unsigned short  close_delay;
	char    reserved_char[2];
	int     hub6;
	unsigned short  closing_wait; /* time to wait before closing */
	unsigned short  closing_wait2; /* no longer used... */
	int     reserved[4];
};

#ifndef _PATH_STDPATH
#define _PATH_STDPATH	"/usr/bin:/bin:/usr/sbin:/sbin"
#endif

#define KERNEL_VERSION(a,b,c) (((a) << 16) + ((b) << 8) + (c))

#if __GNU_LIBRARY__ > 5
	#include <sys/kdaemon.h>
#else
	extern int bdflush (int func, long int data);
#endif


#define SHELL        "/bin/sh"	     /* Default shell */
#define LOGIN_SHELL  "-" SHELL	     /* Default login shell */

#define MAXENV	16		/* Number of env. vars */
static const int LOG = 0x1;
static const int CONSOLE = 0x2;

typedef struct _consoleItem consoleItem;
struct _consoleItem {
    pid_t pid;
    char console[256];
};
static consoleItem consoles[7];

//static char *secondConsole = VC_2;
//static char *thirdConsole  = VC_3;
//static char *fourthConsole = VC_4;
//static char *log           = VC_5;
static char termType[32]   = "TERM=linux";
static char console[32]    = _PATH_CONSOLE;

static int
device_open(char *device, int mode)
{
    int m, f, fd = -1;

    m = mode | O_NONBLOCK;

    /* Retry up to 5 times */
    for (f = 0; f < 5; f++)
        if ((fd = open(device, m, 0600)) >= 0)
            break;
    if (fd < 0)
        return fd;
    /* Reset original flags. */
    if (m != mode)
        fcntl(fd, F_SETFL, mode);
    return fd;
}

static char *
get_last_path_component(char *path)
{
    char *s;
    register char *ptr = path;
    register char *prev = 0;

    while (*ptr)
        ptr++;
    s = ptr - 1;

    /* strip trailing slashes */
    while (s != path && *s == '/') {
        *s-- = '\0';
    }

    /* find last component */
    ptr = path;
    while (*ptr != '\0') {
        if (*ptr == '/')
            prev = ptr;
        ptr++;
    }
    s = prev;

    if (s == NULL || s[1] == '\0')
        return path;
    else
        return s+1;
}

static void loop_forever()
{
	while (1)
		sleep (1);
}

/* Print a message to the specified device.
 * Device may be bitwise-or'd from LOG | CONSOLE */
static void message(int device, char *fmt, ...)
		   __attribute__ ((format (printf, 2, 3)));
static void message(int device, char *fmt, ...)
{
// XXX: This needs to be fixed!!!
    return;
#if 0
    va_list arguments;
    int fd;

    static int log_fd = -1;

    /* Take full control of the log tty, and never close it.
     * It's mine, all mine!  Muhahahaha! */
    if (log_fd < 0) {
        if (log == NULL) {
            /* don't even try to log, because there is no such console */
            log_fd = -2;
            /* log to main console instead */
            device = CONSOLE;
        } else if ((log_fd = device_open(log, O_RDWR|O_NDELAY)) < 0) {
            log_fd = -2;
            fprintf(stderr, "Bummer, can't write to log on %s!\r\n", log);
            log = NULL;
            device = CONSOLE;
        }
    }
    if ((device & LOG) && (log_fd >= 0)) {
        va_start(arguments, fmt);
        vdprintf(log_fd, fmt, arguments);
        va_end(arguments);
    }

    if (device & CONSOLE) {
        /* Always send console messages to /dev/console so people will see them. */
        if (
            (fd =
             device_open(_PATH_CONSOLE,
                         O_WRONLY | O_NOCTTY | O_NDELAY)) >= 0) {
            va_start(arguments, fmt);
            vdprintf(fd, fmt, arguments);
            va_end(arguments);
            close(fd);
        } else {
            fprintf(stderr, "Bummer, can't print: ");
            va_start(arguments, fmt);
            vfprintf(stderr, fmt, arguments);
            va_end(arguments);
        }
    }
#endif
}

/* Set terminal settings to reasonable defaults */
static void set_term(int fd)
{
	struct termios tty;

	tcgetattr(fd, &tty);

	/* set control chars */
	tty.c_cc[VINTR]  = 3;	/* C-c */
	tty.c_cc[VQUIT]  = 28;	/* C-\ */
	tty.c_cc[VERASE] = 127; /* C-? */
	tty.c_cc[VKILL]  = 21;	/* C-u */
	tty.c_cc[VEOF]   = 4;	/* C-d */
	tty.c_cc[VSTART] = 17;	/* C-q */
	tty.c_cc[VSTOP]  = 19;	/* C-s */
	tty.c_cc[VSUSP]  = 26;	/* C-z */

	/* use line dicipline 0 */
	tty.c_line = 0;

	/* Make it be sane */
	tty.c_cflag &= CBAUD|CBAUDEX|CSIZE|CSTOPB|PARENB|PARODD;
	tty.c_cflag |= CREAD|HUPCL|CLOCAL;


	/* input modes */
	tty.c_iflag = ICRNL | IXON | IXOFF;

	/* output modes */
	tty.c_oflag = OPOST | ONLCR;

	/* local modes */
	tty.c_lflag =
		ISIG | ICANON | ECHO | ECHOE | ECHOK | ECHOCTL | ECHOKE | IEXTEN;

	tcsetattr(fd, TCSANOW, &tty);
}

static void console_init()
{
    return;
// XXX: And what about this?
#if 0
    int fd;
    int tried_devcons = 0;
    int tried_vtprimary = 0;
    struct vt_stat vt;
    struct serial_struct sr;
    char *s;

    if ((s = getenv("TERM")) != NULL) {
        snprintf(termType, sizeof(termType) - 1, "TERM=%s", s);
    }

    if ((s = getenv("CONSOLE")) != NULL) {
        safe_strncpy(console, s, sizeof(console));
    }
#if #cpu(sparc)
    /* sparc kernel supports console=tty[ab] parameter which is also 
     * passed to init, so catch it here */
    else if ((s = getenv("console")) != NULL) {
        /* remap tty[ab] to /dev/ttyS[01] */
        if (strcmp(s, "ttya") == 0)
            safe_strncpy(console, SC_0, sizeof(console));
        else if (strcmp(s, "ttyb") == 0)
            safe_strncpy(console, SC_1, sizeof(console));
    }
#endif
    else {
        /* 2.2 kernels: identify the real console backend and try to use it */
        if (ioctl(0, TIOCGSERIAL, &sr) == 0) {
            /* this is a serial console */
            snprintf(console, sizeof(console) - 1, SC_FORMAT, sr.line);
        } else if (ioctl(0, VT_GETSTATE, &vt) == 0) {
            /* this is linux virtual tty */
            snprintf(console, sizeof(console) - 1, VC_FORMAT, vt.v_active);
        } else {
            safe_strncpy(console, _PATH_CONSOLE, sizeof(console));
            tried_devcons++;
        }
    }

    while ((fd = open(console, O_RDONLY | O_NONBLOCK)) < 0) {
        /* Can't open selected console -- try /dev/console */
        if (!tried_devcons) {
            tried_devcons++;
            safe_strncpy(console, _PATH_CONSOLE, sizeof(console));
            continue;
        }
        /* Can't open selected console -- try vt1 */
        if (!tried_vtprimary) {
            tried_vtprimary++;
            safe_strncpy(console, VC_1, sizeof(console));
            continue;
        }
        break;
    }
    if (fd < 0) {
        /* Perhaps we should panic here? */
        safe_strncpy(console, "/dev/null", sizeof(console));
    } else {
        /* check for serial console and disable logging to tty5 & running a
         * shell to tty2-4 */
        if (ioctl(0, TIOCGSERIAL, &sr) == 0) {
            log = NULL;
            secondConsole = NULL;
            thirdConsole = NULL;
            fourthConsole = NULL;
            /* Force the TERM setting to vt102 for serial console --
             * iff TERM is set to linux (the default) */
            if (strcmp( termType, "TERM=linux" ) == 0)
                safe_strncpy(termType, "TERM=vt102", sizeof(termType));
            message(LOG | CONSOLE,
                    "serial console detected.  Disabling virtual terminals.\r\n");
        }
        close(fd);
    }
    message(LOG, "console=%s\n", console);
#endif
}
	
static pid_t
run(char *command, char *terminal, int get_enter)
{
    int i, j;
    int fd;
    pid_t pid;
    char *tmpCmd, *s;
    char *cmd[255], *cmdpath;
    char buf[255];
    struct stat sb;
    static const char press_enter[] =
        "\nPlease press Enter to activate this console. ";

    if ((pid = fork()) == 0) {
        /* Clean up */
        ioctl(0, TIOCNOTTY, 0);
        close(0);
        close(1);
        close(2);
        setsid();

        /* Reset signal handlers set for parent process */
        signal(SIGUSR1, SIG_DFL);
        signal(SIGUSR2, SIG_DFL);
        signal(SIGINT, SIG_DFL);
        signal(SIGTERM, SIG_DFL);
        signal(SIGHUP, SIG_DFL);

        if ((fd = device_open(terminal, O_RDWR)) < 0) {
            if (stat(terminal, &sb) != 0) {
                message(LOG | CONSOLE, "device '%s' does not exist.\n",
                        terminal);
                exit(1);
            }
            message(LOG | CONSOLE, "Bummer, can't open %s\r\n", terminal);
            exit(1);
        }
        dup2(fd, 0);
        dup2(fd, 1);
        dup2(fd, 2);
        ioctl(0, TIOCSCTTY, 1);
        tcsetpgrp(0, getpgrp());
        set_term(0);
        
        /* See if any special /bin/sh requiring characters are present */
        if (strpbrk(command, "~`!$^&*()=|\\{}[];\"'<>?") != NULL) {
            cmd[0] = SHELL;
            cmd[1] = "-c";
            strcpy(buf, "exec ");
            strncat(buf, command, sizeof(buf) - strlen(buf) - 1);
            cmd[2] = buf;
            cmd[3] = NULL;
        } else {
            /* Convert command (char*) into cmd (char**, one word per string) */
            for (tmpCmd = command, i = 0;
                 (tmpCmd = strsep(&command, " \t")) != NULL;) {
                if (*tmpCmd != '\0') {
                    cmd[i] = tmpCmd;
                    tmpCmd++;
                    i++;
                }
            }
            cmd[i] = NULL;
        }
        
        cmdpath = cmd[0];

        /*
          Interactive shells want to see a dash in argv[0].  This
          typically is handled by login, argv will be setup this 
          way if a dash appears at the front of the command path 
          (like "-/bin/sh").
        */

        if (*cmdpath == '-') {
            /* skip over the dash */
            ++cmdpath;

            /* find the last component in the command pathname */
            s = get_last_path_component(cmdpath);

            /* make a new argv[0] */
            if ((cmd[0] = malloc(strlen(s)+2)) == NULL) {
                message(LOG | CONSOLE, "malloc failed");
                cmd[0] = cmdpath;
            } else {
                cmd[0][0] = '-';
                strcpy(cmd[0]+1, s);
            }
        }

        if (get_enter == TRUE) {
            /*
             * Save memory by not exec-ing anything large (like a shell)
             * before the user wants it. This is critical if swap is not
             * enabled and the system has low memory. Generally this will
             * be run on the second virtual console, and the first will
             * be allowed to start a shell or whatever an init script 
             * specifies.
             */

            write(fileno(stdout), press_enter, sizeof(press_enter) - 1);
            getc(stdin);
        }

        /* Now run it.  The new program will take over this PID, 
         * so nothing further in init.c should be run. */
        execv(cmdpath, cmd);

        /* We're still here?  Some error happened. */
        message(LOG | CONSOLE, "Bummer, could not run '%s': %s\n", cmdpath,
                strerror(errno));
        exit(-1);
    }

    return pid;
}

static int waitfor(char *command, char *terminal, int get_enter)
{
	int status, wpid;
	int pid = run(command, terminal, get_enter);

	while (1) {
		wpid = wait(&status);
		if (wpid > 0 && wpid != pid) {
			continue;
		}
		if (wpid == pid)
			break;
	}
	return wpid;
}

static void
handle_sigterm(int sig)
{
    int i;

    for (i = 2; i <= 6; i++) {
        if (consoles[i].pid != 0) {
            kill(consoles[i].pid, SIGTERM);
        }
    }

    sleep(1);

    for (i = 2; i <= 6; i++) {
        if (consoles[i].pid != 0) {
            kill(consoles[i].pid, SIGKILL);
        }
    }

    exit(EX_OK);
}

int
main(int argc, char *argv[])
{
    int i;
    pid_t wpid;
    int status;

    signal(SIGTERM, handle_sigterm);

    /* Figure out where the default console should be */
    console_init();

    /* Close whatever files are open, and reset the console. */
    close(0);
    close(1);
    close(2);
    set_term(0);
    chdir("/");
    setsid();

    /* Set sane environment. */
    putenv("HOME=/");
    putenv("PATH="_PATH_STDPATH);
    putenv("SHELL=" SHELL);
    putenv("USER=root");

    for (i = 2; i <=6; i++) {
        consoles[i].pid = 0;
        snprintf(consoles[i].console, 255, "/dev/tty%d", i);
    }

    /* Now run the looping stuff for the rest of forever */
    while (1) {
        for (i = 2; i <= 6; i++) {
            /* Only run stuff with pid==0.  If they have
             * a pid, that means they are still running */
            if (consoles[i].pid == 0) {
                /* run the askfirst stuff */
                consoles[i].pid = run("/bin/envsh",
                                      consoles[i].console, TRUE);
            }
        }

        /* Wait for a child process to exit */
        wpid = wait(&status);
        if (wpid > 0) {
            /* Find out who died and clean up their corpse */
            for (i = 2; i <= 6; i++) {
                if (consoles[i].pid == wpid) {
                    consoles[i].pid = 0;
                    message(LOG,
                            "Shell (pid %d) exited.  Scheduling it for restart.\n",
                            i, wpid);
                }
            }
        }
        sleep(1);
    }
}

/*
 * Local variables:
 * c-file-style: "progeny"
 * indent-tabs-mode: nil
 * End:
 */
/* vim: set cin fo=tcroq sw=4 et sts=4 tw=75: */
