  /**************************************************************************\
  * phpGroupWare - SyncML Synchronization                                    *
  * http://www.phpgroupware.org                                              *
  * Written by Jens P. Elsner <jpelsner@gmx.net>                             *
  *            Mark Wormgoor <mark@wormgoor.com>                             *
  * --------------------------------------------                             *
  *  This program is free software; you can redistribute it and/or modify it *
  *  under the terms of the GNU General Public License as published by the   *
  *  Free Software Foundation; either version 2 of the License, or (at your  *
  *  option) any later version.                                              *
  \**************************************************************************/

/*
This product includes software developed by The SyncML Initiative.
*/

#include "php.h"
#include "php_ini.h"
#include "php_syncml.h"

/* syncml includes */
#include <sml.h>
#include <smlerr.h>

/* True global resources - no need for thread safety here */
static int le_syncml;

function_entry syncml_functions[] = {
        /* Manager API */
        PHP_FE(sml_init_instance, NULL)
	PHP_FE(sml_set_syncml_options, NULL)
	PHP_FE(sml_set_callbacks, NULL)
	PHP_FE(sml_set_encoding, NULL)
	PHP_FE(sml_set_user_data, NULL)
	PHP_FE(sml_free_proto_element, NULL)
	PHP_FE(sml_write_in_buffer, NULL)
	PHP_FE(sml_read_out_buffer, NULL)
	PHP_FE(sml_get_free_buffer, NULL)
        /* Command Builder API */
        PHP_FE(sml_start_message, NULL)
        PHP_FE(sml_end_message, NULL)
        PHP_FE(sml_start_sync, NULL)
        PHP_FE(sml_end_sync, NULL)
        PHP_FE(sml_start_atomic, NULL)
        PHP_FE(sml_end_atomic, NULL)
        PHP_FE(sml_start_sequence, NULL)
        PHP_FE(sml_end_sequence, NULL)
        PHP_FE(sml_add_cmd, NULL)
        PHP_FE(sml_alert_cmd, NULL)
        PHP_FE(sml_copy_cmd, NULL)
        PHP_FE(sml_delete_cmd, NULL)
        PHP_FE(sml_exec_cmd, NULL)
        PHP_FE(sml_get_cmd, NULL)
        PHP_FE(sml_put_cmd, NULL)
        PHP_FE(sml_map_cmd, NULL)
        PHP_FE(sml_results_cmd, NULL)
        PHP_FE(sml_search_cmd, NULL)
        PHP_FE(sml_status_cmd, NULL)
        PHP_FE(sml_replace_cmd, NULL)
	/* Command Dispatcher API */
        PHP_FE(sml_process_data, NULL) 
	{NULL, NULL, NULL}	/* Must be the last line in syncml_functions[] */
};

zend_module_entry syncml_module_entry = {
	"syncml",
	syncml_functions,
	PHP_MINIT(syncml),
	PHP_MSHUTDOWN(syncml),
	NULL,
	NULL,
	PHP_MINFO(syncml),
	STANDARD_MODULE_PROPERTIES
};

#ifdef COMPILE_DL_SYNCML
ZEND_GET_MODULE(syncml)
#endif

// FIXME: this redefines a macro
#define WRONG_PARAM_COUNT \
        php_error(E_ERROR, "Wrong parameter count.")   

/* small helper macros for the command builder */
#define GET_STR_PROP(object, prop, struc) \
	prop = get_property(object, #prop); \
	convert_to_string_ex(prop); \
	struc.prop = smlString2Pcdata((*prop)->value.str.val); \
	struc.prop->contentType = SML_PCDATA_STRING; \

#define GET_LONG_PROP(object, prop, struc) \
        prop = get_property(object, #prop); \
        convert_to_long_ex(prop); \
        struc.prop = (*prop)->value.lval; \

/* define type for our resource */
typedef struct
{
        InstanceID_t id;
} resource_type;

static void PrintFunc(String_t outputString);

/* Terminates a SyncML instance. */
static void _sml_terminate_instance(resource_type *resource) {
        Ret_t ret;

        //zend_printf("TerminateInstance called, id: 0x%x \n", resource->id);
        ret = smlTerminateInstance(resource->id);
        efree(resource);
}

PHP_MINIT_FUNCTION(syncml) {
        SmlOptions_t syncmlOptions;
        Ret_t ret;

	le_syncml = register_list_destructors(_sml_terminate_instance, NULL);

        syncmlOptions.defaultPrintFunc=PrintFunc;
        syncmlOptions.maxWorkspaceAvailMem=100000;
        ret = smlInit(&syncmlOptions);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlInit failed. Reason: 0x%x.", ret);
		return FAILURE;
        }
	return SUCCESS;
}

PHP_MSHUTDOWN_FUNCTION(syncml) {
        Ret_t ret;
        
        ret = smlTerminate();
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlTerminate failed. Reason: 0x%x.", ret);
        }
	return SUCCESS;
}

PHP_MINFO_FUNCTION(syncml) {
	php_info_print_table_start();
	php_info_print_table_header(2, "SyncML support", "enabled");
	php_info_print_table_end();
}

/* sml callbacks */

/* wrapper for call_user_function */
void CallHandler(char* function_name, VoidPtr_t userData, zval* data) {
        zval *retval;
	zval *function_name_zval;
	zval *args[1];

        //zend_printf("CallHandler: Callback %s called. \n", function_name);
        MAKE_STD_ZVAL(function_name_zval);
        MAKE_STD_ZVAL(retval);
        ZVAL_STRING(function_name_zval, function_name, 1);

	args[0] = data;

        CLS_FETCH();

        if (userData==NULL) {
		if (data==NULL) {
                	if(call_user_function(CG(function_table), NULL, function_name_zval, retval, 0, NULL) != SUCCESS) {
                        	zend_printf("CallHandler: (userData = NULL) Callback %s not defined. \n", function_name);
                	}
		}
		else {
	                        if(call_user_function(CG(function_table), NULL, function_name_zval, retval, 1, args) != SUCCESS) {
                                zend_printf("CallHandler: (userData = NULL) Callback %s not defined. \n", function_name);
                        }
		}
        }
        else {
                if (data==NULL) {
                        if(call_user_function((HashTable *) userData, NULL, function_name_zval, retval, 0, NULL) != SUCCESS) {
                                zend_printf("CallHandler: (userData = NULL) Callback %s not defined. \n", function_name);
                        }
                }
                else {
                	if(call_user_function((HashTable *) userData, NULL, function_name_zval, retval, 1, args) != SUCCESS) {
                	zend_printf("CallHandler: (userData = NULL) Callback %s not defined. \n", function_name);
                        }
                }
        }
}

/* functions to create zval objects */

zval *make_source_or_target(SmlTargetPtr_t target) {
	zend_class_entry *class_entry;
	zval *zval_target=NULL;

	if (target!=NULL) {
		if (zend_hash_find(CG(class_table), "sml_source_or_target", strlen("sml_source_or_target")+1, (void **)&class_entry)==FAILURE) {
			php_error(E_ERROR, "Unable to find the %s class definition.", "sml_source_or_target");
		}
		else {
			MAKE_STD_ZVAL(zval_target);
			object_init_ex(zval_target, class_entry);
			add_property_stringl(zval_target, "locURI", target->locURI->content, target->locURI->length, 1);
			if (target->locName!=NULL) {
				add_property_stringl(zval_target, "locName", target->locName->content, target->locName->length, 1);
			}
		}
	}
	return zval_target;
}

zval *make_cred(SmlCredPtr_t cred) {
	zend_class_entry *class_entry;
	zval *zval_cred=NULL;

	if (cred!=NULL) {
		if (zend_hash_find(CG(class_table), "sml_cred", strlen("sml_cred")+1, (void **)&class_entry)==FAILURE) {
			php_error(E_ERROR, "Unable to find the %s class definition.", "sml_cred");
		}
		else {
			MAKE_STD_ZVAL(zval_cred);
			object_init_ex(zval_cred, class_entry);
			if (cred->meta!=NULL) {
				add_property_stringl(zval_cred, "meta", cred->meta->content, cred->meta->length, 1);
			}
				add_property_stringl(zval_cred, "data", cred->data->content, cred->data->length, 1);
		}
	}
	return zval_cred;
}

zval *make_item_list(SmlItemListPtr_t itemList) {
        zend_class_entry *class_entry;
        zval *zval_item_list=NULL, *item;
	
        if (itemList!=NULL) {

                if (zend_hash_find(CG(class_table), "sml_item", strlen("sml_item")+1, (void **)&class_entry)==FAILURE) {
                        php_error(E_ERROR, "Unable to find the %s class definition.", "sml_item");
                }
		else {
			MAKE_STD_ZVAL(zval_item_list);
			if(array_init(zval_item_list) != SUCCESS) {
				php_error(E_ERROR, "Could not initialize array.");
			}

			while (itemList != NULL) {
				MAKE_STD_ZVAL(item);
				object_init_ex(item, class_entry);
				if (itemList->item->target!=NULL) {
					add_property_zval(item, "target", make_source_or_target(itemList->item->target));
				}
				if (itemList->item->source!=NULL) {
                                        add_property_zval(item, "source", make_source_or_target(itemList->item->source));
				}
				if (itemList->item->meta!=NULL) {
					add_property_stringl(item, "meta", itemList->item->meta->content, itemList->item->meta->length, 1);
                                }
                                if (itemList->item->data!=NULL) { 
                                        add_property_stringl(item, "data", itemList->item->data->content, itemList->item->data->length, 1);
                                }
				add_next_index_zval(zval_item_list, item);
				itemList = itemList->next;
			} 
       	 	}
	}
        return zval_item_list;
}

/* Protocol Management */
Ret_t StartMessage(InstanceID_t id, VoidPtr_t userData, SmlSyncHdrPtr_t pContent) {
        zval *data=NULL;
	zend_class_entry *class_entry;
	
        CLS_FETCH();

	// first step: see if we have access to the right class
        if (zend_hash_find(CG(class_table), "sml_sync_hdr", strlen("sml_sync_hdr")+1, (void **)&class_entry)==FAILURE) {
		php_error(E_WARNING, "Unable to find the %s class definition.", "sml_sync_hdr");
        }
	else {
		// second step: create an object
		MAKE_STD_ZVAL(data);
		object_init_ex(data, class_entry);
		// third step: fill in the data
		add_property_stringl(data, "version", pContent->version->content, pContent->version->length, 1);
                add_property_stringl(data, "proto", pContent->proto->content, pContent->proto->length, 1);
                add_property_stringl(data, "sessionID", pContent->sessionID->content, pContent->sessionID->length, 1);
                add_property_stringl(data, "msgID", pContent->msgID->content, pContent->msgID->length, 1);
		add_property_long(data, "flags", pContent->flags);

                if (pContent->target!=NULL) {
			add_property_zval(data, "target", make_source_or_target(pContent->target));
		}

                if (pContent->source!=NULL) {
			add_property_zval(data, "source", make_source_or_target(pContent->source));
		}

                if (pContent->respURI!=NULL) {
			add_property_stringl(data, "respURI", pContent->respURI->content, pContent->respURI->length, 1);
		}

		if (pContent->cred!=NULL) {
			add_property_zval(data, "cred", make_cred(pContent->cred));
		}

                if (pContent->meta!=NULL) {
			add_property_stringl(data, "meta", pContent->meta->content, pContent->meta->length, 1);
		}
	}
        CallHandler("cb_start_message", userData, data);
        return SML_ERR_OK;
}

Ret_t EndMessage(InstanceID_t id, VoidPtr_t userData, Boolean_t final) {
        zval *data=NULL;

	MAKE_STD_ZVAL(data);
	ZVAL_BOOL(data, final);

        CallHandler("cb_end_message", userData, data);
        return SML_ERR_OK;
}

Ret_t StartSync(InstanceID_t id, VoidPtr_t userData, SmlSyncPtr_t pContent) {
        zval *data=NULL;
        zend_class_entry *class_entry;

        CLS_FETCH();

        if (zend_hash_find(CG(class_table), "sml_sync", strlen("sml_sync")+1, (void **)&class_entry)==FAILURE) {
                php_error(E_WARNING, "Unable to find the %s class definition.", "sml_sync");
        }
        else {
                MAKE_STD_ZVAL(data);
                object_init_ex(data, class_entry);

                add_property_stringl(data, "cmdID", pContent->cmdID->content, pContent->cmdID->length, 1);
                add_property_long(data, "flags", pContent->flags);

                if (pContent->cred!=NULL) {
                        add_property_zval(data, "cred", make_cred(pContent->cred));
                }

                if (pContent->target!=NULL) {
                        add_property_zval(data, "target", make_source_or_target(pContent->target));
                }

                if (pContent->source!=NULL) {
                        add_property_zval(data, "source", make_source_or_target(pContent->source));
                }

                if (pContent->meta!=NULL) {
                        add_property_stringl(data, "meta", pContent->meta->content, pContent->meta->length, 1);
                }
	}

        CallHandler("cb_start_sync", userData, data);
        return SML_ERR_OK;
}

Ret_t EndSync(InstanceID_t id, VoidPtr_t userData) {
        CallHandler("cb_end_sync", userData, NULL);
        return SML_ERR_OK;
}

Ret_t StartAtomic(InstanceID_t id, VoidPtr_t userData, SmlAtomicPtr_t pContent) {
        zval *data=NULL;
        zend_class_entry *class_entry;

        CLS_FETCH();

        if (zend_hash_find(CG(class_table), "sml_atomic_seq", strlen("sml_atomic_seq")+1, (void **)&class_entry)==FAILURE) {
                php_error(E_ERROR, "Unable to find the %s class definition.", "sml_atomic_seq");
        }
        else {
                MAKE_STD_ZVAL(data);
                object_init_ex(data, class_entry);

                add_property_stringl(data, "cmdID", pContent->cmdID->content, pContent->cmdID->length, 1);
                add_property_long(data, "flags", pContent->flags);

                if (pContent->meta!=NULL) {
                        add_property_stringl(data, "meta", pContent->meta->content, pContent->meta->length, 1);
                }
        }

        CallHandler("cb_start_atomic", userData, data);
        return SML_ERR_OK;
}

Ret_t EndAtomic(InstanceID_t id, VoidPtr_t userData) {
        CallHandler("cb_end_atomic", userData, NULL);
        return SML_ERR_OK;
}

Ret_t StartSequence(InstanceID_t id, VoidPtr_t userData, SmlSequencePtr_t pContent) {
        zval *data=NULL;
        zend_class_entry *class_entry;

        CLS_FETCH();

        if (zend_hash_find(CG(class_table), "sml_atomic_seq", strlen("sml_atomic_seq")+1, (void **)&class_entry)==FAILURE) {
                php_error(E_ERROR, "Unable to find the %s class definition.", "sml_atomic_seq");
        }
        else {
                MAKE_STD_ZVAL(data);
                object_init_ex(data, class_entry);

                add_property_stringl(data, "cmdID", pContent->cmdID->content, pContent->cmdID->length, 1);
                add_property_long(data, "flags", pContent->flags);

                if (pContent->meta!=NULL) {
                        add_property_stringl(data, "meta", pContent->meta->content, pContent->meta->length, 1);
                }
        }

        CallHandler("cb_start_sequence", userData, data);
        return SML_ERR_OK;
}

Ret_t EndSequence(InstanceID_t id, VoidPtr_t userData) {
        CallHandler("cb_end_sequence", userData, NULL);
        return SML_ERR_OK;
}

/* Sync Commands */
Ret_t AddCmd(InstanceID_t id, VoidPtr_t userData, SmlAddPtr_t pContent) {
        zval *data=NULL;
        zend_class_entry *class_entry;

        CLS_FETCH();

        if (zend_hash_find(CG(class_table), "sml_generic", strlen("sml_generic")+1, (void **)&class_entry)==FAILURE) {
                php_error(E_WARNING, "Unable to find the %s class definition.", "sml_generic");
        }
        else {
                MAKE_STD_ZVAL(data);
                object_init_ex(data, class_entry);

                add_property_stringl(data, "cmdID", pContent->cmdID->content, pContent->cmdID->length, 1);
                add_property_long(data, "flags", pContent->flags);

                if (pContent->cred!=NULL) {
                        add_property_zval(data, "cred", make_cred(pContent->cred));
                }

                if (pContent->meta!=NULL) {
                        add_property_stringl(data, "meta", pContent->meta->content, pContent->meta->length, 1);
                }

		add_property_zval(data, "itemList", make_item_list(pContent->itemList));
		
        }

        CallHandler("cb_add_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t AlertCmd(InstanceID_t id, VoidPtr_t userData, SmlAlertPtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_alert_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t DeleteCmd(InstanceID_t id, VoidPtr_t userData, SmlDeletePtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_delete_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t GetCmd(InstanceID_t id, VoidPtr_t userData, SmlGetPtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_get_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t PutCmd(InstanceID_t id, VoidPtr_t userData, SmlPutPtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_put_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t MapCmd(InstanceID_t id, VoidPtr_t userData, SmlMapPtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_map_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t ResultsCmd(InstanceID_t id, VoidPtr_t userData, SmlResultsPtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_results_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t StatusCmd(InstanceID_t id, VoidPtr_t userData, SmlStatusPtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_status_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t ReplaceCmd(InstanceID_t id, VoidPtr_t userData, SmlReplacePtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_replace_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t CopyCmd(InstanceID_t id, VoidPtr_t userData, SmlCopyPtr_t param) {
        zval *data=NULL;

        /* put param into data object here */

        CallHandler("cb_copy_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t ExecCmd(InstanceID_t id, VoidPtr_t userData, SmlExecPtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_exec_cmd", userData, data);
        return SML_ERR_OK;
}
Ret_t SearchCmd(InstanceID_t id, VoidPtr_t userData, SmlSearchPtr_t pContent) {
        zval *data=NULL;

        /* put pContent into data object here */

        CallHandler("cb_search_cmd", userData, data);
        return SML_ERR_OK;
}

/* Other Callbacks */
Ret_t HandleError(InstanceID_t id, VoidPtr_t userData) {
        CallHandler("cb_handle_error", userData, NULL);
        return SML_ERR_OK;
}
Ret_t TransmitChunk(InstanceID_t id, VoidPtr_t userData) {
        CallHandler("cb_transmit_chunk", userData, NULL);
        return SML_ERR_OK;
}

/* {{{ proto resource sml_init_instance(int encoding, int wssize, string wsname, [object function_namespace])
   Creates a SyncML instance. */
PHP_FUNCTION(sml_init_instance) {
	zval **encoding, **wssize, **wsname, **resource_value, **object;
	resource_type *resource;
	HashTable *function_table;

        SmlInstanceOptions_t options;
        SmlCallbacks_t callbacks;       
        InstanceID_t id;
        Ret_t ret;

        resource = (resource_type *)emalloc(sizeof(resource_type));

	switch (ZEND_NUM_ARGS()) {
		case 4:
        		if (zend_get_parameters_ex(4, &encoding, &wssize, &wsname, &object) == FAILURE) {
                		WRONG_PARAM_COUNT;
        		}
			function_table = &((**object).value.obj.ce->function_table);
			break;
		case 3:
                        if (zend_get_parameters_ex(3, &encoding, &wssize, &wsname) == FAILURE) {
                        	WRONG_PARAM_COUNT;
                        }
		        CLS_FETCH();
			function_table=CG(function_table);
                        break;
	}

	convert_to_long_ex(encoding);
	convert_to_long_ex(wssize);
	convert_to_string_ex(wsname);

	if ((*encoding)->value.lval != SML_XML && (*encoding)->value.lval != SML_WBXML) {
		php_error(E_ERROR, "sml_init_instance: unknown encoding specified.");
	}

        options.encoding = (*encoding)->value.lval;	// XML==2, WBXML==1
        options.workspaceSize = (*wssize)->value.lval;
        options.workspaceName = (*wsname)->value.str.val; 

        callbacks.startMessageFunc = StartMessage;
        callbacks.endMessageFunc = EndMessage;
        callbacks.startSyncFunc = StartSync;
        callbacks.endSyncFunc = EndSync;
        callbacks.startAtomicFunc = StartAtomic;
        callbacks.endAtomicFunc = EndAtomic;
        callbacks.startSequenceFunc = StartSequence;
        callbacks.endSequenceFunc = EndSequence;
        callbacks.addCmdFunc = AddCmd;
        callbacks.alertCmdFunc = AlertCmd;
        callbacks.deleteCmdFunc = DeleteCmd;
        callbacks.getCmdFunc = GetCmd;
        callbacks.putCmdFunc = PutCmd;
        callbacks.mapCmdFunc = MapCmd;
        callbacks.resultsCmdFunc = ResultsCmd;
        callbacks.statusCmdFunc = StatusCmd;
        callbacks.replaceCmdFunc = ReplaceCmd;
        callbacks.copyCmdFunc = CopyCmd;
        callbacks.execCmdFunc = ExecCmd;
        callbacks.searchCmdFunc = SearchCmd;
        callbacks.handleErrorFunc = HandleError;
        callbacks.transmitChunkFunc = TransmitChunk;

        ret = smlInitInstance(&callbacks, &options, function_table, &id);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlInitInstance failed. Reason: 0x%x.", ret);
        }

	//zend_printf("InitInstance called, id 0x%x\n", id);

	resource->id = id;

	RETURN_RESOURCE(zend_list_insert(resource, le_syncml));
}       
/* }}} */

PHP_FUNCTION(sml_set_syncml_options) {
	php_error(E_WARNING, "sml_set_syncml_options: not yet implemented");
}

/* {{{ */
PHP_FUNCTION(sml_set_callbacks) {
        php_error(E_WARNING, "sml_set_syncml_options: not yet implemented");
}
/* }}} */

PHP_FUNCTION(sml_set_encoding) {
	php_error(E_WARNING, "sml_set_encoding: not yet implemented");
}
PHP_FUNCTION(sml_set_user_data) {
	php_error(E_WARNING, "sml_set_user_data: not yet implemented");
}

PHP_FUNCTION(sml_free_proto_element) {
	php_error(E_WARNING, "sml_free_proto_element: not yet implemented (and never will be)");
}

/* {{{ proto int sml_write_in_buffer(int id, string data) */
PHP_FUNCTION(sml_write_in_buffer) {
        zval **id, **data;

        MemPtr_t writePosition;
        MemSize_t freeSize, writtenBytes;
	Ret_t ret;

        if (ZEND_NUM_ARGS() != 2 || zend_get_parameters_ex(2, &id, &data) == FAILURE) {
                        WRONG_PARAM_COUNT;
        }

        convert_to_long_ex(id);
	convert_to_string_ex(data);

        ret = smlLockWriteBuffer((InstanceID_t)((*id)->value.lval), &writePosition, &freeSize);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlLockWriteBuffer failed. Reason: 0x%x.", ret);
        }

	if ((*data)->value.str.len > freeSize) {
		writtenBytes = freeSize;
	}
	else {
		writtenBytes = (*data)->value.str.len;
	}

	RETVAL_LONG(writtenBytes);

        memcpy(writePosition, (*data)->value.str.val, writtenBytes); 

        ret = smlUnlockWriteBuffer((InstanceID_t)((*id)->value.lval), writtenBytes);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlUnlockReadBuffer failed. Reason: 0x%x.", ret);
        }
}
/* }}} */

/* {{{ proto string sml_read_out_buffer(int id) */
PHP_FUNCTION(sml_read_out_buffer) {
        zval **id, **data;

        Ret_t ret;
	MemPtr_t readPosition;
	MemSize_t usedSize;

        if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &id) == FAILURE) {
                        WRONG_PARAM_COUNT;
        }

        convert_to_long_ex(id);

        ret = smlLockReadBuffer((InstanceID_t)((*id)->value.lval), &readPosition, &usedSize);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlLockReadBuffer failed. Reason: 0x%x.", ret);
        }

        RETVAL_STRINGL(readPosition, usedSize, 1)

        ret = smlUnlockReadBuffer((InstanceID_t)((*id)->value.lval), usedSize);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlUnlockReadBuffer failed. Reason: 0x%x.", ret);
        }
}
/* }}} */

/* {{{ proto int sml_get_free_buffer(int id) */
PHP_FUNCTION(sml_get_free_buffer) {
	zval **id;

        Ret_t ret;

        if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &id) == FAILURE) {
                        WRONG_PARAM_COUNT;
        }

        convert_to_long_ex(id);

        RETURN_LONG(smlGetFreeBuffer((InstanceID_t)((*id)->value.lval)));
}
/* }}} */

/* Command Builder API */

/* functions to get object properties */

zval **get_property(zval **object, char *propname) {
        zval **prop;

        if (zend_hash_find(HASH_OF(*object), propname, strlen(propname)+1, (void **)&prop)==FAILURE) {
                php_error(E_ERROR, "Unable to find the %s property.", propname);
                return NULL;
        }
        return prop;
}

void make_sml_cred(zval **object, SmlCredPtr_t cred) {
        zval **meta, **data;

        meta = get_property(object, "meta");
        convert_to_string_ex(meta);
        cred->meta = smlString2Pcdata((*meta)->value.str.val);
        cred->meta->contentType = SML_PCDATA_STRING;
        data = get_property(object, "data");
        convert_to_string_ex(data);
        cred->data = smlString2Pcdata((*data)->value.str.val);
        cred->data->contentType = SML_PCDATA_STRING;
}

void make_sml_source_or_target(zval **object, SmlTargetPtr_t target) {
        zval **targeturi, **targetname;

        targeturi = get_property(object, "locURI");
        convert_to_string_ex(targeturi);
        target->locURI = smlString2Pcdata((*targeturi)->value.str.val);
        target->locURI->contentType = SML_PCDATA_STRING;
        targetname = get_property(object, "locName");
        convert_to_string_ex(targetname);
        target->locName = smlString2Pcdata((*targetname)->value.str.val);
        target->locName->contentType = SML_PCDATA_STRING;
}

/* {{{ proto void sml_start_message(int id, sml_sync_hdr sync) */
PHP_FUNCTION(sml_start_message) {
        zval **id, **syncHdr, **version, **proto, **sessionID, **msgID, **flags, **respURI, **meta;

        SmlSyncHdr_t smlSyncHdr;
        SmlCred_t cred;
        SmlTarget_t target;
        SmlSource_t source;
        Ret_t ret;

        if (ZEND_NUM_ARGS() != 2 || zend_get_parameters_ex(2, &id, &syncHdr) == FAILURE) {
			WRONG_PARAM_COUNT;
        }

        convert_to_long_ex(id);

        smlSyncHdr.elementType = SML_PE_HEADER;

	GET_STR_PROP(syncHdr, version, smlSyncHdr);
	GET_STR_PROP(syncHdr, proto, smlSyncHdr);
	GET_STR_PROP(syncHdr, sessionID, smlSyncHdr);
	GET_STR_PROP(syncHdr, msgID, smlSyncHdr);
	GET_LONG_PROP(syncHdr, flags, smlSyncHdr);

        make_sml_source_or_target(get_property(syncHdr, "target"), &target);
        smlSyncHdr.target = &target;
        make_sml_source_or_target(get_property(syncHdr, "source"), &source);
        smlSyncHdr.source = &source;

	GET_STR_PROP(syncHdr, respURI, smlSyncHdr);

        make_sml_cred(get_property(syncHdr, "cred"), &cred);
        smlSyncHdr.cred = &cred;

	GET_STR_PROP(syncHdr, meta, smlSyncHdr);

        ret = smlStartMessage((InstanceID_t)((*id)->value.lval), &smlSyncHdr);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlStartMessage failed. Reason: 0x%x.", ret);
        }
}
/* }}} */

/* {{{ proto void sml_end_message(int id, bool final) */
PHP_FUNCTION(sml_end_message) {
        zval **id, **final;

        Ret_t ret;

        if (ZEND_NUM_ARGS() != 2 || zend_get_parameters_ex(2, &id, &final) == FAILURE) {
                        WRONG_PARAM_COUNT;
        }

        convert_to_long_ex(id);
	convert_to_boolean_ex(final);

        ret = smlEndMessage((InstanceID_t)((*id)->value.lval), (Boolean_t)((*final)->value.lval));
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlEndMessage failed. Reason: 0x%x.", ret);
        }
}
/* }}} */

/* {{{ proto void sml_start_sync(int id, sml_sync sync) */
PHP_FUNCTION(sml_start_sync) { // if you call those functions after sml_end_message, you get a segfault! BAD!
        zval **id, **sync, **cmdID, **flags, **meta;

        SmlSync_t startSync;
	SmlCred_t cred;
	SmlTarget_t target;
	SmlSource_t source;
        Ret_t ret;

        if (ZEND_NUM_ARGS() != 2 || zend_get_parameters_ex(2, &id, &sync) == FAILURE) {
                        WRONG_PARAM_COUNT;
        }

        convert_to_long_ex(id);

        startSync.elementType = SML_PE_SYNC_START;

	GET_STR_PROP(sync, cmdID, startSync);
	GET_LONG_PROP(sync, flags, startSync);
	GET_STR_PROP(sync, meta, startSync);

	make_sml_cred(get_property(sync, "cred"), &cred);	
	startSync.cred = &cred;

	make_sml_source_or_target(get_property(sync, "target"), &target);
	startSync.target = &target;
	make_sml_source_or_target(get_property(sync, "source"), &source);
	startSync.source = &source;

        ret = smlStartSync((InstanceID_t)((*id)->value.lval), &startSync);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlStartSync failed. Reason: 0x%x.", ret);
        }
}
/* }}} */

/* {{{ proto void sml_end_sync(int id) */
PHP_FUNCTION(sml_end_sync) {
        zval **id;

        Ret_t ret;

        if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &id) == FAILURE) {
                        WRONG_PARAM_COUNT;
        }

        convert_to_long_ex(id);

        ret = smlEndSync((InstanceID_t)((*id)->value.lval));
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlEndSync failed. Reason: 0x%x.", ret);
        }
}
/* }}} */

/* {{{ proto void sml_start_atomic(int id, string cmdID, int flags, */
PHP_FUNCTION(sml_start_atomic) {
	php_error(E_WARNING, "sml_StartAtomic: not yet implemented");
}
/* }}} */

PHP_FUNCTION(sml_end_atomic) {
	php_error(E_WARNING, "sml_EndAtomic: not yet implemented");
}
PHP_FUNCTION(sml_start_sequence) {
	php_error(E_WARNING, "sml_StartSequence: not yet implemented");
}
PHP_FUNCTION(sml_end_sequence) {
	php_error(E_WARNING, "sml_EndSequence: not yet implemented");
}

/*
void make_sml_item(SmlItemPtr_t pItem, zval **object) {
	SmlTarget_t target;
	SmlSource_t source;

	//allocate space for item

	if ((pItem = (SmlItemPtr_t)emalloc(sizeof(SmlItem_t))) == NULL) {
		php_error(E_ERROR, "Out of mem.");
	}

        // fill up with zeros
        memset(pItem, 0, sizeof(SmlItem_t));

	make_sml_source_or_target(get_property(object, "target"), pItem->target));
	make_sml_source_or_target(get_property(object, "source"), (SmlTargetPtr_t)&(pItem->target));
	
	GET_STR_PROP(object, meta, (*pItem));
	GET_STR_PROP(object, data, (*item));

}
*/

/*
void append_sml_item_list(SmlItemListPtr_t *ppItemList, zval **object) {
        SmlItemListPtr_t pNewItemList, pItemList;

        SmlTarget_t target;
        SmlSource_t source;

	// move to end of list
	pItemList = *ppItemList;
	if (pItemList != NULL) {
		while (pItemList->next != NULL) {
			pItemList = pItemList->next;
		}
	}

	// allocate mem for new item
	if ((pNewItemList = (SmlItemListPtr_t)emalloc(sizeof(SmlItemList_t))) == NULL ) {
		php_error(E_ERROR, "Out of mem.");	// i dont think this will happen
	}

	// fill up with zeros
	memset(pNewItemList, 0, sizeof(SmlItemList_t));

	make_sml_item(pNewItemList->item, object);
		
	if (pItemList == NULL) { // create new list
		*ppItemList = pNewItemList;
	}
	else {	// last item
		pItemList->next = pNewItemList;
	}
}
*/

/*
void make_sml_item_list(zval **object, SmlItemListPtr_t list) {
	zval **entry;

	if ((*object)->type != IS_ARRAY) {
		php_error(E_ERROR, "Argument is not an array. \n");
	}
	else {
		zend_hash_internal_pointer_reset(HASH_OF(*object));
		while(zend_hash_get_current_data(HASH_OF(*object), (void **)&entry)!=FAILURE) {
			append_sml_item_list(&list, entry);
		}
	}
}
*/

/*
void make_sml_item_list(zval **object, SmlItemListPtr_t list) {
        zval **entry, **meta, **data;

	int i=0;
	SmlTarget_t t[100];
	SmlSource_t s[100];
	SmlItemList_t lists[100];
	SmlItem_t items[100];

        if ((*object)->type != IS_ARRAY) {
                php_error(E_ERROR, "Argument is not an array. \n");
        }
        else {
                zend_hash_internal_pointer_reset(HASH_OF(*object));
                while(zend_hash_get_current_data(HASH_OF(*object), (void **)&entry)!=FAILURE) {

			make_sml_source_or_target(get_property(entry, "target"), &(t[i]));
			make_sml_source_or_target(get_property(entry, "source"), &(s[i]));

			GET_STR_PROP(entry, meta, items[i]);
			GET_STR_PROP(entry, data, items[i]);

			items[i].target = &(t[i]);
			items[i].source = &(s[i]);
			lists[i].item = &(items[i]);
			lists[i].next = &(lists[i+1]);

			zend_printf("going thru array... 0x%x \n", i);
			zend_hash_move_forward(HASH_OF(*object));
			i++;
                }
		if (i==0) {
			php_error(E_ERROR, "No item specified.");
		}
		lists[i-1].next=NULL;
        }
}
*/

/* {{{ proto void sml_add_cmd(int id, sml_generic gen) */
PHP_FUNCTION(sml_add_cmd) {
        zval **id, **gen, **cmdID, **flags, **meta, **itemlist;

        SmlGenericCmd_t smlGen;
        SmlCred_t cred;
        SmlItemList_t list1;
	SmlItem_t item1;
        Ret_t ret;

	zval **entry, **object, **data;
        int i=0;
        SmlTarget_t t[100];
        SmlSource_t s[100];
        SmlItemList_t lists[100];
        SmlItem_t items[100];

        if (ZEND_NUM_ARGS() != 2 || zend_get_parameters_ex(2, &id, &gen) == FAILURE) {
                        WRONG_PARAM_COUNT;
        }
        convert_to_long_ex(id);

	GET_STR_PROP(gen, cmdID, smlGen);
	GET_LONG_PROP(gen, flags, smlGen);

        make_sml_cred(get_property(gen, "cred"), &cred);
        smlGen.cred = &cred;

	GET_STR_PROP(gen, meta, smlGen);

	itemlist = get_property(gen, "itemList");

	object=itemlist;
        if ((*object)->type != IS_ARRAY) {
                php_error(E_ERROR, "Argument is not an array. \n");
        }
        else {
                zend_hash_internal_pointer_reset(HASH_OF(*object));
                while(zend_hash_get_current_data(HASH_OF(*object), (void **)&entry)!=FAILURE) {

                        make_sml_source_or_target(get_property(entry, "target"), &(t[i]));
                        make_sml_source_or_target(get_property(entry, "source"), &(s[i]));

                        GET_STR_PROP(entry, meta, items[i]);
                        GET_STR_PROP(entry, data, items[i]);

                        items[i].target = &(t[i]);
                        items[i].source = &(s[i]);
                        lists[i].item = &(items[i]);
                        lists[i].next = &(lists[i+1]);

                        zend_printf("going thru array... 0x%x \n", i);
                        zend_hash_move_forward(HASH_OF(*object));
                        i++;
                }
                if (i==0) {
                        php_error(E_ERROR, "No item specified.");
                }
                lists[i-1].next=NULL;
        }

/*
        item1.target = NULL;
        item1.source = NULL;
        item1.meta = NULL;
        item1.data = smlString2Pcdata("this a whole lot of data. :-)");
        item1.data->contentType = SML_PCDATA_STRING;
        smlGen.itemList = &list1;
        smlGen.itemList->next = NULL;
        smlGen.itemList->item = &item1;
*/

	smlGen.itemList = &(lists[0]);

        ret = smlAddCmd((InstanceID_t)((*id)->value.lval), &smlGen);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlAddCmd failed. Reason: 0x%x.", ret);
        }
}
/* }}} */

PHP_FUNCTION(sml_alert_cmd) {
	php_error(E_WARNING, "smlAlertCmd: not yet implemented");
}
PHP_FUNCTION(sml_copy_cmd) {
	php_error(E_WARNING, "smlCopyCmd: not yet implemented");
}
PHP_FUNCTION(sml_delete_cmd) { 
	php_error(E_WARNING, "smlDeleteCmd: not yet implemented");
}
PHP_FUNCTION(sml_exec_cmd) {
	php_error(E_WARNING, "smlExecCmd: not yet implemented");
}
PHP_FUNCTION(sml_get_cmd) {
	php_error(E_WARNING, "smlGetCmd: not yet implemented");
}
PHP_FUNCTION(sml_put_cmd) {
	php_error(E_WARNING, "smlPutCmd: not yet implemented");
}
PHP_FUNCTION(sml_map_cmd) {
	php_error(E_WARNING, "smlMapCmd: not yet implemented");
}
PHP_FUNCTION(sml_results_cmd) {
	php_error(E_WARNING, "smlResultsCmd: not yet implemented");
}
PHP_FUNCTION(sml_search_cmd) {
	php_error(E_WARNING, "smlSearchCmd: not yet implemented");
}
PHP_FUNCTION(sml_status_cmd) {
	php_error(E_WARNING, "smlStatusCmd: not yet implemented");
}
PHP_FUNCTION(sml_replace_cmd) {
	php_error(E_WARNING, "smlReplaceCmd: not yet implemented");
}

/* Command Dispatcher API */
/* {{{ proto void sml_process_data(int id, int mode) */
PHP_FUNCTION(sml_process_data) {
        zval **id;

        Ret_t ret;

        if (ZEND_NUM_ARGS() != 1 || zend_get_parameters_ex(1, &id) == FAILURE) {
                        WRONG_PARAM_COUNT;
        }
        convert_to_long_ex(id);

        ret = smlProcessData((InstanceID_t)((*id)->value.lval), SML_ALL_COMMANDS);
        if (ret!=SML_ERR_OK) {
                php_error(E_ERROR, "smlProcessData failed. Reason: 0x%x.", ret);
        }
}
/* }}} */

/* sml printfunc callback */
void PrintFunc(String_t outputString) {
        zval *data=NULL;

        //zend_printf("%s", outputString);

        MAKE_STD_ZVAL(data);
        ZVAL_STRING(data, outputString, 1);

        CallHandler("cb_end_message", NULL, data);
}
