/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment GStreamer image sink
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmimagesink
 * @short_description: A GStreamer image sink outputing frames to image
 * drawables.
 * @see_also: #PgmImage
 *
 * #PgmImageSink is a GStreamer image sink element allowing to build GStreamer
 * pipelines outputing frames directly to #PgmImage.
 *
 * You can explicitly use the pgm_image_sink_new() function to instanciate a
 * new sink, but you can alos use the standard GStreamer functions using
 * gst_element_factory_make() or gst_parse_launch(), the factory name of the
 * sink being "pgmimagesink".
 *
 * Internally, the sink is simply outputing GStreamer buffers (#GstBuffer) to
 * images (#PgmImage) using the pgm_image_set_from_gst_buffer() function.
 *
 * Last reviewed on 2008-01-28 (0.3.4)
 */

#if HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <gst/video/video.h> /* Helper functions */
#include <string.h>
#include "pgmimagesink.h"
#include "pgmcanvas.h"
#include "pgmenumtypes.h"

static GstElementDetails pgm_image_sink_details =
    GST_ELEMENT_DETAILS ("Video Sink", "Sink/Video",
        "Video output to PgmImage object",
        "Loïc Molinari <loic@fluendo.com>, "
        "Julien Moutte <julien@fluendo.com>");

static GstStaticPadTemplate pgm_image_sink_template =
    GST_STATIC_PAD_TEMPLATE ("sink", GST_PAD_SINK, GST_PAD_ALWAYS,
        GST_STATIC_CAPS ("video/x-raw-rgb, "
            "framerate = (fraction) [ 0, MAX ], "
            "width = (int) [ 1, MAX ], "
            "height = (int) [ 1, MAX ]; "
            "video/x-raw-yuv, "
            "framerate = (fraction) [ 0, MAX ], "
            "width = (int) [ 1, MAX ], "
            "height = (int) [ 1, MAX ]"));

GST_DEBUG_CATEGORY_STATIC (pgm_image_sink_debug);
#define GST_CAT_DEFAULT pgm_image_sink_debug

enum {
  ARG_0,
  ARG_IMAGE,
  ARG_EVENTS
};

static GstVideoSinkClass *parent_class = NULL;

/* Private methods */

/* Called with LOCK */
static void
update_caps_from_image (PgmImageSink *sink)
{
  PgmCanvas *canvas = NULL;
  GstCaps *caps = NULL;
  gulong pixel_formats = 0;
  PgmImagePixelFormat format;
  guint i;

  /* Unref last caps */
  if (G_UNLIKELY (sink->our_caps))
    gst_caps_unref (sink->our_caps);

  /* And init our caps to EMPTY */
  sink->our_caps = gst_caps_new_empty ();

  /* Get the supported pixel formats */
  canvas = PGM_CANVAS (GST_OBJECT_PARENT (sink->image));
  if (canvas)
    pgm_canvas_get_pixel_formats (canvas, &pixel_formats);

  /* Parse the pixel formats mask */
  for (i = 0; i < PGM_IMAGE_NB_PIXEL_FORMATS; i++)
    {
      format = pixel_formats & (1 << i);

      if (format == 0)
        continue;

      /* Create a caps of the format available */
      switch (format)
        {
        case PGM_IMAGE_RGB:
          caps = gst_caps_new_simple
            ("video/x-raw-rgb",
             "bpp", G_TYPE_INT, 24, "depth", G_TYPE_INT, 24,
             "red_mask",   G_TYPE_INT, 0x00ff0000,
             "green_mask", G_TYPE_INT, 0x0000ff00,
             "blue_mask",  G_TYPE_INT, 0x000000ff,
             "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, G_MAXINT, 1,
             NULL);
          break;

        case PGM_IMAGE_BGR:
          caps = gst_caps_new_simple
            ("video/x-raw-rgb",
             "bpp", G_TYPE_INT, 24, "depth", G_TYPE_INT, 24,
             "red_mask",   G_TYPE_INT, 0x000000ff,
             "green_mask", G_TYPE_INT, 0x0000ff00,
             "blue_mask",  G_TYPE_INT, 0x00ff0000,
             "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, G_MAXINT, 1,
             NULL);
          break;

        case PGM_IMAGE_RGBA:
          caps = gst_caps_new_simple
            ("video/x-raw-rgb",
             "bpp", G_TYPE_INT, 32, "depth", G_TYPE_INT, 32,
             "red_mask",   G_TYPE_INT, 0xff000000,
             "green_mask", G_TYPE_INT, 0x00ff0000,
             "blue_mask",  G_TYPE_INT, 0x0000ff00,
             "alpha_mask", G_TYPE_INT, 0x000000ff,
             "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, G_MAXINT, 1,
             NULL);
          break;

        case PGM_IMAGE_BGRA:
          caps = gst_caps_new_simple
            ("video/x-raw-rgb",
             "bpp", G_TYPE_INT, 32, "depth", G_TYPE_INT, 32,
             "red_mask",   G_TYPE_INT, 0x0000ff00,
             "green_mask", G_TYPE_INT, 0x00ff0000,
             "blue_mask",  G_TYPE_INT, 0xff000000,
             "alpha_mask", G_TYPE_INT, 0x000000ff,
             "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, G_MAXINT, 1,
             NULL);
          break;

        case PGM_IMAGE_I420:
          caps = gst_caps_new_simple
            ("video/x-raw-yuv",
             "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('I', '4', '2', '0'),
             "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, G_MAXINT, 1,
             NULL);
          break;

        case PGM_IMAGE_YV12:
          caps = gst_caps_new_simple
            ("video/x-raw-yuv",
             "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('Y', 'V', '1', '2'),
             "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, G_MAXINT, 1,
             NULL);
          break;

        case PGM_IMAGE_UYVY:
          caps = gst_caps_new_simple
            ("video/x-raw-yuv",
             "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y'),
             "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, G_MAXINT, 1,
             NULL);
          break;

        case PGM_IMAGE_YUYV:
          caps = gst_caps_new_simple
            ("video/x-raw-yuv",
             "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('Y', 'U', 'Y', 'V'),
             "width",  GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "height", GST_TYPE_INT_RANGE, 1, G_MAXINT,
             "framerate", GST_TYPE_FRACTION_RANGE, 0, 1, G_MAXINT, 1,
             NULL);
          break;

        default:
          break;
        }

      /* And append it to our caps */
      gst_caps_append (sink->our_caps, caps);
    }
}

static gboolean
update_format_from_caps (PgmImageSink *sink,
                         GstStructure *s)
{
  gboolean ret = FALSE;

  /* RGB format */
  if (gst_structure_has_name (s, "video/x-raw-rgb"))
    {
      gint depth, bpp, red_mask, green_mask, blue_mask, alpha_mask;

      gst_structure_get_int (s, "bpp", &bpp);
      gst_structure_get_int (s, "depth", &depth);
      gst_structure_get_int (s, "red_mask", &red_mask);
      gst_structure_get_int (s, "green_mask", &green_mask);
      gst_structure_get_int (s, "blue_mask", &blue_mask);

      switch (depth)
        {
          /* RGB or BGR */
        case 24:
          if (red_mask      == 0x00ff0000
              && green_mask == 0x0000ff00
              && blue_mask  == 0x000000ff)
            {
              sink->format = PGM_IMAGE_RGB;
            }
          else if (red_mask      == 0x000000ff
                   && green_mask == 0x0000ff00
                   && blue_mask  == 0x00ff0000)
            {
              sink->format = PGM_IMAGE_BGR;
            }
          else
            {
              GST_WARNING_OBJECT (sink, "unsupported RGB 24 bits masks");
            }
          break;

          /* RGBA or BGRA */
        case 32:
          gst_structure_get_int (s, "alpha_mask", &alpha_mask);
          if (red_mask      == 0xff000000
              && green_mask == 0x00ff0000
              && blue_mask  == 0x0000ff00)
            {
              sink->format = PGM_IMAGE_RGBA;
            }
          else if (red_mask      == 0x0000ff00
                   && green_mask == 0x00ff0000
                   && blue_mask  == 0xff000000)
            {
              sink->format = PGM_IMAGE_BGRA;
            }
          else
            {
              GST_WARNING_OBJECT (sink, "unsupported RGB 32 bits masks");
              goto beach;
            }
          break;
        }
    }

  /* YUV format */
  else if (gst_structure_has_name (s, "video/x-raw-yuv"))
    {
      guint32 fourcc;

      gst_structure_get_fourcc (s, "format", &fourcc);

      switch (fourcc)
        {
        case GST_MAKE_FOURCC ('I', '4', '2', '0'):
        case GST_MAKE_FOURCC ('I', 'Y', 'U', 'V'):
          sink->format = PGM_IMAGE_I420;
          break;

        case GST_MAKE_FOURCC ('Y', 'V', '1', '2'):
          sink->format = PGM_IMAGE_YV12;
          break;

        case GST_MAKE_FOURCC ('Y', 'U', 'Y', 'V'):
        case GST_MAKE_FOURCC ('Y', 'U', 'Y', '2'):
        case GST_MAKE_FOURCC ('V', '4', '2', '2'):
        case GST_MAKE_FOURCC ('Y', 'U', 'N', 'V'):
          sink->format = PGM_IMAGE_YUYV;
          break;

        case GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y'):
        case GST_MAKE_FOURCC ('Y', '4', '2', '2'):
        case GST_MAKE_FOURCC ('U', 'Y', 'N', 'V'):
          sink->format = PGM_IMAGE_UYVY;
          break;

        default:
          GST_WARNING_OBJECT (sink,
                              "unsupported YUV format %" GST_FOURCC_FORMAT,
                              GST_FOURCC_ARGS (fourcc));
          goto beach;
        }
    }

  /* Unsupported format */
  else
    {
      GST_WARNING_OBJECT (sink, "unsupported caps name %s",
                          gst_structure_get_name (s));
      goto beach;
    }

  ret = TRUE;

 beach:
  return ret;
}

static gboolean
push_navigation_event (PgmImageSink *image_sink,
                       const gchar *event_name,
                       gfloat x,
                       gfloat y,
                       gfloat z,
                       PgmButtonType button)
{
  gboolean ret = FALSE;
  GstStructure *structure;
  GstEvent *event;
  GstPad *pad;
  gfloat x_drawable, y_drawable;
  gint x_image, y_image;
  guint gst_button;

  /* Convert from canvas space to image space */
  GST_OBJECT_LOCK (image_sink);
  pgm_drawable_from_canvas (PGM_DRAWABLE (image_sink->image),
                            &x_drawable, &y_drawable, x, y, z);
  pgm_image_from_drawable (image_sink->image, &x_image, &y_image,
                           x_drawable, y_drawable);
  GST_OBJECT_UNLOCK (image_sink);

  /* Convert from Pigment button to GStreamer button */
  switch (button)
    {
    case PGM_BUTTON_LEFT:
      gst_button = 1;
    case PGM_BUTTON_MIDDLE:
      gst_button = 2;
    case PGM_BUTTON_RIGHT:
      gst_button = 3;
    default:
      gst_button = 0;
    }

  /* Build and push the navigation event upstream */
  if (gst_button != 0)
    structure = gst_structure_new ("application/x-gst-navigation",
                                   "event",     G_TYPE_STRING, event_name,
                                   "pointer_x", G_TYPE_DOUBLE, (gdouble) x_image,
                                   "pointer_y", G_TYPE_DOUBLE, (gdouble) y_image,
                                   "button",    G_TYPE_INT,    gst_button,
                                   NULL);
  else
    structure = gst_structure_new ("application/x-gst-navigation",
                                   "event",     G_TYPE_STRING, event_name,
                                   "pointer_x", G_TYPE_DOUBLE, (gdouble) x_image,
                                   "pointer_y", G_TYPE_DOUBLE, (gdouble) y_image,
                                   NULL);
  event = gst_event_new_navigation (structure);
  pad = gst_element_get_pad (GST_ELEMENT (image_sink), "sink");
  ret = gst_pad_push_event (pad, event);

  return ret;
}

static gboolean
drawable_motion_cb (PgmDrawable *drawable,
                    gfloat x,
                    gfloat y,
                    gfloat z,
                    guint time,
                    gpointer user_data)
{
  PgmImageSink *image_sink = PGM_IMAGE_SINK (user_data);

  return push_navigation_event (image_sink, "mouse-move", x, y, z, 0);
}

static gboolean
drawable_pressed_cb (PgmDrawable  *drawable,
                     gfloat x,
                     gfloat y,
                     gfloat z,
                     PgmButtonType button,
                     guint time,
                     guint pressure,
                     gpointer user_data)
{
  PgmImageSink *image_sink = PGM_IMAGE_SINK (user_data);

  return push_navigation_event (image_sink, "mouse-button-press",
                                x, y, z, button);
}

static gboolean
drawable_released_cb (PgmDrawable  *drawable,
                      gfloat x,
                      gfloat y,
                      gfloat z,
                      PgmButtonType button,
                      guint time,
                      gpointer user_data)
{
  PgmImageSink *image_sink = PGM_IMAGE_SINK (user_data);

  return push_navigation_event (image_sink, "mouse-button-release",
                                x, y, z, button);
}

/* Called with LOCK */
static void
update_event_mask (PgmImageSink *sink)
{
  if (NULL == sink->image)
    return;

  /* Update connection to the motion events */
  if ((sink->events & PGM_IMAGE_SINK_MOTION) && !sink->motion_handler)
    sink->motion_handler = g_signal_connect (sink->image, "motion",
                                             G_CALLBACK (drawable_motion_cb),
                                             sink);
  else if (!(sink->events & PGM_IMAGE_SINK_MOTION) && sink->motion_handler)
    {
      g_signal_handler_disconnect (sink->image, sink->motion_handler);
      sink->motion_handler = 0;
    }

  /* Update connection to the pressed events */
  if ((sink->events & PGM_IMAGE_SINK_PRESSED) && !sink->pressed_handler)
    sink->pressed_handler = g_signal_connect (sink->image, "pressed",
                                              G_CALLBACK (drawable_pressed_cb),
                                              sink);
  else if (!(sink->events & PGM_IMAGE_SINK_PRESSED) && sink->pressed_handler)
    {
      g_signal_handler_disconnect (sink->image, sink->pressed_handler);
      sink->pressed_handler = 0;
    }

  /* Update connection to the released events */
  if ((sink->events & PGM_IMAGE_SINK_RELEASED) && !sink->released_handler)
    sink->released_handler = g_signal_connect (sink->image, "released",
                                               G_CALLBACK (drawable_released_cb),
                                               sink);
  else if (!(sink->events & PGM_IMAGE_SINK_RELEASED)
           && sink->released_handler)
    {
      g_signal_handler_disconnect (sink->image, sink->released_handler);
      sink->released_handler = 0;
    }
}

/* GstBaseSink methods */

static void
pgm_image_sink_get_times (GstBaseSink *bsink,
                          GstBuffer *buf,
                          GstClockTime *start,
                          GstClockTime *end)
{
  PgmImageSink *sink = PGM_IMAGE_SINK (bsink);

  if (GST_BUFFER_TIMESTAMP_IS_VALID (buf))
    {
      *start = GST_BUFFER_TIMESTAMP (buf);

      if (GST_BUFFER_DURATION_IS_VALID (buf))
        *end = *start + GST_BUFFER_DURATION (buf);
      else if (sink->framerate_n > 0)
        *end = *start + gst_util_uint64_scale_int (GST_SECOND,
                                                   sink->framerate_d,
                                                   sink->framerate_n);
    }
}

static GstFlowReturn
pgm_image_sink_show (GstBaseSink *bsink,
                     GstBuffer *buf)
{
  PgmImageSink *sink = PGM_IMAGE_SINK (bsink);

  GST_OBJECT_LOCK (sink);

  if (sink->image && buf)
    {
      /* Send the GstBuffer */
      pgm_image_set_from_gst_buffer (sink->image, sink->format, sink->width,
                                     sink->height, sink->stride, buf);
      /* And specify the pixel-aspect-ratio */
      pgm_image_set_aspect_ratio (sink->image, (guint) ((gfloat) sink->width *
                                  ((gfloat) sink->par_n / (gfloat) sink->par_d)),
                                  (guint) sink->height);
    }

  GST_OBJECT_UNLOCK (sink);

  return GST_FLOW_OK;
}

static GstCaps *
pgm_image_sink_getcaps (GstBaseSink *bsink)
{
  PgmImageSink *sink = PGM_IMAGE_SINK (bsink);
  GstCaps *caps = NULL;

  GST_OBJECT_LOCK (sink);

  /* If we don't have any PgmImage yet, return the template */
  if (G_UNLIKELY (!sink->image))
    caps = gst_caps_copy (gst_pad_get_pad_template_caps
                          (GST_VIDEO_SINK_PAD (sink)));

  /* Else we update the capacities of the image's canvas if any and return it */
  else
    {
      update_caps_from_image (sink);
      caps = gst_caps_ref (sink->our_caps);
    }

  GST_OBJECT_UNLOCK (sink);

  return caps;
}

static gboolean
pgm_image_sink_setcaps (GstBaseSink *bsink,
                        GstCaps *caps)
{
  PgmImageSink *sink = PGM_IMAGE_SINK (bsink);
  GstStructure *structure = NULL;
  GstCaps *intersection = NULL;
  GstCaps *our_caps = NULL;
  gboolean ret = FALSE;

  GST_DEBUG_OBJECT (sink, "setcaps called with %" GST_PTR_FORMAT, caps);

  GST_OBJECT_LOCK (sink);

  /* Try intersecting this caps with ours or with pad template if N/A */
  if (sink->our_caps)
    our_caps = sink->our_caps;
  else
    /* That gives us no ref */
    our_caps = gst_static_pad_template_get_caps (&pgm_image_sink_template);

  GST_OBJECT_UNLOCK (sink);

  intersection = gst_caps_intersect (our_caps, caps);
  GST_DEBUG_OBJECT (sink, "intersection returned %" GST_PTR_FORMAT,
                    intersection);

  if (gst_caps_is_empty (intersection))
    goto beach;

  /* Now configure ourself with those caps */
  structure = gst_caps_get_structure (caps, 0);

  /* Fill the pixel-aspect-ratio if any */
  ret = gst_structure_get_fraction (structure, "pixel-aspect-ratio",
                                    &sink->par_n, &sink->par_d);
  if (!ret)
    {
      sink->par_n = 1;
      sink->par_d = 1;
    }

  /* Fill the other necessary properties */
  ret = gst_structure_get_int (structure, "width", &sink->width);
  ret &= gst_structure_get_int (structure, "height", &sink->height);
  ret &= gst_structure_get_fraction (structure, "framerate",
                                     &sink->framerate_n, &sink->framerate_d);

  if (G_UNLIKELY (!ret))
    {
      GST_WARNING_OBJECT (sink, "incomplete caps structure");
      goto beach;
    }

  ret = update_format_from_caps (sink, structure);

 beach:
  return ret;
}

/* GstElement method */

static GstStateChangeReturn
pgm_image_sink_change_state (GstElement *element,
                             GstStateChange transition)
{
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

/*   switch (transition) */
/*     { */
/*     default: */
/*       break; */
/*     } */

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);
  if (ret == GST_STATE_CHANGE_FAILURE)
    return ret;

/*   switch (transition) */
/*     { */
/*     default: */
/*       break; */
/*     } */

  return ret;
}

/* GObject methods */

static void
pgm_image_sink_set_property (GObject *object,
                             guint prop_id,
                             const GValue *value,
                             GParamSpec *pspec)
{
  PgmImageSink *sink;

  g_return_if_fail (PGM_IS_IMAGE_SINK (object));

  sink = PGM_IMAGE_SINK (object);

  switch (prop_id)
    {
    case ARG_IMAGE:
      {
        GST_OBJECT_LOCK (sink);

        /* Unref last image */
        if (G_UNLIKELY (sink->image))
          {
            if (sink->motion_handler)
              g_signal_handler_disconnect (sink->image, sink->motion_handler);
            if (sink->pressed_handler)
              g_signal_handler_disconnect (sink->image, sink->pressed_handler);
            if (sink->released_handler)
              g_signal_handler_disconnect (sink->image, sink->released_handler);
            gst_object_unref (sink->image);
          }

        /* And get the new image */
        sink->image = gst_object_ref (PGM_IMAGE (g_value_get_object (value)));
        update_event_mask (sink);

        GST_OBJECT_UNLOCK (sink);
        break;
      }

    case ARG_EVENTS:
      {
        PgmImageSinkEventMask events = g_value_get_flags (value);

        GST_OBJECT_LOCK (sink);
        if (events != sink->events)
          {
            sink->events = events;
            update_event_mask (sink);
          }
        GST_OBJECT_UNLOCK (sink);
        break;
      }

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
pgm_image_sink_get_property (GObject *object,
                             guint prop_id,
                             GValue *value,
                             GParamSpec *pspec)
{
  PgmImageSink *sink;

  g_return_if_fail (PGM_IS_IMAGE_SINK (object));

  sink = PGM_IMAGE_SINK (object);

  switch (prop_id)
    {
    case ARG_IMAGE:
      GST_OBJECT_LOCK (sink);
      if (sink->image)
        g_value_set_object (value, sink->image);
      GST_OBJECT_UNLOCK (sink);
      break;

    case ARG_EVENTS:
      GST_OBJECT_LOCK (sink);
      g_value_set_flags (value, sink->events);
      GST_OBJECT_UNLOCK (sink);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
pgm_image_sink_dispose (GObject *object)
{
  PgmImageSink *sink = PGM_IMAGE_SINK (object);

  GST_OBJECT_LOCK (sink);

  if (sink->image)
    {
      if (sink->motion_handler)
        {
          g_signal_handler_disconnect (sink->image, sink->motion_handler);
          sink->motion_handler = 0;
        }
      if (sink->pressed_handler)
        {
          g_signal_handler_disconnect (sink->image, sink->pressed_handler);
          sink->pressed_handler = 0;
        }
      if (sink->released_handler)
        {
          g_signal_handler_disconnect (sink->image, sink->released_handler);
          sink->released_handler = 0;
        }
      gst_object_unref (sink->image);
      sink->image = NULL;
    }

  if (sink->our_caps)
    {
      gst_caps_unref (sink->our_caps);
      sink->our_caps = NULL;
    }

  GST_OBJECT_UNLOCK (sink);

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_image_sink_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details (element_class, &pgm_image_sink_details);
  gst_element_class_add_pad_template (element_class,
                                      gst_static_pad_template_get
                                      (&pgm_image_sink_template));
}

static void
pgm_image_sink_class_init (PgmImageSinkClass *klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbasesink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesink_class = (GstBaseSinkClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_image_sink_dispose);
  gobject_class->set_property = GST_DEBUG_FUNCPTR (pgm_image_sink_set_property);
  gobject_class->get_property = GST_DEBUG_FUNCPTR (pgm_image_sink_get_property);

  /**
   * PgmImageSink:image:
   *
   * A #PgmImage on which the sink output video frames.
   */
  g_object_class_install_property
    (gobject_class, ARG_IMAGE, g_param_spec_object
     ("image", "PgmImage", "A PgmImage on which the sink output video frames",
      PGM_TYPE_IMAGE, G_PARAM_READWRITE));
  /**
   * PgmImageSink:events:
   *
   * A #PgmImageSinkEventMask specifying what input events should be sent
   * upstream in the pipeline.
   */
  g_object_class_install_property
    (gobject_class, ARG_EVENTS, g_param_spec_flags
     ("events", "PgmImageSinkEventMask", "A mask specifying what input events "
      "should be sent upstream in the pipeline",
      PGM_TYPE_IMAGE_SINK_EVENT_MASK, 0,
      G_PARAM_READWRITE));

  gstelement_class->change_state =
    GST_DEBUG_FUNCPTR (pgm_image_sink_change_state);

  gstbasesink_class->get_caps = GST_DEBUG_FUNCPTR (pgm_image_sink_getcaps);
  gstbasesink_class->set_caps = GST_DEBUG_FUNCPTR (pgm_image_sink_setcaps);
  gstbasesink_class->get_times = GST_DEBUG_FUNCPTR (pgm_image_sink_get_times);
  gstbasesink_class->preroll = GST_DEBUG_FUNCPTR (pgm_image_sink_show);
  gstbasesink_class->render = GST_DEBUG_FUNCPTR (pgm_image_sink_show);

  GST_DEBUG_CATEGORY_INIT (pgm_image_sink_debug, "pgm_image_sink", 0,
                           "pigment image sink gstreamer element");
}

static void
pgm_image_sink_init (PgmImageSink *sink)
{
  sink->image = NULL;
  sink->our_caps = NULL;
  sink->motion_handler = 0;
  sink->pressed_handler = 0;
  sink->released_handler = 0;
  sink->events = 0;
}

GType
pgm_image_sink_get_type (void)
{
  static GType sink_type = 0;

  if (G_UNLIKELY (!sink_type))
    {
      static const GTypeInfo sink_info = {
        sizeof (PgmImageSinkClass),
        pgm_image_sink_base_init,
        NULL,
        (GClassInitFunc) pgm_image_sink_class_init,
        NULL,
        NULL,
        sizeof (PgmImageSink),
        0,
        (GInstanceInitFunc) pgm_image_sink_init,
        NULL
      };

      sink_type = g_type_register_static (GST_TYPE_VIDEO_SINK, "PgmImageSink",
                                          &sink_info, 0);
    }

  return sink_type;
}

/* PgmImageSink methods */

/**
 * pgm_image_sink_new:
 * @name: the name of the new image sink.
 *
 * Creates a new image sink with the given name.
 *
 * Returns: a new #PgmImageSink.
 */
GstElement*
pgm_image_sink_new (const gchar *name)
{
  return gst_element_factory_make ("pgmimagesink", name);
}

/**
 * pgm_image_sink_set_image:
 * @sink: a #PgmImageSink object.
 * @image: the #PgmImage object.
 *
 * Sets @image as the #PgmImage object which will receive the frames
 * (#GstBuffer) outputed by @sink.
 *
 * MT Safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_sink_set_image (PgmImageSink *sink,
                          PgmImage *image)
{
  g_return_val_if_fail (PGM_IS_IMAGE_SINK (sink), PGM_ERROR_X);
  g_return_val_if_fail (PGM_IS_IMAGE (image), PGM_ERROR_X);

  g_object_set (G_OBJECT (sink), "image", image, NULL);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_sink_get_image:
 * @sink: a #PgmImageSink object.
 * @image: a pointer to a #PgmImage pointer where the image will be stored.
 * Unref after usage.
 *
 * Retrieves the #PgmImage object in @image which @sink is using to output its
 * frames (#GstBuffer).
 *
 * MT Safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_sink_get_image (PgmImageSink *sink,
                          PgmImage **image)
{
  g_return_val_if_fail (PGM_IS_IMAGE_SINK (sink), PGM_ERROR_X);
  g_return_val_if_fail (image != NULL, PGM_ERROR_X);

  *image = gst_object_ref (sink->image);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_sink_set_events:
 * @sink: a #PgmImageSink object.
 * @events: a #PgmImageSinkEventMask.
 *
 * Sets the mouse events interacting with the currently set image that will be
 * sent upstream in the pipeline.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_sink_set_events (PgmImageSink *sink,
                           PgmImageSinkEventMask events)
{
  g_return_val_if_fail (PGM_IS_IMAGE_SINK (sink), PGM_ERROR_X);

  g_object_set (sink, "events", events, NULL);

  return PGM_ERROR_OK;
}

/**
 * pgm_image_sink_get_events:
 * @sink: a #PgmImageSink object.
 * @events: a #PgmImageSinkEventMask pointer where the event mask will be
 * stored.
 *
 * Retrieves in @events the mouse events sent upstream in the pipeline.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_image_sink_get_events (PgmImageSink *sink,
                           PgmImageSinkEventMask *events)
{
  g_return_val_if_fail (PGM_IS_IMAGE_SINK (sink), PGM_ERROR_X);
  g_return_val_if_fail (NULL != events, PGM_ERROR_X);

  g_object_get (sink, "events", events, NULL);

  return PGM_ERROR_OK;
}
