/*

Copyright (C) 2000  Paul Wilkins

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
/* draw_image.c by Paul Wilkins 1/2/2000 */

#include <stdio.h>
#include <math.h>
#include <gtk/gtk.h>
#include <gdk_imlib.h>

#include "globals.h"
#include "render_image.h"
#include "read_db.h"
#include "draw_image.h"


int check_options(struct IMAGE_OPTIONS *im_opt){
   int ret = 1;
   if(im_opt->opt_alg == PIX_SIZE) {
      if(im_opt->pixW < 1){
         printf("Image size X: invalid number\n");
         ret = 0;
      }
      if(im_opt->pixH < 1){
         printf("Image size Y: invalid number\n");
         ret = 0;
      }
   } else {
      if(im_opt->nPixW < 1){
         printf("Image count X: invalid number\n");
         ret = 0;
      }
      if(im_opt->nPixH < 1){
         printf("Image count Y: invalid number\n");
         ret = 0;
      }
   }
   if(-1 == im_opt->proximity){
      printf("Image proximity: invalid number\n");
      ret = 0;
   }

   return ret;
}

int calc_dimensions(struct IMAGE_OPTIONS *im_opt){

   if(0 == check_options(im_opt)){
      printf("error with options\n");
      return 0;
   }

   if(im_opt->opt_alg == PIX_SIZE) {
      if(globals.in_im == NULL){
         im_opt->nPixW = 0;
         im_opt->nPixH = 0;
      } else {
	 im_opt->nPixW = ceil((double)globals.in_im->rgb_width / (double)im_opt->pixW);
	 im_opt->nPixH = ceil((double)globals.in_im->rgb_height / (double)im_opt->pixH);
      }
   } else if (im_opt->opt_alg == PIX_COUNT) {
      if(globals.in_im == NULL){
         im_opt->pixW = 0;
         im_opt->pixH = 0;
      } else {
	 im_opt->pixW = ceil((double)globals.in_im->rgb_width / (double)im_opt->nPixW);
	 im_opt->pixH = ceil((double)globals.in_im->rgb_height / (double)im_opt->nPixH);
      }
   } else {
      fprintf(stderr, "calc_dimensions: I shouldn't get here.\n");
      exit(1);
   }

   im_opt->width = im_opt->nPixW * im_opt->pixW;
   im_opt->height = im_opt->nPixH * im_opt->pixH;

   return 1;
}

int copy_opt_data(){
   int ret = 0;
   if(globals.cur_opt.width != globals.new_opt.width) ret = 1;
   if(globals.cur_opt.height != globals.new_opt.height) ret = 1;

   globals.cur_opt.pixW = globals.new_opt.pixW;
   globals.cur_opt.pixH = globals.new_opt.pixH;
   globals.cur_opt.nPixW = globals.new_opt.nPixW;
   globals.cur_opt.nPixH = globals.new_opt.nPixH;
   globals.cur_opt.width = globals.new_opt.width;
   globals.cur_opt.height = globals.new_opt.height;
   globals.cur_opt.opt_alg = globals.new_opt.opt_alg;
   globals.cur_opt.proximity = globals.new_opt.proximity;

   return ret;
}

void redraw_screen(int x, int y, int w, int h){
   GtkStyle *style;

   /* if we have opened or rendered an image */
   if(globals.in_im != NULL){

      /* we may need to create the pixmap it */
      if(globals.pixmap == NULL){

         /* has the picture been rendered yet? */
         if(globals.image){
            render_image_pixmap(globals.out_im);
         } else {
            render_image_pixmap(globals.in_im);
         }
      }

      /* copy it to the screen */
      style = gtk_widget_get_style(GTK_WIDGET(globals.picDA));
      gdk_draw_pixmap(GTK_WIDGET(globals.picDA)->window,
                      style->fg_gc[GTK_WIDGET_STATE(GTK_WIDGET(globals.picDA))],
                      globals.pixmap,
                      x, y, x, y, w, h);
   }
}

void render_image_pixmap(GdkImlibImage *im){

   /* Render the original 24-bit Image data into a pixmap of size w * h */
   gdk_imlib_render(im,
      globals.cur_opt.width, globals.cur_opt.height);

   if(globals.pixmap != NULL){
      gdk_imlib_free_pixmap(globals.pixmap);
   }

   /* Extract the Image and mask pixmaps from the Image */
   globals.pixmap = gdk_imlib_move_image(im);
}


int draw_big_image(GdkImlibImage *im){

   render_image_pixmap(im);

   gdk_draw_pixmap(globals.picDA->window,
		   globals.picDA->style->fg_gc[GTK_WIDGET_STATE(globals.picDA)],
		   globals.pixmap,
                   0, 0, 
                   0, 0, 
                   globals.cur_opt.width, globals.cur_opt.height);

   return 1;
}


int change_small_image(int xx, int yy){
   struct PIC_DB *db;
   struct IMAGE_INFO *inf;
   GdkImlibImage *im;
   GdkImlibImage *scale_im;

   if(globals.image != NULL){
      inf = &(globals.image[yy][xx]);
      db = inf->matches[inf->match_no];

      /* load the image */
      if(NULL == (im=gdk_imlib_load_image(db->fname))){
	 fprintf(stderr, "Error: Unable to open %s\n", db->fname);
	 return 0;
      }

      /* scale the image */
      if(NULL == (scale_im=gdk_imlib_clone_scaled_image(im,
	 globals.cur_opt.pixW, globals.cur_opt.pixH))){
	 fprintf(stderr, "Error: Unable to scale image %s\n", db->fname);
	 return 0;
      }

      draw_small_image(scale_im, xx, yy);
      copy_image_RGB(scale_im->rgb_data, globals.out_im->rgb_data, xx, yy);

      /* tell imlib we messed with the RGB data */
      gdk_imlib_changed_image(globals.out_im);

      /* this is no longer valid */
      if(globals.pixmap != NULL){
	 gdk_imlib_free_pixmap(globals.pixmap);
      }
      globals.pixmap = NULL;

      /* free the images we are done with */
      gdk_imlib_destroy_image(im);
      gdk_imlib_destroy_image(scale_im);

   }
   return 1;
}

int draw_small_image(GdkImlibImage *scale_im, int x, int y){

   gdk_imlib_paste_image(scale_im, 
      globals.picDA->window, 
      globals.cur_opt.pixW * x, 
      globals.cur_opt.pixH * y, 
      globals.cur_opt.pixW, 
      globals.cur_opt.pixH);

   return 1;
}
