#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <unistd.h>
#include <fcntl.h>
#include <sys/file.h>
#include <limits.h>
#include <errno.h>

#ifndef _POSIX_PATH_MAX
#include <posix1_lim.h>
#endif

#include <time.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "client.h"
#include "lk.h"
#include "p_db.h"

static P_DB *timestampdb = NULL;
static int lockfilefd = -1;

int pkspxyc_init_db (const char *fname)
{
  char dbfile[_POSIX_PATH_MAX];
  char lockfile[_POSIX_PATH_MAX];
  const char *home;

  if (fname)
  {
    strncpy (dbfile, fname, sizeof (dbfile));
    dbfile[sizeof(dbfile) - 1] = '\0';
  }
  else
  {
    if (!(home = getenv ("HOME")))
    {
      fprintf (stderr, "%s: I'm homeless.\n", Progname);
      return -1;
    }
    snprintf (dbfile, sizeof (dbfile), "%s/.pkspxy", home);
  }

  snprintf (lockfile, sizeof (lockfile), "%s.lock", dbfile);
  
  if ((lockfilefd = open (lockfile, O_CREAT | O_WRONLY, 0600)) == -1)
  {
    fprintf (stderr, "%s: Can't open %s (%s).\n", Progname, lockfile, strerror (errno));
    goto bail;
  }
  
  if (lk_lock (lockfilefd, 1) == -1)
  {
    fprintf (stderr, "%s: Can't lock %s (%s).\n", Progname, lockfile, strerror (errno));
    goto bail;
  }

  if ((timestampdb = p_dbopen (dbfile)) == NULL)
  {
    fprintf (stderr, "%s: Can't open %s (%s).\n", Progname, dbfile, strerror (errno));
    goto bail;
  }
  
  return 0;
  
  bail:
  
  if (lockfilefd >= 0)
  {
    lk_unlock (lockfilefd);
    close (lockfilefd);
  }
  
  return -1;
}

void pkspxyc_close_db (void)
{
  if (timestampdb == NULL)
    return;

  p_dbclose (timestampdb);
  lk_unlock (lockfilefd);
  close (lockfilefd);
}

time_t pkspxyc_get_timestamp (const char *query)
{
  P_DBT key, datum;
  time_t t = 0;

  key.p_dptr = (char *) query;
  key.p_dsize = strlen (query);

  if (p_dbget (timestampdb, &key, &datum) == 0)
  {
    if (Debug) 
      fprintf (stderr, "%s: get: datum.size = %d\n", Progname, datum.p_dsize);
    if (datum.p_dsize == sizeof (t))
    {
      memcpy (&t, datum.p_dptr, datum.p_dsize);
      if (Debug) 
	fprintf (stderr, "%s: get: t = %d\n", Progname, t);
    }
  }
  
  return t;
}

int pkspxyc_set_timestamp (const char *query, time_t t)
{
  P_DBT key, datum;

  if (t == 0)
  {
    t = time (NULL);
    if (Debug) fprintf (stderr, "%s: setting time stamp to: %d\n", Progname, t);
  }

  key.p_dptr = (char *) query;
  key.p_dsize = strlen (query);

  datum.p_dptr = malloc (sizeof (t));
  memcpy (datum.p_dptr, &t, sizeof (t));
  datum.p_dsize = sizeof (t);

  return p_dbput (timestampdb, &key, &datum);
}

