/*
  Plee the Bear

  Copyright (C) 2005-2009 Julien Jorge, Sebastien Angibaud

  This program is free software; you can redistribute it and/or modify it
  under the terms of the GNU General Public License as published by the
  Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along
  with this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

  contact: plee-the-bear@gamned.org

  Please add the tag [PTB] in the subject of your mails.
*/
/**
 * \file frame.cpp
 * \brief Implementation of the ptb::frame class.
 * \author Julien Jorge
 */
#include <climits>
#include <limits>

#include "ptb/frame/frame.hpp"

#include "engine/game.hpp"
#include "ptb/layer/windows_layer.hpp"

/*----------------------------------------------------------------------------*/
/**
 * \brief Constructor.
 * \param owning_layer The layer onwing the window.
 */
ptb::frame::frame( windows_layer* owning_layer )
  : bear::gui::window( NULL, create_frame(owning_layer) ),
    m_owning_layer(owning_layer)
{
  CLAW_PRECOND(owning_layer != NULL);
  set_size( m_owning_layer->get_size() );
} // frame::frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Method called when the frame is displayed.
 */
void ptb::frame::on_focus()
{
  // nothing to do
} // frame::on_focus()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite to use for an arrow.
 */
bear::visual::sprite ptb::frame::get_arrow() const
{
  return get_frame_sprite_at( 64, 0, 18, 20 );
} // frame::get_cursor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite to use for the cursor.
 */
bear::visual::sprite ptb::frame::get_cursor() const
{
  return get_frame_sprite_at( 40, 0, 24, 25 );
} // frame::get_cursor()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite to use for a not checked checkbox.
 */
bear::visual::sprite ptb::frame::get_checkbox_off() const
{
  return get_frame_sprite_at( 82, 0, 23, 23 );
} // frame::get_checkbox_off()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite to use for a checked checkbox.
 */
bear::visual::sprite ptb::frame::get_checkbox_on() const
{
  return get_frame_sprite_at( 105, 0, 23, 23 );
} // frame::get_checkbox_on()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the background.
 */
bear::visual::sprite ptb::frame::get_background() const
{
  return get_frame_sprite_at(32, 32, 32, 32);
} // frame::get_background()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the slider bar.
 */
bear::visual::sprite ptb::frame::get_slider_bar() const
{
  return get_frame_sprite_at(40, 25, 1, 7);
} // frame::get_slider_bar()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the slider.
 */
bear::visual::sprite ptb::frame::get_slider() const
{
  return get_frame_sprite_at(64, 20, 9, 21);
} // frame::get_slider()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the font to use in the frames.
 */
bear::gui::static_text::font_type ptb::frame::get_font() const
{
  return m_owning_layer->get_level().get_globals().get_font
    ("font/fixed_yellow-10x20.fnt");
} // frame::get_font()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the color of the borders of the controls.
 */
claw::graphic::rgba_pixel ptb::frame::get_border_color() const
{
  const claw::graphic::rgba_pixel::component_type full_color =
    std::numeric_limits<claw::graphic::rgba_pixel::component_type>::max();

  return claw::graphic::rgba_pixel
    (full_color, full_color / 2, full_color / 4, full_color);
} // frame::get_border_color()

/*----------------------------------------------------------------------------*/
/**
 * \brief Show a window, centered in the scrren, to replace this window.
 * \param wnd The class name of the window to show.
 */
void ptb::frame::show_window( frame* wnd ) const
{
  CLAW_PRECOND(wnd != NULL);
  m_owning_layer->show_centered_window(wnd);
} // frame::show_window()

/*----------------------------------------------------------------------------*/
/**
 * \brief Close the current window (come back to the previous one).
 */
void ptb::frame::close_window() const
{
  m_owning_layer->close_window();
} // frame::close_window()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the frame in the window.
 */
bear::gui::frame& ptb::frame::get_frame()
{
  return *m_content_frame;
} // frame::get_frame()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the frame contained in the window.
 * \param owner The layer from which we can get the resources.
 */
bear::gui::frame* ptb::frame::create_frame(const windows_layer* owner)
{
  m_content_frame = new bear::gui::frame
    ( NULL, create_corner(owner), create_horizontal_border(owner),
      create_vertical_border(owner), create_background(owner) );

  m_content_frame->set_background_alpha( 0.75 );

  return m_content_frame;
} // frame::create_corner()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the corners.
 * \param owner The layer from which we can get the resources.
 */
bear::visual::sprite*
ptb::frame::create_corner(const windows_layer* owner) const
{
  return new bear::visual::sprite( get_frame_sprite_at(owner, 0, 0, 32, 32) );
} // frame::create_corner()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the horizontal borders.
 * \param owner The layer from which we can get the resources.
 */
bear::visual::sprite*
ptb::frame::create_horizontal_border(const windows_layer* owner) const
{
  return new bear::visual::sprite( get_frame_sprite_at(owner, 0, 32, 32, 8) );
} // frame::create_horizontal_border()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the vertical borders.
 * \param owner The layer from which we can get the resources.
 */
bear::visual::sprite*
ptb::frame::create_vertical_border(const windows_layer* owner) const
{
  return new bear::visual::sprite( get_frame_sprite_at(owner, 32, 0, 8, 32) );
} // frame::create_vertical_border()

/*----------------------------------------------------------------------------*/
/**
 * \brief Create the sprite of the background.
 * \param owner The layer from which we can get the resources.
 */
bear::visual::sprite*
ptb::frame::create_background(const windows_layer* owner) const
{
  return new bear::visual::sprite( get_frame_sprite_at(owner, 32, 32, 32, 32) );
} // frame::create_background()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite of the frame.
 * \param x X-position of the sprite in the resource image.
 * \param y Y-position of the sprite in the resource image.
 * \param w Width of the sprite in the resource image.
 * \param h Height of the sprite in the resource image.
 */
bear::visual::sprite ptb::frame::get_frame_sprite_at
( unsigned int x, unsigned int y, unsigned int w, unsigned int h ) const
{
  return get_frame_sprite_at(m_owning_layer, x, y, w, h);
} // frame::get_frame_sprite_at()

/*----------------------------------------------------------------------------*/
/**
 * \brief Get the sprite of the frame.
 * \param owner The layer from which we can get the resources.
 * \param x X-position of the sprite in the resource image.
 * \param y Y-position of the sprite in the resource image.
 * \param w Width of the sprite in the resource image.
 * \param h Height of the sprite in the resource image.
 */
bear::visual::sprite ptb::frame::get_frame_sprite_at
( const windows_layer* owner, unsigned int x, unsigned int y, unsigned int w,
  unsigned int h ) const
{
  bear::engine::level_globals& glob = owner->get_level().get_globals();

  const bear::visual::image& img = glob.get_image( "gfx/frame.tga" );
  claw::math::rectangle<unsigned int> rect( x, y, w, h );

  return bear::visual::sprite( img, rect );
} // frame::get_frame_sprite_at()
