/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  proc_ctrl.c:  emulation of processor control instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"

static void SetCR0(vm_t *, Bit32u val32);


  void
SGDT_Ms(vm_t *vm)
{
  Bit16u limit_16;
  Bit32u base_32;
 
  /* op1 is a register or memory reference */
  if (vm->i.mod == 0xc0) {
    monpanic(vm, "SGDT: use of register causes #UD.\n");
    UndefinedOpcode(vm);
    }
 
  limit_16 = vm->guest_cpu.gdtr.limit;
  base_32  = vm->guest_cpu.gdtr.base;
  /* +++ upper bits for 16bit opsize (24bit base)? */
  write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &limit_16);
  write_virtual_dword(vm, vm->i.seg, vm->i.rm_addr+2, &base_32);
}

  void
SIDT_Ms(vm_t *vm)
{
  Bit16u limit_16;
  Bit32u base_32;
 
  if (V8086Mode(vm))
    monpanic(vm, "SIDT: v8086 mode unsupported\n");
 
  /* op1 is a register or memory reference */
  if (vm->i.mod == 0xc0) {
    /* undefined opcode exception */
    monpanic(vm, "SIDT: use of register is undefined opcode.\n");
    UndefinedOpcode(vm);
    }
 
  limit_16 = vm->guest_cpu.idtr.limit;
  base_32  = vm->guest_cpu.idtr.base;
 
  /* 386+: ??? regardless of operand size, all 32bits of base are stored */
 
  write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &limit_16);
  write_virtual_dword(vm, vm->i.seg, vm->i.rm_addr+2, &base_32);
}

  void
SMSW_Ew(vm_t *vm)
{
  Bit16u msw;
 
  /* +++ reserved bits 0 ??? */
  /* should NE bit be included here ??? */
  /* should ET bit be included here (AW) */
  msw = vm->guest_cpu.cr0.raw & 0x0000000f;
 
  if (vm->i.mod == 0xc0) {
    if (vm->i.os_32) {
      WriteReg32(vm, vm->i.rm, msw);  /* zeros out high 16bits */
      }
    else {
      WriteReg16(vm, vm->i.rm, msw);
      }
    }
  else {
    write_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &msw);
    }
}

  void
LMSW_Ew(vm_t *vm)
{
  Bit16u msw;
  Bit32u cr0;
 
  if (V8086Mode(vm))
    monpanic(vm, "LMSW_Ew: v8086 mode.\n");
 
  if ( ProtectedMode(vm) ) {
    if ( G_CPL(vm) != 0 ) {
      monpanic(vm, "LMSW: CPL=%u\n", (unsigned) G_CPL(vm));
      exception(vm, ExceptionGP, 0);
      }
    }
 
  if (vm->i.mod == 0xc0) {
    msw = ReadReg16(vm, vm->i.rm);
    }
  else {
    read_virtual_word(vm, vm->i.seg, vm->i.rm_addr, &msw);
    }
 
  /* LMSW does not affect PG,CD,NW,AM,WP,NE,ET bits, and
   * cannot clear PE.
   */
 
  if ( ((msw & 0x0001)==0) && vm->guest_cpu.cr0.fields.pe ) {
    msw |= 0x0001; /* adjust PE bit to current value of 1 */
    }
 
  msw &= 0x000f; /* LMSW only affects last 4 flags */
  cr0 = (vm->guest_cpu.cr0.raw & 0xfffffff0) | msw;
  SetCR0(vm, cr0);
}

  void
INVLPG(vm_t *vm)
{
  monprint(vm, "INVLPG:\n");
}

  void
MOV_CdRd(vm_t *vm)
{
  Bit32u val32;


  if (V8086Mode(vm))
    monpanic(vm, "MOV_CdRd: v8086 mode unsupported\n");

  if (vm->i.mod != 0xc0) {
    monpanic(vm, "MOV_CdRd: rm field not a register!\n");
    }

  invalidate_prefetch_q();

  if (ProtectedMode(vm) && (G_CPL(vm)!=0)) {
    monpanic(vm, "MOV_CdRd: CPL!=0\n");
    exception(vm, ExceptionGP, 0);
    }

  val32 = ReadReg32(vm, vm->i.rm);

  switch (vm->i.nnn) {
    case 0: /* CR0 (MSW) */
      /* monprint(vm, "MOV_CdRd: CR0: 0x%x\n", val32); */
      SetCR0(vm, val32);
      break;

    case 1: /* CR1 */
      monpanic(vm, "MOV_CdRd: CR1 not implemented yet\n");
      break;
    case 2: /* CR2 */
      monpanic(vm, "MOV_CdRd: CR2 not implemented yet\n");
      vm->guest_cpu.cr2 = val32;
      break;
    case 3: /* CR3 */
      CR3_change(vm, val32);
      break;
    case 4: /* CR4 */
      monpanic(vm, "MOV_CdRd: CR4\n");
      /*  Protected mode: #GP(0) if attempt to write a 1 to */
      /*  any reserved bit of CR4 */

      monprint(vm, "MOV_CdRd: ignoring write to CR4 of 0x%08x\n",
        val32);
      if (val32) {
        monpanic(vm, "MOV_CdRd: (CR4) write of 0x%08x\n",
          val32);
        }
      /* Only allow writes of 0 to CR4 for now. */
      /* Writes to bits in CR4 should not be 1s as CPUID */
      /* returns not-supported for all of these features. */
      vm->guest_cpu.cr4.raw = 0;
      break;
    default:
      monpanic(vm, "MOV_CdRd: control register index out of range\n");
      break;
    }
}

  void
MOV_RdCd(vm_t *vm)
{
  /* mov control register data to register */
  Bit32u val32;

  if (V8086Mode(vm))
    monpanic(vm, "MOV_RdCd: v8086 mode unsupported\n");

  if (vm->i.mod != 0xc0) {
    monpanic(vm, "MOV_RdCd: rm field not a register!\n");
    }

  if (ProtectedMode(vm) && (G_CPL(vm)!=0)) {
    monpanic(vm, "MOV_RdCd: CPL!=0\n");
    exception(vm, ExceptionGP, 0);
    }

  switch (vm->i.nnn) {
    case 0: /* CR0 (MSW) */
      val32 = vm->guest_cpu.cr0.raw;
      break;
    case 1: /* CR1 */
      monpanic(vm, "MOV_RdCd: CR1\n");
      break;
    case 2: /* CR2 */
      val32 = vm->guest_cpu.cr2;
      break;
    case 3: /* CR3 */
      val32 = vm->guest_cpu.cr3;
      break;
    case 4: /* CR4 */
      monpanic(vm, "MOV_RdCd: CR4\n");
      val32 = vm->guest_cpu.cr4.raw;
      break;
    default:
      monpanic(vm, "MOV_RdCd: nnn=%u\n", vm->i.nnn);
    }
  WriteReg32(vm, vm->i.rm, val32);
}

  void
SetCR0(vm_t *vm, Bit32u val32)
{
  unsigned prev_pe, prev_pg;

  /* Before we change the CR0 bits, we need to make the
   * selectors and selector caches current.
   */
  cache_sreg(vm, SRegCS);
  cache_sreg(vm, SRegSS);
  cache_sreg(vm, SRegDS);
  cache_sreg(vm, SRegES);
  cache_sreg(vm, SRegFS);
  cache_sreg(vm, SRegGS);

  prev_pe = vm->guest_cpu.cr0.fields.pe;
  prev_pg = vm->guest_cpu.cr0.fields.pg;

  if (cpuid_info.procSignature.fields.family==5) {
    /* ET bit hardwired, reserved bits retain value written to them. */
    vm->guest_cpu.cr0.raw = val32 | 0x00000010;
    }
  else if (cpuid_info.procSignature.fields.family==6) {
    /* ET bit hardwired, reserved bits zeroed. */
    vm->guest_cpu.cr0.raw = (val32 | 0x00000010) & 0xe005003f;
    }
  else {
    monpanic(vm, "SetCR0: family=%u\n",
      cpuid_info.procSignature.fields.family);
    }

if ((val32 & 0x00000020) == 0) {
#warning "guest CR0 hack"
  /* monprint(vm, "SetCR0: NE=0\n"); */
  /* +++ SetCR0 NE forced to 1 hack */
  vm->guest_cpu.cr0.raw |= 0x20;
  }

  if (prev_pe==0 && vm->guest_cpu.cr0.fields.pe) {
    /* Transition to Protected Mode */
    vm->modeChange |= ModeChangeEventTransition;
    }
  else if (prev_pe==1 && vm->guest_cpu.cr0.fields.pe==0) {
    /* Transition to Real Mode */
    vm->modeChange |= ModeChangeEventTransition;
    /* +++ Start new mode marked with small segments, but the rest
     * of the legacy values remain.
     */
    vm->guest_cpu.desc_cache[SRegCS].desc.d_b = 0;
    vm->guest_cpu.desc_cache[SRegSS].desc.d_b = 0;
#warning "fix this"
    }

  if (prev_pg==0 && vm->guest_cpu.cr0.fields.pg) {
    /* Transition to paging enabled */
    vm->modeChange |= ModeChangeEventPaging | ModeChangeRequestPaging;
    }
  else if (prev_pg==1 && vm->guest_cpu.cr0.fields.pg==0) {
    /* Transition to paging disabled */
    vm->modeChange |= ModeChangeEventPaging | ModeChangeRequestPaging;
    }
}


  void
CPUID(vm_t *vm)
{
  unsigned type, family, model, stepping, features;

  invalidate_prefetch_q();

  switch (G_EAX(vm)) {
    case 0:
      /* EAX: highest input value understood by CPUID */
      /* EBX: vendor ID string */
      /* EDX: vendor ID string */
      /* ECX: vendor ID string */
      G_EAX(vm) = 1; /* 486 or pentium */
      G_EBX(vm) = 0x756e6547; /* "Genu" */
      G_EDX(vm) = 0x49656e69; /* "ineI" */
      G_ECX(vm) = 0x6c65746e; /* "ntel" */
      break;

    case 1:
      /* EAX[3:0]   Stepping ID */
      /* EAX[7:4]   Model: starts at 1 */
      /* EAX[11:8]  Family: 4=486, 5=Pentium, 6=PPro */
      /* EAX[13:12] Type: 0=OEM,1=overdrive,2=dual cpu,3=reserved */
      /* EAX[31:14] Reserved */
      /* EBX:       Reserved (0) */
      /* ECX:       Reserved (0) */
      /* EDX:       Feature Flags */
      /*   [0:0]   FPU on chip */
      /*   [1:1]   VME: Virtual-8086 Mode enhancements */
      /*   [2:2]   DE: Debug Extensions (I/O breakpoints) */
      /*   [3:3]   PSE: Page Size Extensions */
      /*   [4:4]   TSC: Time Stamp Counter */
      /*   [5:5]   MSR: RDMSR and WRMSR support */
      /*   [6:6]   PAE: Physical Address Extensions */
      /*   [7:7]   MCE: Machine Check Exception */
      /*   [8:8]   CXS: CMPXCHG8B instruction */
      /*   [9:9]   APIC: APIC on Chip */
      /*   [11:10] Reserved */
      /*   [12:12] MTRR: Memory Type Range Reg */
      /*   [13:13] PGE/PTE Global Bit */
      /*   [14:14] MCA: Machine Check Architecture */
      /*   [15:15] CMOV: Cond Mov/Cmp Instructions */
      /*   [22:16] Reserved */
      /*   [23:23] MMX Technology */
      /*   [31:24] Reserved */

      features = 0; /* start with none */
      type = 0; /* OEM */

      family = cpuid_info.procSignature.fields.family;
      if (family==4) {
        if (cpuid_info.featureFlags.fields.fpu) {
          /* 486dx */
          model = 1;
          stepping = 3;
          features |= 0x01;
          }
        else {
          /* 486sx */
          model = 2;
          stepping = 3;
          }
        }
      else if (family==5) {
        model = 1; /* Pentium (60,66) */
        stepping = 3; /* ??? */
        if (cpuid_info.featureFlags.fields.fpu)
          features |= 0x01;
        }
      else {
        monpanic(vm, "CPUID: not implemented for > 5\n");
        }

      G_EAX(vm) = (family <<8) | (model<<4) | stepping;
      G_EBX(vm) = 0; /* reserved */
      G_ECX(vm) = 0; /* reserved */
      G_EDX(vm) = features;
      break;

    default:
      G_EAX(vm) = 0; /* Reserved, undefined */
      G_EBX(vm) = 0; /* Reserved, undefined */
      G_ECX(vm) = 0; /* Reserved, undefined */
      G_EDX(vm) = 0; /* Reserved, undefined */
      break;
    }
}

  void
CLTS(vm_t *vm)
{
  if (V8086Mode(vm)) monpanic(vm, "clts: v8086 mode unsupported\n");
 
  /* +++ read errata file */
  /* +++ does CLTS also clear NT flag??? */
 
  /* #GP(0) if CPL is not 0 */
  if (G_CPL(vm)!=0) {
    monprint(vm, "CLTS(): CPL!=0\n");
    exception(vm, ExceptionGP, 0);
    }
 
  vm->guest_cpu.cr0.fields.ts = 0;
}

  void
CLI_guest(vm_t *vm)
{
  if (ProtectedMode(vm)) {
    if (G_CPL(vm) > G_GetIOPL(vm)) {
      exception(vm, ExceptionGP, 0);
      }
    }
  else if (V8086Mode(vm)) {
    if (G_GetIOPL(vm) != 3) {
      exception(vm, ExceptionGP, 0);
      }
    }
 
  G_SetIF(vm, 0);
}

  void
STI_guest(vm_t *vm)
{
  if (ProtectedMode(vm)) {
    if (G_CPL(vm) > G_GetIOPL(vm)) {
      exception(vm, ExceptionGP, 0);
      }
    }
  else if (V8086Mode(vm)) {
    if (G_GetIOPL(vm) != 3) {
      exception(vm, ExceptionGP, 0);
      }
    }
 
  if (!G_GetIF(vm)) {
    G_SetIF(vm, 1);
    vm->guest_cpu.inhibit_mask |= INHIBIT_INTERRUPTS;
    vm->guest_cpu.async_event = 1;
    }
}

  void
MOV_DdRd(vm_t *vm)
{
  Bit32u val_32;

  if (V8086Mode(vm)) monpanic(vm, "MOV_DdRd: v8086 mode unsupported\n");

  /* NOTES:
   *   32bit operands always used
   *   r/m field specifies general register
   *   mod field should always be 11 binary
   *   reg field specifies which special register
   */

  if (vm->i.mod != 0xc0) {
    monpanic(vm, "MOV_DdRd(): rm field not a register!\n");
    }

  invalidate_prefetch_q();

  if (ProtectedMode(vm) && G_CPL(vm)!=0) {
    monpanic(vm, "MOV_DdRd: CPL!=0\n");
    /* #GP(0) if CPL is not 0 */
    exception(vm, ExceptionGP, 0);
    }

  val_32 = ReadReg32(vm, vm->i.rm);
  /*monprint(vm, "MOV_DdRd: DR[%u]=0x%08x\n", */
  /*    (unsigned) vm->i.nnn, (unsigned) val_32); */

  switch (vm->i.nnn) {
    case 0: /* DR0 */
      vm->guest_cpu.dr0 = val_32;
      break;
    case 1: /* DR1 */
      vm->guest_cpu.dr1 = val_32;
      break;
    case 2: /* DR2 */
      vm->guest_cpu.dr2 = val_32;
      break;
    case 3: /* DR3 */
      vm->guest_cpu.dr3 = val_32;
      break;

    case 4: /* DR4 */
    case 6: /* DR6 */
      /* DR4 aliased to DR6 by default.  With Debug Extensions on, */
      /* access to DR4 causes #UD */
      if (cpuid_info.procSignature.fields.family >= 4) {
        if ( (vm->i.nnn == 4) && vm->guest_cpu.cr4.fields.de ) {
          /* Debug extensions on */
          monprint(vm, "MOV_DdRd: access to DR4 causes #UD\n");
          UndefinedOpcode(vm);
          }
        }
      if (cpuid_info.procSignature.fields.family <= 4) {
        /* On 386/486 bit12 is settable */
        vm->guest_cpu.dr6 = (vm->guest_cpu.dr6 & 0xffff0ff0) |
                            (val_32 & 0x0000f00f);
        }
      else {
        /* On Pentium+, bit12 is always zero */
        vm->guest_cpu.dr6 = (vm->guest_cpu.dr6 & 0xffff0ff0) |
                            (val_32 & 0x0000e00f);
        }
      break;

    case 5: /* DR5 */
    case 7: /* DR7 */
      /* Note: 486+ ignore GE and LE flags.  On the 386, exact */
      /* data breakpoint matching does not occur unless it is enabled */
      /* by setting the LE and/or GE flags. */

      /* DR5 aliased to DR7 by default.  With Debug Extensions on, */
      /* access to DR5 causes #UD */
      if (cpuid_info.procSignature.fields.family >= 4) {
        if ( (vm->i.nnn == 5) && vm->guest_cpu.cr4.fields.de ) {
          /* Debug extensions (CR4.DE) on */
          monprint(vm, "MOV_DdRd: access to DR5 causes #UD\n");
          UndefinedOpcode(vm);
          }
        }
      /* Some sanity checks... */
      if ( val_32 & 0x00002000 ) {
        monpanic(vm, "MOV_DdRd: GD bit not supported yet\n");
        /* Note: processor clears GD upon entering debug exception */
        /* handler, to allow access to the debug registers */
        }
      if ( (((val_32>>16) & 3)==2) ||
           (((val_32>>20) & 3)==2) ||
           (((val_32>>24) & 3)==2) ||
           (((val_32>>28) & 3)==2) ) {
        /* IO breakpoints (10b) are not yet supported. */
        monpanic(vm, "MOV_DdRd: write of %08x contains IO breakpoint\n",
          val_32);
        }
      if ( (((val_32>>18) & 3)==2) ||
           (((val_32>>22) & 3)==2) ||
           (((val_32>>26) & 3)==2) ||
           (((val_32>>30) & 3)==2) ) {
        /* LEN0..3 contains undefined length specifier (10b) */
        monpanic(vm, "MOV_DdRd: write of %08x contains undefined LENx\n",
          val_32);
        }
      if ( ((((val_32>>16) & 3)==0) && (((val_32>>18) & 3)!=0)) ||
           ((((val_32>>20) & 3)==0) && (((val_32>>22) & 3)!=0)) ||
           ((((val_32>>24) & 3)==0) && (((val_32>>26) & 3)!=0)) ||
           ((((val_32>>28) & 3)==0) && (((val_32>>30) & 3)!=0)) ) {
        /* Instruction breakpoint with LENx not 00b (1-byte length) */
        monpanic(vm, "MOV_DdRd: write of %08x, R/W=00b LEN!=00b\n",
          val_32);
        }
      if (cpuid_info.procSignature.fields.family <= 4) {
        /* 386/486: you can play with all the bits except b10 is always 1 */
        vm->guest_cpu.dr7 = val_32 | 0x00000400;
        }
      else {
        /* Pentium+: bits15,14,12 are hardwired to 0, rest are settable. */
        /* Even bits 11,10 are changeable though reserved. */
        vm->guest_cpu.dr7 = (val_32 & 0xffff2fff) | 0x00000400;
        }
      break;

    default:
      monpanic(vm, "MOV_DdRd: control register index out of range\n");
      break;
    }
}

  void
MOV_RdDd(vm_t *vm)
{
  Bit32u val_32;

  if (V8086Mode(vm)) {
    monprint(vm, "MOV_RdDd: v8086 mode causes #GP\n");
    exception(vm, ExceptionGP, 0);
    }

  if (vm->i.mod != 0xc0) {
    monpanic(vm, "MOV_RdDd: rm field not a register!\n");
    UndefinedOpcode(vm);
    }

  if (ProtectedMode(vm) && (G_CPL(vm)!=0)) {
    monprint(vm, "MOV_RdDd: CPL!=0 causes #GP\n");
    exception(vm, ExceptionGP, 0);
    }

  switch (vm->i.nnn) {
    case 0: /* DR0 */
      val_32 = vm->guest_cpu.dr0;
      break;
    case 1: /* DR1 */
      val_32 = vm->guest_cpu.dr1;
      break;
    case 2: /* DR2 */
      val_32 = vm->guest_cpu.dr2;
      break;
    case 3: /* DR3 */
      val_32 = vm->guest_cpu.dr3;
      break;

    case 4: /* DR4 */
    case 6: /* DR6 */
      /* DR4 aliased to DR6 by default.  With Debug Extensions on, */
      /* access to DR4 causes #UD */
      if ( (vm->i.nnn == 4) && vm->guest_cpu.cr4.fields.de ) {
        /* Debug extensions on */
        monprint(vm, "MOV_RdDd: access to DR4 causes #UD\n");
        UndefinedOpcode(vm);
        }
      val_32 = vm->guest_cpu.dr6;
      break;

    case 5: /* DR5 */
    case 7: /* DR7 */
      /* DR5 aliased to DR7 by default.  With Debug Extensions on, */
      /* access to DR5 causes #UD */
      if ( (vm->i.nnn == 5) && vm->guest_cpu.cr4.fields.de ) {
        /* Debug extensions on */
        monprint(vm, "MOV_RdDd: access to DR5 causes #UD\n");
        UndefinedOpcode(vm);
        }
      val_32 = vm->guest_cpu.dr7;
      break;

    default:
      monpanic(vm, "MOV_RdDd: control register index out of range\n");
      val_32 = 0;
    }
  WriteReg32(vm, vm->i.rm, val_32);
}

  void
HLT(vm_t *vm)
{
  /* hack to ignore HLT if cosimulating as slave */
#warning "HLT hack for cosimulation"
  if (vm->executeN) return;

  cache_selector(vm, SRegCS);

  /* +++ hack to panic if HLT comes from BIOS */
  if ( vm->guest_cpu.selector[SRegCS].raw == 0xf000 )
    monpanic(vm, "HALT instruction encountered\n");
 
  if (G_CPL(vm)!=0) {
    monpanic(vm, "HLT: CPL!=0\n");
    exception(vm, ExceptionGP, 0);
    }
 
  if ( ! G_GetIF(vm) ) {
    monpanic(vm, "HLT instruction with IF=0!\n");
    }
 
  /* stops instruction execution and places the processor in a
   * HALT state.  An enabled interrupt, NMI, or reset will resume
   * execution.  If interrupt (including NMI) is used to resume
   * execution after HLT, the saved CS:eIP points to instruction
   * following HLT.
   */
 
  /* artificial trap bit, why use another variable. */
  vm->guest_cpu.debug_trap |= 0x80000000; /* artificial trap */
  vm->guest_cpu.async_event = 1; /* so processor knows to check */
  /* Execution of this instruction completes.  The processor
   * will remain in a halt state until one of the above conditions
   * is met.
   */
}

  void
WBINVD(vm_t *vm)
{
  invalidate_prefetch_q();

  if ( V8086Mode(vm) ) {
    monprint(vm, "WBINVD: v86 #GP\n");
    exception(vm, ExceptionGP, 0);
    }
 
  if (vm->guest_cpu.cr0.fields.pe) {
    if (G_CPL(vm)!=0) {
      monprint(vm, "WBINVD: CPL!=0\n");
      exception(vm, ExceptionGP, 0);
      }
    }
  /* else RM, no checks */

  /* +++ WBINVD ignored */
  monprint(vm, "WBINVD: (ignoring)\n");
}

  void
RDTSC(vm_t *vm)
{
#warning "RDTSC hack"
  if ( (vm->guest_cpu.cr4.fields.tsd==0) ||
       (vm->guest_cpu.cr4.fields.tsd && (G_CPL(vm)==0)) ) {
    Bit32u eax, edx;
    asm volatile (
      "rdtsc"
      : "=a" (eax), "=d" (edx)
      );
    G_EAX(vm) = eax;
    G_EDX(vm) = edx;
    }
  else
    exception(vm, ExceptionGP, 0);
}
