// ------------------------------------------------------------------------- //
// $Id: nodecontroller.cpp,v 1.25 2003/12/16 22:33:21 pandr Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

// ------------------------------------------------------------------------- //

#include "nodecontroller.h"
#include "node.h"
#include "mainwindow.h"	

// ------------------------------------------------------------------------- //

Parameters::Parameters(PyObject *py_dict)
{
	assert(PyDict_Check(py_dict));
	_py_dict = py_dict;
}

float Parameters::get_float(const char* name, float def) const
{
	PyObject* o;
	o = PyDict_GetItemString(_py_dict, (char*)name);
	if (o) {
		return PyFloat_AsDouble(PyNumber_Float(o));
	}
	else {
		return def;
	}
}

float Parameters::get_float(const char* name) const
{
	PyObject* o;

	o = PyDict_GetItemString(_py_dict, (char*)name);
	assert(o != NULL);
	return PyFloat_AsDouble(PyNumber_Float(o));
}

void Parameters::set_float(const char* name, float f)
{
	PyObject *new_val = Py_BuildValue((char*)"d", f);
	PyDict_SetItemString(_py_dict, (char*)name, new_val);
}

int Parameters::get_int(const char* name) const
{
	return PyInt_AsLong(PyNumber_Int(PyDict_GetItemString(_py_dict, 
					(char*)name)));
}

int Parameters::get_int(const char* name, int def) const
{
	PyObject* o;
	o = PyDict_GetItemString(_py_dict, (char*)name);
	if (o) {
		return PyInt_AsLong(PyNumber_Int(o));
	}
	else {
		return def;
	}
}

void Parameters::set_int(const char* name, int i)
{
	PyObject *new_val = Py_BuildValue((char*)"i", i);
	PyDict_SetItemString(_py_dict, (char*)name, new_val);
}

// ------------------------------------------------------------------------- //

NodeController::NodeController(Node* node) : _node(node), _active(true)
{
}

// ------------------------------------------------------------------------- //

EffectController::EffectController(Node* node, const Parameters& p)
	: NodeController(node), _state(at_start), _parms(p)
{
	// Default activate at step 1
	_activate_at_step = _parms.get_int("activate_step", 1);

	// Listen for notify's from viewer
	//g_renderer->_viewer.attach(this);
}
// ------------------------------------------------------------------------- //

Rotator::Rotator(Node* node, const Parameters& p)
	: EffectController(node, p),
	  _rotation(v3(0.0f, 1.0f, 0.0f))
{
}

void Rotator::frame_update()
{
	if (_state == at_start || _state == at_end) return;

	float t = g_clock.time() - _start_time;
	float r = tan(t*.25f);
	m33 m;
	m.makeRotate(_rotation.x()*r, _rotation.y()*r, _rotation.z()*r);
	_node->set_mat(m);

	if (t > 3.1415f*4.0f) {
		_state = at_end;
	}
}

void Rotator::start()
{
	_state = start_to_end;
	_start_time = g_clock.time();
}

// ------------------------------------------------------------------------- //

Fader::Fader(Node* node, const Parameters& p) : EffectController(node, p)
{
	_duration    = p.get_float("duration",    1.0f);
	_start_alpha = p.get_float("start_alpha", 0.0f);
	_end_alpha   = p.get_float("end_alpha",   1.0f);
}

void Fader::start()
{
	_state      = start_to_end;
	_start_time = g_clock.time();
}

void Fader::frame_update()
{
	if (_state == at_start || _state == at_end) return;

	float t = (g_clock.time() - _start_time) / _duration;

	if (t > 1.0f) {
		t = 1.0f;
		_state = at_end;
	}
	_node->set_alpha(_start_alpha + t*(_end_alpha - _start_alpha));
}

// ------------------------------------------------------------------------- //


Slider::Slider(Node* node, const Parameters& p)
	: EffectController(node, p),
	  _start_pos(v3(0.0, 0.0, 0.0)),
	  _end_pos(v3(100.0, 100.0, 0.0)),
	  _speed(10.0)
{
	assert(_speed > 0.0f);

	_node->set_pos(_start_pos);

	_direction = _end_pos - _start_pos;
	_length    = _direction.length();
	_direction.make_norm();

	_state      = at_start;
	_start_time = 0;
}

void Slider::frame_update()
{
	float so_far = 0.0f;
	switch (_state) {
		case start_to_end:
			so_far = (g_renderer->_frame_time - _start_time) * _speed * 2.0f;
			if (so_far > _length) {
				so_far = _length;
				_state      = at_end;
				_start_time = g_renderer->_frame_time;
			}
			_node->set_pos(_start_pos + _direction * so_far);
			break;
		case end_to_start:
			so_far = (g_renderer->_frame_time - _start_time) * _speed;
			if (so_far > _length) {
				so_far = _length;
				_state = at_start;
			}
			_node->set_pos(_end_pos - _direction * so_far);
			break;
		case at_end:
			if (g_renderer->_frame_time - 3.0f > _start_time) {
				_state      = end_to_start;
				_start_time = g_renderer->_frame_time;
			}
		case at_start:
			return;
	}
}

void Slider::start()
{
	if (_state != at_start)
		return;
	_state      = start_to_end;
	_start_time = g_renderer->_frame_time;
}

// ------------------------------------------------------------------------- //

PythonNodeController::PythonNodeController(Node* node, PyObject* py_controller)
	: NodeController(node), _py_controller(py_controller)
{
	if (!PyCallable_Check(_py_controller)) {
		assert(0); // Must be a callable object
	}
	Py_INCREF(py_controller);
}

PythonNodeController::~PythonNodeController()
{
}

void PythonNodeController::frame_update()
{
	PyObject *args = Py_BuildValue((char*)"ff", g_renderer->_frame_time, 
		g_renderer->_frame_duration);
	PyObject_CallObject(_py_controller, args);
	Py_DECREF(args);

	if (PyErr_Occurred()) {
		PyErr_Print();
	}
}

// ------------------------------------------------------------------------- //

PythonEffectController::PythonEffectController(Node* node, 
		const Parameters& p) : EffectController(node,p)
{
}

EffectController* PythonEffectController::EffectControllerFactory(
		Node *n, const char *effect_name, const Parameters& p)
{
	EffectController* e;
	if (!strcmp(effect_name, "Rotater")) {
		e = new Rotator(n, p);
	}
	else if (!strcmp(effect_name, "Fader")) {
		e = new Fader(n, p);
	}
	else {
		return 0;
	}
	n->add_controller(e);
	return e;
}

// ------------------------------------------------------------------------- //

