/*****
*
* Copyright (C) 2001, 2002 Yoann Vandoorselaere <yoann@prelude-ids.org>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Yoann Vandoorselaere <yoann@prelude-ids.org>
*
*****/


#include <stdio.h>
#include <stdlib.h>

#include "protocol.h"
#include "rpc-prv.h"


int rpc_plugin_id;
int is_enabled = 0;
port_list_t *port_list;


static int add_rpc_rule(rpc_test_t *ptr, rules_t **rules) 
{
        int ret;
        rule_t *rule;

        rule = make_new_rule(0, NULL);
        if ( ! rule )
                return -1;
        
        ret = add_rule_leaf_match(rule, 1, ptr, &match_rpc_rule);
        if ( ret < 0 )
                return -1;
        
        *rules = make_new_rules(rule, NULL);
        if ( ! *rules )
                return -1;
        
        return 0;
}



static int parse_rpc_rule(parameters_t *plist, rules_t **rules)
{
        int i;
        rpc_test_t *ptr;
        
        ptr = calloc(1, sizeof(rpc_test_t));
        if ( ! ptr ) {
                log(LOG_ERR, "memory exhausted.\n");
                return -1;
        }

        for ( i = 0; plist != NULL; plist = plist->next ) {

                /*
                 * This parameter is empty.
                 */
                if ( *plist->str == '*' ) {
                        i++;
                        continue;
                }                       

                if ( i == 0 ) {
                        ptr->procedure = strtoul(plist->str, NULL, 0);
                        ptr->flags |= RPC_CHECK_PROCEDURE;
                }
                
                else if ( i == 1 ) {
                        ptr->version = strtoul(plist->str, NULL, 0);
                        ptr->flags |= RPC_CHECK_VERSION;
                }
                
                else if ( i == 2 ) {
                        ptr->program = strtoul(plist->str, NULL, 0);
                        ptr->flags |= RPC_CHECK_PROGRAM;
                }

                i++;
        }

        if ( ptr->flags )
                return add_rpc_rule(ptr, rules);
        
        free(ptr);
        return -1;
}




static plugin_protocol_t plugin;



static int set_port(prelude_option_t *opt, const char *optarg) 
{
        /*
         * reset port list
         */
        protocol_plugin_port_list_free(port_list);

        port_list = protocol_plugin_port_list_new();
        if ( ! port_list )
                return prelude_option_error;
        
        protocol_plugin_add_string_port_to_list(port_list, optarg);

        return prelude_option_success;
}



static int set_enable_state(prelude_option_t *opt, const char *optarg) 
{
        int ret;
        
        if ( is_enabled == 1 ) {
                ret = plugin_unsubscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;        
                is_enabled = 0;
        }

        else {
                ret = plugin_subscribe((plugin_generic_t *) &plugin);
                if ( ret < 0 )
                        return prelude_option_error;
                is_enabled = 1;
        }
        
        return prelude_option_success;
}




static int get_enable_state(char *buf, size_t size) 
{
        snprintf(buf, size, "%s", (is_enabled == 1) ? "enabled" : "disabled");
        return prelude_option_success;
}




plugin_generic_t *plugin_init(int argc, char **argv)
{
        prelude_option_t *opt;

        opt = prelude_option_add(NULL, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 0, "rpcmod",
                                 "enable RpcMod and precede it's option",
                                 optionnal_argument, set_enable_state, get_enable_state);
        
        prelude_option_add(opt, CLI_HOOK|CFG_HOOK|WIDE_HOOK, 'p', "port-list",
                           "List of port this plugin should look at",
                           required_argument, set_port, NULL);
        
	rpc_plugin_id = plugin_request_new_id();

        port_list = protocol_plugin_port_list_new();
        if ( ! port_list )
                return NULL;
        
        protocol_plugin_add_port_to_list(port_list, 111);

        plugin_set_name(&plugin, "RpcMod");        
        plugin_set_protocol(&plugin, "rpc");
        plugin_set_author(&plugin, "Yoann Vandoorselaere");
        plugin_set_contact(&plugin, "yoann@prelude-ids.org");
        plugin_set_desc(&plugin, "Decode the RPC protocol and provide rpc key for the signature engine.");
        plugin_set_running_func(&plugin, decode_rpc);
        
        signature_parser_add_multiple_args_key("rpc", &parse_rpc_rule);
        
        return (plugin_generic_t *) &plugin;
}



