/*****
*
* Copyright (C) 2001 Jeremie Brebec <flagg@ifrance.com>
* All Rights Reserved
*
* This file is part of the Prelude program.
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by 
* the Free Software Foundation; either version 2, or (at your option)
* any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; see the file COPYING.  If not, write to
* the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*
* Written by Jeremie Brebec <flagg@ifrance.com>
*
*****/


#include <stdio.h>

#include <libprelude/prelude-log.h>

#include "rules.h"
#include "rules-type.h"
#include "rules-stats.h"


static int stats_total_depth;
static int stats_max_path, stats_min_path;
static double stats_nodes;
static double stats_rules;
static double stats_leaf;



static void compute_leaf_stats(double increment, int depth) 
{
        stats_leaf += increment;
        stats_total_depth += depth;
        
        if ( depth > stats_max_path )
                stats_max_path = depth;
        if ( depth < stats_min_path )
                stats_min_path = depth;
}




static void get_node_stats(rules_node_t *node, int depth)
{
	l_run_t* run;
        const double inc = 1 / ((double) node->ref);
        
        if ( ! node ) {
                compute_leaf_stats(1, depth);
                return;
        }

        if ( node->ref == 0 )
                log(LOG_ERR, "node without reference.\n");
        
        /*
         * Count number of rules
         */
        for ( run = node->run; run != NULL; run = run->next )
                stats_rules += inc;
        
        /*
         * leaf
         */
        if ( node->var.id == ID_LEAF_NODE ) 
                compute_leaf_stats(inc, depth);
        else {
                /*
                 * count number of node
                 */
                stats_nodes += inc;
                get_node_stats(node->brother, depth + 1);
                get_node_stats(node->child, depth + 1);
        }
}




int signature_engine_get_nodes_stats(rules_node_t *root, stats_t *stats)
{
	if ( ! root || root->var.id != ID_ROOT_NODE )
		return -1;

	stats_min_path = 9999; /* +oo */
	stats_max_path = 0;

	stats_total_depth = 0;
	stats_nodes = 0;
	stats_leaf = 0;
	stats_rules = 0;

	get_node_stats(root->child, 0);

	if ( stats->leaf > 0 )
		stats->average_path = ((double)stats_total_depth) / ((double)stats_leaf);
	else
		stats->average_path = 0;

	stats->node = (int) stats_nodes;
	stats->leaf = (int) stats_leaf;
	stats->rule = (int) stats_rules;
	stats->max_path = stats_max_path;
	stats->min_path = stats_min_path;

	return 0;
}




static void print_run_node(l_run_t *run) 
{
        fprintf(stderr, "R: ");

        for ( ; run != NULL; run = run->next ) 
                fprintf(stderr, "[%p,%p] ", run->run, run->data);

        fprintf(stderr, "\n");
}




static void print_leaf_node(rules_node_t *node) 
{       
        fprintf(stderr, "+++ leaf node [%d]\n", node->ref);
        
        if ( node->run )
                print_run_node(node->run);
}




void signature_engine_print_nodes(rules_node_t *node)
{
	generic_type_t *type;
        
	if ( node->var.id == ID_LEAF_NODE ) {
                print_leaf_node(node);
                return;
	}
        
	else if ( node->var.id == ID_IGNORE_NODE ) {
		fprintf(stderr, "+++ ignore node [%d]\n", node->ref);
		return;
	}
        
	type = signature_engine_get_type_by_id(node->var.id);
	if ( type ) {
		fprintf(stderr, "+++ node: %d [%d]\n", node->var.id, node->ref);
                
                if ( node->run )
                        print_run_node(node->run);

                type->print(node->var.set);
	} else
                fprintf(stderr, "+++ root node [%d]\n", node->ref);
        

        if ( node->brother ) {
                fprintf(stderr, "=> brother\n");
                signature_engine_print_nodes(node->brother);
        }

        if ( node->child ) {
                fprintf(stderr, "=> child\n");
                signature_engine_print_nodes(node->child);
        }
        
	fprintf(stderr, "+++ end of node\n");
}




/*
 * print rules
 */
void signature_engine_print_rules(rules_t *rules)
{
        test_t *test;
        
	if ( ! rules ) {
		fprintf(stderr, "===== \n");
		return;
	} else
                fprintf(stderr, "===== rule\n");

	if ( ! rules->rule ) {
		fprintf(stderr, "no rule...\n");
                return;
        }
	
        for ( test = rules->rule->test; test != NULL; test = test->next ) {
                generic_type_t *type;
                
                type = signature_engine_get_type_by_id(test->var.id);
                if ( ! type )
                        fprintf(stderr, "error, no type\n");
                else {
                        if (test->inversed)
                                fprintf(stderr, "inversed test\n");
                        type->print(test->var.set);
                }
        }
        
	signature_engine_print_rules(rules->next);
}
