/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: enumeratenode.cc,v $
 * Revision 1.3  2001/04/27 20:02:50  slash
 * Make use of xmlChildrenNode and xmlRootNode.
 *
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:29  slash
 * Initial release.
 *
 */

#include "main.h"

#include "itemnode.h"
#include "itemizenode.h"
#include "enumeratenode.h"


// The following macros are used to provide an easy handling,
// if the apropriate propertiy isn't given in the XML-file.
#define TYPE_       (type_ != UNDEFINED ? type_ : DECIMAL)
#define STEP_       (step_ != -1 ? step_ : 0)
#define CLOSING_P_  (closing_p_ != -1 ? closing_p_ : 1)


//---------------------------------------------------------------------------
EnumerateNode::EnumerateNode(xmlNodePtr node,
                             unsigned int level,
                             int with_left_margin)
  : ListNode(level, with_left_margin)
{
    type_ = UNDEFINED;
    step_ = -1;
    closing_p_ = -1;

    // Extract the properties of the itemize-tag.
    char *type = (char*)xmlGetProp(node, (xmlChar*)"type");
    if (type) {
        if      (!strcmp(type, "decimal"))     type_ = DECIMAL;
        else if (!strcmp(type, "upper-alpha")) type_ = UPPER_ALPHA;
        else if (!strcmp(type, "lower-alpha")) type_ = LOWER_ALPHA;
        else if (!strcmp(type, "upper-roman")) type_ = UPPER_ROMAN;
        else if (!strcmp(type, "lower-roman")) type_ = LOWER_ROMAN;
    }
    char *step = (char*)xmlGetProp(node, (xmlChar*)"step");
    if (step) {
        if      (!strcmp(step, "yes")) step_ = 1;
        else if (!strcmp(step, "no"))  step_ = 0;
    }
    char *closing_p = (char*)xmlGetProp(node, (xmlChar*)"closing_p");
    if (closing_p) {
        if      (!strcmp(closing_p, "yes")) closing_p_ = 1;
        else if (!strcmp(closing_p, "no"))  closing_p_ = 0;
    }


    // Extract the sub-nodes.
    for (node = node->xmlChildrenNode; node; node = node->next) {
        if (!strcmp((char*)node->name,"item")) {
            child_nodes_.push_back(new ItemNode(node, CLOSING_P_));
            is_sub_list_.push_back(0);
            number_of_html_steps_++;

        } else if (!strcmp((char*)node->name,"itemize")) {
            ListNode *sub_list = new ItemizeNode(node, level_+1);
            child_nodes_.push_back(sub_list);
            is_sub_list_.push_back(1);
            if (sub_list->isHTMLStepped()) {
                number_of_html_steps_ += sub_list->getNumberOfHTMLSteps();
            } else {
                number_of_html_steps_++;
            }

        } else if (!strcmp((char*)node->name,"enumerate")) {
            ListNode *sub_list = new EnumerateNode(node, level_+1);
            child_nodes_.push_back(sub_list);
            is_sub_list_.push_back(1);
            if (sub_list->isHTMLStepped()) {
                number_of_html_steps_ += sub_list->getNumberOfHTMLSteps();
            } else {
                number_of_html_steps_++;
            }

        } else {
            Number_Of_Errors_++;
        }
    }
}

//---------------------------------------------------------------------------
EnumerateNode::~EnumerateNode()
{
}


//---------------------------------------------------------------------------
void EnumerateNode::writeLaTeX() const
{
    // Set the enumeration-type.
    char *level;
    switch (level_) {
      // LaTeX supports only 4 levels.
      case 0:  level = "i"; break;
      case 1:  level = "ii"; break;
      case 2:  level = "iii"; break;
      case 3:  level = "iv"; break;
      default: level = ""; break;
    }
    Output_ << "\\renewcommand{\\labelenum" << level << "}{";
    switch (TYPE_) {
      case DECIMAL:     Output_ << "\\arabic"; break;
      case UPPER_ALPHA: Output_ << "\\Alph"; break;
      case LOWER_ALPHA: Output_ << "\\alph"; break;
      case UPPER_ROMAN: Output_ << "\\Roman"; break;
      case LOWER_ROMAN: Output_ << "\\roman"; break;
      case UNDEFINED:   break;  // To prevent a compiler-warning.
    }
    Output_ << "{enum" << level << "}.}" << endl;

    // Write the enumeration.
    Output_ << "\\begin{enumerate}" << endl;
    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeLaTeX();
    }
    Output_ << "\\end{enumerate}" << endl;
}

//---------------------------------------------------------------------------
void EnumerateNode::writeXML() const
{
    Output_ << "<enumerate";
    switch (type_) {
      case UNDEFINED:   break;
      case DECIMAL:     Output_ << " type=\"decimal\""; break;
      case UPPER_ALPHA: Output_ << " type=\"upper-alpha\""; break;
      case LOWER_ALPHA: Output_ << " type=\"lower-alpha\""; break;
      case UPPER_ROMAN: Output_ << " type=\"upper-roman\""; break;
      case LOWER_ROMAN: Output_ << " type=\"lower-roman\""; break;
    }
    if (step_ != -1) {
        Output_ << " step=\"" << (step_ == 1 ? "yes" : "no") << "\"";
    }
    if (closing_p_ != -1) {
        Output_ << " closing_p=\"" << (closing_p_ == 1 ? "yes" : "no") << "\"";
    }
    Output_ << ">" << endl;

    for (unsigned int i=0; i<child_nodes_.size(); i++) {
        child_nodes_[i]->writeXML();
    }

    Output_ << "</enumerate>" << endl;
}


//---------------------------------------------------------------------------
int EnumerateNode::isHTMLStepped() const
{
    return STEP_;
}

//---------------------------------------------------------------------------
unsigned int EnumerateNode::getNumberOfHTMLSteps() const
{
    return number_of_html_steps_;
}


//---------------------------------------------------------------------------
void EnumerateNode::writeHTMLHeader() const
{
    if (with_left_margin_ && (level_ == 0)) {
        Output_ << "<DIV class=\"";
        switch (getLength() / 16) {
          case 0:  Output_ << "listtiny"; break;
          case 1:  Output_ << "listmedium"; break;
          default: Output_ << "listlarge"; break;
        }
        Output_ << "\">" << endl;
    }

    Output_ << "<OL style=\"list-style-type:";
    switch (TYPE_) {
      case DECIMAL:     Output_ << "decimal"; break;
      case UPPER_ALPHA: Output_ << "upper-alpha"; break;
      case LOWER_ALPHA: Output_ << "lower-alpha"; break;
      case UPPER_ROMAN: Output_ << "upper-roman"; break;
      case LOWER_ROMAN: Output_ << "lower-roman"; break;
      case UNDEFINED:   break;  // To prevent a compiler-warning.
    }
    Output_ << "\">" << endl;
}

//---------------------------------------------------------------------------
void EnumerateNode::writeHTMLFooter() const
{
    Output_ << "</OL>" << endl;

    if (with_left_margin_ && (level_ == 0)) {
        Output_ << "</DIV>" << endl;
    }
}
