/* Copyright (C) 1999-2000 Bernhard Trummer
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 *
 * $Log: main.cc,v $
 * Revision 1.2  2001/04/20 15:40:36  slash
 * Make use of the function xmlGetProp().
 *
 * Revision 1.1.1.1  2001/01/30 18:36:38  slash
 * Initial release.
 *
 */

#include <config.h>
#include <stdlib.h>
#include <stdio.h>
#include <string>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif

// The following defines are needed for the AC_HEADER_DIRENT macro.
// (See the autoconf documentation)
#if HAVE_DIRENT_H
# include <dirent.h>
#else
# define dirent direct
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif

#include <libintl.h>
#define _(String) gettext(String)

#include <fstream.h>

#include "main.h"
#include "parameters.h"
#include "presentation.h"

#include <magick/magick.h>


char Buffer_[BUFFER_SIZE];

unsigned int Current_Line_Number_;
unsigned int Number_Of_Errors_;

Parameters Parameters_;
Presentation Presentation_;
ofstream Output_;

extern int xmlDoValidityCheckingDefaultValue;


//---------------------------------------------------------------------------
void print_help_text()
{
    ENDL;
    printf(_("Usage: prestimel [options] XML-file"));
    ENDL;

    printf("  -v, --%s    ", OPTION_VERSION);
    printf(_("Print version and exit."));
    ENDL;

    printf("  -h, --%s       ", OPTION_HELP);
    printf(_("Print help and exit."));
    ENDL;

//  printf("  -l[pdf], --%s\n", OPTION_LATEX_OUTPUT);
    printf("  -l, --%s\n", OPTION_LATEX_OUTPUT);
    printf(_("                   Create a LaTeX-file rather than HTML-slides."));
    ENDL;
//  printf(_("                   If the parameter \"pdf\" is given, the output will be\n                   optimized for pdflatex.")
//  ENDL;

    printf("  -p DIR, --%s=DIR\n", OPTION_PATH_TO_THEMES);
    printf(_("                   Set the path to the themes."));
    ENDL;

    printf("  -t[theme], --%s[=theme]\n", OPTION_THEME);
    printf(_("                   Use the theme \"theme\". If no theme is given, print a\n                   list of all available themes."));
    ENDL;

    printf("  -c FILE, --%s=FILE\n", OPTION_CONFIG_FILE);
    printf(_("                   Set the config-file, which should be used for the\n                   initialization."));
    ENDL;

    printf("  -x, --%s\n", OPTION_DISABLE_XML_SETTINGS);
    printf(_("                   Disable the settings stored in the XML-file."));
    ENDL;

    printf("  -R RES, --%s=RES\n", OPTION_RESOLUTION);
    printf(_("                   Set the resolution (640x480, 800x600 or 1024x768)."));
    ENDL;

    printf("  -B, --%s\n", OPTION_DISABLE_BACKGROUND);
    printf(_("                   Disable the background-graphic for the slides."));
    ENDL;

    printf("  -H, --%s\n", OPTION_DISABLE_HEADER);
    printf(_("                   Disable the background-graphic for the header."));
    ENDL;

    printf("  -F, --%s\n", OPTION_DISABLE_FIRST_LAST);
    printf(_("                   Disable the buttons to the first and last slide."));
    ENDL;

    printf("  -N, --%s\n", OPTION_DISABLE_NOTE_BUTTON);
    printf(_("                   Disable the note-button."));
    ENDL;

    printf("  -S, --%s\n", OPTION_SLOPPY_BUTTONS);
    printf(_("                   Draw the buttons below the content of the slides rather\n                   than at fixed positions."));
    ENDL;

    printf("  -T, --%s\n", OPTION_TOP_NAVIGATION);
    printf(_("                   Draw the buttons on the top of the slides."));
    ENDL;

    printf("  -O, --%s\n", OPTION_PRINT_OUTLINE);
    printf(_("                   Write the LaTeX-file as outline rather than slides."));
    ENDL;
}


//---------------------------------------------------------------------------
void list_available_themes()
{
    DIR *directory,*directory2 = NULL;
    dirent *entry,*entry2;
    string dir_name;

    ENDL;

    if (Parameters_.getPathToThemes().empty()) {
        printf(_("I cannot list the available themes, because the theme-path is uninitialized."));
        ENDL;
        return;
    }
    if ((directory = opendir(Parameters_.getPathToThemes().c_str())) == NULL) {
        printf(_("The theme-path \"%s\" doesn't exist."),
               Parameters_.getPathToThemes().c_str());
        ENDL;
        return;
    }

    printf(_("Available themes in \"%s\" :"),
           Parameters_.getPathToThemes().c_str());
    ENDL;

    unsigned int number_of_themes = 0;
    while ((entry = readdir(directory)) != NULL) {
        if (directory2)  closedir(directory2);
        dir_name = Parameters_.getPathToThemes() + "/" + entry->d_name;
        directory2 = opendir(dir_name.c_str());

        if (directory2) {
            while ((entry2 = readdir(directory2)) != NULL) {
                // Check, if the config-file exists
                if (!strcmp("config", entry2->d_name)) {
                    printf("    %s\n", entry->d_name);
                    number_of_themes++;
                }
            }
        }
    }
    closedir(directory);

    ENDL;
    if (number_of_themes > 0) {
        printf(_("Select a theme with the option \"-ttheme\"."));
    } else {
        printf(_("There are no themes available."));
    }
    ENDL;
}


//---------------------------------------------------------------------------
int parse_commandline_options(int argc, char **argv)
{
    int c;

#ifdef HAVE_GETOPT_LONG
    option long_options[] = {
        { OPTION_VERSION,0,0,'v' },
        { OPTION_HELP,0,0,'h' },
        { OPTION_LATEX_OUTPUT,2,0,'l' },
        { OPTION_PATH_TO_THEMES,1,0,'p' },
        { OPTION_THEME,2,0,'t' },
        { OPTION_CONFIG_FILE,1,0,'c' },
        { OPTION_DISABLE_XML_SETTINGS,0,0,'x' },

        { OPTION_RESOLUTION,1,0,'R' },
        { OPTION_DISABLE_BACKGROUND,0,0,'B' },
        { OPTION_DISABLE_HEADER,0,0,'H' },
        { OPTION_DISABLE_FIRST_LAST,0,0,'F' },
        { OPTION_DISABLE_NOTE_BUTTON,0,0,'N' },
        { OPTION_SLOPPY_BUTTONS,0,0,'S' },
        { OPTION_TOP_NAVIGATION,0,0,'T' },

        { OPTION_PRINT_OUTLINE,0,0,'O' },

        { 0,0,0,0 }
    };

    int option_index;

//  while ((c = getopt_long(argc, argv, "vhl::p:t::d:c:xR:BHFNSTO",
    while ((c = getopt_long(argc, argv, "vhlp:t::d:c:xR:BHFNSTO",
                            long_options, &option_index)) != EOF) {
#else
//  while ((c = getopt(argc, argv, "vhl::p:t::d:c:xR:BHFNSTO")) != EOF) {
    while ((c = getopt(argc, argv, "vhlp:t::d:c:xR:BHFNSTO")) != EOF) {
#endif //HAVE_GETOPT_LONG

        switch (c) {
          case 'v':
            ENDL;
            printf(_("This is %s version %s."), PACKAGE, VERSION);
            ENDL;
            return -1;
            break;
            
          case 'l':
//          if (optarg == NULL) {
                Parameters_.output_format_ = LATEX;
//          } else if (!strcmp(optarg, "pdf")) {
//              Parameters_.output_format_ = PDFLATEX;
//          } else {
//              cerr << _("Error :") << " \"" << optarg << "\" "
//                   << _("is an invalid argument for -l.") << endl;
//              return -1;
//          }
            break;

          case 'p':
            Parameters_.cl_path_to_themes_ = optarg;
            break;

          case 't':
            if (optarg == NULL) {
                list_available_themes();
                return -1;
            }
            Parameters_.cl_html_theme_ = optarg;
            break;

          case 'c':
            Parameters_.cl_config_file_ = optarg;
            break;

          case 'x':
            Parameters_.cl_disable_xml_settings_ = 1;
            break;

          case 'R':
            if (!strcmp(optarg, "640x480")) {
                Parameters_.cl_html_resolution_ = 640;
            } else if (!strcmp(optarg, "800x600")) {
                Parameters_.cl_html_resolution_ = 800;
            } else if (!strcmp(optarg, "1024x768")) {
                Parameters_.cl_html_resolution_ = 1024;
            }
            break;

          case 'B':
            Parameters_.cl_html_disable_background_graphic_ = 1;
            break;

          case 'H':
            Parameters_.cl_html_disable_header_graphic_ = 1;
            break;

          case 'F':
            Parameters_.cl_html_disable_first_last_button_ = 1;
            break;

          case 'N':
            Parameters_.cl_html_disable_note_button_ = 1;
            break;

          case 'S':
            Parameters_.cl_html_sloppy_buttons_ = 1;
            break;

          case 'T':
            Parameters_.cl_html_top_navigation_ = 1;
            break;

          case 'O':
            Parameters_.cl_latex_print_outline_ = 1;
            break;

          default:
            print_help_text();
            return -1;
        }
    }

    // Check, if an XML-file is given.
    if (optind < argc) {
        Parameters_.xml_file_ = argv[optind];
    } else {
        ENDL;
        printf(_("Error : No XML-file specified."));
        ENDL;
        return -1;
    }

    return 0;
}


//---------------------------------------------------------------------------
int main(int argc, char **argv)
{
    // locale setup.
    setlocale(LC_MESSAGES, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);

    // Enable validity checking of the XML-file.
    xmlDoValidityCheckingDefaultValue = 1;

    // Disable errors and warnings from ImageMagick.
    SetErrorHandler(NULL);
    SetWarningHandler(NULL);

    // Initialize the global variables.
    Current_Line_Number_ = 0;
    Number_Of_Errors_ = 0;

    // Evaluate the commandline-options.
    if (parse_commandline_options(argc, argv)) {
        return -1;
    }

    // Read the config-file and override the given settings.
    Parameters_.readConfigFile();

    // read this earlier, so that HTML node with theme comes in earlier
    if (Presentation_.parseXMLFile() != 0) {
        return -1;
    }

    // Do some final initializations.
    if (Parameters_.initPathToTheme()) {
        return -1;
    }
    Parameters_.initImageSizes();
    ENDL;

    switch (Parameters_.output_format_) {
      case HTML:  Presentation_.writeHTML(); break;
      case LATEX: Presentation_.writeLaTeX(); break;
      case XML:   Presentation_.writeXML(); break;
    }

    return 0;
}


//---------------------------------------------------------------------------
int fcopy(const char *src,
          const char *dst)
{
    char copy_buffer[4096];
    size_t characters_read, characters_written;
    int return_value = 0;

    FILE *f_src = fopen(src, "r");
    if (!f_src) {
        printf(_("The file \"%s\" could not be opened."), src);
        ENDL;
        return -1;
    }
    FILE *f_dst = fopen(dst, "w");
    if (!f_dst) {
        printf(_("The file \"%s\" could not be created."), dst);
        ENDL;
        return -1;
    }

    while (!feof(f_src)) {
        characters_read = fread(copy_buffer, sizeof(char), 4096, f_src);
        if (characters_read > 0) {
            characters_written = fwrite(copy_buffer, sizeof(char),
                                        characters_read, f_dst);
            if (characters_written != characters_read) {
                printf(_("Error while writing to the file \"%s\"."), dst);
                ENDL;
                printf(_("Maybe the disk is full."));
                ENDL;
                return_value = -1;
                break;
            }
        }
    }

    fclose(f_src);
    fclose(f_dst);

    return return_value;
}


//---------------------------------------------------------------------------
int ping_image(const char *filename,
               ImageSize *size)
{
    Image *image;
    ImageInfo ii;
    GetImageInfo(&ii);

    strncpy(ii.filename, filename, MaxTextExtent);
    ii.filename[MaxTextExtent-1] = '\0';

#ifdef IMAGE_MAGICK_52
    ExceptionInfo exception;
    image = PingImage(&ii, &exception);
#else
    image = PingImage(&ii);
#endif

    if (image) {
        size->height_ = image->rows;
        size->width_ = image->columns;
        DestroyImage(image);
    } else {
        printf(_("The image \"%s\" doesn't exist."), filename);
        ENDL;
        return -1;
    }

    return 0;
}
