/****************************************************************************
** chatdlg.cpp - a single window chat dialog
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"chatdlg.h"

#include<qsplitter.h>
#include<qlayout.h>
#include<qdatetime.h>
#include<qtooltip.h>
#include<qpalette.h>

#ifdef Q_WS_WIN
#include<windows.h>
#endif


ChatDlg::ChatDlg(UserListItem *item, const QString &_whoami, int _localStatus, QWidget *parent)
:QWidget(parent, "chatwindow"), UniqueWindow(TRUE, "ChatDlg", this, cleanJid(item->jid))
{
	//setWFlags(getWFlags() | WDestructiveClose);
	resize(option.sizeChatDlg);

	// save everything locally
	v_jid = item->jid;
	v_nick = item->nick;
	v_status = item->state;

	whoami = _whoami;
	localStatus = _localStatus;
	pending = 0;
	keepOpen = FALSE;
	selfDestruct = 0;
	flashTimer = 0;

	setIcon(anim_chat->base());

        QVBoxLayout *vb1 = new QVBoxLayout(this, 4);
        QSplitter *sp = new QSplitter(Vertical, this);
        vb1->addWidget(sp);

	// ---- single
	lo_single = new QWidget(sp);
	QVBoxLayout *lo_single_vb = new QVBoxLayout(lo_single, 4);
	QHBoxLayout *lo_single_hb = new QHBoxLayout(lo_single_vb);
	lb_statusIcon = new QLabel(lo_single);
	lo_single_hb->addWidget(lb_statusIcon);

	le_nick = new QLineEdit(lo_single);
	le_nick->setReadOnly(TRUE);
	le_nick->setFixedWidth(120);
	lo_single_hb->addWidget(le_nick);

	lo_single_hb->addSpacing(16);

	QLabel *lo_single_jid = new QLabel(tr("Jabber ID"), lo_single);
	lo_single_hb->addWidget(lo_single_jid);
	le_jid = new QLineEdit(lo_single);
	le_jid->setReadOnly(TRUE);
	le_jid->setText(v_jid);
	le_jid->setCursorPosition(0);
	QToolTip::add(le_jid, v_jid);
	lo_single_hb->addWidget(le_jid);

	t = new MsgMle(lo_single);
	connect(t, SIGNAL(linkClicked(const QString &)), SIGNAL(aOpenURL(const QString &)));
	t->setReadOnly(TRUE);
	t->setUndoRedoEnabled(FALSE);
	lo_single_vb->addWidget(t);

	//vb1->addWidget(t);
	QWidget *w = new QWidget(sp);
	//vb1->addWidget(w);
	QVBoxLayout *vb2 = new QVBoxLayout(w, 4);
	QHBoxLayout *hb1 = new QHBoxLayout(vb2);

	QToolButton *tb_clear = new QToolButton(w);
	tb_clear->setIconSet(QIconSet(*pix_chatclear));
	connect(tb_clear, SIGNAL(clicked()), SLOT(doClearButton()));
	QToolTip::add(tb_clear, tr("Clear chat window"));
	hb1->addWidget(tb_clear);

	hb1->addStretch(1);

	tb_info = new QToolButton(w);
	tb_info->setIconSet(QIconSet(*pix_info));
	connect(tb_info, SIGNAL(clicked()), SLOT(doInfo()));
	QToolTip::add(tb_info, tr("User info"));
	hb1->addWidget(tb_info);

	tb_history = new QToolButton(w);
	tb_history->setIconSet(QIconSet(*pix_history));
	connect(tb_history, SIGNAL(clicked()), SLOT(doHistory()));
	QToolTip::add(tb_history, tr("Message history"));
	hb1->addWidget(tb_history);

	QHBoxLayout *hb2 = new QHBoxLayout(vb2);
	mle = new ChatMle(w);
	mle->setMinimumHeight(48);
	hb2->addWidget(mle);
	//hb2->addStretch(1);

	/*(tb_send = new QToolButton(w);
	QPalette mypal = tb_send->palette();
	mypal.setColor(QColorGroup::Button, palette().color(QPalette::Normal, QColorGroup::Background));
	tb_send->setPalette(mypal);

	QIconSet is(*pix_chatsend1, QIconSet::Large);
	//is.setPixmap(*pix_chatsend1, QIconSet::Large, QIconSet::Normal);
	is.setPixmap(*pix_chatsend2, QIconSet::Large, QIconSet::Active);
	tb_send->setIconSet(is);
	tb_send->setAutoRaise(TRUE);
	tb_send->setUsesBigPixmap(TRUE);
	//tb_send->setFixedSize(56,56);
	connect(tb_send, SIGNAL(clicked()), SLOT(doSend()));
	hb2->addWidget(tb_send);*/

	vb2->activate();

	QValueList<int> list;
	//list << 276;
	//list << 144;
	list << 324;
	list << 96;
	sp->setSizes(list);

	updateContact(item);

	mle->setTextFormat(PlainText);
	t->setTextFormat(RichText);
	mle->setFocus();

	//if(localStatus == STATUS_OFFLINE)
	//	tb_send->setEnabled(FALSE);

	setLooks();

	QString msg = expandEntities(jidnick(v_jid, v_nick)) + " is " + status2txt(v_status);
	if(!item->statusString.isEmpty()) {
		msg += QString(" [%1]").arg(item->statusString);
	}
	appendSysMsg(msg);
}

ChatDlg::~ChatDlg()
{
}

QSize ChatDlg::defaultSize()
{
	return QSize(380, 420);
}

void ChatDlg::resizeEvent(QResizeEvent *e)
{
	if(option.keepSizes)
		option.sizeChatDlg = e->size();
}

void ChatDlg::keyPressEvent(QKeyEvent *e)
{
        if(e->key() == Key_Escape)
                close();
	else if(e->key() == Key_Return || (e->key() == Key_S && (e->state() & AltButton)))
		doSend();
        else if(e->key() == Key_H && (e->state() & ControlButton))
                doHistory();
	else if(e->key() == Key_PageUp && (e->state() & ShiftButton))
		t->setContentsPos(t->contentsX(), t->contentsY() - t->visibleHeight()/2);
	else if(e->key() == Key_PageDown && (e->state() & ShiftButton))
		t->setContentsPos(t->contentsX(), t->contentsY() + t->visibleHeight()/2);
        else
                e->ignore();
}

void ChatDlg::closeEvent(QCloseEvent *e)
{
	if(keepOpen) {
		int n = QMessageBox::information(this, tr("Warning"), tr("A new chat message was just received.\nDo you still want to close the window?"), tr("&Yes"), tr("&No"));
		if(n != 0)
			return;
	}

	// destroy the dialog if delChats is dcClose
	if(option.delChats == dcClose)
		setWFlags(getWFlags() | WDestructiveClose);
	else {
		if(option.delChats == dcHour)
			setSelfDestruct(60);
		else if(option.delChats == dcDay)
			setSelfDestruct(60 * 24);
	}

	mle->setFocus();
	e->accept();
}

void ChatDlg::showEvent(QShowEvent *)
{
	setSelfDestruct(0);
}

void ChatDlg::windowActivationChange(bool oldstate)
{
	QWidget::windowActivationChange(oldstate);

	// if we're bringing it to the front, get rid of the '*' if necessary
	if(isActiveWindow()) {
		if(pending > 0) {
			pending = 0;
			updateCaption();
		}
		doFlash(FALSE);

		mle->setFocus();
	}
}

/* static */ ChatDlg *ChatDlg::find(const QString &xjid)
{
        return (ChatDlg *)UniqueWindowBank::find("ChatDlg", cleanJid(xjid));
}

void ChatDlg::localUpdate(const JabRosterEntry &e)
{
	localStatus = e.localStatus();
	whoami = e.nick;

	//if(localStatus == STATUS_OFFLINE)
	//	tb_send->setEnabled(FALSE);
	//else
	//	tb_send->setEnabled(TRUE);
}

void ChatDlg::updateContact(UserListItem *item)
{
        if(jidcmp(item->jid, v_jid)) {
		int oldstate = v_status;

                v_jid = item->jid;
                v_nick = item->nick;
                v_status = item->state;

                QString nick = jidnick(item->jid, item->nick);

                if(nick != le_nick->text()) {
                        le_nick->setText(nick);
			le_nick->setCursorPosition(0);
			QToolTip::add(le_nick, nick);
		}
		if(nick != caption()) {
			dispNick = nick;
			updateCaption();
		}

		lb_statusIcon->setPixmap(jidIsAgent(v_jid) ? transport2icon(v_jid, v_status) : status2pix(v_status));

		if(oldstate != v_status && !isHidden()) {
			QString msg = expandEntities(jidnick(v_jid, v_nick)) + " is " + status2txt(v_status);
			if(!item->statusString.isEmpty()) {
				msg += QString(" [%1]").arg(item->statusString);
			}
			appendSysMsg(msg);
		}
        }
}

void ChatDlg::incomingMessage(const Message &msg)
{
	//if(!jidcmp(msg.from, v_jid))
	//	return;

	appendMessage(msg);
}

void ChatDlg::doSend()
{
	if(mle->text().isEmpty())
		return;

	if(mle->text() == "/clear") {
		mle->setText("");
		doClear();
		return;
	}

	if(localStatus == STATUS_OFFLINE)
		return;

	//if(!tb_send->isEnabled())
	//	return;

	Message msg;
	msg.to = v_jid;
	msg.type = MESSAGE_CHAT;
	if(option.outgoingAs == 1)
		msg.type = MESSAGE_NORM;
	msg.text = mle->text();
	msg.timeStamp = QDateTime::currentDateTime();
	msg.originLocal = TRUE;
	msg.late = FALSE;

	aSend(msg);
	appendMessage(msg);

	mle->setText("");
}

void ChatDlg::appendMessage(const Message &msg)
{
	QString who, color;

	if(msg.originLocal) {
		who = whoami;
		color = "#FF0000";
	}
	else {
		who = jidnick(v_jid, v_nick);
		color = "#0000FF";
	}
	if(msg.late)
		color = "#008000";

	QString timestr;
	QDateTime time = msg.timeStamp;
	timestr.sprintf("%02d:%02d:%02d", time.time().hour(), time.time().minute(), time.time().second());

	int y = t->contentsHeight() - t->visibleHeight();
	if(y < 0)
		y = 0;
	bool atBottom = (t->contentsY() < y - 32) ? FALSE: TRUE;

	//t->setUpdatesEnabled(FALSE);
	if(msg.text.left(4) == "/me ") {
		QString txt = plain2rich(msg.text.mid(4));
		t->append(QString("<font color=\"%1\">").arg(color) + QString("[%1]").arg(timestr) + QString(" *%1 ").arg(expandEntities(who)) + txt + "</font><br>\n");
	}
	else {
		QString txt = linkify(plain2rich(msg.text));
		t->append(QString("<font color=\"%1\">").arg(color) + QString("[%1] &lt;").arg(timestr) + expandEntities(who) + QString("&gt;</font> ") + txt + "<br>\n");
	}

	if(!(!msg.originLocal && !atBottom)) {
		// scroll after some event loop processing
		//QTimer::singleShot(0, this, SLOT(deferredScroll()));
		deferredScroll();
	}
	//if(msg.originLocal)
	//	deferredScroll();

	// if we're not active, notify the user by changing the title
	if(!isActiveWindow()) {
		++pending;
		updateCaption();
		doFlash(TRUE);
	}

	if(!msg.originLocal) {
		keepOpen = TRUE;
		QTimer::singleShot(1000, this, SLOT(setKeepOpenFalse()));
	}
}

void ChatDlg::doHistory()
{
	aHistory(v_jid);
}

void ChatDlg::doAuth()
{
}

void ChatDlg::doAdd()
{
}

void ChatDlg::doInfo()
{
	aInfo(v_jid);
}

void ChatDlg::setLooks()
{
	// update the font
	QFont f;
	f.fromString(option.font[fChat]);
	t->setFont(f);
	mle->setFont(f);

	// update the status icon
	lb_statusIcon->setPixmap(jidIsAgent(v_jid) ? transport2icon(v_jid, v_status) : status2pix(v_status));

	// toolbuttons
	tb_info->setIconSet(QIconSet(*pix_info));

	// update the widget icon
	setIcon(anim_chat->base());
}

void ChatDlg::optionsUpdate()
{
	setLooks();

	if(isHidden()) {
		if(option.delChats == dcClose) {
			deleteLater();
			return;
		}
		else {
			if(option.delChats == dcHour)
				setSelfDestruct(60);
			else if(option.delChats == dcDay)
				setSelfDestruct(60 * 24);
			else
				setSelfDestruct(0);
		}
	}
}

void ChatDlg::appendSysMsg(const QString &str)
{
	QString timestr;
	QDateTime time = QDateTime::currentDateTime();
	timestr.sprintf("%02d:%02d:%02d", time.time().hour(), time.time().minute(), time.time().second());

	//t->setUpdatesEnabled(FALSE);
	t->append(QString("<font color=\"#00A000\">[%1]").arg(timestr) + QString(" *** %1</font><br>\n").arg(str));

	deferredScroll();

	// scroll after some event loop processing
	//QTimer::singleShot(0, this, SLOT(deferredScroll()));
}

void ChatDlg::updateCaption()
{
	QString cap = "";

	if(pending > 0) {
		cap += "* ";
		if(pending > 1)
			cap += QString("[%1] ").arg(pending);
	}
	cap += dispNick;

	// if taskbar flash, then we need to erase first and redo
#ifdef Q_WS_WIN
	bool on = FALSE;
	if(flashTimer)
		on = flashCount & 1;
	if(on)
		FlashWindow(winId(), TRUE);
#endif
	setCaption(cap);
#ifdef Q_WS_WIN
	if(on)
		FlashWindow(winId(), TRUE);
#endif
}

void ChatDlg::deferredScroll()
{
	//t->moveCursor(QTextEdit::MoveEnd, FALSE);
	t->scrollToBottom();
	//t->setUpdatesEnabled(TRUE);
	//t->update();
}

void ChatDlg::doClearButton()
{
	int n = QMessageBox::information(this, tr("Warning"), tr("Are you sure you want to clear the chat window?\n(note: does not affect saved history)"), tr("&Yes"), tr("&No"));
	if(n == 0)
		doClear();
}

void ChatDlg::doClear()
{
	t->setText("");
}

void ChatDlg::setKeepOpenFalse()
{
	keepOpen = FALSE;
}

void ChatDlg::setSelfDestruct(int minutes)
{
	if(minutes <= 0) {
		if(selfDestruct) {
			delete selfDestruct;
			selfDestruct = 0;
		}
		return;
	}

	if(!selfDestruct) {
		selfDestruct = new QTimer(this);
		connect(selfDestruct, SIGNAL(timeout()), SLOT(deleteLater()));
	}

	selfDestruct->start(minutes * 60000);
}

#ifdef Q_WS_WIN
void ChatDlg::doFlash(bool yes)
{
	if(yes) {
		if(flashTimer)
			return;
		flashTimer = new QTimer(this);
		connect(flashTimer, SIGNAL(timeout()), SLOT(flashAnimate()));
		flashCount = 0;
		flashAnimate(); // kick the first one immediately
		flashTimer->start(500);
	}
	else {
		if(flashTimer) {
			delete flashTimer;
			flashTimer = 0;
			FlashWindow(winId(), FALSE);
		}
	}
}
#else
void ChatDlg::doFlash(bool)
{
}
#endif

void ChatDlg::flashAnimate()
{
#ifdef Q_WS_WIN
	FlashWindow(winId(), TRUE);
	++flashCount;
	//if(flashCount == 10)
	//	flashCount = 0;
	if(flashCount == 5)
		flashTimer->stop();
#endif
}

