/****************************************************************************
** contactview.h - an Licq-like contact list
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#ifndef CONTACTVIEW_H
#define CONTACTVIEW_H

#include<qlistview.h>
#include<qpainter.h>
#include<qpixmap.h>
#include<qstringlist.h>
#include<qptrlist.h>
#include<qtooltip.h>
#include<qtimer.h>
#include<qdict.h>
#include<qaction.h>
#include"jabcommon.h"
#include"userlist.h"


/****************************************************************************
  ContactView

  Holds a contact list.

****************************************************************************/


#define CV_PROFILE	0
#define CV_GROUP        1
#define CV_CONTACT      2

#define CVG_SELF        0
#define CVG_ONLINE      1
#define CVG_OFFLINE     2
#define CVG_NOTINLIST   3
#define CVG_AUTHWAIT    4
#define CVG_AGENTS      5
#define CVG_USER        6


#define LVI2CVI(x) (static_cast<ContactViewItem*>(x))


class ContactView;
class ContactViewItem;

struct ContactProfileItem
{
	UserListItem uli;
	ContactViewItem *cvi;
};

// ContactProfile:  this holds/manipulates a roster profile for an account
class ContactProfile : public QObject
{
	Q_OBJECT

public:
	ContactProfile(const QString &, const QString &, int _localStatus, bool, ContactView *);
	~ContactProfile();

	QString jid, name;
	ContactView *cv;
	ContactViewItem *cvi, *cvi_online, *cvi_offline;
	QDict<ContactProfileItem> *roster;

	ContactView *contactView() { return cv; }
	ContactViewItem *online() { return cvi_online; }
	ContactViewItem *offline() { return cvi_offline; }

	void clear();
	ContactProfileItem *cvi2cpi(ContactViewItem *);
	ContactProfileItem *findByJid(const QString &jid);
	bool isMine(ContactViewItem *);
	ContactViewItem *findCVIByJid(const QString &jid);

	void doContextMenu(ContactViewItem *, const QPoint &);

	ContactViewItem *addGroup(int group);
	ContactViewItem *addGroup(const QString &group, ContactViewItem *par);

	ContactViewItem *checkGroup(int group);
	ContactViewItem *checkGroup(const QString &group, ContactViewItem *par);

	ContactViewItem *ensureGroup(int group);
	ContactViewItem *ensureGroup(const QString &group, int status);

	void changeGroup(ContactViewItem *item, ContactViewItem *group_new);
	void changeGroup(ContactViewItem *item, const QString &group);

	void checkDestroyGroup(ContactViewItem *group);
	void checkDestroyGroup(const QString &group);

	ContactProfileItem *addEntry(const UserListItem &i);
	void updateEntry(const UserListItem &);
	void removeEntry(const QString &jid);

	void showAlert(const QString &jid, int type);
	void clearAlert(const QString &jid, int at=0);
	void clearAllAlerts(const QString &jid);
	void animateNick(const QString &jid);

	void removeEntry(ContactProfileItem *pi);
	void clearAlert(ContactViewItem *, int at=0);
	void clearAllAlerts(ContactViewItem *);

	// useful functions to grab groups of users
	QStringList contactListFromCVGroup(ContactViewItem *);
	QStringList contactListFromGroup(const QString &groupName);
	int contactSizeFromCVGroup(ContactViewItem *);
	int contactSizeFromGroup(const QString &groupName);

	void updateGroupInfo(ContactViewItem *group);

	// shortcuts
	void scActionDefault(ContactViewItem *);
	void scRecvEvent(ContactViewItem *);
	void scSendMessage(ContactViewItem *);
	void scRename(ContactViewItem *);
	void scGroupChange(const QString &, const QString &);
	void scHistory(ContactViewItem *);
	void scOpenChat(ContactViewItem *);
	void scAgentSetStatus(ContactViewItem *, int);
	void scRemove(ContactViewItem *);
	void doItemRenamed(ContactViewItem *, const QString &);

signals:
	void actionDefault(ContactViewItem *);
	void actionRecvEvent(ContactViewItem *);
	void actionSendMessage(ContactViewItem *);
	void actionSendMessage(const QStringList &);
	void actionSendUrl(const QString &);
	void actionRemove(const QString &);
	void actionAuthorize(ContactViewItem *);
	void actionAdd(ContactViewItem *);
	void actionRename(ContactViewItem *, const QString &);
	void actionGroupChange(const QString &, const QString &);
	void actionGroupRename(const QString &, const QString &);
	void actionHistory(const QString &);
	void actionStatusShow(const QString &);
	void actionOpenChat(const QString &);
	void actionAgentSetStatus(const QString &, int);
	void actionInfo(const QString &);
	void actionOfferFile(const QString &);

public slots:
	void localUpdate(const JabRosterEntry &e);

private:
	QStringList buildGroupList();
	int localStatus;
};

// ContactViewItem:  this can be either a contact or a group
class ContactViewItem : public QListViewItem
{
public:
	// create profile
	ContactViewItem(const QString &xgroupname, int xgrouptype, QListView *parent);
	// create group
	ContactViewItem(const QString &xgroupname, int xgrouptype, QListViewItem *parent);
	// create user
	ContactViewItem(const QString &xjid, const QString &xname, bool xisAgent, QListViewItem *parent);

	~ContactViewItem();

	// read functions
	int type() { return v_type; }
	int status() { return v_status; }
	QString & jid() { return v_jid; }
	QString & nick() { return v_nick; }
	QString & group() { return v_group; }
	QString & sub() { return v_sub; }
	QString & statusString() { return v_statusString; }
	bool inList() { return v_inList; }
	bool isAgent() { return v_isAgent; }
	bool isAlert() { return v_isAlert; }
	bool isAnimateNick() { return v_isAnimateNick; }
	int groupType() { return v_groupType; }
	QString & groupName() { return v_groupName; }
	JabResourceList & resList() { return v_resList; }

	int parentGroupType(); // use with contacts: returns grouptype of parent group

	// write functions
	void setStatus(int);
	void resetStatus();
	void setNick(const QString &);
	void resetNick(); // use this to cancel a rename
	void setGroup(const QString &);
	void setSub(const QString &);
	void setStatusString(const QString &);
	void setInList(bool=TRUE);
	void setIsAgent(bool=TRUE);
	void setAlert(int type);
	void clearAlert(int at=0);
	void setAttr(const QString &xnick, const QString xgroup, const QString &sub, int status);
	void setAnimateNick();
	void stopAnimateNick();

	void setGroupName(const QString &);
	void setGroupInfo(const QString &);
	void resetGroupName(bool showInfo=TRUE);

	void animate();
	void resetAnim();

	void animateNick();
	void optionsUpdate();

	bool isAway();

	bool needSorting; // dirty flag for group items

private:
	int v_type, v_status;
	QString v_jid;
	QString v_nick;
	QString v_group;
	QString v_sub;
	QString v_statusString;
	bool v_inList, v_isAgent;
	JabResourceList v_resList;

	int v_groupType;
	QString v_groupName, v_groupInfo;

	bool v_isAlert, v_isAnimateNick;
	int alertType;
	int animStep;
	QPtrList<int> alertQueue;
	int animateNickX, animateNickColor;

	void init();
	void drawGroupIcon();

public:
	int rankGroup(int groupType) const;
	int rankStatus(int status) const;

	// reimplemented functions
	void paintFocus(QPainter *, const QColorGroup &, const QRect &);
	void paintBranches(QPainter *, const QColorGroup &, int, int, int);
	void paintCell(QPainter *, const QColorGroup & cg, int column, int width, int alignment);
	void setOpen(bool o);
	int compare(QListViewItem *, int, bool) const;
	virtual void insertItem(QListViewItem * newChild);
	virtual void takeItem(QListViewItem * item);
	void sort();

	//virtual bool acceptDrop(const QMimeSource *) const;
	//void dropped(QDropEvent *);
};


class ContactView : public QListView, public QToolTip
{
	Q_OBJECT
public:

	ContactView(QWidget *parent=0, const char *name=0);
	~ContactView();

	ContactProfile *createProfile(const QString &jid, const QString &name, bool unique=FALSE);
	ContactProfile *findProfile(const QString &jid);
	void removeProfile(const QString &jid);
	void removeProfile(ContactProfile *p);
	ContactProfile *defPro();
	ContactProfile *owner(ContactViewItem *);

	void clear();   // override QListView

	bool isShowOffline() { return v_showOffline; }
	bool isShowAgents() { return v_showAgents; }
	bool isShowAway() { return v_showAway; }

	void sort();
	void deferredContactSort();

	// very handy
	void ensureVisible(ContactViewItem *);
	void setVisibility(ContactViewItem *);

	QAction *qa_send, *qa_chat, *qa_ren, *qa_hist, *qa_logon, *qa_recv, *qa_rem;

private:
	bool v_showOffline, v_showAway, v_showAgents;

	// reimplemented
	void resizeEvent(QResizeEvent *);
	void keyPressEvent(QKeyEvent *);
	void viewportMousePressEvent(QMouseEvent *);
	void viewportMouseMoveEvent(QMouseEvent *);

	QPoint mousePressPos; // store pressed position, idea taken from Licq

	bool lcto_active; // double click active?
	QPoint lcto_pos;
	QListViewItem *lcto_item;

	QTimer *sortTimer;

	QPtrList<ContactProfile> profiles;

protected:
	// reimplemented
	void maybeTip(const QPoint &);
	//QDragObject *dragObject();
	void dragEnterEvent(QDragEnterEvent *);
	void dropEvent(QDropEvent *);

signals:
	void showOffline(bool);
	void showAway(bool);
	void showAgents(bool);

public slots:
	void setShowOffline(bool);
	void setShowAgents(bool);
	void setShowAway(bool);

	void optionsUpdate();

private slots:
	void qlv_singleclick(int, QListViewItem *, const QPoint &, int);
	void qlv_doubleclick(QListViewItem *);
	void qlv_contextPopup(QListViewItem *, const QPoint &, int);
	void qlv_itemRenamed(QListViewItem *, int, const QString &);
	void leftClickTimeOut();
	void animAlert();

	void doRecvEvent();
	void doRename();
	void doEnter();
	void doContext();
	void doSendMessage();
	void doOpenChat();
	void doHistory();
	void doLogon();
	void doRemove();

	void actualContactSort();
};

#endif
