/****************************************************************************
** jabber.cpp - communicates with the Jabber network
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"jabber.h"
#include"common.h"
#include<time.h>
#include<qregexp.h>
#include<qtimer.h>


/****************************************************************************
  Jabber
****************************************************************************/
Jabber::Jabber()
{
	host = "";
	port = 5222;

	v_status = STATUS_OFFLINE;
	v_isActive = FALSE;
	v_offlineEnabled = FALSE;

	connect(&stream, SIGNAL(connected()), SLOT(streamConnected()));
	connect(&stream, SIGNAL(error(int)), SLOT(streamError(int)));
	connect(&stream, SIGNAL(receivePacket(const QDomElement &)), SLOT(receivePacket(const QDomElement &)));

	//io.stream = &stream;
	connect(&io, SIGNAL(done(JabTask *)), SLOT(ioDone(JabTask *)));
	connect(&io, SIGNAL(outgoingPacket(const QDomElement &)), SLOT(ioOutgoingPacket(const QDomElement &)));
	connect(&io, SIGNAL(anyDone(JabTask *)), SLOT(doUpdate()));

	// queue
	queue = new JT_Queue(io.root());
	connect(queue, SIGNAL(countChanged()), SLOT(doUpdate()));
}

Jabber::~Jabber()
{
	stream.disc();
	delete queue;
}

void Jabber::setHost(const QString &xhost, int xport)
{
	host = xhost;
	port = xport;

	me.jid = QString("%1@%2").arg(user).arg(host);
}

void Jabber::setAccount(const QString &xuser, const QString &xpass, const QString &xresource)
{
	user = xuser;
	pass = xpass;
	resource = xresource;

	me.jid = QString("%1@%2").arg(user).arg(host);
	me.nick = user;
}

void Jabber::setNoop(int mills)
{
	stream.setNoop(mills);
}

void Jabber::send(const QString &str)
{
	if(stream.isConnected()) {
		stream.sendString(str.latin1());
		pdb(DEBUG_JABBER, QString("Jabber: wrote: [\n%1]\n").arg(str.latin1()));
	}
}

void Jabber::streamConnected()
{
	pdb(DEBUG_JABBER, "Jabber: stream connected\n");

	// tell the world
	connected();

	if(connType == JABCONN_LOGIN) {
		JT_Login *l = new JT_Login(io.root());
		connect(l, SIGNAL(receiveAuth()), SLOT(ioAuth()));
		connect(l, SIGNAL(receivePresence(int, const QString &)), SLOT(ioPresence(int, const QString &)));
		connect(l, SIGNAL(receiveRoster(JabRoster &)), SLOT(ioReceiveRoster(JabRoster &)));
		connect(l, SIGNAL(stepChanged()), SLOT(doUpdate()));

		l->login(user, pass, stream.id(), resource, i_status, i_statusString, i_priority, usePlain);
		l->go();

		// presence daemon
		JT_PushPresence *pp = new JT_PushPresence(io.root());
		connect(pp, SIGNAL(subscription(const Jid &, int)), SLOT(pushSubscription(const Jid &, int)));
		connect(pp, SIGNAL(presence(const JT_PushPresence::Info &)), SLOT(pushPresence(const JT_PushPresence::Info &)));

		// roster daemon
		JT_PushRoster *pr = new JT_PushRoster(io.root());
		connect(pr, SIGNAL(roster(JabRoster &)), SLOT(pushRoster(JabRoster &)));

		// message daemon
		JT_PushMessage *pm = new JT_PushMessage(io.root());
		connect(pm, SIGNAL(message(const JabMessage &)), SLOT(pushMessage(const JabMessage &)));

		// other stuff
		new JT_ServInfo(io.root());

		// user branch
		userTask = new JabTask(io.root());
		connect(userTask, SIGNAL(childAdded(JabTask *)), SLOT(doUpdate()));
		connect(userTask, SIGNAL(childRemoved(JabTask *)), SLOT(doUpdate()));
	}
	else if(connType == JABCONN_REGISTER) {
		JT_Register *r = new JT_Register(io.root());
		r->reg(user, pass);
		r->go();
	}
}

void Jabber::streamError(int x)
{
	pdb(DEBUG_JABBER, QString("Jabber: stream error=[%1]\n").arg(x));

	switch(x) {
		case JABSTREAM_ERR_DNS:
			err.msg = tr("Unable to resolve hostname");
			err.type = JABERR_CONNECT;
			break;
		case JABSTREAM_ERR_CONNREFUSED:
			err.msg = tr("Connection refused");
			err.type = JABERR_CONNECT;
			break;
		case JABSTREAM_ERR_CONNTIMEOUT:
			err.msg = tr("Connection timeout");
			err.type = JABERR_CONNECT;
			break;
		case JABSTREAM_ERR_HANDSHAKE:
			err.msg = tr("Unable to establish stream handshake");
			err.type = JABERR_CONNECT;
			break;
		case JABSTREAM_ERR_DISC:
		case JABSTREAM_ERR_SOCKET:
		default:
			err.type = JABERR_DISCONNECT;
			break;
	}

	disconnected();

	cleanup();
	doError();
}

void Jabber::cleanup(bool fast)
{
	pdb(DEBUG_JABBER, "Jabber: cleanup\n");

	queue->stop();
	//printf("JT_Queue::toString() = [%s]\n", queue->toString().latin1());
	io.reset();
	userTask = 0;

	if(!fast) {
		// set all contacts to offline
		for(JabRosterEntry *i = roster.first(); i; i = roster.next()) {
			QPtrListIterator<JabResource> it(i->res);
			JabResource *r;
			while((r = it.current())) {
				// clean the jid first, in case it has an attached resource already (like /registered)
				Jid j = QString("%1/%2").arg(cleanJid(i->jid)).arg(r->name);
				i->res.remove(r);
				resourceUnavailable(j);
			}
			// not offline mode? delete!
			if(!v_offlineEnabled)
				contactRemove(i);
		}
		// not offline mode?  clear our local roster
		if(!v_offlineEnabled)
			roster.clear();

		// remove 'me' resources (in reverse order)
		QPtrListIterator<JabResource> it(me.res);
		it.toLast();
		JabResource *r;
		while((r = it.current())) {
			Jid j = QString("%1/%2").arg(cleanJid(me.jid)).arg(r->name);
			me.res.remove(r);
			resourceUnavailable(j);
		}
	}

        stream.disc();

        v_isActive = FALSE;
        v_status = STATUS_OFFLINE;
        //presenceChanged(v_status, "");

        // update the offline status since we're now offline
        /*JabUpdate x;
        x.req = 0;
        x.pending = olr.numInteresting();
        statusUpdate(&x);*/

	JabUpdate x = makeUpdate();
	statusUpdate(&x);
}

void Jabber::conn()
{
	v_isActive = TRUE;
	stream.connectToHost(host, port);

	doUpdate();
}

void Jabber::disc(bool fast)
{
        if(v_isActive) {
		//JT_Presence *p = new JT_Presence(io.root());
		//p->pres(STATUS_OFFLINE, "Logged out", 0);
		//p->go();

                stream.disc();
                pdb(DEBUG_JABBER, "Jabber: disconnecting.\n");
        }

        v_isActive = FALSE;

        cleanup(fast);
}

// this should clear the contact list (and signal the changes), and clear the offline queue  (FIXME:  shouldn't this just be cleanup() ?)
void Jabber::reset()
{
        // remove everyone
        for(JabRosterEntry *i = roster.first(); i; i = roster.next())
                contactRemove(i);
        roster.clear();

	io.reset();
}

void Jabber::ioAuth()
{
}

void Jabber::ioPresence(int status, const QString &statusString)
{
	JabResource *r;
	r = me.res.find(resource);
	if(!r) {
		r = new JabResource;
		r->name = resource;
		me.res.append(r);
	}
	r->status = status;
	r->statusString = statusString;
	r->timeStamp = QDateTime::currentDateTime();
	Jid myjid = QString("%1/%2").arg(me.jid).arg(r->name);
	resourceAvailable(myjid, *r);
}

void Jabber::ioReceiveRoster(JabRoster &r)
{
	JabRosterEntry *i;
	for(i = roster.first(); i; i = roster.next())
		i->flagForDelete = TRUE;

	pdb(DEBUG_JABBER, QString("[Roster item(s) received: %1]\n").arg(r.size()));
	JabRosterEntry *item;
	for(item = r.first(); item != 0; item = r.next()) {
		processRosterEntryRecv(*item);
	}

	// remove flagged entries
	for(i = roster.first(); i; ) {
		if(i->flagForDelete) {
			//printf("Would have deleted: [%s]\n", i->jid.latin1());
			contactRemove(i);
			roster.remove(i);
			i = roster.current();
		}
		else
			i = roster.next();
	}

	// release the hounds!
	queue->go();
}

void Jabber::pushRoster(JabRoster &r)
{
	pdb(DEBUG_JABBER, QString("[Roster item(s) received: %1]\n").arg(r.size()));
	JabRosterEntry *item;
	for(item = r.first(); item != 0; item = r.next()) {
		processRosterEntryRecv(*item);
	}
}

void Jabber::pushSubscription(const Jid &from, int subType)
{
	if(subType == JABSUB_SUBSCRIBE) {
		pdb(DEBUG_JABBER, QString("[%1 wants to add you to their list]\n").arg(from.full()));
		authRequest(from);
	}
	else if(subType == JABSUB_SUBSCRIBED) {
		pdb(DEBUG_JABBER, QString("[%1 has authorized you]\n").arg(from.full()));
		authGrant(from);
	}
	else if(subType == JABSUB_UNSUBSCRIBE) {
		pdb(DEBUG_JABBER, QString("[%1 has deleted you from their list]\n").arg(from.full()));
		authRemove(from);
	}
	else if(subType == JABSUB_UNSUBSCRIBED) {
		pdb(DEBUG_JABBER, QString("[%1 has removed your authorization!]\n").arg(from.full()));
		//authRemove(from);
	}
}

void Jabber::pushPresence(const JT_PushPresence::Info &i)
{
	if(i.status == STATUS_OFFLINE) {
		pdb(DEBUG_JABBER, QString("[%1 went offline]\n").arg(i.jid.full()));
	}
	else {
		pdb(DEBUG_JABBER, QString("[%1 is online]\n").arg(i.jid.full()));
	}

	JabRosterEntry *entry;
	Jid j = i.jid;

	// is it me?
	Jid myjid = QString("%1@%2").arg(user).arg(host);
	if(j == myjid)
		entry = &me;
	else
		entry = roster.findByJid(j.s());

	if(entry) {
		JabResource *r;

		r = entry->res.find(j.resource());

		// unavailable?  remove the resource
		if(i.status == STATUS_OFFLINE) {
			if(r) {
				entry->res.remove(r);
				entry->unavailableStatusString = i.statusString;
				resourceUnavailable(j);
			}
		}
		// available? add/update the resource
		else {
			if(!r) {
				r = new JabResource;
				r->name = j.resource();
				entry->res.append(r);

				QString dstr;
				dstr = QString("Adding resource to [%1]: name=[%2], status=[%3]\n").arg(j.s()).arg(j.resource()).arg(status2txt(i.status));
				pdb(DEBUG_JABBER, dstr);
			}
			else {
				QString dstr;
				dstr = QString("Updating resource to [%1]: name=[%2], status=[%3]\n").arg(j.s()).arg(j.resource()).arg(status2txt(i.status));
				pdb(DEBUG_JABBER, dstr);
			}

			if(!i.songTitle.isEmpty())
				pdb(DEBUG_JABBER, QString("Listening to: %1\n").arg(i.songTitle));

			r->status = i.status;
			r->statusString = i.statusString;
			r->priority = i.priority;
			r->songTitle = i.songTitle;
			r->timeStamp = i.timeStamp;

			resourceAvailable(j, *r);
		}
	}
}

void Jabber::pushMessage(const JabMessage &message)
{
	pdb(DEBUG_JABBER, QString("[Message]\nFrom: %1\nText: %2\n").arg(message.from.full()).arg(message.body));

	messageReceived(message);
}

void Jabber::processRosterEntryRecv(const JabRosterEntry &item)
{
	QString str;
	if(item.sub == "none")
		str = "----";
	if(item.sub == "both")
		str = "<-->";
	if(item.sub == "from")
		str = "  ->";
	if(item.sub == "to")
		str = "<-  ";
	if(item.sub == "remove")
		str = "xxxx";

	QString dstr; dstr.sprintf("  %s %-32s\n", str.latin1(), item.jid.latin1());
	pdb(DEBUG_JABBER, dstr);

	// remove from contact list
	if(item.sub == "remove") {
		JabRosterEntry *entry = roster.findByJid(item.jid);
		if(entry) {
			contactRemove(entry);
			roster.remove(entry);
		}
	}
	// update/add to contact list
	else {
		JabRosterEntry *entry = roster.findByJid(item.jid);
		if(entry) {
			entry->flagForDelete = FALSE;

			// don't process if we have an outgoing entry (FIXME)
			bool ok = TRUE;
			//JabReq *r = out.findSetRoster(item.jid);
			//if(r && !r->active)
			//	ok = FALSE;

			if(ok) {
				// don't copy, only update "roster-related" entries
				entry->jid = item.jid;
				entry->nick = item.nick;
				entry->groups = item.groups;
				entry->sub = item.sub;
				entry->ask = item.ask;

				contactChanged(entry);
			}
		}
		else {
			// add the item
			JabRosterEntry *newEntry = new JabRosterEntry(item);
			roster.add(newEntry);

			// signal it
			contactNew(newEntry);
		}
	}
}

void Jabber::login(int status, const QString &statusStr, int priority, bool plain)
{
	i_status = status;
	i_statusString = statusStr;
	i_priority = priority;
	usePlain = plain;

	connType = JABCONN_LOGIN;
	conn();
}

void Jabber::accRegister()
{
	connType = JABCONN_REGISTER;
	conn();
}

void Jabber::setPresence(int status, const QString &statusStr, int priority)
{
	// if we're currently logging in, alter the login
	JT_Login *l = (JT_Login*)io.find("JT_Login");
	if(l) {
		l->status = status;
		l->statusString = statusStr;
		l->priority = priority;
		return;
        }

	// send a presence change
	JT_Presence *p = new JT_Presence(io.root());
	connect(p, SIGNAL(receivePresence(int, const QString &)), SLOT(ioPresence(int, const QString &)));
	p->pres(status, statusStr, priority);
	p->go();
}

void Jabber::setRoster(const QString &jid, const QString &nick, const QString &group)
{
	/*JabReq *req = new JabReq(JABREQ_ROSTER);
	req->action = JABACT_SET;
	req->jid = jid;
	req->nick = nick;*/

	QStringList groups;

	// see if we have the entry already
	JabRosterEntry *entry = roster.findByJid(jid);
	if(entry) {
		// copy over the groups
		groups = entry->groups;

		if(!groups.isEmpty()) {
			// new group not the same as the base group?
			if(group != groups[0]) {
				// nuke the grouplist, and start fresh with just the one group
				groups.clear();
				if(!group.isEmpty())
					groups.append(group);
			}
		}
		else {
			if(!group.isEmpty()) {
				groups.append(group);
			}
		}
	}
	else {
		if(!group.isEmpty())
			groups.append(group);
	}

	JT_Roster *p = new JT_Roster(queue);
	p->set(jid, nick, groups);
	queue->addTask(p);

	// make the effect happen locally
	if(v_offlineEnabled) {
		if(entry) {
			JabRosterEntry item( *entry );
			item.nick = nick;
			item.groups = groups;
			processRosterEntryRecv(item);
		}
		else {
			// FIXME: make new entry?
		}
	}
}

void Jabber::remove(const QString &jid)
{
	JabRosterEntry *entry = roster.findByJid(jid);
	if(!entry)
		return;

	JT_Roster *p = new JT_Roster(io.root());
	p->remove(entry->jid);
	p->go();

	// make the effect happen locally
	if(v_offlineEnabled) {
		JabRosterEntry item(*entry);
		item.sub = "remove";
		processRosterEntryRecv(item);
	}
}


void Jabber::sendMessage(const JabMessage &msg)
{
	JT_Message *p = new JT_Message(io.root(), msg);
	p->go();
}

// FIXME: change these to Jids instead of QString
void Jabber::subscribe(const QString &to)
{
	JT_Presence *p = new JT_Presence(io.root());
	p->sub(to, JABSUB_SUBSCRIBE);
	p->go();
}

void Jabber::subscribed(const QString &to)
{
	JT_Presence *p = new JT_Presence(io.root());
	p->sub(to, JABSUB_SUBSCRIBED);
	p->go();
}

void Jabber::unsubscribe(const QString &to)
{
	JT_Presence *p = new JT_Presence(io.root());
	p->sub(to, JABSUB_UNSUBSCRIBE);
	p->go();

	remove(to);
}

void Jabber::unsubscribed(const QString &to)
{
	JT_Presence *p = new JT_Presence(io.root());
	p->sub(to, JABSUB_UNSUBSCRIBED);
	p->go();
}

void Jabber::cancelTransaction(const QString &id)
{
	JabTask *j = userTask->findById(id);
	if(j)
		j->deleteLater();
}

QCString Jabber::encodeXML(const QString &str)
{
        QString data(str);

        data.replace(QRegExp("&"), "&amp;"); // This _must_ come first
        data.replace(QRegExp("<"), "&lt;");
        data.replace(QRegExp(">"), "&gt;");
        data.replace(QRegExp("\""), "&quot;");
        data.replace(QRegExp("'"), "&apos;");

        QCString locallyEncoded = data.utf8();
        return locallyEncoded;
        //return data;
}

void Jabber::insertXml(const QString &str)
{
        if(!isConnected()) {
                pdb(DEBUG_JABBER, "Jabber: can't send Xml unless connected.\n");
                return;
        }

        send(str);
}

void Jabber::agentSetStatus(const QString &jid, int x)
{
	JabRosterEntry *r = roster.findByJid(jid);
	if(!r)
		return;

	Jid j = r->jid;

	QString str;
	if(x == STATUS_OFFLINE)
		str = QString("<presence to=\"%1\" type='unavailable'/>\n").arg(encodeXML(j.full()));
	else
		str = QString("<presence to=\"%1\"/>\n").arg(encodeXML(j.full()));

	send(str);
	pdb(DEBUG_JABBER, QString("Jabber: agentSetStatus: [%1] [%2]\n").arg(r->jid).arg(x));
}

JabRosterEntry * Jabber::findByJid(const QString &jid)
{
        return roster.findByJid(jid);
}

void Jabber::setOfflineEnabled(bool x)
{
        v_offlineEnabled = x;
}

void Jabber::setCurrentRoster(JabRoster *x_roster)
{
        roster = *x_roster;

	//printf("rolo disk begin: [%d]\n", (int)clock() / 1000);
        // alert the world of the new entries
        for(JabRosterEntry *i = roster.first(); i; i = roster.next())
                contactNew(i);
	//printf("rolo disk end: [%d]\n", (int)clock() / 1000);
}

JabRoster *Jabber::getCurrentRoster()
{
        return &roster;
}

// this signals a delayed error, by putting it in the Qt event loop
void Jabber::delayedError()
{
        QTimer *t = new QTimer(this);
        connect(t, SIGNAL(timeout()), SLOT(doError()));
        t->start(0, TRUE);
}

void Jabber::doError()
{
        error(&err);
}

void Jabber::setOLR(const QString &str)
{
	/*QStringList list = lineDecodeList(str);

	for(QStringList::Iterator it = list.begin(); it != list.end(); ++it) {
		JabReq *req = JabReq::fromString(*it);
		if(req)
			olr.enqueue(req);
	}*/

	queue->fromString(str);

	// update the offline status since we're offline (or we should be, no sane person should call setOLR when online)
	/*JabUpdate x;
	x.req = 0;
	x.pending = olr.numInteresting();
	statusUpdate(&x);*/

	JabUpdate x = makeUpdate();
	statusUpdate(&x);
}

QString Jabber::getOLR()
{
	/*QStringList list;

	for(JabReq *req = olr.last(); req; req = olr.prev()) {
		QString str = req->toString();
		if(!str.isEmpty())
			list.append(str);
	}

	return lineEncodeList(list);*/

	return queue->toString();
}

void Jabber::receivePacket(const QDomElement &chunk)
{
	io.incomingPacket(chunk);
}

void Jabber::ioDone(JabTask *p)
{
	if(p->isA("JT_Login")) {
		if(!p->success()) {
			JT_Login *j = (JT_Login *)p;
			if(j->type == 0)
				err.type = JABERR_AUTH;
			else
				err.type = JABERR_CREATE;
			err.msg = j->errorString();

			disc();
			//printf("jabber: failed!\n");
			doError();
			return;
		}
		else {
			//printf("jabber: success!\n");
		}
	}
	else if(p->isA("JT_Register")) {
		JT_Register *r = (JT_Register *)p;
		bool b = r->success();
		QString err = r->errorString();
		r->deleteLater();
		accRegisterFinished(b, err);
		return;
	}

	//printf("jabber: deleting task\n");
	p->deleteLater();

	doUpdate();
}

void Jabber::ioOutgoingPacket(const QDomElement &x)
{
	stream.sendPacket(x);
}

JabUpdate Jabber::makeUpdate()
{
	QString str;

	if(isActive()) {
		JabTask *j;
		int x;

		// are we connected yet?
		if(!stream.isConnected())
			str = tr("Connecting...");
		// so we must be logging in then
		else if((j = io.find("JT_Login")) && !j->isDone()) {
			JT_Login *l = (JT_Login *)j;
			if(l->reg)
				str = tr("Registering...");
			else if(l->rost)
				str = tr("Requesting roster...");
			else
				str = tr("Authorizing...");
		}
		// waiting for outgoing roster requests to process?
		else if(queue->count() > 0) {
			str = tr("Updating Roster...");
		}
		// some other type of request somewhere?
		else if((x = notSpecial()) > 0) {
			str = tr("Awaiting response...");
			//printf("notspecial = [%d]\n", x);
		}
		else {
			str = tr("Connected.");
		}
	}
	else {
		if(queue->count() > 0)
			str = tr("Requests pending.");
		else
			str = tr("Not connected.");
	}

	JabUpdate x;
	x.str = str;
	x.queuePending = queue->count();
	return x;
}

JabUpdate Jabber::getJU()
{
	return makeUpdate();
}

void Jabber::doUpdate()
{
	JabUpdate x = makeUpdate();
	statusUpdate(&x);
}

int Jabber::notSpecial()
{
	int total = 0;

	// count the root
	const QObjectList *p = io.root()->children();
	if(!p)
		return total;
	QObjectListIt it(*p);
	for(JabTask *j; (j = (JabTask *)it.current()); ++it) {
		if(j->isA("JT_Login") || j->isA("JT_Queue") || j->isDone() || j->isDaemon() || (userTask && j->id() == userTask->id()))
			continue;
		++total;
	}

	// count userTask
	if(userTask) {
		const QObjectList *p = userTask->children();
		if(!p)
			return total;
		QObjectListIt it(*p);
		for(JabTask *j; (j = (JabTask *)it.current()); ++it) {
			if(j->isDone())
				continue;
			++total;
		}
	}

	return total;
}


/****************************************************************************
  SHA1 - from a public domain implementation by Steve Reid (steve@edmweb.com)
****************************************************************************/

#define rol(value, bits) (((value) << (bits)) | ((value) >> (32 - (bits))))
#define blk(i) (block->l[i&15] = rol(block->l[(i+13)&15]^block->l[(i+8)&15]^block->l[(i+2)&15]^block->l[i&15],1))

/* (R0+R1), R2, R3, R4 are the different operations used in SHA1 */
#define R0(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk0(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R1(v,w,x,y,z,i) z+=((w&(x^y))^y)+blk(i)+0x5A827999+rol(v,5);w=rol(w,30);
#define R2(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0x6ED9EBA1+rol(v,5);w=rol(w,30);
#define R3(v,w,x,y,z,i) z+=(((w|x)&y)|(w&x))+blk(i)+0x8F1BBCDC+rol(v,5);w=rol(w,30);
#define R4(v,w,x,y,z,i) z+=(w^x^y)+blk(i)+0xCA62C1D6+rol(v,5);w=rol(w,30);


SHA1::SHA1()
{
	int wordSize;

	qSysInfo(&wordSize, &bigEndian);
}

unsigned long SHA1::blk0(Q_UINT32 i)
{
	if(bigEndian)
		return block->l[i];
	else
		return (block->l[i] = (rol(block->l[i],24)&0xFF00FF00) | (rol(block->l[i],8)&0x00FF00FF));
}

// Hash a single 512-bit block. This is the core of the algorithm.
void SHA1::transform(Q_UINT32 state[5], unsigned char buffer[64])
{
	Q_UINT32 a, b, c, d, e;

	block = (CHAR64LONG16*)buffer;

	// Copy context->state[] to working vars
	a = state[0];
	b = state[1];
	c = state[2];
	d = state[3];
	e = state[4];

	// 4 rounds of 20 operations each. Loop unrolled.
	R0(a,b,c,d,e, 0); R0(e,a,b,c,d, 1); R0(d,e,a,b,c, 2); R0(c,d,e,a,b, 3);
	R0(b,c,d,e,a, 4); R0(a,b,c,d,e, 5); R0(e,a,b,c,d, 6); R0(d,e,a,b,c, 7);
	R0(c,d,e,a,b, 8); R0(b,c,d,e,a, 9); R0(a,b,c,d,e,10); R0(e,a,b,c,d,11);
	R0(d,e,a,b,c,12); R0(c,d,e,a,b,13); R0(b,c,d,e,a,14); R0(a,b,c,d,e,15);
	R1(e,a,b,c,d,16); R1(d,e,a,b,c,17); R1(c,d,e,a,b,18); R1(b,c,d,e,a,19);
	R2(a,b,c,d,e,20); R2(e,a,b,c,d,21); R2(d,e,a,b,c,22); R2(c,d,e,a,b,23);
	R2(b,c,d,e,a,24); R2(a,b,c,d,e,25); R2(e,a,b,c,d,26); R2(d,e,a,b,c,27);
	R2(c,d,e,a,b,28); R2(b,c,d,e,a,29); R2(a,b,c,d,e,30); R2(e,a,b,c,d,31);
	R2(d,e,a,b,c,32); R2(c,d,e,a,b,33); R2(b,c,d,e,a,34); R2(a,b,c,d,e,35);
	R2(e,a,b,c,d,36); R2(d,e,a,b,c,37); R2(c,d,e,a,b,38); R2(b,c,d,e,a,39);
	R3(a,b,c,d,e,40); R3(e,a,b,c,d,41); R3(d,e,a,b,c,42); R3(c,d,e,a,b,43);
	R3(b,c,d,e,a,44); R3(a,b,c,d,e,45); R3(e,a,b,c,d,46); R3(d,e,a,b,c,47);
	R3(c,d,e,a,b,48); R3(b,c,d,e,a,49); R3(a,b,c,d,e,50); R3(e,a,b,c,d,51);
	R3(d,e,a,b,c,52); R3(c,d,e,a,b,53); R3(b,c,d,e,a,54); R3(a,b,c,d,e,55);
	R3(e,a,b,c,d,56); R3(d,e,a,b,c,57); R3(c,d,e,a,b,58); R3(b,c,d,e,a,59);
	R4(a,b,c,d,e,60); R4(e,a,b,c,d,61); R4(d,e,a,b,c,62); R4(c,d,e,a,b,63);
	R4(b,c,d,e,a,64); R4(a,b,c,d,e,65); R4(e,a,b,c,d,66); R4(d,e,a,b,c,67);
	R4(c,d,e,a,b,68); R4(b,c,d,e,a,69); R4(a,b,c,d,e,70); R4(e,a,b,c,d,71);
	R4(d,e,a,b,c,72); R4(c,d,e,a,b,73); R4(b,c,d,e,a,74); R4(a,b,c,d,e,75);
	R4(e,a,b,c,d,76); R4(d,e,a,b,c,77); R4(c,d,e,a,b,78); R4(b,c,d,e,a,79);

	// Add the working vars back into context.state[]
	state[0] += a;
	state[1] += b;
	state[2] += c;
	state[3] += d;
	state[4] += e;

	// Wipe variables
	a = b = c = d = e = 0;
}

// SHA1Init - Initialize new context
void SHA1::init(SHA1_CONTEXT* context)
{
	// SHA1 initialization constants
	context->state[0] = 0x67452301;
	context->state[1] = 0xEFCDAB89;
	context->state[2] = 0x98BADCFE;
	context->state[3] = 0x10325476;
	context->state[4] = 0xC3D2E1F0;
	context->count[0] = context->count[1] = 0;
}

// Run your data through this
void SHA1::update(SHA1_CONTEXT* context, unsigned char* data, Q_UINT32 len)
{
	Q_UINT32 i, j;

	j = (context->count[0] >> 3) & 63;
	if((context->count[0] += len << 3) < (len << 3))
		context->count[1]++;

	context->count[1] += (len >> 29);

	if((j + len) > 63) {
		memcpy(&context->buffer[j], data, (i = 64-j));
		transform(context->state, context->buffer);
		for ( ; i + 63 < len; i += 64) {
			transform(context->state, &data[i]);
		}
		j = 0;
	}
	else i = 0;
	memcpy(&context->buffer[j], &data[i], len - i);
}

// Add padding and return the message digest
void SHA1::final(unsigned char digest[20], SHA1_CONTEXT* context)
{
	Q_UINT32 i, j;
	unsigned char finalcount[8];

	for (i = 0; i < 8; i++) {
		finalcount[i] = (unsigned char)((context->count[(i >= 4 ? 0 : 1)]
		>> ((3-(i & 3)) * 8) ) & 255);  // Endian independent
	}
	update(context, (unsigned char *)"\200", 1);
	while ((context->count[0] & 504) != 448) {
		update(context, (unsigned char *)"\0", 1);
	}
	update(context, finalcount, 8);  // Should cause a transform()
	for (i = 0; i < 20; i++) {
		digest[i] = (unsigned char) ((context->state[i>>2] >> ((3-(i & 3)) * 8) ) & 255);
	}

	// Wipe variables
	i = j = 0;
	memset(context->buffer, 0, 64);
	memset(context->state, 0, 20);
	memset(context->count, 0, 8);
	memset(&finalcount, 0, 8);
}


/* static */ QString SHA1::digest(QString in)
{
	SHA1_CONTEXT context;
	unsigned char digest[20];

	SHA1 s;
	QCString data = in.latin1();

	s.init(&context);
	s.update(&context, (unsigned char *)data.data(), (unsigned int)data.length());
	s.final(digest, &context);

	QString out;
	for(int n = 0; n < 20; ++n) {
		QString str;
		str.sprintf("%02x", digest[n]);
		out.append(str);
	}

	return out;
}
