/****************************************************************************
** jabcommon.cpp - common classes and functions for handling Jabber data
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"jabcommon.h"
#include<qregexp.h>
#include<qobject.h>


QString lineEncode(QString str)
{
	str.replace(QRegExp("\\\\"), "\\\\");   // backslash to double-backslash
	str.replace(QRegExp("\\|"), "\\p");     // pipe to \p
	str.replace(QRegExp("\n"), "\\n");      // newline to \n
	return str;
}

QString lineDecode(const QString &str)
{
	QString ret;

	for(unsigned int n = 0; n < str.length(); ++n) {
		if(str.at(n) == '\\') {
			++n;
			if(n >= str.length())
				break;

			if(str.at(n) == 'n')
				ret.append('\n');
			if(str.at(n) == 'p')
				ret.append('|');
			if(str.at(n) == '\\')
				ret.append('\\');
		}
		else {
			ret.append(str.at(n));
		}
	}

	return ret;
}

QString lineEncodeList(QStringList list)
{
	QString str = "";

	int n = 0;
	for(QStringList::Iterator it = list.begin(); it != list.end(); ++it) {
		if(n > 0)
			str += '|';
		str += lineEncode(*it);
		++n;
	}

	return lineEncode(str);
}

QStringList lineDecodeList(const QString &in)
{
	QString str = lineDecode(in);
	QStringList list;

	int p1 = 0;
	int p2;
	while(p1 != -1) {
		p2 = str.find('|', p1);
		QString chunk;
		if(p2 == -1) {
			chunk = str.mid(p1);
			p1 = -1;
		}
		else {
			chunk = str.mid(p1, p2-p1);
			p1 = p2 + 1;
		}

		list.append(lineDecode(chunk));
	}

	return list;
}


QString status2jabprestxt(int status)
{
	switch(status) {
		case STATUS_AWAY:       return "away";
		case STATUS_XA:         return "xa";
		case STATUS_DND:        return "dnd";
	}

	return "";
}

int jabprestxt2status(const QString &str)
{
	if(str == "away")       return STATUS_AWAY;
	else if(str == "xa")    return STATUS_XA;
	else if(str == "dnd")   return STATUS_DND;
	else
		return STATUS_ONLINE;
}

QString cleanJid(QString jid)
{
	int x = jid.find('/');
	if(x) {
		jid.truncate(x);
	}

	return jid;
}

bool jidcmp(const QString &jid1, const QString &jid2)
{
	QString jid1clean = cleanJid(jid1);
	QString jid2clean = cleanJid(jid2);

	if(jid1clean.length() != jid2clean.length())
		return 0;

	for(unsigned int n = 0; n < jid1clean.length(); ++n) {
		if(jid1clean.at(n).lower() != jid2clean.at(n).lower())
			return 0;
	}

	return 1;
}

bool jidIsAgent(const QString &jid)
{
	return (jid.find('@') == -1) ? TRUE : FALSE;
}


/****************************************************************************
  Jid
****************************************************************************/
Jid::Jid()
{
}

Jid::Jid(const QString &s)
{
	set(s);
}

void Jid::set(const QString &s)
{
	int n, n2;

	realJid = s;
	if(s.isEmpty())
		return;

	// user
	v_user = "";
	n = realJid.find('@');
	if(n != -1)
		v_user = realJid.mid(0, n);

	// host: locate the '@'
	n = realJid.find('@');
	if(n == -1)
		n = 0;
	else
		++n;
	// host: locate the '/'
	n2 = realJid.find('/', n);
	if(n2 == -1)
		v_host = realJid.mid(n);
	else
		v_host = realJid.mid(n, n2-n);

	// resource
	v_resource = "";
	n = realJid.find('/');
	if(n != -1)
		v_resource = realJid.mid(n+1);

	// s
	n = realJid.find('/');
	if(n == -1)
		v_s = realJid;
	else
		v_s = realJid.mid(0, n);

	// isAgent
	v_isAgent = realJid.find('@') == -1 ? TRUE: FALSE;
}

Jid & Jid::operator=(const QString &s)
{
	this->set(s);
	return *this;
}

bool operator==(const Jid &j1, const Jid &j2)
{
	if(j1.s().length() != j2.s().length())
		return FALSE;

	for(unsigned int n = 0; n < j1.s().length(); ++n) {
		if(j1.s().at(n).lower() != j2.s().at(n).lower())
			return FALSE;
	}

	return TRUE;
}


/****************************************************************************
  JabFormField
****************************************************************************/
JabFormField::JabFormField()
{
	v_type = misc;
}

QString JabFormField::realName()
{
	switch(v_type) {
		case username:  return "username";
		case nick:      return "nick";
		case password:  return "password";
		case name:      return "name";
		case first:     return "first";
		case last:      return "last";
		case email:     return "email";
		case address:   return "address";
		case city:      return "city";
		case state:     return "state";
		case zip:       return "zipcode";
		case phone:     return "phone";
		case url:       return "url";
		case date:      return "date";
		case misc:      return "misc";
		default:        return "";
	};
}

QString JabFormField::fieldName()
{
	switch(v_type) {
		case username:  return QObject::tr("Username");
		case nick:      return QObject::tr("Nickname");
		case password:  return QObject::tr("Password");
		case name:      return QObject::tr("Name");
		case first:     return QObject::tr("First Name");
		case last:      return QObject::tr("Last Name");
		case email:     return QObject::tr("E-mail");
		case address:   return QObject::tr("Address");
		case city:      return QObject::tr("City");
		case state:     return QObject::tr("State");
		case zip:       return QObject::tr("Zipcode");
		case phone:     return QObject::tr("Phone");
		case url:       return QObject::tr("URL");
		case date:      return QObject::tr("Date");
		case misc:      return QObject::tr("Misc");
		default:        return "";
	};
}

void JabFormField::setType(int x)
{
	v_type = x;
}

bool JabFormField::setType(const QString &in)
{
	int x = str2type(in);
	if(x == -1)
		return FALSE;

	v_type = x;
	return TRUE;
}

void JabFormField::setValue(const QString &in)
{
	v_value = in;
}

int JabFormField::str2type(const QString &in)
{
	if(!in.compare("username")) return username;
	if(!in.compare("nick"))     return nick;
	if(!in.compare("password")) return password;
	if(!in.compare("name"))     return name;
	if(!in.compare("first"))    return first;
	if(!in.compare("last"))     return last;
	if(!in.compare("email"))    return email;
	if(!in.compare("address"))  return address;
	if(!in.compare("city"))     return city;
	if(!in.compare("state"))    return state;
	if(!in.compare("zip"))      return zip;
	if(!in.compare("phone"))    return phone;
	if(!in.compare("url"))      return url;
	if(!in.compare("date"))     return date;
	if(!in.compare("misc"))     return misc;

	return -1;
}


/****************************************************************************
  JabForm
****************************************************************************/
JabForm::JabForm()
{
	setAutoDelete(TRUE);
};

JabForm::JabForm(const JabForm &from)
:QPtrList<JabFormField>()
{
	*this = from;
}

// deep copy
JabForm & JabForm::operator=(const JabForm &from)
{
	clear();

	QPtrListIterator<JabFormField> it(from);
	JabFormField *f;
	for(; (f = it.current()); ++it)
		append(new JabFormField(*f));

	jid = from.jid;
	instructions = from.instructions;
	key = from.key;

	return *this;
}

JabFormField *JabForm::find(int type) const
{
	QPtrListIterator<JabFormField> it(*this);
	JabFormField *f;
	for(; (f = it.current()); ++it) {
		if(f->type() == type)
			return f;
	}

	return 0;
}


/****************************************************************************
  JabRoster - holds a bunch of JabRosterEntries
****************************************************************************/
JabRoster::JabRoster()
{
	list.setAutoDelete(TRUE);
}

JabRoster::~JabRoster()
{
	//printf("JabRoster: destructing [%d]\n", list.count());
}

JabRoster::JabRoster(const JabRoster &from)
{
	//printf("JabRoster: copy()\n");

	*this = from;
/*	list.clear();
	QPtrList<JabRosterEntry> fromlist = from.list;
	for(JabRosterEntry *entry = fromlist.first(); entry; entry = fromlist.next())
		add(entry);*/
}

JabRoster & JabRoster::operator=(const JabRoster &from)
{
	//printf("JabRoster: assignment\n");

	//printf("JabRoster: deleting %d contacts.\n", list.count());
	list.clear();
	QPtrList<JabRosterEntry> fromlist = from.list;
	for(JabRosterEntry *entry = fromlist.first(); entry; entry = fromlist.next())
		add(new JabRosterEntry(*entry));

	//printf("JabRoster: list now has %d contacts.\n", list.count());

	return *this;
}

int JabRoster::size()
{
        return list.count();
}

JabRosterEntry *JabRoster::first()
{
        return list.first();
}

JabRosterEntry *JabRoster::current()
{
        return list.current();
}

JabRosterEntry *JabRoster::next()
{
        return list.next();
}

JabRosterEntry *JabRoster::getFirst() const
{
        return list.getFirst();
}

void JabRoster::add(JabRosterEntry *entry)
{
        list.append(entry);
}

void JabRoster::remove(JabRosterEntry *entry)
{
        list.remove(entry);
}

void JabRoster::clear()
{
        list.clear();
}

JabRosterEntry *JabRoster::findByJid(const QString &jid)
{
        QPtrListIterator<JabRosterEntry> it(list);
        JabRosterEntry *entry;

        for( ; it.current(); ++it) {
                entry = it.current();
                if(jidcmp(entry->jid,jid))
                        return entry;
        }
        return 0;
}

JabRosterEntry *JabRoster::findByNick(const QString &nick)
{
        QPtrListIterator<JabRosterEntry> it(list);
        JabRosterEntry *entry;

        for( ; it.current(); ++it) {
                entry = it.current();
                if(entry->nick == nick)
                        return entry;
        }
        return 0;
}


/****************************************************************************
  JabRosterEntry - holds minimal contact info
****************************************************************************/
JabRosterEntry::JabRosterEntry()
{
        ask = 0;
	push = FALSE;

	isRegisterable = isSearchable = isGCCapable = hasSubAgents = isTransport = FALSE;

        flagForDelete = FALSE;
}

JabRosterEntry::~JabRosterEntry()
{
}

bool JabRosterEntry::isAvailable() const
{
	return res.isEmpty() ? FALSE: TRUE;
}

JabResource *JabRosterEntry::local() const
{
	return res.local();
}

JabResource *JabRosterEntry::priority() const
{
	return res.priority();
}

int JabRosterEntry::localStatus() const
{
	if(isAvailable())
		return res.local()->status;
	else
		return STATUS_OFFLINE;
}



/****************************************************************************
  VCard
****************************************************************************/
VCard::VCard()
{
}

QString VCard::tagContent(const QDomElement &e)
{
	// look for some tag content
	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomText i = n.toText();
		if(i.isNull())
			continue;
		return i.data();
	}

	return "";
}

QDomElement VCard::findSubTag(const QDomElement &e, const QString &name, bool *found)
{
	if(found)
		*found = FALSE;

	for(QDomNode n = e.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;
		if(i.tagName() == name) {
			if(found)
				*found = TRUE;
			return i;
		}
	}

	QDomElement tmp;
	return tmp;
}

QDomElement VCard::textTag(const QString &name, const QString &content)
{
	QDomElement tag = doc.createElement(name);
	QDomText text = doc.createTextNode(content);
	tag.appendChild(text);

	return tag;
}

QDomElement VCard::toXml()
{
	QDomElement v = doc.createElement("vCard");
	v.setAttribute("version", "3.0");
	v.setAttribute("prodid", "-//HandGen//NONSGML vGen v1.0//EN");
	v.setAttribute("xmlns", "vcard-temp"); // <-- FIXME: how to do namespace attributes?

	if(!field[vFullname].isEmpty())
		v.appendChild(textTag("FN", field[vFullname]));
	if(!field[vNickname].isEmpty())
		v.appendChild(textTag("NICKNAME", field[vNickname]));
	if(!field[vDesc].isEmpty())
		v.appendChild(textTag("DESC", field[vDesc]));
	if(!field[vEmail].isEmpty())
		v.appendChild(textTag("EMAIL", field[vEmail]));
	if(!field[vBday].isEmpty())
		v.appendChild(textTag("BDAY", field[vBday]));
	if(!field[vHomepage].isEmpty())
		v.appendChild(textTag("URL", field[vHomepage]));

	if(1) {
		QDomElement w = doc.createElement("ORG");
		if(!field[vOrgName].isEmpty())
			w.appendChild(textTag("ORGNAME", field[vOrgName]));
		if(!field[vOrgUnit].isEmpty())
			w.appendChild(textTag("ORGUNIT", field[vOrgUnit]));
		v.appendChild(w);
	}

	if(!field[vTitle].isEmpty())
		v.appendChild(textTag("TITLE", field[vTitle]));
	if(!field[vRole].isEmpty())
		v.appendChild(textTag("ROLE", field[vRole]));
	if(!field[vPhone].isEmpty()) {
		QDomElement w = doc.createElement("TEL");

		QDomElement x = doc.createElement("HOME");
		w.appendChild(x);

		w.appendChild(textTag("VOICE", field[vPhone]));

		v.appendChild(w);
	}

	if(1) {
		QDomElement w = doc.createElement("ADR");

		QDomElement x = doc.createElement("HOME");
		w.appendChild(x);

		if(!field[vStreet].isEmpty())
			w.appendChild(textTag("STREET", field[vStreet]));
		if(!field[vExt].isEmpty())
			w.appendChild(textTag("EXTADD", field[vExt]));
		if(!field[vCity].isEmpty())
			w.appendChild(textTag("LOCALITY", field[vCity]));
		if(!field[vState].isEmpty())
			w.appendChild(textTag("REGION", field[vState]));
		if(!field[vPcode].isEmpty())
			w.appendChild(textTag("PCODE", field[vPcode]));
		if(!field[vCountry].isEmpty())
			w.appendChild(textTag("COUNTRY", field[vCountry]));

		v.appendChild(w);
	}

	xml = v;
	return xml;
}

bool VCard::fromXml(const QDomElement &q)
{
	if(q.tagName() != "vcard")
		return FALSE;
	//if(q.attribute("version") != "3.0")
	//	return FALSE;

	xml = q;
	for(QDomNode n = q.firstChild(); !n.isNull(); n = n.nextSibling()) {
		QDomElement i = n.toElement();
		if(i.isNull())
			continue;

		if(i.tagName() == "fn")
			field[vFullname] = tagContent(i);
		else if(i.tagName() == "nickname")
			field[vNickname] = tagContent(i);
		else if(i.tagName() == "desc")
			field[vDesc] = tagContent(i);
		else if(i.tagName() == "email") {
			// email tag is weird.  check parent first
			QString str = tagContent(i);
			if(str.isEmpty()) {
				// check children
				for(QDomNode n = i.firstChild(); !n.isNull(); n = n.nextSibling()) {
					QDomElement e = n.toElement();
					if(e.isNull())
						continue;
					str = tagContent(e);
					if(!str.isEmpty())
						break;
				}
			}
			field[vEmail] = str;
		}
		else if(i.tagName() == "bday")
			field[vBday] = tagContent(i);
		else if(i.tagName() == "url")
			field[vHomepage] = tagContent(i);
		else if(i.tagName() == "org") {
			bool found;
			QDomElement tag = findSubTag(i, "orgname", &found);
			if(found)
				field[vOrgName] = tagContent(tag);
			tag = findSubTag(i, "orgunit", &found);
			if(found)
				field[vOrgUnit] = tagContent(tag);
		}
		else if(i.tagName() == "title")
			field[vTitle] = tagContent(i);
		else if(i.tagName() == "role")
			field[vRole] = tagContent(i);
		else if(i.tagName() == "tel") {
			bool found;
			int type = 0;
			QDomElement tag;

			findSubTag(i, "home", &found);
			if(found)
				type = 1;

			tag = findSubTag(i, "voice", &found);
			if(found) {
				if(type == 1)
					field[vPhone] = tagContent(tag);
			}
		}
		else if(i.tagName() == "adr") {
			bool found;
			int type = 0;
			QDomElement tag;

			findSubTag(i, "home", &found);
			if(found)
				type = 1;
			findSubTag(i, "work", &found);
			if(found)
				type = 2;

			if(type == 1) {
				tag = findSubTag(i, "extadd", &found);
				if(found)
					field[vExt] = tagContent(tag);
				tag = findSubTag(i, "street", &found);
				if(found)
					field[vStreet] = tagContent(tag);
				tag = findSubTag(i, "locality", &found);
				if(found)
					field[vCity] = tagContent(tag);
				tag = findSubTag(i, "region", &found);
				if(found)
					field[vState] = tagContent(tag);
				tag = findSubTag(i, "pcode", &found);
				if(found)
					field[vPcode] = tagContent(tag);
				tag = findSubTag(i, "country", &found);
				if(found)
					field[vCountry] = tagContent(tag);
			}
		}
	}

	return TRUE;
}

bool VCard::isIncomplete()
{
	for(int n = 0; n < NUM_VCARDFIELDS; ++n) {
		if(!field[n].isEmpty())
			return FALSE;
	}
	return TRUE;
}


/****************************************************************************
  JabResource
****************************************************************************/
JabResource::JabResource()
{
	name = "";
	priority = 0;
	status = STATUS_OFFLINE;
	statusString = "";
}


/****************************************************************************
  JabResourceList - holds a list of resources
****************************************************************************/
JabResourceList::JabResourceList()
{
	setAutoDelete(TRUE);
};

JabResourceList::JabResourceList(const JabResourceList &from)
:QPtrList<JabResource>()
{
	*this = from;
}

// deep copy
JabResourceList & JabResourceList::operator=(const JabResourceList &from)
{
	clear();

	QPtrListIterator<JabResource> it(from);
	JabResource *r;
	for(; (r = it.current()); ++it)
		append(new JabResource(*r));

	return *this;
}

JabResource *JabResourceList::find(const QString &name) const
{
	QPtrListIterator<JabResource> it(*this);
	JabResource *r;
	for(; (r = it.current()); ++it) {
		if(r->name == name)
			return r;
	}

	return 0;
}

JabResource *JabResourceList::local() const
{
	return getFirst();
}

JabResource *JabResourceList::priority() const
{
	JabResource *highest = 0;
	int highest_val;

	QPtrListIterator<JabResource> it(*this);
	JabResource *r;
	for(; (r = it.current()); ++it) {
		if(!highest || (r->priority > highest->priority)) {
			highest = r;
			highest_val = r->priority;
		}
	}

	// this can be zero
	return highest;
}
