/****************************************************************************
** message.cpp - messages and history
** Copyright (C) 2001, 2002  Justin Karneges
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License
** as published by the Free Software Foundation; either version 2
** of the License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,USA.
**
****************************************************************************/

#include"message.h"
#include"common.h"

#include<qfile.h>
#include<qregexp.h>
#include"profiles.h"
#include<stdio.h>


Message::Message()
{
	unread = FALSE;
}

MessageHistory::MessageHistory(const QString &jid, int xmode)
{
	v_jid = cleanJid(jid);

	QString fname = getHistoryDir() + "/" + qstrlower(jidEncode(v_jid)) + ".history";
	pdb(DEBUG_MAINWIN, QString("MessageHistory: fname=[%1]\n").arg(fname.latin1()));
	f.setName(fname);

	mode = xmode;
	bool ok = 0;
	if(xmode == HISTORY_WRITE)
		ok = f.open(IO_WriteOnly | IO_Append);
	else if(xmode == HISTORY_READ)
		ok = f.open(IO_ReadOnly);
	else if(xmode == HISTORY_FLAG)
		ok = f.open(IO_ReadWrite);

	if(!ok) {
		pdb(DEBUG_MAINWIN, QString("Error opening message history.  Actions will be ignored. [mode=%1]\n").arg(xmode));
		mode = 0;
	}

	if(mode == HISTORY_READ || mode == HISTORY_FLAG)
		f.at(f.size()-2); // last byte should be a newline.  we want the byte before it.
}

MessageHistory::~MessageHistory()
{
	pdb(DEBUG_MAINWIN, "MessageHistory: closing.\n");
	if(mode)
		f.close();
}

void MessageHistory::stepForward()
{
	if(mode != HISTORY_FLAG && mode != HISTORY_READ)
		return;

	if(f.at() == 0)
		return;

	// go backwards till we find a newline (or the beginning of the file)
	while(f.at() > 0) {
		int x = f.getch();
		if(x == -1)
			return;
		QChar c = x;

		if(c == '\n')
			break;
		if(f.at() == 1) {
			f.at(0);
			break;
		}
		f.at(f.at()-2);
	}

	// skip past the newline
	int x = f.at() - 2;
	if(x < 0)
		x = 0;
	f.at(x);
}

void MessageHistory::stepBack()
{
	if(mode != HISTORY_FLAG && mode != HISTORY_READ)
		return;

	// at end?
	if(f.at() >= f.size()-2)
		return;

	// skip past the newline if we're not at the beginning
	if(f.at() > 0) {
		f.at(f.at() + 2);
	}

	// cursor should always be after the current message, so just read forward
	QTextStream t;
	t.setDevice(&f);
	t.setEncoding(QTextStream::UnicodeUTF8);
	t.readLine();
	t.unsetDevice();

	// we want the position before the newline
	f.at(f.at() - 2);
}

Message *MessageHistory::readCurrent()
{
	if(mode != HISTORY_READ && mode != HISTORY_FLAG)
		return 0;

	if(f.at() == 0)
		return 0;

	int at = f.at();

	// go backwards till we find a newline (or the beginning of the file)
	while(f.at() > 0) {
		int x = f.getch();
		if(x == -1) {
			f.at(at);
			return 0;
		}
		QChar c = x;

		if(c == '\n')
			break;
		if(f.at() == 1) {
			f.at(0);
			break;
		}
		f.at(f.at()-2);
	}

	QTextStream t;
	t.setDevice(&f);
	t.setEncoding(QTextStream::UnicodeUTF8);
	QString line = t.readLine();
	t.unsetDevice();

	f.at(at);

	return parseLine(line);
}

Message *MessageHistory::parseLine(const QString &line)
{
	// -- read the line --
	QString sTime, sType, sOrigin, sFlags, sText, sSubj, sUrl, sUrlDesc;
	int x1, x2;
	x1 = line.find('|') + 1;

	x2 = line.find('|', x1);
	sTime = line.mid(x1, x2-x1);
	x1 = x2 + 1;

	x2 = line.find('|', x1);
	sType = line.mid(x1, x2-x1);
	x1 = x2 + 1;

	x2 = line.find('|', x1);
	sOrigin = line.mid(x1, x2-x1);
	x1 = x2 + 1;

	x2 = line.find('|', x1);
	sFlags = line.mid(x1, x2-x1);
	x1 = x2 + 1;

	// check for extra fields
	if(sFlags[1] != '-') {
		int subflags = hexChar2int(sFlags[1].latin1());

		// have subject?
		if(subflags & 1) {
			x2 = line.find('|', x1);
			sSubj = line.mid(x1, x2-x1);
			x1 = x2 + 1;
		}
		// have url?
		if(subflags & 2) {
			x2 = line.find('|', x1);
			sUrl = line.mid(x1, x2-x1);
			x1 = x2 + 1;
			x2 = line.find('|', x1);
			sUrlDesc = line.mid(x1, x2-x1);
			x1 = x2 + 1;
		}
	}

	// body text is last
	sText = line.mid(x1);

	// -- read end --


	// populate the message class
	Message *msg = new Message;
	msg->timeStamp = QDateTime::fromString(sTime, Qt::ISODate);
	msg->type = sType.toInt();
	msg->originLocal = (sOrigin == "to") ? TRUE: FALSE;
	if(msg->originLocal)
		msg->to = v_jid;
	else
		msg->from = v_jid;
	if(sFlags[0] == 'N')
		msg->text = logdecode(sText);
	else
		msg->text = logdecode(QString::fromUtf8(sText));
	if(sFlags[2] == 'U')
		msg->unread = TRUE;
	msg->subject = logdecode(sSubj);
	msg->url = logdecode(sUrl);
	msg->url_desc = logdecode(sUrlDesc);
	msg->late = TRUE;

	return msg;
}

Message *MessageHistory::readEntry()
{
	Message *msg = readCurrent();
	stepForward();
	return msg;
}

void MessageHistory::writeEntry(const Message &msg)
{
	if(mode != HISTORY_WRITE)
		return;

	int subflags = 0;
	if(!msg.subject.isEmpty())
		subflags |= 1;
	if(!msg.url.isEmpty())
		subflags |= 2;

	QString sTime, sType, sOrigin, sFlags;
	sTime = msg.timeStamp.toString(Qt::ISODate);
	sType = sType.setNum(msg.type);
	sOrigin = msg.originLocal ? "to": "from";
	sFlags = "N---";
	if(subflags != 0)
		sFlags[1] = int2hexChar(subflags);
	if(msg.unread)
		sFlags[2] = 'U';

	//  | date | type | To/from | flags | text
	QString line = "|" + sTime + "|" + sType + "|" + sOrigin + "|" + sFlags + "|";

	if(subflags & 1) {
		line += logencode(msg.subject) + "|";
	}
	if(subflags & 2) {
		line += logencode(msg.url) + "|";
		line += logencode(msg.url_desc) + "|";
	}

	line += logencode(msg.text);

	pdb(DEBUG_MAINWIN, QString("MessageHistory::writeEntry: line=[%1]\n").arg(line.latin1()));

	QTextStream t;
	t.setDevice(&f);
	t.setEncoding(QTextStream::UnicodeUTF8);
	t << line << endl;
	t.unsetDevice();
	f.flush();
}

void MessageHistory::setFlagsCurrent(const QString &flags)
{
	if(mode != HISTORY_FLAG)
		return;

	if(flags.length() != 4)
		return;

	if(f.at() == 0)
		return;

	int at = f.at();

	// go backwards till we find a newline (or the beginning of the file)
	while(f.at() > 0) {
		int x = f.getch();
		if(x == -1) {
			f.at(at);
			return;
		}
		QChar c = x;

		if(c == '\n')
			break;
		if(f.at() == 1) {
			f.at(0);
			break;
		}
		f.at(f.at()-2);
	}

	QTextStream t;
	t.setDevice(&f);
	t.setEncoding(QTextStream::UnicodeUTF8);
	QString line = t.readLine();
	t.unsetDevice();

	f.at(at);

	// find 4 pipes in a row
	int x1 = 0;
	int n;
	for(n = 0; n < 4; ++n) {
		x1 = line.find('|', x1);
		if(x1 == -1)
			break;
		++x1;
	}
	if(x1 == -1)
		return;
	QString oldflags = line.mid(x1, 4);
	QString newflags;

	for(n = 0; n < 4; ++n) {
		if(flags[n] == '*')
			newflags[n] = oldflags[n];
		else
			newflags[n] = flags[n];
	}

	line.replace(x1, 4, newflags);

	// save the updated entry
	at = f.at();

	// go backwards till we find a newline (or the beginning of the file)
	while(f.at() > 0) {
		int x = f.getch();
		if(x == -1) {
			f.at(at);
			return;
		}
		QChar c = x;

		if(c == '\n')
			break;
		if(f.at() == 1) {
			f.at(0);
			break;
		}
		f.at(f.at()-2);
	}

	t.setDevice(&f);
	t.setEncoding(QTextStream::UnicodeUTF8);
	t << line << endl;
	t.unsetDevice();

	f.flush();
	f.at(at);
}
