# arch-tag: 6d27c8ec-a8aa-48dd-9be0-66968204b14e
# Copyright (C) 2004 David Allouche <david@allouche.net>
#               2005 Canonical Ltd.
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Testing framework fixtures.
"""

import os
import tempfile
import shutil
import copy

import pybaz as arch
from pybaz.util import DirName


class FixtureMemento(object):
    pass


class BaseFixture(object):
    """Base class for all fixtures."""

    memento_attrs = ()

    def setUp(self):
        pass

    def tearDown(self):
        pass

    def memento_key(self):
        return type(self)

    def create_memento(self):
        self.memento = FixtureMemento()
        self.save_memento()
        memento = copy.deepcopy(self.memento)
        self.memento = None
        return memento

    def save_memento(self):
        attrs = [(k, getattr(self, k)) for k in self.memento_attrs]
        self.memento.__attrs = attrs

    def set_memento(self, memento):
        self.memento = memento
        self.load_memento()
        self.memento = None

    def load_memento(self):
        for k, v in self.memento.__attrs:
            setattr(self, k, v)


class NullFixture(BaseFixture):
    """A fixture that does nothing."""

    def create_memento(self):
        return None

    def set_memento(self, memento):
        unused = memento


class EmptySandboxFixture(BaseFixture):
    """A fixture that sets up a sandbox."""

    def __init__(self):
        self.sandbox = None
        self.arch_dir_base = DirName(r'baz tests')
        self._hash_extra = None

    def setUp(self):
        """Setup a sandbox.

        Set HOME to an (empty) temporary directory and clears EDITOR.
        """
        self.sandbox = Sandbox()
        self.sandbox.setUp()

    def tearDown(self):
        """Restore the environment.

        Restore HOME, EDITOR and and the current directory to their value when
        setUp was run, and remove the sandbox directory.
        """
        self.sandbox.tearDown()
        self.sandbox = None

    def save_memento(self):
        super(EmptySandboxFixture, self).save_memento()
        self.memento.__sandbox = self.sandbox.create_memento()

    def load_memento(self):
        super(EmptySandboxFixture, self).load_memento()
        self.sandbox = Sandbox()
        self.sandbox.setUp()
        self.sandbox.set_memento(self.memento.__sandbox)


class SandboxDir(object):

    _instance = None

    def __init__(self):
        self.path = None

    def __del__(self):
        # this is meant to be run on python exit
        shutil.rmtree(self.path)

    def instance():
        if SandboxDir._instance is None:
            SandboxDir._instance = SandboxDir()
            path = DirName(tempfile.mkdtemp(prefix='pyarch-')).realpath()
            SandboxDir._instance.path = path
        return SandboxDir._instance

    instance = staticmethod(instance)

    def clear(self):
        for name in os.listdir(self.path):
            shutil.rmtree(os.path.join(self.path, name))


class Sandbox(object):

    def setUp(self):
        self.home_dir = os.environ.get('HOME')
        self.tmp_dir = SandboxDir.instance().path
        self.saved_editor = None
        self.here = os.getcwd()
        os.environ['HOME'] = self.tmp_dir
        if os.environ.has_key('EDITOR'):
            self.saved_editor = os.environ['EDITOR']
            del(os.environ['EDITOR'])
        os.chdir (self.tmp_dir)

    def tearDown(self):
        os.environ['HOME'] = self.home_dir
        SandboxDir.instance().clear()
        if self.saved_editor is not None:
            os.environ['EDITOR'] = self.saved_editor
        os.chdir(self.here)

    def create_memento(self):
        listing = os.listdir(self.tmp_dir)
        if len(listing) == 0:
            return NullSandboxMemento()
        else:
            return SandboxMemento(self.tmp_dir, listing)

    def set_memento(self, memento):
        memento.restore(self.tmp_dir)

from tarfile import TarFile
from StringIO import StringIO

class SandboxMemento(object):

    def __init__(self, tmp_dir, listing):
        strio = StringIO()
        tar = TarFile.open('', 'w', strio)
        for name in listing:
            tar.add(os.path.join(tmp_dir, name), name)
        tar.close()
        self.__tardata = strio.getvalue()

    def restore(self, tmp_dir):
        strio = StringIO(self.__tardata)
        tar = TarFile.open('', 'r', strio)
        for member in tar.getmembers():
            tar.extract(member, tmp_dir)
        tar.close()


class NullSandboxMemento(object):

    def restore(self, tmp_dir):
        pass


class MyIdFixture(EmptySandboxFixture):
    """A fixture that sets my-id in a sandbox."""

    def __init__(self):
        super(MyIdFixture, self).__init__()
        self.my_id = "John Doe <jdoe@example.com>"

    def setUp(self):
        """Setup and sandbox and sets my-id."""
        super(MyIdFixture, self).setUp()
        arch.set_my_id(self.my_id)


class InventoryTreeFixture(MyIdFixture):
    """A fixture that creates a working tree without associated archive."""

    memento_attrs = MyIdFixture.memento_attrs + ('tree',)

    def setUp(self):
        super(InventoryTreeFixture, self).setUp()
        arch_dir = self.sandbox.tmp_dir / self.arch_dir_base
        os.mkdir(arch_dir)
        tree_dir = arch_dir / 'workingtree'
        os.mkdir(tree_dir)
        self.tree = arch.init_tree(tree_dir)


class ArchiveDirFixture(EmptySandboxFixture):
    """A fixture that allows creating an archive in a sandbox."""

    memento_attrs = EmptySandboxFixture.memento_attrs + ('arch_dir',)

    def __init__(self):
        super(ArchiveDirFixture, self).__init__()
        self.arch_name = 'jdoe@example.com'

    def location(self, archive):
        archive = str(archive)
        return self.arch_dir / archive

    def setUp(self):
        """Sets up the attribute and sandbox for creating an archive."""
        super(ArchiveDirFixture, self).setUp()
        self.arch_dir = self.sandbox.tmp_dir / self.arch_dir_base
        os.mkdir(self.arch_dir)


class UnregisteredArchiveFixture(EmptySandboxFixture):
    """A fixture that provides an Archive that's garanteed not registered."""

    memento_attrs = EmptySandboxFixture.memento_attrs + ('archive',)

    def setUp(self):
        EmptySandboxFixture.setUp(self)
        self.archive = arch.Archive('jdoe@example.com')


class ArchiveFixture(ArchiveDirFixture):
    """A fixture that creates an archive in a sandbox."""

    memento_attrs = ArchiveDirFixture.memento_attrs + ('archive', 'master')

    def __init__(self, tla=False):
        super(ArchiveFixture, self).__init__()
        self.tla_archive_format = tla

    def setUp(self):
        """Create an archive in a sandbox."""
        super(ArchiveFixture, self).setUp()
        self.archive = arch.Archive(self.arch_name)
        self.master = arch.ArchiveLocation(self.location(self.arch_name))
        params = arch.ArchiveLocationParams()
        if self.tla_archive_format:
            params.tla_format()
        self.master.create_master(self.archive, params)

    def memento_key(self):
        inherited = super(ArchiveFixture, self).memento_key()
        return (self.tla_archive_format, inherited)


class MirrorFixture(ArchiveFixture):
    """A fixture that creates an archive and a mirror in a sandbox.

    The mirror has a different registered name (old style archive
    registration).
    """

    memento_attrs = ArchiveFixture.memento_attrs + ('mirror',)

    def setUp(self):
        super(MirrorFixture, self).setUp()
        master = self.archive
        mirror_name = master.name + '-MIRROR'
        location = self.location(mirror_name)
        self.mirror = master.make_mirror(mirror_name, location)


class MirrorLocationFixture(ArchiveFixture):
    """A fixture that creates an archive and a mirror in a sandbox.

    The mirror does not have a different registered name (new style archive
    registration).
    """

    memento_attrs = ArchiveFixture.memento_attrs + ('mirror',)

    def setUp(self):
        super(MirrorLocationFixture, self).setUp()
        mirror_name = self.archive.name + '-MIRROR'
        self.mirror = arch.ArchiveLocation(self.location(mirror_name))
        params = arch.ArchiveLocationParams()
        self.mirror.create_mirror(self.archive, params)


class MirroringFixture(ArchiveFixture, MyIdFixture):
    """A fixture that creates mirrors and versions to test mirroring."""

    memento_attrs = (ArchiveFixture.memento_attrs
                     + MyIdFixture.memento_attrs
                     + ('mirror1', 'mirror2', 'version1', 'version2',
                        'all_revisions', 'other_master', 'other_archive'))

    def setUp(self):
        super(MirroringFixture, self).setUp()
        mirror1_url = self.location(self.archive.name + '-1')
        mirror2_url = self.location(self.archive.name + '-2')
        self.mirror1 = arch.ArchiveLocation(mirror1_url)
        self.mirror2 = arch.ArchiveLocation(mirror2_url)
        params = arch.ArchiveLocationParams()
        self.mirror1.create_mirror(self.archive, params)
        self.mirror2.create_mirror(self.archive, params)
        self.version1 = self.archive['cat']['brn']['1']
        self.version2 = self.archive['cat']['brn']['2']
        tree_path = self.sandbox.tmp_dir/self.arch_dir_base/'tree'
        os.mkdir(tree_path)
        tree = arch.init_tree(tree_path, self.version1)
        tree.import_()
        self.version1['base-0'].make_continuation(self.version2)
        m = tree.log_message()
        m['Summary'] = 'first revision'
        tree.commit(m, just_commit=True)
        archive, master = self.archive, self.master
        self.all_revisions = list(archive.iter_location_revisions(master))
        self.other_archive = arch.Archive('bob@example.com')
        other_master_url = self.location(self.other_archive.name)
        self.other_master = arch.ArchiveLocation(other_master_url)
        params = arch.ArchiveLocationParams()
        self.other_master.create_master(self.other_archive, params)


class WorkingTreeFixture(ArchiveFixture, MyIdFixture):
    """A fixture that creates a commitable working tree in a sandbox."""

    memento_attrs = (ArchiveFixture.memento_attrs
                     + MyIdFixture.memento_attrs
                     + ('version', 'tree'))

    def setUp(self):
        super(WorkingTreeFixture, self).setUp()
        dirname = self.sandbox.tmp_dir/self.arch_dir_base/'workingtree'
        os.mkdir(dirname)
        self.version = self.archive['cat']['brn']['0']
        self.tree = arch.init_tree(dirname, self.version)


class SimpleRevisionFixture(WorkingTreeFixture):
    """A fixture that creates base-0 and patch-1 with no ancillary files."""

    def setUp(self):
        super(SimpleRevisionFixture, self).setUp()
        self.tree.import_()
        m = self.tree.log_message()
        m['Summary'] = 'first revision'
        self.tree.commit(m, just_commit=True)


class SimpleTlaRevisionFixture(SimpleRevisionFixture):
    """Creates base-0 and patch-1 with no ancillary files in a tla archive."""

    def __init__(self):
        super(SimpleTlaRevisionFixture, self).__init__()
        self.tla_archive_format = True


def _hash_dict(adict):
    items = adict.items()
    items.sort()
    return tuple(items)


class HistoryFixture(WorkingTreeFixture):
    """A fixture that creates a tree with a specified history of changes."""

    def __init__(self, *steps):
        super(HistoryFixture, self).__init__()
        self.__history = History(steps)

    def setUp(self):
        super(HistoryFixture, self).setUp()
        self.__history.run(self.tree)

    def memento_key(self):
        inherited = super(HistoryFixture, self).memento_key()
        return (self.__history, inherited)


class History(object):

    def __init__(self, steps):
        self.steps = [HistoryStep(i, s) for i, s in enumerate(steps)]

    def run(self, tree):
        for step in self.steps:
            step.run(tree)

    def __hash__(self):
        return hash(tuple(self.steps))


class HistoryStep(object):

    def __init__(self, index, changes):
        self.index = index
        self.actions = []
        for name, content in changes.items():
            if name.startswith('%'):
                continue
            self.actions.append(HistoryWrite(name, content))
        if '%add' in changes:
            self.actions.append(HistoryAdds(changes['%add']))
        if '%mv' in changes:
            self.actions.append(HistoryRenames(changes['%mv']))
        if '%rm' in changes:
            self.actions.append(HistoryDeletes(changes['%rm']))

    def run(self, tree):
        for action in self.actions:
            action.run(tree)
        if self.index == 0:
            tree.import_()
        else:
            msg = tree.log_message()
            msg['Summary'] = 'revision %d' % self.index
            tree.commit(msg)

    def __hash__(self):
        return hash(tuple([self.index] + self.actions))


class HistoryWrite(object):

    def __init__(self, name, content):
        self.name = name
        self.content = content

    def run(self, tree):
        open(tree/self.name, 'w').write(self.content)

    def __hash__(self):
        return hash((self.name, self.content))


class HistoryAdds(object):

    def __init__(self, names):
        self.names = names

    def run(self, tree):
        for name in self.names:
            if not os.path.isfile(tree/name):
                open(tree/name, 'w').close()
            tree.add_tag(name)

    def __hash__(self):
        return hash(tuple(self.names))


class HistoryRenames(object):

    def __init__(self, renames):
        self.renames = renames

    def run(self, tree):
        for old, new in self.renames:
            tree.move_file(old, new)
            tree.move_tag(old, new)

    def __hash__(self):
        return hash(tuple([tuple(x) for x in self.renames]))


class HistoryDeletes(object):

    def __init__(self, names):
        self.names = names

    def run(self, tree):
        for name in self.names:
            tree.delete(name)

    def __hash__(self):
        return hash(tuple(self.names))
