#include "routines.h"
#include "number.h"
#include <string.h>
#include <ctype.h>
#include <stdio.h>

/*

DecimalFormat provides decimal formatting for the XSLT format-number
extension function.  It is modeled on the Java equivalent (as per XSLT spec)

See:

http://java.sun.com/products/jdk/1.1/docs/api/java.text.DecimalFormatSymbols.html
http://java.sun.com/products/jdk/1.1/docs/api/java.text.DecimalFormat.html

 */

PyObject *DecimalFormat(PyObject *self, PyObject *args) {
  char *pattern;
  double number;
  DigitList digitList;
  PatternInfo patternInfo;
  DecimalFormatSymbols symbols;
  char temp[1024];
  PyObject *result;

  int i, pos;

  char *prefix, *suffix;

  short digitIndex, count, sizeBeforeIntegerPart;

  char isNegative, fractionPresent;

  symbols.decimalSeparator = '.';
  symbols.groupingSeparator = ',';
  symbols.infinity = "Infinity";
  symbols.minus = '-';
  symbols.NaN = "NaN";
  symbols.percent = '%';
  symbols.perMill = '%';  /* FIXME: should be unichr(0x2030) */
  symbols.zeroDigit = '0';
  symbols.digit = '#';
  symbols.separator = ';';

  if (!PyArg_ParseTuple(args, "ds|(ccscsccccc)", &number, &pattern,
                        &symbols.decimalSeparator, &symbols.groupingSeparator,
                        &symbols.infinity, &symbols.minus, &symbols.NaN,
                        &symbols.percent, &symbols.perMill, &symbols.zeroDigit,
                        &symbols.digit, &symbols.separator))
    return NULL;

  /*
    fprintf(stderr, "pattern: %s,  number: %f\n", pattern, number);
  */
  patternInfo.positivePattern = strdup(pattern);
  patternInfo.negativePattern = NULL;
  if (!parsePattern(&patternInfo, &symbols)) {
      /* There was an error while parsing pattern */
      free(patternInfo.positivePattern);
      if (patternInfo.negativePattern)
          free(patternInfo.negativePattern);
      return NULL;
  }
  /*
    fprintf(stderr, "posPrefix: %s  posSuffix: %s\n",
    patternInfo.positivePrefix,
    patternInfo.positiveSuffix);
    fprintf(stderr, "negPrefix: %s  negSuffix: %s\n",
    patternInfo.negativePrefix,
    patternInfo.negativeSuffix);
  */
  if (isnan(number)) {
      result = PyString_FromString(symbols.NaN);
      Py_XINCREF(result);
      free(patternInfo.positivePattern);
      if (patternInfo.negativePattern)
          free(patternInfo.negativePattern);
      return result;
  }

  /* Detecting whether a double is negative is easy with the exception of
   * the value -0.0.  This is a double which has a zero mantissa (and
   * exponent), but a negative sign bit.  It is semantically distinct from
   * a zero with a positive sign bit, and this distinction is important
   * to certain kinds of computations.  However, it's a little tricky to
   * detect, since (-0.0 == 0.0) and !(-0.0 < 0.0).  How then, you may
   * ask, does it behave distinctly from +0.0?  Well, 1/(-0.0) == -Inf. */

  isNegative = ((number < 0.0) || (number == 0.0 && 1/number) < 0.0);

  if (isNegative) {
      number = -number;
      prefix = patternInfo.negativePrefix;
      suffix = patternInfo.negativeSuffix;
  } else {
      prefix = patternInfo.positivePrefix;
      suffix = patternInfo.negativeSuffix;
  }

  if (patternInfo.multiplier != 1) number *= patternInfo.multiplier;

  /* Make sure the result string starts empty */
  *temp = 0;

  if (number == atof("inf")) {
      strcat(temp, prefix);
      strcat(temp, symbols.infinity);
      strcat(temp, suffix);
      result = PyString_FromString(temp);
      Py_XINCREF(result);
      free(patternInfo.positivePattern);
      if (patternInfo.negativePattern)
          free(patternInfo.negativePattern);
      return result;
  }

  parseNumber(number, patternInfo.maxFractionDigits, &digitList);

  strcat(temp, prefix);
  pos = strlen(temp);

  count = patternInfo.minIntegerDigits;
  digitIndex = 0;
  sizeBeforeIntegerPart = pos;

  if (digitList.decimalAt > 0 && count < digitList.decimalAt)
      count = digitList.decimalAt;

  /* Handle the case where maxIntegerDigits is smaller
   * than the real number of integer digits.  If this is so, we
   * output the least significant max integer digits.  For example,
   * the value 1997 printed with 2 max integer digits is just "97". */
  if (count > patternInfo.maxIntegerDigits) {
      count = patternInfo.maxIntegerDigits;
      digitIndex = digitList.decimalAt - count;
  }

  for (i = count - 1; i >= 0; i--) {
      if (i < digitList.decimalAt && digitIndex < digitList.count)
          /* Output a real digit */
          temp[pos++] = digitList.digits[digitIndex++] + symbols.zeroDigit;
      else
          /* Output a leading zero */
          temp[pos++] = symbols.zeroDigit;

      /* Output grouping separator if necessary.  Don't output a
       * grouping separator if i==0 though; that's at the end of
       * the integer part. */
      if (patternInfo.groupingUsed && i > 0 &&
          (patternInfo.groupingSize != 0) &&
          (i % patternInfo.groupingSize == 0))
          temp[pos++] = symbols.groupingSeparator;
  }

  /* Determine whether or not there are any printable fractional
   * digits.  If we've used up the digits we know there aren't. */
  fractionPresent = (patternInfo.minFractionDigits > 0) ||
      (digitIndex < digitList.count);

  /* If there is no fraction present, and we haven't printed any
   * integer digits, then print a zero.  Otherwise we won't print
   * _any_ digits, and we won't be able to parse this string. */
  if (!fractionPresent && pos == sizeBeforeIntegerPart) {
      temp[pos++] = symbols.zeroDigit;
  }

  /* Output the decimal separator if we always do so. */
  if (patternInfo.showDecimalPoint || fractionPresent) {
      temp[pos++] = symbols.decimalSeparator;
  }

  for (i = 0; i < patternInfo.maxFractionDigits; i++) {
      /* Here is where we escape from the loop.  We escape if we've output
       * the maximum fraction digits (specified in the for expression above).
       * We also stop when we've output the minimum digits and either:
       * we have an integer, so there is no fractional stuff to display,
       * or we're out of significant digits. */
      if (i >= patternInfo.minFractionDigits && (digitIndex >= digitList.count)) {
          break;
      }

      /* Output leading fractional zeros.  These are zeros that come after
       * the decimal but before any significant digits.  These are only
       * output if abs(number being formatted) < 1.0. */
      if (-1 - i > (digitList.decimalAt - 1)) {
          temp[pos++] = symbols.zeroDigit;
          continue;
      }
      /* Output a digit, if we have any precision left, or a
       * zero if we don't.  We don't want to output noise digits. */
      if (digitIndex < digitList.count)
          temp[pos++] = digitList.digits[digitIndex++] + symbols.zeroDigit;
      else
          temp[pos++] = symbols.zeroDigit;
  }
  temp[pos] = 0;
  strcat(temp, suffix);
  result = PyString_FromString(temp);
  Py_XINCREF(result);
  free(patternInfo.positivePattern);
  if (patternInfo.negativePattern)
      free(patternInfo.negativePattern);
  return result;
}

int parsePattern(PatternInfo *pi, DecimalFormatSymbols *symbols) {
    char *pattern;
    char noNegative = 1; /* true/false */

    /* Two variables are used to record the subrange of the pattern
     * occupied by phase 1.  This is used during the processing of the
     * second pattern (the one representing negative numbers) to ensure
     * that no deviation exists in phase 1 between the two patterns. */
    int phaseOneStart = 0;
    int phaseOneLength = 0;

    int start = 0;
    int end = 0;
    int j, pos;
    char ch;

    char inQuote;
    char *prefix, *suffix, *affix;
    char *prefixEnd, *suffixEnd;
    int decimalPos = -1;
    int multiplier = 1;

    int digitLeftCount = 0;
    int digitRightCount = 0;
    int zeroDigitCount = 0;

    char groupingCount = -1;

    int phase = 0;

    prefixEnd = NULL;

    pattern = pi->positivePattern;
    start = 0;
    end = strlen(pattern);
    if (strchr(pattern, symbols->separator)) noNegative--;
    for (j = 1; j >= noNegative; j--) {
      inQuote = 0;
      decimalPos = -1;
      multiplier = 1;
      digitLeftCount = zeroDigitCount = digitRightCount = 0;
      groupingCount = -1;
      /* The phase ranges from 0 to 2.  Phase 0 is the prefix.  Phase 1 is
       * the section of the pattern with digits, decimal separator,
       * grouping characters.  Phase 2 is the suffix.  In phases 0 and 2,
       * percent, permille, and currency symbols are recognized and
       * translated.  The separation of the characters into phases is
       * strictly enforced; if phase 1 characters are to appear in the
       * suffix, for example, they must be quoted. */
      phase = 0;

      /* The affix is either the prefix or the suffix. */
      prefix = pattern + start;
      suffix = suffixEnd = pattern + end;
      affix = prefix;

      for (pos = start; pos < end; pos++) {
          ch = pattern[pos];
          switch (phase) {
          case 0:
          case 2:
              /* Process the prefix/suffix characters */
              if (inQuote) {
                  /* A quote within quotes indicates either the closing
                   * quote or two quotes, which is a quote literal.  That is,
                   * we have the second quote in 'do' or 'don''t'. */
                  if (ch == QUOTE) {
                      if ((pos+1) < end && pattern[pos+1] == QUOTE) {
                          pos++;
                          *affix++ = ch;
                      } else {
                          inQuote = 0;        /* 'do' */
                      }
                      continue;
                  }
              } else {
                  /* Process unquoted characters seen in prefix or suffix phase. */
                  if (ch == symbols->digit ||
                      ch == symbols->zeroDigit ||
                      ch == symbols->groupingSeparator ||
                      ch == symbols->decimalSeparator) {
                      /* Any of these characters implicitly begins the next
                       * phase.  If we are in phase 2, there is no next phase,
                       * so these characters are illegal. */
                      phaseOneStart = pos;
                      phaseOneLength = 0;
                      prefixEnd = affix;
                      phase = 1;
                      pos--;            /* Reprocess this character */
                      continue;
                  } else if (ch == QUOTE) {
                      /* A quote outside quotes indicates either the opening
                       * quote or two quotes, which is a quote literal.  That is,
                       * we have the first quote in 'do' or o''clock. */
                      if (ch == QUOTE) {
                          if ((pos+1) < end && pattern[pos+1] == QUOTE) {
                              pos++;
                              *affix++ = ch;
                          } else {
                              inQuote = 1;
                          }
                          continue;
                      }
                  } else if (ch == symbols->separator) {
                      /* Don't allow separators before we see digit characters of phase
                       * 1, and don't allow separators in the second pattern (j == 0). */
                      if (phase == 0 || j == 0) {
                          PyErr_Format(PyExc_SyntaxError,
                                       "Unquoted special character '%c' in pattern '%s'",
                                       ch, pattern);
                          return 0;
                      }
                      suffixEnd = affix;
                      start = pos + 1;
                      pos = end;
                      continue;
                  }

                  /* Next handle characters which are appended directly. */
                  else if (ch == symbols->percent) {
                      if (multiplier != 1) {
                          PyErr_Format(PyExc_SyntaxError,
                                       "Too many percent/permille characters in pattern '%s'",
                                       pattern);
                          return 0;
                      }
                      multiplier = 100;
                  } else if (ch == symbols->perMill) {
                      if (multiplier != 1) {
                          PyErr_Format(PyExc_SyntaxError,
                                       "Too many percent/permille characters in pattern '%s'",
                                       pattern);
                          return 0;
                      }
                      multiplier = 1000;
                  }
              }
              *affix++ = ch;
              break;
          case 1:
              if (j == 1) {
                  /* Process the digits, decimal, and grouping characters.  We
                   * record five pieces of information.  We expect the digits
                   * to occur in the pattern ####0000.####, and we record the
                   * number of left digits, zero (central) digits, and right
                   * digits.  The position of the last grouping character is
                   * recorded (should be somewhere within the first two blocks
                   * of characters), as is the position of the decimal point,
                   * if any (should be in the zero digits).  If there is no
                   * decimal point, then there should be no right digits. */
                  if (ch == symbols->digit) {
                      if (zeroDigitCount > 0) digitRightCount++;
                      else digitLeftCount++;
                      if (groupingCount >= 0 && decimalPos < 0) groupingCount++;
                  } else if (ch == symbols->zeroDigit) {
                      if (digitRightCount > 0) {
                          PyErr_Format(PyExc_SyntaxError,
                                       "Unexpected '0' in pattern '%s'",
                                       pattern);
                          return 0;
                      }
                      zeroDigitCount++;
                      if (groupingCount >= 0 && decimalPos < 0) groupingCount++;
                  } else if (ch == symbols->groupingSeparator) {
                      groupingCount = 0;
                  } else if (ch == symbols->decimalSeparator) {
                      if (decimalPos >= 0) {
                          PyErr_Format(PyExc_SyntaxError,
                                       "Multiple decimal separators in pattern '%s'",
                                       pattern);
                          return 0;
                      }
                      decimalPos = digitLeftCount + zeroDigitCount + digitRightCount;
                  } else {
                      /* Save this position as start of suffix */
                      suffix = pattern + pos;
                      affix = suffix;
                      phase = 2;
                      pos--;
                      continue;
                  }
              } else {
                  /* Phase one must be identical in the two sub-patterns.  We
                   * enforce this by doing a direct comparison.  While
                   * processing the second, we compare characters. */
                  if (ch != pattern[phaseOneStart + phaseOneLength]) {
                      PyErr_Format(PyExc_SyntaxError,
                                   "Subpattern mismatch in pattern '%s'",
                                   pattern);
                      return 0;
                  }
                  /* Ignore formating in the negative pattern */
                  if (ch == symbols->digit ||
                      ch == symbols->zeroDigit ||
                      ch == symbols->groupingSeparator ||
                      ch == symbols->decimalSeparator) {
                      phaseOneLength++;
                  } else {
                      suffix = pattern + pos;
                      affix = suffix;
                      phase = 2;
                      pos--;
                      continue;
                  }
              }
              break;
          }
      }
      /* Handle patterns with no '0' pattern character.  These patterns
       * are legal, but must be interpreted.  "##.###" -> "#0.###".
       * ".###" -> ".0##".
       * We allow patterns of the form "####" to produce a zeroDigitCount of
       * zero (got that?); although this seems like it might make it possible
       * for format() to produce empty strings, format() checks for this
       * condition and outputs a zero digit in this situation.  Having a
       * zeroDigitCount of zero yields a minimum integer digits of zero, which
       * allows proper round-trip patterns.  That is, we don't want "#" to
       * become "#0" when toPattern() is called (even though that's what it
       * really is, semantically).  */
      if (zeroDigitCount == 0 && digitLeftCount > 0 && decimalPos >= 0) {
          /* Handle "###.###" and "###." and ".###" */
          int n = decimalPos;
          if (n == 0) ++n; /* Handle ".###" */
          digitRightCount = digitLeftCount - n;
          digitLeftCount = n - 1;
          zeroDigitCount = 1;
      }

      /* Do syntax checking on the digits. */
      if ((decimalPos < 0 && digitRightCount > 0) ||
          (decimalPos >= 0 && (decimalPos < digitLeftCount
                               || decimalPos > (digitLeftCount + zeroDigitCount)))
          || groupingCount == 0 || inQuote) {
          PyErr_Format(PyExc_SyntaxError, "Malformed pattern '%s'", pattern);
          return 0;
      }

      /* Terminate the prefix/suffix strings. */
      *prefixEnd = 0;
      *suffixEnd = 0;

      if (j == 1) {
          int digitTotalCount = digitLeftCount + zeroDigitCount + digitRightCount;
          int effectiveDecimalPos = decimalPos >= 0 ? decimalPos : digitTotalCount;
          pi->positivePrefix = prefix;
          pi->positiveSuffix = suffix;
          pi->negativePrefix = pi->negativeSuffix = NULL;
          /* The effectiveDecimalPos is the position the decimal is at or
           * would be at if there is no decimal.  Note that if decimalPos<0,
           * then digitTotalCount == digitLeftCount + zeroDigitCount.  */
          pi->minIntegerDigits = effectiveDecimalPos - digitLeftCount;
          pi->maxIntegerDigits = DOUBLE_INTEGER_DIGITS;
          if (decimalPos >= 0) {
              pi->maxFractionDigits = digitTotalCount - decimalPos;
              pi->minFractionDigits = digitLeftCount + zeroDigitCount - decimalPos;
          } else {
              pi->minFractionDigits = 0;
              pi->maxFractionDigits = 0;
          }
          pi->groupingUsed = (groupingCount > 0);
          pi->groupingSize = (groupingCount > 0) ? groupingCount : 0;
          pi->showDecimalPoint = (decimalPos == 0 || decimalPos == digitTotalCount);
          pi->multiplier = multiplier;
      } else {
          pi->negativePrefix = prefix;
          pi->negativeSuffix = suffix;
      }
    }

    if (end == 0) {
        pi->positivePrefix = pi->positiveSuffix = "";
        pi->negativePrefix = pi->negativeSuffix = NULL;
        pi->minIntegerDigits = 0;
        pi->maxIntegerDigits = DOUBLE_INTEGER_DIGITS;
        pi->minFractionDigits = 0;
        pi->maxFractionDigits = DOUBLE_FRACTION_DIGITS;
    }

    /* If there was no negative pattern, then prepend the minus sign to
     * the positive pattern to form the negative pattern. */
    if (!pi->negativePrefix && !pi->negativeSuffix) {
        pi->negativePattern = (char*)malloc(strlen(pi->positivePrefix)+2);
        pi->negativePrefix = pi->negativePattern;
        *(pi->negativePrefix) = symbols->minus;
        strcpy(pi->negativePattern + 1, pi->positivePrefix);
        pi->negativeSuffix = pi->positiveSuffix;
    } else {
        pi->negativePattern = NULL;
    }

    return 1; /* SUCCESS */
}

void parseNumber(double number, int maximumDigits, DigitList *dl) {
    int i;
    int printedDigits;
    char ch;
    char nonZeroFound = 0;  /* boolean */
    char rounding;
    int zerosSkipped = 0;

    /* One more decimal digit than required for rounding */
    printedDigits = snprintf(dl->digits, DOUBLE_FRACTION_DIGITS, "%.*f",
                             maximumDigits + 1, number);

    dl->decimalAt = -1;
    dl->count = 0;
    for (i = 0; i < printedDigits; i++) {
        ch = dl->digits[i];
        if (ch == '.') {
            dl->decimalAt = dl->count;
	    zerosSkipped = 0;
        } else {
	  if (!nonZeroFound) {
	    nonZeroFound = (ch != '0'); zerosSkipped ++;
	  }
	  if (nonZeroFound) {
	    dl->digits[dl->count++] = ch - '0';
	  }
        }
    }

    if (zerosSkipped != 0) {
      dl->decimalAt -= (zerosSkipped - 1);
    }

    i = dl->count;
    rounding = 0;
    do {
        i--;
        if (!rounding && dl->digits[i] >= 5) {
            rounding = 1;
        } else if (rounding) {
            if (dl->digits[i] == 9) {
                /* continue rounding up */
                dl->digits[i] = 0;
            } else {
                dl->digits[i]++;
                rounding = 0;
            }
        }
    } while (rounding);


    /* Remove the rounding digit */
    dl->count--;

    /* Remove any trailing zeros from the decimal part */
    for (i = dl->count-1; i > dl->decimalAt; i--) {
        if (dl->digits[i] == 0) dl->count--;
        else break;
    }
}

static PyMethodDef routinesMethods[] = {
     { "FormatNumber", DecimalFormat, 1 },
     { NULL, NULL }
};

void initroutines(void) {
  PyObject *module;

  module = Py_InitModule("routines", routinesMethods);
};


