/***************************************************************************
  grainpage.cpp
  -------------------
  A dialog page for grains
  -------------------
  begin         October 10th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include "resource.h"
#include "grainpage.h"

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>

#include "fixedspinbox.h"

#include "qbrewdoc.h"
#include "qbrewview.h"
#include "qbrewcalc.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// GrainPage()
// -----------
// Constructor

GrainPage::GrainPage(QWidget *parent, const char *name, QObject *doc, QObject *calc)
    : IngredientPage(parent, name, doc, calc), extractlabel_(0), extractspin_(0),
    colorlabel_(0), colorspin_(0), uselabel_(0), usecombo_(0), currentitem_(0)
{
    // setup listview
    itemslistview_->addColumn(tr("Extract"));
    itemslistview_->addColumn(tr("Color"));
    itemslistview_->addColumn(tr("Use"));
    itemslistview_->setColumnAlignment(2, AlignRight);
    itemslistview_->setColumnAlignment(3, AlignRight);
    itemslistview_->setColumnAlignment(4, AlignRight);
    itemslistview_->setSorting(0);
    itemslistview_->setSorting(1);
    itemslistview_->setSorting(2);
    itemslistview_->setSorting(3);
    itemslistview_->setSorting(4);

    // setup widgets
    quantityspin_->setSuffix(tr(" lbs"));
    extractlabel_ = new QLabel(tr("Extract"), this, "extractlabel_");
    extractspin_ = new FixedSpinBox(1.000, 1.100, 0.001, 3, this, "extractspin_");
    colorlabel_ = new QLabel(tr("Color"), this, "colorlabel_");
    colorspin_ = new FixedSpinBox(0.0, 500.0, 1.0, 1, this, "colorspin_");
    colorspin_->setSuffix(AppResource::CHAR_LATIN_DEGREE);
    uselabel_ = new QLabel(tr("Use"), this, "uselabel_");
    usecombo_ = new QComboBox(false, this, "usecombo_");
    usecombo_->insertStringList(Grain::useStringList());

    // setup layout
    rowtwolayout_->addWidget(quantitylabel_, 0);
    rowtwolayout_->addWidget(quantityspin_, 0, AlignRight);
    rowtwolayout_->addWidget(extractlabel_, 0);
    rowtwolayout_->addWidget(extractspin_, 0);
    rowtwolayout_->addWidget(colorlabel_, 0);
    rowtwolayout_->addWidget(colorspin_, 0);
    rowtwolayout_->addWidget(uselabel_, 0);
    rowtwolayout_->addWidget(usecombo_, 0);

    rowtwolayout_->addWidget(spacer_, 1);
    rowtwolayout_->addWidget(addbutton_, 0);
    rowtwolayout_->addWidget(applybutton_, 0);
    rowtwolayout_->addWidget(removebutton_, 0);

    // set the leftmost labels to the same minimum size
    quantitylabel_->adjustSize();
    quantitylabel_->setMinimumWidth(quantitylabel_->width());
    itemlabel_->setMinimumWidth(quantitylabel_->width());

    // fixup tab orders
    QWidget::setTabOrder(itemcombo_, quantityspin_);
    QWidget::setTabOrder(quantityspin_, extractspin_);
    QWidget::setTabOrder(extractspin_, colorspin_);
    QWidget::setTabOrder(colorspin_, usecombo_);

    // setup connections
    connect(itemcombo_,SIGNAL(activated(const QString &)), this, SLOT(slotUpdateEntryItems(const QString &)));
    connect(addbutton_, SIGNAL(clicked()), SLOT(slotAddGrain()));
    connect(applybutton_, SIGNAL(clicked()), SLOT(slotApplyGrain()));
    connect(removebutton_, SIGNAL(clicked()), SLOT(slotRemoveGrain()));
    connect(itemslistview_, SIGNAL(selectionChanged(QListViewItem *)), SLOT(slotItemSelected(QListViewItem *)));

    slotDocumentChanged();
}

GrainPage::~GrainPage() {}

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotUpdateEntryItems()
// ----------------------
// Received to update the items in the entry area

void GrainPage::slotUpdateEntryItems(const QString &name)
{
    Grain *thisgrain = ((QBrewCalc *)calc_)->grain(name);
    quantityspin_->setValue(thisgrain->quantity());
    extractspin_->setValue(thisgrain->extract());
    colorspin_->setValue(thisgrain->color());
    // vary default use according to recipe type
    int utemp = thisgrain->use();
    QString ustr = thisgrain->useString();
    if (((QBrewDoc *)document_)->mash()) {
        if (utemp == GRAIN_STEEPED) ustr = tr("mashed");
    } else {
        if (utemp == GRAIN_MASHED) ustr = tr("steeped");
    }
    setComboItem(usecombo_, ustr);
}

//////////////////////////////////////////////////////////////////////////////
// slotDocumentChanged()
// ---------------------
// Received if document has changed

void GrainPage::slotDocumentChanged()
{
    itemcombo_->clear();
    itemcombo_->insertStringList(((QBrewCalc *)calc_)->grainsList());
    slotUpdateEntryItems(itemcombo_->currentText());
    updateListView();
}

//////////////////////////////////////////////////////////////////////////////
// slotAddGrain()
// --------------
// Received when the user adds a grain to the recipe

void GrainPage::slotAddGrain()
{
    GrainIterator it;
    GrainListItem *item;
    it = ((QBrewDoc *)document_)->addGrain(itemcombo_->currentText(), quantityspin_->value(),
        extractspin_->value(), colorspin_->value(), usecombo_->currentText());
    item = new GrainListItem( itemslistview_, (*it).name(),
        (*it).quantityString(),
        (*it).extractString(),
        (*it).colorString(),
        (*it).useString() );
    item->setGrainIterator(it);
}

//////////////////////////////////////////////////////////////////////////////
// slotApplyGrain()
// ----------------
// Received when the user applies a grain to the recipe

void GrainPage::slotApplyGrain()
{
     if (currentitem_ != 0) {
        ((QBrewDoc *)document_)->applyGrain(currentitem_->grainIterator(),
                itemcombo_->currentText(), quantityspin_->value(), extractspin_->value(),
                colorspin_->value(), usecombo_->currentText());
        currentitem_->setToIterator();
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotRemoveGrain()
// -----------------
// Received when the user removes a grain from the recipe

void GrainPage::slotRemoveGrain()
{
    if (currentitem_ != 0) {
        ((QBrewDoc *)document_)->removeGrain(currentitem_->grainIterator());
        delete currentitem_; currentitem_ = 0;
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotItemSelected()
// ------------------
// Received when a grain is selected in the listview

void GrainPage::slotItemSelected(QListViewItem *item)
{
    currentitem_ = (GrainListItem *)item;
    if (item != 0) {
        itemcombo_->setEditText(item->text(0));
        quantityspin_->setValue(item->text(1).toDouble());
        extractspin_->setValue(item->text(2).toDouble());
        colorspin_->setValue(item->text(3).toDouble());
        setComboItem(usecombo_, item->text(4));
    }
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// setComboItem()
// --------------
// Set a combo item to the given text

// TODO: this is a duplicate from QBrewView... fix
void GrainPage::setComboItem(QComboBox *combo, const QString &text)
{
    int item;
    // search combo box for style string
    for (item = 0; item < combo->count(); item++) {
        if (combo->text(item) == text) {
            combo->setCurrentItem(item);
            return;
        }
    }
    // if item not found, will set it to the first one
    combo->setCurrentItem(0);
}

//////////////////////////////////////////////////////////////////////////////
// updateListView()
// ----------------
// Update the list view

void GrainPage::updateListView()
{
    GrainList *list = ((QBrewDoc *)document_)->grainList();
    GrainIterator it;
    GrainListItem *item;
    itemslistview_->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new GrainListItem( itemslistview_, (*it).name(),
            (*it).quantityString(),
            (*it).extractString(),
            (*it).colorString(),
            (*it).useString() );
        item->setGrainIterator(it);
    }
}
