/****************************************************************************
** helpwindow.cpp
**
** Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
** This file is part of an example program for Qt.  This example
** program may be used, distributed and modified without limitation.
**
** Modified by David Johnson for use in QBrew
*****************************************************************************/

#include <qapplication.h>
#include <qiconset.h>
#include <qmenubar.h>
#include <qpaintdevicemetrics.h>
#include <qpixmap.h>
#include <qpopupmenu.h>
#include <qprinter.h>
#include <qtextbrowser.h>
#include <qtoolbar.h>
#include <qtoolbutton.h>
#include <qsimplerichtext.h>
#include <qstatusbar.h>
#include <qstylesheet.h>

#include "helpwindow.h"
#include "home.xpm"
#include "back.xpm"
#include "forward.xpm"

//////////////////////////////////////////////////////////////////////////////
// Construction                                                             //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// HelpWindow()
// ------------
// Constructor

HelpWindow::HelpWindow(const QString& home, const QString& path, QWidget* parent, const char *name)
    : QMainWindow(parent, name, WDestructiveClose), browser_(0), _backward(0), forward_(0)
{
    browser_ = new QTextBrowser(this);
    browser_->mimeSourceFactory()->setFilePath(path);
    browser_->setFrameStyle(QFrame::Panel | QFrame::Sunken);
    connect(browser_, SIGNAL(textChanged()), this, SLOT(textChanged()));

    setCentralWidget(browser_);
    if (!home.isEmpty()) browser_->setSource(home);
    connect(browser_, SIGNAL(highlighted(const QString&)), statusBar(), SLOT(message(const QString&)));

    // set a reasonable width
    int width = 480;
    if ((width * 3) > (QApplication::desktop()->width() * 2))
        width = QApplication::desktop()->width() * 2 / 3;
    else if ((width * 2) < (QApplication::desktop()->width()))
        width = QApplication::desktop()->width() / 2;
    // set a reasonable height
    int height = 640;
    if ((height * 3) > (QApplication::desktop()->height() * 2))
        height = QApplication::desktop()->height() * 2 / 3;
    else if ((height * 2) < (QApplication::desktop()->height()))
        height = QApplication::desktop()->height() / 2;
    resize(width, height);

    // setup menus
    QPopupMenu* file = new QPopupMenu(this);
    file->insertItem(tr("&Print"), this, SLOT(print()), ALT | Key_P);
    file->insertSeparator();
    file->insertItem(tr("&Quit"), this, SLOT(close()), ALT | Key_Q);
    QPopupMenu* go = new QPopupMenu(this);
    _backward = go->insertItem(QPixmap(backxpm),
        tr("&Backward"), browser_, SLOT(backward()), ALT | Key_Left);
    forward_ = go->insertItem(QPixmap(forwardxpm),
        tr("&Forward"), browser_, SLOT(forward()), ALT | Key_Right);
    go->insertItem(QPixmap(homexpm), tr("&Home"), browser_, SLOT(home()));

    menuBar()->insertItem(tr("&File"), file);
    menuBar()->insertItem(tr("&Go"), go);
    menuBar()->insertSeparator();

    menuBar()->setItemEnabled(forward_, false);
    menuBar()->setItemEnabled(_backward, false);
    connect(browser_, SIGNAL(backwardAvailable(bool)), this, SLOT(setBackwardAvailable(bool)));
    connect(browser_, SIGNAL(forwardAvailable(bool)), this, SLOT(setForwardAvailable(bool)));

    // setup tool bar
    QToolBar* toolbar = new QToolBar(this);
    addToolBar(toolbar, "Toolbar");
    QToolButton* button;
    button = new QToolButton(QPixmap(backxpm), tr("Backward"), "", browser_, SLOT(backward()), toolbar);
    connect(browser_, SIGNAL(backwardAvailable(bool)), button, SLOT(setEnabled(bool)));
    button->setEnabled(false);
    button = new QToolButton(QPixmap(forwardxpm), tr("Forward"), "", browser_, SLOT(forward()), toolbar);
    connect(browser_, SIGNAL(forwardAvailable(bool)), button, SLOT(setEnabled(bool)));
    button->setEnabled(false);
    button = new QToolButton(QPixmap(homexpm), tr("Home"), "", browser_, SLOT(home()), toolbar);

    browser_->setFocus();
}

//////////////////////////////////////////////////////////////////////////////
// ~HelpWindow()
// -------------
// Destructor

HelpWindow::~HelpWindow() { ; }

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// setBackwardAvailable()
// ----------------------
// Set the backward button available

void HelpWindow::setBackwardAvailable(bool b)
{
    menuBar()->setItemEnabled(_backward, b);
}

//////////////////////////////////////////////////////////////////////////////
// setForwardAvailable()
// ----------------------
// Set the forward button available

void HelpWindow::setForwardAvailable(bool b)
{
    menuBar()->setItemEnabled(forward_, b);
}

//////////////////////////////////////////////////////////////////////////////
// textChanged()
// -------------
// The displayed document has changed, so change caption

void HelpWindow::textChanged()
{
    if (browser_->documentTitle().isNull())
        setCaption("QBrew " + tr("Help") + " - " + browser_->context());
    else
        setCaption("QBrew " + tr("Help") + " - " + browser_->documentTitle());
}

//////////////////////////////////////////////////////////////////////////////
// print()
// -------
// Print the contents of the help window

void HelpWindow::print()
{
    QPrinter printer;
    printer.setFullPage(true);
    if (printer.setup()) {
        QPainter p(&printer);
        QPaintDeviceMetrics metrics(p.device());
        int dpix = metrics.logicalDpiX();
        int dpiy = metrics.logicalDpiY();
        const int margin = 72; // pt
        QRect body(margin * dpix / 72, margin * dpiy / 72,
            metrics.width() - margin * dpix / 72 * 2,
            metrics.height() - margin * dpiy / 72 * 2);
        QFont font("times", 10);
        QSimpleRichText richText(browser_->text(), font, browser_->context(), browser_->styleSheet(),
            browser_->mimeSourceFactory(), body.height());
        richText.setWidth(&p, body.width());
        QRect view(body);
        int page = 1;
        while (true) {
            richText.draw(&p, body.left(), body.top(), view, colorGroup());
            view.moveBy(0, body.height());
            p.translate(0, -body.height());
            p.setFont(font);
            p.drawText(view.right() - p.fontMetrics().width(QString::number(page)),
                view.bottom() + p.fontMetrics().ascent() + 5, QString::number(page));
            if (view.top() >= richText.height()) break;
            printer.newPage();
            ++page;
        }
    }
}
