/***************************************************************************
  hopspage.cpp
  -------------------
  description   A dialog page for hops
  -------------------
  begin         October 20th, 1999
  author        David Johnson <david@usermode.org>
  -------------------
  Copyright 1999, 2001, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include "hopspage.h"

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlistview.h>
#include <qpushbutton.h>

#include "fixedspinbox.h"

#include "qbrewdoc.h"
#include "qbrewview.h"
#include "qbrewcalc.h"

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// HopsPage()
// ----------
// Constructor

HopsPage::HopsPage(QWidget *parent, const char *name, QObject *doc, QObject *calc)
    : IngredientPage(parent, name, doc, calc), formlabel_(0), formcombo_(0),
    alphalabel_(0), alphaspin_(0), timelabel_(0), timespin_(0), currentitem_(0)
{
    // setup listview
    itemslistview_->addColumn(tr("Form"));
    itemslistview_->addColumn(tr("Alpha"));
    itemslistview_->addColumn(tr("Time"));
    itemslistview_->setColumnAlignment(2, AlignRight);
    itemslistview_->setColumnAlignment(3, AlignRight);
    itemslistview_->setColumnAlignment(4, AlignRight);
    itemslistview_->setSorting(0);
    itemslistview_->setSorting(1);
    itemslistview_->setSorting(2);
    itemslistview_->setSorting(3);
    itemslistview_->setSorting(4);

    // setup widgets
    quantityspin_->setSuffix(tr(" oz"));
    formlabel_ = new QLabel(tr("Form"), this, "formlabel_");
    formcombo_ = new QComboBox(true, this, "formcombo_");
    formcombo_->insertStringList(((QBrewCalc *)calc_)->formsList());
    alphalabel_ = new QLabel(tr("Alpha"), this, "alphalabel_");
    alphaspin_ = new FixedSpinBox(0.0, 50.0, 0.1, 1, this, "alphaspin_");
    alphaspin_->setSuffix("%");
    timelabel_ = new QLabel(tr("Time"), this, "timelabel_");
    timespin_ = new QSpinBox(0, 120, 5, this, "time");
    timespin_->setSuffix(tr(" min"));

    // setup layout
    rowtwolayout_->addWidget(quantitylabel_, 0);
    rowtwolayout_->addWidget(quantityspin_, 0, AlignRight);
    rowtwolayout_->addWidget(formlabel_, 0);
    rowtwolayout_->addWidget(formcombo_, 0);
    rowtwolayout_->addWidget(alphalabel_, 0);
    rowtwolayout_->addWidget(alphaspin_, 0);
    rowtwolayout_->addWidget(timelabel_, 0);
    rowtwolayout_->addWidget(timespin_, 0);

    rowtwolayout_->addWidget(spacer_, 1);
    rowtwolayout_->addWidget(addbutton_, 0);
    rowtwolayout_->addWidget(applybutton_, 0);
    rowtwolayout_->addWidget(removebutton_, 0);

    // set the leftmost labels to the same minimum size
    quantitylabel_->adjustSize();
    quantitylabel_->setMinimumWidth(quantitylabel_->width());
    itemlabel_->setMinimumWidth(quantitylabel_->width());

    // fixup tab orders
    QWidget::setTabOrder(itemcombo_, quantityspin_);
    QWidget::setTabOrder(quantityspin_, formcombo_);
    QWidget::setTabOrder(formcombo_, alphaspin_);
    QWidget::setTabOrder(alphaspin_, timespin_);

    // setup connections
    connect(itemcombo_,SIGNAL(activated(const QString &)), this, SLOT(slotUpdateEntryItems(const QString &)));
    connect(addbutton_, SIGNAL(clicked()), SLOT(slotAddHops()));
    connect(applybutton_, SIGNAL(clicked()), SLOT(slotApplyHops()));
    connect(removebutton_, SIGNAL(clicked()), SLOT(slotRemoveHops()));
    connect(itemslistview_, SIGNAL(selectionChanged(QListViewItem *)), SLOT(slotItemSelected(QListViewItem *)));
    // initialize widgets with values
    slotDocumentChanged();
}

HopsPage::~HopsPage() { ; }

//////////////////////////////////////////////////////////////////////////////
// Slots                                                                    //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// slotUpdateEntryItems()
// ----------------------
// Received to update the hop entry widgets

void HopsPage::slotUpdateEntryItems(const QString &name)
{
    // highlight the appropriate entry in the listview, if available
    QListViewItem *item = itemslistview_->firstChild();
    while (item != 0) {
        if (item->text(0) == name) {
            itemslistview_->setSelected(item, true);
            // this sends the signal selectionChanged()
            return;
        }
        item = item->nextSibling();
    }
    // wasn't in the listview, update from database
    Hops *thisHops = ((QBrewCalc *)calc_)->hop(name);
    quantityspin_->setValue(thisHops->quantity());
    formcombo_->setEditText(thisHops->form());
    alphaspin_->setValue(thisHops->alpha());
    timespin_->setValue(thisHops->time());
}

//////////////////////////////////////////////////////////////////////////////
// slotDocumentChanged()
// ---------------------
// Received when the document has changed

void HopsPage::slotDocumentChanged()
{
    itemcombo_->clear();
    itemcombo_->insertStringList(((QBrewCalc *)calc_)->hopsList());
    slotUpdateEntryItems(itemcombo_->currentText());
    updateListView();
}

//////////////////////////////////////////////////////////////////////////////
// slotAddHops()
// -------------
// Received when user adds a hop to recipe

void HopsPage::slotAddHops()
{
    HopIterator it;
    HopsListItem *item;
    it = ((QBrewDoc *)document_)->addHop(itemcombo_->currentText(), quantityspin_->value(),
        formcombo_->currentText(), alphaspin_->value(), timespin_->value());
    item = new HopsListItem( itemslistview_, (*it).name(),
        (*it).quantityString(),
        (*it).form(),
        (*it).alphaString(),
        (*it).timeString() );
    item->setHopIterator(it);
}

//////////////////////////////////////////////////////////////////////////////
// slotApplyHops()
// ---------------
// Received when user applies a hop to recipe

void HopsPage::slotApplyHops()
{
     if (currentitem_ != 0) {
        ((QBrewDoc *)document_)->applyHop(currentitem_->hopIterator(),
                itemcombo_->currentText(), quantityspin_->value(),
                formcombo_->currentText(), alphaspin_->value(), timespin_->value());
        currentitem_->setToIterator();
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotRemoveHops()
// ----------------
// Received when user removes a hop from recipe

void HopsPage::slotRemoveHops()
{
    if (currentitem_ != 0) {
        ((QBrewDoc *)document_)->removeHop(currentitem_->hopIterator());
        delete currentitem_; currentitem_ = 0;
    }
}

//////////////////////////////////////////////////////////////////////////////
// slotItemSelected()
// -----------------
// Received when user selects an item in listview

void HopsPage::slotItemSelected(QListViewItem *item)
{
    currentitem_ = (HopsListItem *)item;
    if (item != 0) {
        itemcombo_->setEditText(item->text(0));
        quantityspin_->setValue(item->text(1).toDouble());
        formcombo_->setEditText(item->text(2));
        alphaspin_->setValue(item->text(3).toDouble());
        timespin_->setValue(item->text(4).toUInt());
    }
}

//////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                            //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// updateListView()
// ----------------
// Update the listview

void HopsPage::updateListView()
{
    HopsList *list = ((QBrewDoc *)document_)->hopsList();
    HopIterator it;
    HopsListItem *item;
    itemslistview_->clear();
    for (it=list->begin(); it != list->end(); ++it) {
        item = new HopsListItem( itemslistview_, (*it).name(),
            (*it).quantityString(),
            (*it).form(),
            (*it).alphaString(),
            (*it).timeString() );
        item->setHopIterator(it);
    }
}
